/* variant that works with ZLIB instead of PACKFILEs
 *
 * This gets much better compression but doesn't allow you do use packfiles
 * or datafiles with your pictures.
 *
 * you must link against 'z' (i.e use '-lz' in your compile) to get ZLIB.
 *
 * Tom St Denis
*/
#include "pic.h"
#include <zlib.h>

/* we scan the elements in this order */
static int ZigZag[] = {
0, 0, 
0, 1, 1, 0, 
0, 2, 1, 1, 2, 0, 
0, 3, 1, 2, 2, 1, 3, 0, 
0, 4, 1, 3, 2, 2, 3, 1, 4, 0, 
0, 5, 1, 4, 2, 3, 3, 2, 4, 1, 5, 0, 
0, 6, 1, 5, 2, 4, 3, 3, 4, 2, 5, 1, 6, 0, 
0, 7, 1, 6, 2, 5, 3, 4, 4, 3, 5, 2, 6, 1, 7, 0, 
1, 7, 2, 6, 3, 5, 4, 4, 5, 3, 6, 2, 7, 1, 
2, 7, 3, 6, 4, 5, 5, 4, 6, 3, 7, 2, 
3, 7, 4, 6, 5, 5, 6, 4, 7, 3, 
4, 7, 5, 6, 6, 5, 7, 4, 
5, 7, 6, 6, 7, 5, 
6, 7, 7, 6, 
7, 7 };

long jpg_quality=3;

static void put_pixel_data(BITMAP *bmp, int x, int y, int pass, float color)
{
    int r, g, b, c, d;

    c = getpixel(bmp, x, y);
    d = bitmap_color_depth(bmp);
    r = getr_depth(d, c);
    g = getg_depth(d, c);
    b = getb_depth(d, c);

    switch (pass) {
        case 0: r = (int)color; break;
        case 1: g = (int)color; break;
        case 2: b = (int)color; break; }

    putpixel(bmp, x, y, makecol_depth(d, r, g, b));
}

static float get_pixel_data(BITMAP *b, int x, int y, int pass)
{
    switch (pass) {
        case 0: return (float)getr_depth(bitmap_color_depth(b), getpixel(b, x, y));
        case 1: return (float)getg_depth(bitmap_color_depth(b), getpixel(b, x, y));
        case 2: return (float)getb_depth(bitmap_color_depth(b), getpixel(b, x, y));
        default: return 0;
    }
}

static void output_vint(short code, gzFile *output, int *bitbuf, int *bitcnt)
{
    int out, x, bits, xc;
    static int codes[] = {
        (2<<16)|0,(3<<16)|2,(3<<16)|3,(4<<16)|8,(4<<16)|9,
        (4<<16)|10,(4<<16)|11,(4<<16)|12,(4<<16)|13,(4<<16)|14,(4<<16)|15 };

    /* find range of code */
    x = code<0 ? -code : code;
    if (x >= 512)      { bits = 10; xc = x - 512; }
    else if (x >= 256) { bits = 9; xc = x - 256; }
    else if (x >= 128) { bits = 8; xc = x - 128; }
    else if (x >= 64)  { bits = 7; xc = x - 64; }
    else if (x >= 32)  { bits = 6; xc = x - 32; }
    else if (x >= 16)  { bits = 5; xc = x - 16; }
    else if (x >= 8)   { bits = 4; xc = x - 8;  }
    else if (x >= 4)   { bits = 3; xc = x - 4;  }
    else if (x >= 2)   { bits = 2; xc = x - 2;  }
    else if (x == 1)   { bits = 1; xc = 0;      }
    else               { bits = 0; xc = 0;      }
    if (code < 0) xc |= (1<<(bits-1));

    /* append xc + codes[bits] to form a single word */
    out = ((codes[bits]&65535)<<bits) | xc;
    x = (bits += codes[bits]>>16);

    while (bits--) {
        if (!*bitcnt) {
            gzputc(output, *bitbuf);
            *bitbuf = 0; *bitcnt = 8; }
        *bitbuf = (*bitbuf << 1) | ((out>>(x-1))&1);
        --(*bitcnt);
        out <<= 1;
    }
}

static int input_vint(gzFile *input, int *bitbuf, int *bitcnt)
{
    static int codes[] = {
        (0<<16)|(2<<8)|1, 0, (1<<16)|(3<<8)|1, (2<<16)|(3<<8)|1, 0, 0, 0, 0,
        (3<<16)|(4<<8)|1, (4<<16)|(4<<8)|1, (5<<16)|(4<<8)|1,
        (6<<16)|(4<<8)|1, (7<<16)|(4<<8)|1, (8<<16)|(4<<8)|1,
        (9<<16)|(4<<8)|1,
        (10<<16)|(4<<8)|1 };
    int sofar, bits, xc, c;

    /* try to find how many bits are in this code */
    sofar = c = 0;
    for (;;) {
        if (!*bitcnt) {
            *bitbuf = gzgetc(input);
            *bitcnt = 8; }
        sofar = (sofar << 1) | (*bitbuf >> 7);
        *bitbuf = (*bitbuf << 1) & 255;
        (*bitcnt)--;
        ++c;
        if ((((codes[sofar]>>8)&255)==c) && (codes[sofar] & 1)) break;
    }

    /* get the value */
    xc = bits = codes[sofar] >> 16;
    if (bits == 0) return 0;
    c = 0;
    while (xc--) { 
        if (!*bitcnt) {
            *bitbuf = gzgetc(input);
            *bitcnt = 8; }
        c = (c << 1) | (*bitbuf >> 7);
        *bitbuf = (*bitbuf << 1) & 255;
        (*bitcnt)--;
    }

    /* package it */
    if (bits == 1)
        xc = 0;
    else
        xc = c & ((1 << (bits - 1)) - 1);
    xc += (1 << (bits - 1));
    if (c & (1 << (bits - 1))) xc = -xc;
    return xc;
}


void ddct8x8s(int isgn, float **a)
{
    int j;
    float x0r, x0i, x1r, x1i, x2r, x2i, x3r, x3i;
    float xr, xi;
    
    if (isgn < 0) {
        for (j = 0; j <= 7; j++) {
            x0r = a[0][j] + a[7][j];
            x1r = a[0][j] - a[7][j];
            x0i = a[2][j] + a[5][j];
            x1i = a[2][j] - a[5][j];
            x2r = a[4][j] + a[3][j];
            x3r = a[4][j] - a[3][j];
            x2i = a[6][j] + a[1][j];
            x3i = a[6][j] - a[1][j];
            xr = x0r + x2r;
            xi = x0i + x2i;
            a[0][j] = C8_4R * (xr + xi);
            a[4][j] = C8_4R * (xr - xi);
            xr = x0r - x2r;
            xi = x0i - x2i;
            a[2][j] = C8_2R * xr - C8_2I * xi;
            a[6][j] = C8_2R * xi + C8_2I * xr;
            xr = W8_4R * (x1i - x3i);
            x1i = W8_4R * (x1i + x3i);
            x3i = x1i - x3r;
            x1i += x3r;
            x3r = x1r - xr;
            x1r += xr;
            a[1][j] = C8_1R * x1r - C8_1I * x1i;
            a[7][j] = C8_1R * x1i + C8_1I * x1r;
            a[3][j] = C8_3R * x3r - C8_3I * x3i;
            a[5][j] = C8_3R * x3i + C8_3I * x3r;
        }
        for (j = 0; j <= 7; j++) {
            x0r = a[j][0] + a[j][7];
            x1r = a[j][0] - a[j][7];
            x0i = a[j][2] + a[j][5];
            x1i = a[j][2] - a[j][5];
            x2r = a[j][4] + a[j][3];
            x3r = a[j][4] - a[j][3];
            x2i = a[j][6] + a[j][1];
            x3i = a[j][6] - a[j][1];
            xr = x0r + x2r;
            xi = x0i + x2i;
            a[j][0] = C8_4R * (xr + xi);
            a[j][4] = C8_4R * (xr - xi);
            xr = x0r - x2r;
            xi = x0i - x2i;
            a[j][2] = C8_2R * xr - C8_2I * xi;
            a[j][6] = C8_2R * xi + C8_2I * xr;
            xr = W8_4R * (x1i - x3i);
            x1i = W8_4R * (x1i + x3i);
            x3i = x1i - x3r;
            x1i += x3r;
            x3r = x1r - xr;
            x1r += xr;
            a[j][1] = C8_1R * x1r - C8_1I * x1i;
            a[j][7] = C8_1R * x1i + C8_1I * x1r;
            a[j][3] = C8_3R * x3r - C8_3I * x3i;
            a[j][5] = C8_3R * x3i + C8_3I * x3r;
        }
    } else {
        for (j = 0; j <= 7; j++) {
            x1r = C8_1R * a[1][j] + C8_1I * a[7][j];
            x1i = C8_1R * a[7][j] - C8_1I * a[1][j];
            x3r = C8_3R * a[3][j] + C8_3I * a[5][j];
            x3i = C8_3R * a[5][j] - C8_3I * a[3][j];
            xr = x1r - x3r;
            xi = x1i + x3i;
            x1r += x3r;
            x3i -= x1i;
            x1i = W8_4R * (xr + xi);
            x3r = W8_4R * (xr - xi);
            xr = C8_2R * a[2][j] + C8_2I * a[6][j];
            xi = C8_2R * a[6][j] - C8_2I * a[2][j];
            x0r = C8_4R * (a[0][j] + a[4][j]);
            x0i = C8_4R * (a[0][j] - a[4][j]);
            x2r = x0r - xr;
            x2i = x0i - xi;
            x0r += xr;
            x0i += xi;
            a[0][j] = x0r + x1r;
            a[7][j] = x0r - x1r;
            a[2][j] = x0i + x1i;
            a[5][j] = x0i - x1i;
            a[4][j] = x2r - x3i;
            a[3][j] = x2r + x3i;
            a[6][j] = x2i - x3r;
            a[1][j] = x2i + x3r;
        }
        for (j = 0; j <= 7; j++) {
            x1r = C8_1R * a[j][1] + C8_1I * a[j][7];
            x1i = C8_1R * a[j][7] - C8_1I * a[j][1];
            x3r = C8_3R * a[j][3] + C8_3I * a[j][5];
            x3i = C8_3R * a[j][5] - C8_3I * a[j][3];
            xr = x1r - x3r;
            xi = x1i + x3i;
            x1r += x3r;
            x3i -= x1i;
            x1i = W8_4R * (xr + xi);
            x3r = W8_4R * (xr - xi);
            xr = C8_2R * a[j][2] + C8_2I * a[j][6];
            xi = C8_2R * a[j][6] - C8_2I * a[j][2];
            x0r = C8_4R * (a[j][0] + a[j][4]);
            x0i = C8_4R * (a[j][0] - a[j][4]);
            x2r = x0r - xr;
            x2i = x0i - xi;
            x0r += xr;
            x0i += xi;
            a[j][0] = x0r + x1r;
            a[j][7] = x0r - x1r;
            a[j][2] = x0i + x1i;
            a[j][5] = x0i - x1i;
            a[j][4] = x2r - x3i;
            a[j][3] = x2r + x3i;
            a[j][6] = x2i - x3r;
            a[j][1] = x2i + x3r;
        }
    }
}

static void gzputl(long x, gzFile *out)
{
    gzputc(out, x&255);
    gzputc(out, (x>>8)&255);
    gzputc(out, (x>>16)&255);
    gzputc(out, (x>>24)&255);
}

static long gzgetl(gzFile *in)
{
    long a, b, c, d;
    a = gzgetc(in);
    b = gzgetc(in);
    c = gzgetc(in);
    d = gzgetc(in);
    return (a) | (b << 8) | (c << 16) | (d << 24);
}

int save_bitmap_pic(char *name, BITMAP *bmp, RGB *pal)
{
    float dblk[8][8], *_dblk[8];
    int x, y, xx, yy, zz, pass, bitlen=8, bitbuf=0;
    long td;
    gzFile *output;

    /* setup file */
    for (x = 0; x < 8; x++) _dblk[x] = dblk[x];
    output = gzopen(name, "wb9");
    if (!output || (bmp->w & 7) || (bmp->h & 7))
        return -1;

    /* output a header */
    gzputl(0x12874532, output);
    gzputl(bmp->w, output);
    gzputl(bmp->h, output);
    gzputl(bitmap_color_depth(bmp), output);
    gzputl(jpg_quality, output);

    for (pass = 0; pass < 3; pass++) {
        for (y = 0; y < bmp->h; y += 8)
        for (x = 0; x < bmp->w; x += 8) {
            /* copy pixels into dblk */
            for (xx = 0; xx < 8; xx++)
            for (yy = 0; yy < 8; yy++)
                dblk[xx][yy] = get_pixel_data(bmp, x+xx,y+yy, pass);

            /* do a DCT on it */
            ddct8x8s(-1, _dblk);

            /* scan along zig zag */
            for (xx = 0; xx < 64; xx++) {
                /* get position */
                zz = ZigZag[(xx*2) + 0];
                yy = ZigZag[(xx*2) + 1];

                /* get data and quantise */
                td = (long)(floor(dblk[zz][yy]+0.5) / (float)(1 + (1 + zz + yy) * jpg_quality));
                output_vint(td, output, &bitbuf, &bitlen);
            }
        }
    }
    while (bitlen--) bitbuf <<= 1;
    gzputc(output, bitbuf);
    gzclose(output);
    return 0;
}

BITMAP *load_bitmap_pic(char *name, RGB *pal)
{
    float dblk[8][8], *_dblk[8];
    long td, q;
    int x, y, xx, yy, zz, pass, bitbuf=0, bitcnt=0;
    gzFile *input;
    BITMAP *b;

    /* setup file */
    for (x = 0; x < 8; x++) _dblk[x] = dblk[x];
    input  = gzopen(name, "rb");
    if (gzgetl(input) != 0x12874532) { gzclose(input); return NULL; }

    /* setup bmp */
    x = gzgetl(input);
    y = gzgetl(input);
    xx = gzgetl(input);
    q = gzgetl(input);
    b = create_bitmap_ex(xx, x, y);
    clear(b);

    /* decode data */
    for (pass = 0; pass < 3; pass++) {
        for (y = 0; y < b->h; y += 8)
        for (x = 0; x < b->w; x += 8) {
            /* de quantise it and stuff in the DCT block */
            for (xx = 0; xx < 64; xx++) {
                /* get position */
                zz = ZigZag[(xx*2) + 0];
                yy = ZigZag[(xx*2) + 1];

                /* de quantise */
                td = input_vint(input, &bitbuf, &bitcnt);
                dblk[zz][yy] = (float)(td * (1 + ((1 + zz + yy) * q)));
            }

            /* do an IDCT on it */
            ddct8x8s(1, _dblk);

            /* store the pixels */
            for (xx = 0; xx < 8; xx++)
            for (yy = 0; yy < 8; yy++) {
                if (dblk[xx][yy] < 0.0)
                    dblk[xx][yy] = 0;
                else if (dblk[xx][yy] > 255.0)
                    dblk[xx][yy] = 255.0;
                put_pixel_data(b, x+xx, y+yy, pass, dblk[xx][yy]);
            }
        }
    }
    gzclose(input);
    return b;
}

