/*

   Example 'hello world' program for the MX-Windows Library

   (C) 1999 Achillefs Margaritis

*/


#include "mx.h"                        /* using MX Windows */


typedef struct Root {                  /* a root window object */
   Window window;
   Dialog dialog;
   int background_color;
} Root;


typedef struct PushButton {            /* a push button object */
   Window window;
   char *caption;
   Bool pressed;
   void (*action)();
} PushButton;


Bool application_runs = True;          /* when this variable is reset to */
                                       /* false root dialog stops and */
                                       /* program exits */

Root *root;                            /* the root object */
PushButton *hello_button;              /* the hello button object */
PushButton *exit_button;               /* the button that exits application */
PushButton *random_background_button;  /* changes background color */
PushButton *moveme_button;             /* moves the window at the mouse position */
PushButton *restack_button;            /* brings button on top or at bottom */
PushButton *random_draw_button;        /* draws something temporary into a button */


/* draws a 3d rectangle */
void rect3d(
   BITMAP *bmp,
   int x1,
   int y1,
   int x2,
   int y2,
   int width,
   int back_color,
   int light_color,
   int dark_color)
{
   int i;

   for(i = 0; i < width; i++) {
      vline(bmp, x1, y1, y2, light_color);
      hline(bmp, x1, y1, x2, light_color);
      vline(bmp, x2, y1, y2, dark_color);
      hline(bmp, x1, y2, x2, dark_color);
      x1++;
      y1++;
      x2--;
      y2--;
   }
   rectfill(bmp, x1, y1, x2, y2, back_color);
}


/* draws text centered into the given rectangle */
void textcenter(
   BITMAP *bmp,
   FONT *f,
   char *text,
   int x1,
   int y1,
   int x2,
   int y2,
   int color)
{
   int text_x = x1 + (((x2 - x1 + 1) - text_length(f, text)) >> 1);
   int text_y = y1 + (((y2 - y1 + 1) - text_height(f)) >> 1);

   textout(bmp, f, text, text_x, text_y, color);
}


/* event handler for root window */
void root_event_handler(
   WindowInfo *winfo,
   Event *event)
{
   ExposeEvent *exp;                   /* used to translate given event */
                                       /* to exposure event */

   Root *root = winfo->extension;      /* root window returned here */

   switch (event->type) {

      case WindowExpose:               /* when some parts of it are exposed */
         exp = (ExposeEvent *)event;   /* get expose event */
         solid_mode();                 /* draw background */
         rectfill(                     /* fill exposed area - no need to */
            screen,                    /* draw on the whole screen */
            exp->x1,
            exp->y1,
            exp->x2,
            exp->y2,
            root->background_color);
         break;

      case KeyPress:                   /* if a key is pressed exit */
         application_runs = False;
         break;
   }
}


/* creates a root window */
Root *create_root(int color)
{
   Root *r = calloc(1, sizeof(Root));  /* allocate root object */

   if (r) {
      r->window = CreateRootWindow(r,  /* create root window */
         root_event_handler);
      r->dialog = CreateDialog(        /* create basic dialog */
         Null, Null, r->window, Null); /* this dialog is run */
      r->background_color = color;     /* setup roots background color */
      RedirectKeyboard(r->window);     /* root window accepts key presses */
   }
   return r;
}


/* button event handler */
void button_event_handler(
   WindowInfo *winfo,
   Event *event)
{
   ExposeEvent *exp;                   /* used to translate given event */
                                       /* to exposure event */

   ButtonEvent *btn;                   /* used to translate given event
                                          to button event */

   int light_shadow_color = 15;
   int dark_shadow_color = 8;
   int pushed_fx = 0;

   PushButton *b = winfo->extension;   /* push button returned here */

   switch (event->type) {

      case WindowExpose:               /* when some parts of it are exposed */
         exp = (ExposeEvent *)event;   /* get expose event */

         if (b->pressed) {             /* if button is pressed swap */
            light_shadow_color = 8;    /* foreground and background colors */
            dark_shadow_color = 15;
            pushed_fx = 2;
         }

         solid_mode();

         rect3d(                       /* draw 3d rectangle */
            screen,
            winfo->x1,
            winfo->y1,
            winfo->x2,
            winfo->y2,
            winfo->border_width,
            7,                         /* background color */
            light_shadow_color,
            dark_shadow_color);

         text_mode(-1);
         textcenter(                   /* draw text centered into the */
            screen,                    /* window rectangle */
            font,
            b->caption,
            winfo->x1 + pushed_fx,
            winfo->y1 + pushed_fx,
            winfo->x2 + pushed_fx,
            winfo->y2 + pushed_fx,
            0);
         break;

      case ButtonPress:                /* when mouse button is pressed */
         GrabEvents(b->window);        /* grab events */
         b->pressed = True;            /* show button pressed */
         RefreshWindow(b->window);
         break;

      case ButtonRelease:              /* on mouse button release */
         ReleaseEvents(b->window);     /* release events */
         b->pressed = False;           /* show button de-pressed */
         RefreshWindow(b->window);
         b->action();                  /* and execute its action */
         break;

   }
}


/* creates a push button */
PushButton *create_push_button(
   Window parent,
   int x,
   int y,
   char *text,
   void (*action)())
{
   PushButton *b = calloc(1,           /* allocate push button */
      sizeof(PushButton));

   if (b) {
      b->window = CreateWindow(        /* create the push button window */
         parent,                       /* insert in given parent */
         b,                            /* pass button structure */
         button_event_handler,         /* event handler for buttons */
         x,                            /* given position */
         y,
         text_length(font, text) + 16, /* the size of given text */
         text_height(font) + 16,       /* the height of the default font */
         2,                            /* border width of 2 pixels size */
         0xFFFFFFFF,                   /* events accepted */
         0);                           /* it does not propagate any events */
      b->caption = text;               /* set up rest of fields */
      b->pressed = False;
      b->action = action;
   }
   return b;
}


/* returns a random number from 0 to i */
int rnd(int i)
{
   return random() % i;
}


/* exits application */
void exit_application()
{
   application_runs = False;
}


/* prints the hello world message */
void hello_world()
{
   textout(screen, font, "Hello world!", rnd(600), rnd(480), rnd(256));
}


/* changes the background color */
void change_background_color()
{
   root->background_color = rnd(256);
   PaintWindow(root->window);
}


/* moves the button at the mouse position */
void move_button()
{
   SetWindowX(moveme_button->window, mouse_x);
   SetWindowY(moveme_button->window, mouse_y);
   UpdateWindowGeometry(moveme_button->window);
}


/* changes the depth of the restack_button */
void change_button_depth()
{
   Window higher = GetWindowHigher(restack_button->window);
   int depth;

   if (!higher) {
      SetWindowZOrder(restack_button->window, -1);
   }
   else {
      depth = GetWindowZOrder(restack_button->window, 1);
      SetWindowZOrder(restack_button->window, depth - 1);
   }
}


/* random drawing variables */
int random_x;
int random_y;
int random_width;
int random_height;
int random_background_color;
int random_light_color;
int random_dark_color;


/* random drawing function */
void random_drawing(WindowInfo *winfo, ExposeEvent *expose)
{
   rect3d(
      screen,
      random_x,
      random_y,
      random_x + random_width - 1,
      random_y + random_height - 1,
      2,
      random_background_color,
      random_light_color,
      random_dark_color);
}


/* draws a temporary 3d rectangle into the restack button */
void draw_random()
{
   random_x = rnd(GetWindowWidth(restack_button->window));
   random_y = rnd(GetWindowHeight(restack_button->window));
   random_width = rnd(GetWindowWidth(restack_button->window));
   random_height = rnd(GetWindowHeight(restack_button->window));
   TranslateCoords(                    /* translate random coords to */
      restack_button->window,           /* global coords */
      Null,
      random_x,
      random_y,
      &random_x,
      &random_y);
   random_background_color = rnd(256);
   random_light_color = rnd(256);
   random_dark_color = rnd(256);
   DoWindowDrawing(restack_button->window, False, random_drawing);
}


/* shows instructions */
void show_instructions()
{
   clrscr();
   printf("Welcome to MX-Windows...\n\n");
   printf("This program shows the following objects :\n\n");
   printf("- a root window and use of dialog.\n");
   printf("- a button that prints at random positions 'Hello world'.\n");
   printf("- a button that changes the background color.\n");
   printf("- a button that can be moved around.\n");
   printf("- a button that draws something temporary into another button.\n");
   printf("- a button that is raised and then send to back.\n");
   printf("- finally an exit button.\n\n");
   printf("Objects appear at random positions so if any of them is invisible\n");
   printf("quit the program and re-run it.\n");
   printf("In any given time press any key to exit.\n");
   readkey();
}


/* main function */
main()
{
   InitMX();                           /* initialize library */

   show_instructions();

   set_gfx_mode(GFX_AUTODETECT,        /* open 640 x 480 x 256 colors */
      640, 480, 0, 0);

   root = create_root(rnd(16));        /* create root with random color */

   restack_button =                    /* changes window depth */
      create_push_button(
      root->window,
      40,
      30,
      "Raise me!",
      change_button_depth);
   SetWindowWidth(restack_button->window,
      500);
   SetWindowHeight(restack_button->window,
      400);

   exit_button = create_push_button(   /* create exit button */
      root->window,
      2,
      2,
      "Exit",
      exit_application);

   hello_button = create_push_button(  /* hello world button */
      root->window,
      rnd(600),
      rnd(400),
      "Hello World",
      hello_world);

   random_background_button =          /* changes background color */
      create_push_button(
      root->window,
      rnd(600),
      rnd(400),
      "Change background color",
      change_background_color);

   moveme_button =                     /* moves a button at the mouse position */
      create_push_button(
      root->window,
      rnd(600),
      rnd(400),
      "Drag me!",
      move_button);

   random_draw_button =                /* draws something temporary */
      create_push_button(              /* into the restack button */
      root->window,
      rnd(600),
      rnd(400),
      "Random Drawing",
      draw_random);

   while (application_runs) {
      RunDialog(&root->dialog);
   }

   allegro_exit();                     /* exit allegro */
}

