//----------------------------------------------------------------------------
//
// C++ Objects for Allegro's gui - C only routines
//
// Douglas Eleveld (D.J.Eleveld@anest.azg.nl)
//
//----------------------------------------------------------------------------
#include <stdio.h>
#include "allegro.h"
#include "internal.h"

//----------------------------------------------------------------------------
// Some global graphics functions - Some were taken directly from Allegro
//----------------------------------------------------------------------------

/*--------------------------draw_3d_frame-------------------------
  Draws a filled frame in a 3D look.

  Input: bmp   - Allegro Bitmap where to draw the frame
         x1/y1 - Left top position
         x2/y2 - Right bottom position
         c1    - Inside color (<0 = transparent)
         c2    - Left top frame color
         c3    - Right bottom frame color
----------------------------------------------------------------*/
void draw_3d_frame(BITMAP *bmp,int x1,int y1,int x2,int y2,int c1,int c2,int c3)
   {
   if (c1>=0) rectfill(bmp,x1,y1,x2,y2,c1);
   hline(bmp,x1,y2,x2,c3);
   vline(bmp,x2,y1,y2,c3);
   vline(bmp,x1,y1,y2,c2);
   hline(bmp,x1,y1,x2,c2);
   }
/*--------------------------draw_3d_circle-------------------------
  Draws a filled circle in a 3D look.

  Input: bmp   - Allegro Bitmap where to draw the circle
         x1/y1 - Left top position
         x2/y2 - Right bottom position
         c1    - Inside color (<0 = transparent)
         c2    - Left top frame color
         c3    - Right bottom frame color
----------------------------------------------------------------*/
static int _cx, _cy;
static int _c2, _c3;

static void _3d_circle_helper (BITMAP* bmp, int x, int y, int c)
   {
   c = c;
   if((x-_cx)>(_cy-y)) putpixel(bmp,x,y,_c3);
   if((x-_cx)<(_cy-y)) putpixel(bmp,x,y,_c2);
   }

void draw_3d_circle (BITMAP *bmp,int x1,int y1,int x2,int c1,int c2,int c3)
   {
   int r = (x2-x1)/2;
   _cx = x1 + r;
   _cy = y1 + r;
   _c2 = c2;
   _c3 = c3;

   if (c1>=0) circlefill(bmp,_cx,_cy,r,c1);

   do_circle(bmp,_cx,_cy,r,0,_3d_circle_helper);
   }
//----------------------------------------------------------------------------
/* gui_textout:
 *  Wrapper function for drawing text to the screen, which interprets the
 *  & character as an underbar for displaying keyboard shortcuts. Returns
 *  the width of the output string in pixels.
 */
int degui_textout(BITMAP *bmp, unsigned char *s, int x, int y, int color, int centre)
{
   char tmp[160];
   int c;
   int len = 0;
   int pix_len;
   int hline_pos = -1;

   for (c=0; (s[c]) && (len<159); c++) {
      if (s[c] == '&') {
	 if (s[c+1] != '&')
	    hline_pos = len;
	 else {
	    tmp[len++] = '&';
	    c++;
	 }
      }
      else
	 tmp[len++] = s[c];
   }

   tmp[len] = 0;
   pix_len = text_length(font, tmp);

   if (centre)
      x -= pix_len / 2;

   if (bmp) {
      textout(bmp, font, tmp, x, y, color);

      if (hline_pos >= 0) {
	 c = tmp[hline_pos];
	 tmp[hline_pos] = 0;
	 hline_pos = text_length(font, tmp);
	 tmp[0] = c;
	 tmp[1] = 0;
	 c = text_length(font, tmp);
	 hline(bmp, x+hline_pos, y+text_height(font)-gui_font_baseline, x+hline_pos+c-1, color);
      }
   }

   return pix_len;
}
//----------------------------------------------------------------------------
/* gui_strlen:
 *  Returns the length of a string in pixels, ignoring '&' characters.
 */
int degui_strlen(unsigned char *s)
{
   return degui_textout(NULL, s, 0, 0, 0, 0);
}

/*-------------------------draw_dotted_rect-----------------------
  Draws a dotted rectangle.

  Input: same as Allegro's rect()
----------------------------------------------------------------*/
void draw_dotted_rect(BITMAP *bmp,int x1,int y1,int x2,int y2,int c)
{
  int x, y;
  for(x=x1; x<=x2; x+=2)
  {
    putpixel(bmp,x,y1,c);
    putpixel(bmp,x,y2,c);
  };
  for(y=y1; y<=y2; y+=2)
  {
    putpixel(bmp,x1,y,c);
    putpixel(bmp,x2,y,c);
  };
};

//----------------------------------------------------------------------------
// Some helper functions
//----------------------------------------------------------------------------
int inside_rect(int x, int y,int x1,int y1,int x2,int y2)
   {
   if (x>=x1 && x<=x2 && y>=y1 && y<=y2) return 1; else return 0;
   }

//----------------------------------------------------------------------------
// Some global error handling procedures
//----------------------------------------------------------------------------
// General error handling procedure
void degui_error (const char *message)
   {
   allegro_exit();
   fprintf(stderr,"%s\n",message);
   exit(1);
   }
//----------------------------------------------------------------------------
// Error from a failed new
void degui_no_memory (void)
   {
   degui_error("DEGUI: No memory, new has failed");
   }
//----------------------------------------------------------------------------



