indexing
    description    : "Allegro con Eiffel: bitmap objects"
    status         : "Initial development"
    author         : "Peter Monks (pmonks@iname.com)"
    allegro_author : "Shawn Hargreaves (shawn@talula.demon.co.uk)"
    names          : bitmap
    date_started   : "1st December, 1996"
    version        : "0.1 beta"
    platforms      : "MS-DOS"
    dependencies   : "Allegro v2.2, DJGPP v2.01"


class BITMAP


inherit
    ACE_INFORMATION_SINGLETON  -- implementation inheritance
    end                        -- inherit ACE_INFORMATION_SINGLETON


creation { ANY }
    make,
    make_as_sub_bitmap,
    load

creation { DATAFILE }
    make_from_external


------------------------------------------------------ Creation features
feature { ANY }

    make(bitmap_width, bitmap_height : INTEGER) is
    -- Create a new memory bitmap
    require
        ace_initialised        : info.ace_initialised
        graphics_initialised   : info.graphics_initialised
        bitmap_width_is_valid  : bitmap_width >= 0
        bitmap_height_is_valid : bitmap_height >= 0
    do
        if is_valid then
            discard
        end  -- if

        c_inline_c("C->_data=create_bitmap(a1,a2);")
    ensure
        is_valid : is_valid
    end  -- feature make


    make_as_sub_bitmap(parent : BITMAP;
                       x, y, desired_width, desired_height : INTEGER) is
    -- Create a new sub bitmap of the specified bitmap
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        parent_is_valid      : parent /= Void and then
                               parent /= Current and then
                               parent.is_valid
        x_is_valid           : x >= 0 and x <= parent.width and
                               (info.graphics_driver /= info.graphics_modex or
                               (x \\ 4) = 0)
        y_is_valid           : y >= 0 and y <= parent.height
        desired_width_is_valid : desired_width >= 0 and desired_width <= (parent.width-x) and
                               (info.graphics_driver /= info.graphics_modex or
                               (desired_width \\ 4) = 0)
        desired_height_is_valid : desired_height >= 0 and desired_height <= (parent.height-y)
    local
        p : POINTER
    do
        if is_valid then
            discard
        end  -- if

        p := parent.data
        c_inline_c("C->_data=create_sub_bitmap(_p,a2,a3,a4,a5);")
    ensure
        is_valid : is_valid
    end  -- feature make_as_sub_bitmap


    load(filename : STRING; palette : PALETTE) is
    -- Create a bitmap from the specified file
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        filename_is_valid    : filename /= Void and then
                               ace_file_tools.exists(filename) and then
                               (ace_file_tools.extension(filename).same_as("BMP") or
                                ace_file_tools.extension(filename).same_as("LBM") or
                                ace_file_tools.extension(filename).same_as("PCX") or
                                ace_file_tools.extension(filename).same_as("TGA"))
        palette_is_valid     : palette /= Void and then
                               palette.is_valid
    local
        p1 : POINTER
        p2 : POINTER
    do
        if is_valid then
            discard
        end  -- if

        p1 := filename.to_external
        p2 := palette.data

        c_inline_c("C->_data=load_bitmap(_p1,_p2);")
    ensure
        is_valid      : is_valid
        memory_bitmap : is_memory
    end  -- feature load


------------------------------------------------------ Creation features (DATAFILE)
feature { DATAFILE }

    make_from_external(p : POINTER) is
    -- Create a bitmap from a C pointer
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        p_is_valid           : p.is_not_void
    do
        if is_valid then
            discard
        end  -- if

        data := p
    ensure
        is_valid : is_valid
    end  -- feature make_from_external


------------------------------------------------------ Limited exported features
feature { BITMAP, FLIC, RLE_SPRITE, COMPILED_SPRITE }

    -- A pointer to an Allegro bitmap structure
    data : POINTER


------------------------------------------------------ Internal features
feature { NONE }

    ace_file_tools : ACE_FILE_TOOLS


------------------------------------------------------ Bitmap features
feature { ANY }

    is_valid : BOOLEAN is
    -- Indicates whether this bitmap is valid or not
    do
        Result := data.is_not_void
    end  -- feature is_valid


    discard is
    -- Discard this bitmap
    require
        ace_initialised : info.ace_initialised
        is_valid        : is_valid
    do
        if is_memory or is_sub_bitmap then
            c_inline_c("destroy_bitmap(C->_data);C->_data=NULL;")
        end  -- if
    ensure
        not_valid : not is_valid
    end  -- feature discard


    width : INTEGER is
    -- Return the width of this bitmap
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=((BITMAP *)(C->_data))->w;")
    end  -- feature width


    height : INTEGER is
    -- Return the height of the bitmap
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=((BITMAP *)(C->_data))->h;")
    end  -- feature height


    clip_left : INTEGER is
    -- Return the left of the clipping region
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=((BITMAP *)(C->_data))->cl;")
    end  -- feature clip_left


    clip_top : INTEGER is
    -- Return the top of the clipping region
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=((BITMAP *)(C->_data))->ct;")
    end  -- feature clip_top


    clip_right : INTEGER is
    -- Return the right of the clipping region
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=(((BITMAP *)(C->_data))->cr)-1;")
    end  -- feature clip_right


    clip_bottom : INTEGER is
    -- Return the bottom of the clipping region
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=(((BITMAP *)(C->_data))->cb)-1;")
    end  -- feature clip_bottom


    clipping : BOOLEAN is
    -- Is this bitmap being clipped?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=((BITMAP *)(C->_data))->clip;")
    end  -- feature clipping


    is_linear : BOOLEAN is
    -- Is this a linear bitmap?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=is_linear_bitmap(C->_data);")
    end  -- feature is_linear


    is_planar : BOOLEAN is
    -- Is this a planar bitmap?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=is_planar_bitmap(C->_data);")
    end  -- feature is_planar


    is_memory : BOOLEAN is
    -- Is this a memory bitmap?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=is_memory_bitmap(C->_data);")
    end  -- feature is_memory


    is_screen : BOOLEAN is
    -- Is this a screen bitmap?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=is_screen_bitmap(C->_data);")
    end  -- feature is_screen


    is_sub_bitmap : BOOLEAN is
    -- Is this a sub bitmap?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
    do
        c_inline_c("R=is_sub_bitmap(C->_data);")
    end  -- feature is_sub_bitmap


    is_same_bitmap(other : BITMAP) : BOOLEAN is
    -- Is this bitmap the same as the 'other' bitmap?
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        other_is_valid       : other /= Void and then
                               other.is_valid
    local
        p : POINTER
    do
        p := other.data
        c_inline_c("R=is_same_bitmap(C->_data,_p);")
    end  -- feature is_same_bitmap


    set_clipping_region(left, top, right, bottom : INTEGER) is
    -- Set the clipping region for this bitmap.
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        left_is_valid        : left   >= 0    and left   <= width
        top_is_valid         : top    >= 0    and top    <= height
        right_is_valid       : right  >= left and right  <= width
        bottom_is_valid      : bottom >= top  and bottom <= height
    do
        c_inline_c("set_clip(C->_data,a1,a2,a3,a4);")
    ensure
        clip_left_correct   : clip_left   = left
        clip_top_correct    : clip_top    = top
        clip_right_correct  : clip_right  = right
        clip_bottom_correct : clip_bottom = bottom
    end  -- feature set_clipping_region


    clipping_on is
    -- Set clipping on for this bitmap
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        not_clipping         : not clipping
    do
        c_inline_c("((BITMAP *)(C->_data))->clip=1;")
    ensure
        clipping : clipping
    end  -- feature clipping_on


    clipping_off is
    -- Set clipping off for this bitmap
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        clipping             : clipping
    do
        c_inline_c("((BITMAP *)(C->_data))->clip=0;")
    ensure
        not_clipping : not clipping
    end  -- feature clipping_off


    put_pixel(x, y, colour_index : INTEGER) is
    -- Display a pixel at the specified location of this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y_is_valid            : y >= 0 and y <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("putpixel(C->_data,a1,a2,a3);")
    end  -- feature put_pixel


    get_pixel(x, y : INTEGER) : INTEGER is
    -- Read the pixel at the specified location of this bitmap
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        x_is_valid           : x >= 0 and x <= width
        y_is_valid           : y >= 0 and y <= height
    do
        c_inline_c("R=getpixel(C->_data,a1,a2);")
    ensure
        result_is_valid : Result >= 0 and Result <= 255
    end  -- feature get_pixel


    vertical_line(x, y1, y2, colour_index : INTEGER) is
    -- Draw a vertical line from y1 to y2 on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y1_is_valid           : y1 >= 0 and y1 <= height
        y2_is_valid           : y2 >= 0 and y2 <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("vline(C->_data,a1,a2,a3,a4);")
    end  -- feature vertical_line


    horizontal_line(x1, y, x2, colour_index : INTEGER) is
    -- Draw a vertical line from y1 to y2 on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x1_is_valid           : x1 >= 0 and x1 <= width
        y_is_valid            : y >= 0 and y <= height
        x2_is_valid           : x2 >= 0 and x2 <= width
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("hline(C->_data,a1,a2,a3,a4);")
    end  -- feature vertical_line


    line(x1, y1, x2, y2, colour_index : INTEGER) is
    -- Draw a line from (x1,y1) to (x2,y2) on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x1_is_valid           : x1 >= 0 and x1 <= width
        y1_is_valid           : y1 >= 0 and y1 <= height
        x2_is_valid           : x2 >= 0 and x2 <= width
        y2_is_valid           : y2 >= 0 and y2 <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("line(C->_data,a1,a2,a3,a4,a5);")
    end  -- feature line


    filled_triangle(x1, y1, x2, y2, x3, y3, colour_index : INTEGER) is
    -- Draw a filled triangle on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x1_is_valid           : x1 >= 0 and x1 <= width
        y1_is_valid           : y1 >= 0 and y1 <= height
        x2_is_valid           : x2 >= 0 and x2 <= width
        y2_is_valid           : y2 >= 0 and y2 <= height
        x3_is_valid           : x3 >= 0 and x3 <= width
        y3_is_valid           : y3 >= 0 and y3 <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("triangle(C->_data,a1,a2,a3,a4,a5,a6,a7);")
    end  -- feature filled_triangle


    rectangle(x1, y1, x2, y2, colour_index : INTEGER) is
    -- Draw a rectangle on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x1_is_valid           : x1 >= 0 and x1 <= width
        y1_is_valid           : y1 >= 0 and y1 <= height
        x2_is_valid           : x2 >= 0 and x2 <= width
        y2_is_valid           : y2 >= 0 and y2 <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("rect(C->_data,a1,a2,a3,a4,a5);")
    end  -- feature line


    filled_rectangle(x1, y1, x2, y2, colour_index : INTEGER) is
    -- Draw a filled rectangle on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x1_is_valid           : x1 >= 0 and x1 <= width
        y1_is_valid           : y1 >= 0 and y1 <= height
        x2_is_valid           : x2 >= 0 and x2 <= width
        y2_is_valid           : y2 >= 0 and y2 <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("rectfill(C->_data,a1,a2,a3,a4,a5);")
    end  -- feature line


    circle(x, y, radius, colour_index : INTEGER) is
    -- Draw a circle on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y_is_valid            : y >= 0 and y <= height
        radius_is_valid       : radius >= 0
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("circle(C->_data,a1,a2,a3,a4);")
    end  -- feature line


    filled_circle(x, y, radius, colour_index : INTEGER) is
    -- Draw a filled circle on this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y_is_valid            : y >= 0 and y <= height
        radius_is_valid       : radius >= 0
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("circlefill(C->_data,a1,a2,a3,a4);")
    end  -- feature line


    flood_fill(x, y, colour_index : INTEGER) is
    -- Flood fill this bitmap
    require
        ace_initialised       : info.ace_initialised
        graphics_initialised  : info.graphics_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y_is_valid            : y >= 0 and y <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("floodfill(C->_data,a1,a2,a3);")
    end  -- feature line


    masked_blit(source : BITMAP; x, y : INTEGER) is
    -- Blit the source bitmap to this bitmap, ignoring all pixels of colour 0
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        source_is_valid      : source /= Void and then
                               source.is_valid   and then
                               (source.is_memory and
                                not source.is_sub_bitmap)
        x_is_valid           : x >= 0 and x <= width
        y_is_valid           : y >= 0 and y <= height
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("draw_sprite(C->_data,_p,a2,a3);")
    end  -- feature masked_blit


    masked_blit_vertical_flip(source : BITMAP; x, y : INTEGER) is
    -- Blit the source bitmap to this bitmap upside down, ignoring all pixels
    -- of colour 0
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        source_is_valid      : source /= Void and then
                               source.is_valid   and then
                               (source.is_memory and
                                not source.is_sub_bitmap)
        x_is_valid           : x >= 0 and x <= width
        y_is_valid           : y >= 0 and y <= height
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("draw_sprite_v_flip(C->_data,_p,a2,a3);")
    end  -- feature masked_blit


    masked_blit_horizontal_flip(source : BITMAP; x, y : INTEGER) is
    -- Blit the source bitmap to this bitmap reversed, ignoring all pixels
    -- of colour 0
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        source_is_valid      : source /= Void and then
                               source.is_valid   and then
                               (source.is_memory and
                                not source.is_sub_bitmap)
        x_is_valid           : x >= 0 and x <= width
        y_is_valid           : y >= 0 and y <= height
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("draw_sprite_h_flip(C->_data,_p,a2,a3);")
    end  -- feature masked_blit


    masked_blit_vertical_horizontal_flip(source : BITMAP; x, y : INTEGER) is
    -- Blit the source bitmap to this bitmap upside down and reversed,
    -- ignoring all pixels of colour 0
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        source_is_valid      : source /= Void and then
                               source.is_valid   and then
                               (source.is_memory and
                                not source.is_sub_bitmap)
        x_is_valid           : x >= 0 and x <= width
        y_is_valid           : y >= 0 and y <= height
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("draw_sprite_vh_flip(C->_data,_p,a2,a3);")
    end  -- feature masked_blit


    masked_rotate_blit(source : BITMAP; x, y : INTEGER; angle : FIXED) is
    -- Blit the source bitmap to this bitmap rotated at the specified angle,
    -- ignoring all pixels of colour 0
    require
        ace_initialised      : info.ace_initialised
        graphics_initialised : info.graphics_initialised
        is_valid             : is_valid
        source_is_valid      : source /= Void and then
                               source.is_valid   and then
                               (source.is_memory and
                                not source.is_sub_bitmap)
        x_is_valid           : x >= 0 and x <= width
        y_is_valid           : y >= 0 and y <= height
        angle_is_valid       : angle.to_double >= 0.0 and
                               angle.to_double <= 256.0
    local
        p : POINTER
        i : INTEGER
    do
        p := source.data
        i := angle.data
        c_inline_c("rotate_sprite(C->_data,_p,a2,a3,_i);")
    end  -- feature masked_rotate_blit


    masked_stretch_blit(source : BITMAP; x, y, stretch_width, stretch_height : INTEGER) is
    -- Blit the source bitmap to this bitmap stretch to the given destination
    -- rectangle, ignoring all pixels of colour 0
    require
        ace_initialised         : info.ace_initialised
        graphics_initialised    : info.graphics_initialised
        is_valid                : is_valid
        source_is_valid         : source /= Void and then
                                  source.is_valid   and then
                                  (source.is_memory and
                                   not source.is_sub_bitmap)
        x_is_valid              : x >= 0 and x <= width
        y_is_valid              : y >= 0 and y <= height
        stretch_width_is_valid  : stretch_width >= 0 stretch_width  <= width-x
        stretch_height_is_valid : stretch_height >= 0 stretch_height <= height-y
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("stretch_sprite(C->_data,_p,a2,a3,a4,a5);")
    end  -- feature masked_stretch_blit


    blit(source : BITMAP; source_x, source_y,
                          source_width, source_height,
                          dest_x, dest_y : INTEGER) is
    -- Blit from the source bitmap using the specified rectangle.
    -- Performs clipping on both source and Current.
    require
        ace_initialised        : info.ace_initialised
        graphics_initialised   : info.graphics_initialised
        is_valid               : is_valid
        source_is_valid        : source /= Void and then
                                 source.is_valid
        source_x_is_valid      : source_x >= 0 and source_x <= source.width
        source_y_is_valid      : source_y >= 0 and source_y <= source.height
        source_width_is_valid  : source_width >= 0 and source_width <= source.width
        source_height_is_valid : source_height >= 0 and source_height <= source.height
        dest_x_is_valid        : dest_x <= width
        dest_y_is_valid        : dest_y <= height
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("blit(_p,C->_data,a2,a3,a6,a7,a4,a5);")
    end  -- feature blit


    stretch_blit(source : BITMAP; source_x, source_y,
                                  source_width, source_height,
                                  dest_x, dest_y,
                                  dest_width, dest_height : INTEGER) is
    -- Blit the given rectangle from the source bitmap to Current, stretching
    -- to fit the given destination rectangle
    require
        ace_initialised         : info.ace_initialised
        graphics_initialised    : info.graphics_initialised
        is_valid                : is_valid
        source_is_valid         : source /= Void and then
                                  source.is_valid   and then
                                  (source.is_memory and
                                   not is_equal(source))
        source_x_is_valid       : source_x >= 0 and source_x <= source.width
        source_y_is_valid       : source_y >= 0 and source_y <= source.height
        source_width_is_valid   : source_width >= 0 and source_width <= source.width-source_x
        source_height_is_valid  : source_height >= 0 and source_width <= source.height-source_y
        dest_x_is_valid         : dest_x >= 0 and dest_y <= width
        dest_y_is_valid         : dest_y >= 0 and dest_y <= height
        dest_width_is_valid     : dest_width >= 0 and dest_width <= width-dest_x
        dest_height_is_valid    : dest_height >= 0 and dest_height <= height-dest_y
    local
        p : POINTER
    do
        p := source.data
        c_inline_c("stretch_blit(_p,C->_data,a2,a3,a4,a5,a6,a7,a8,a9);")
    end  -- feature stretch_blit


    clear_to_colour(colour_index : INTEGER) is
    -- Clear this bitmap to the specified colour_index
    require
        ace_initialised       : info.ace_initialised
        is_valid              : is_valid
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("clear_to_color(C->_data,a1);")
    end  -- feature clear_to_colour


    clear is
    -- Clear this bitmap to colour 0
    require
        ace_initialised : info.ace_initialised
        is_valid        : is_valid
    do
        c_inline_c("clear(C->_data);")
    end  -- feature clear


    save(filename : STRING; palette : PALETTE) : BOOLEAN is
    -- Save this bitmap into the specified file, with the specified palette
    require
        ace_initialised   : info.ace_initialised
        is_valid          : is_valid
        palette_is_valid  : palette.is_valid
        filename_is_valid : filename /= Void and then
                            (ace_file_tools.extension(filename).same_as("PCX") or
                             ace_file_tools.extension(filename).same_as("TGA"))
    local
        p1 : POINTER
        p2 : POINTER
    do
        p1 := filename.to_external
        p2 := palette.data
        c_inline_c("R=(save_bitmap(_p1,C->_data,_p2)>0?0:-1);")
    end  -- feature save


    set_text_background(colour_index : INTEGER) is
    -- Make all text output have an opaque background
    require
        ace_initialised       : info.ace_initialised
        is_valid              : is_valid
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
    do
        c_inline_c("text_mode(a1);")
    end  -- feature set_text_background


    set_text_background_transparent is
    -- Make all text output have a transparent background
    require
        ace_initialised : info.ace_initialised
        is_valid        : is_valid
    do
        c_inline_c("text_mode(-1);")
    end  -- feature set_text_background_transparent


    put_string(x, y, colour_index : INTEGER; font : FONT; string : STRING) is
    -- Output string at (x,y)
    require
        ace_initialised       : info.ace_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y_is_valid            : y >= 0 and y <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
        string_is_valid       : string /= void
        font_is_valid         : font /= Void and then
                                font.is_valid
    local
        p1 : POINTER
        p2 : POINTER
    do
        p1 := string.to_external
        p2 := font.data
        c_inline_c("textout(C->_data,_p2,_p1,a1,a2,a3);")
    end  -- feature put_string


    put_string_centred(x, y, colour_index : INTEGER; font : FONT; string : STRING) is
    -- Output string at (x,y)
    require
        ace_initialised       : info.ace_initialised
        is_valid              : is_valid
        x_is_valid            : x >= 0 and x <= width
        y_is_valid            : y >= 0 and y <= height
        colour_index_is_valid : colour_index >= 0 and colour_index <= 255
        string_is_valid       : string /= void
        font_is_valid         : font /= Void and then
                                font.is_valid
    local
        p1 : POINTER
        p2 : POINTER
    do
        p1 := string.to_external
        p2 := font.data
        c_inline_c("textout_centre(C->_data,_p2,_p1,a1,a2,a3);")
    end  -- feature put_string_centred


------------------------------------------------------ Class invariant
invariant

    consistency : is_valid implies data.is_not_void and
                  not is_valid implies data.is_void


end  -- class BITMAP
