/* Project SWORD
   V2.0

   SubSystem : Basic functions usefull for all the system
   File      : Src/Common/Error.CC
   ErrNos    : 100..199
   Author    : Eric NICOLAS
   Overview  : Error management. Driver for showing errors in text mode.
   UpDate    : Oct 02, 1995

** Copyright (C) 1993,1995 The SWORD Group
**
** This file is distributed under the terms listed in the document
** "copying.en". A copy of "copying.en" should accompany this file.
** if not, a copy should be available from where this file was obtained.
** This file may not be distributed without a verbatim copy of "copying.en".
**
** This file is distributed WITHOUT ANY WARRANTY; without even the implied
** warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
*/

#include "Common/Types.H"
#include "Common/Error.H"

#define MaxErrorHandlers  50

// ===== Variables globales =================================================

int          Error;
PErrorViewer ErrorViewer;

// ===== Variables locales ==================================================

static PErrorHandler  ErrorHandlers[MaxErrorHandlers];
static int            NbErrorHandlers;
static char          *Messages[10] =
      { "Out of Memory",
        "Impossible to open a disk file",
        "Impossible to create a disk file",
        "Impossible to read in a disk file",
        "Impossible to write in a disk file",
        "Impossible to close a disk file",
        "Impossible to seek in a disk file",
        "Found unexpected End Of File",
        "Line Too long in a disk file",
        "Programmer: No more error handlers"
      };

// ===== Gestion globale des erreurs ========================================

//---------------------------------------------------------------------------
// SafeMAlloc

void *SafeMAlloc(int32 Size)
{ // Tente l'allocation
  void *Temp=malloc((size_t)Size);
  // Vrifie si le rsulat est correct
  if (Temp==NULL) Error=erLowMemory;
  // Renvoie une valeur, NULL si il y a eu un problme
  return Temp;
}

//---------------------------------------------------------------------------
// SafeFree

void SafeFree(void **Block)
{ // Vrifie que le bloc  libr existe bien
  if (*Block==NULL) return;
  // Libre le bloc
  free(*Block);
  // Marque le bloc comme libre
  *Block=NULL;
}

//---------------------------------------------------------------------------
// GetErrorMessage

void GetErrorMessage(char *Message)
{ int i;
  for(i=0;i<NbErrorHandlers;i++)
    if (ErrorHandlers[i](Message)) return;
  sprintf(Message,"Unknown Error n%d",Error);
}

//---------------------------------------------------------------------------
// AddErrorHandler

void AddErrorHandler(PErrorHandler ErrorHandler)
{ // Vrifie que la liste des Handlers n'est pas pleine
  if (NbErrorHandlers==MaxErrorHandlers)
  { Error=erNoMoreErrorHandlers;
    ErrorViewer();
  }
  // Ajoute le handler
  NbErrorHandlers++;
  ErrorHandlers[NbErrorHandlers-1]=ErrorHandler;
}

//---------------------------------------------------------------------------
// FatalError

void FatalError(void)
{ if (Error)
  { char Message[100];
    GetErrorMessage(Message);
    textmode(C80);
    textattr(0x4C);
    clreol();cprintf("Fatal system error\r\n");
    clreol();cprintf("%s\r\n",Message);
    textattr(0x07);
    cprintf("\r\n\r\n");
    _exit(1);
  }
}

// ===== Gestion locale des erreurs =========================================

static boolean ErrorMessages(char *Message)
{ int MessageNo;
  switch(Error)
  { case erLowMemory:              MessageNo=0;   break;
    case erDiskAccess_Open:        MessageNo=1;   break;
    case erDiskAccess_Create:      MessageNo=2;   break;
    case erDiskAccess_Read:        MessageNo=3;   break;
    case erDiskAccess_Write:       MessageNo=4;   break;
    case erDiskAccess_Close:       MessageNo=5;   break;
    case erDiskAccess_Seek:        MessageNo=6;   break;
    case erDiskAccess_EOF:         MessageNo=7;   break;
    case erDiskAccess_LineTooLong: MessageNo=8;   break;
    case erNoMoreErrorHandlers:    MessageNo=9;   break;
    default:
      return FALSE;
  }
  strcpy(Message,Messages[MessageNo]);
  return TRUE;
}

// ===== Init/Done de cette unit ===========================================

void InitError(void)
{ // Initialise le gestionnaire d'erreurs
  Error=0;
  NbErrorHandlers=1;
  ErrorHandlers[0]=ErrorMessages;
  ErrorViewer=FatalError;
}

void DoneError(void)
{
}
