; xgrep.s   1.03
;
; Fast Unix-compatible grep for MS-DOS
;
; Copyright (c) 1994 Robert Nordier
; All rights reserved
;
; Redistribution and use in source and binary forms are freely
; permitted provided that the above copyright notice and this
; paragraph and the following disclaimer are duplicated in all
; such forms.
;
; This software is provided "AS IS" and without any express or
; implied warranties, including, without limitation, the implied
; warranties of merchantability and fitness for a particular
; purpose.


; Segments and groups (.com file)

_TEXT		segment byte public 'CODE'
_TEXT		ends

_DATA		segment word public 'DATA'
_DATA		ends

_BSS		segment page public 'BSS'
_BSS		ends

DGROUP		group _TEXT, _DATA, _BSS
		assume CS:DGROUP, DS:DGROUP, SS:DGROUP

OFS		equ offset DGROUP:

; Macros

SETDEF		macro lo, hi			;Set definition
		db 01H + hi - lo, lo
		endm

SKIP2		macro				;Skip two bytes
		db 0A9H
		endm

; DOS equates

STDOUT		equ 1				;Standard output
STDERR		equ 2				;Standard error

HT		equ 09H 			;Horizontal tab
LF		equ 0AH 			;Linefeed
CR		equ 0DH 			;Carriage return
EOF		equ 1AH 			;DOS end of file

DOS_INT 	equ 21H 			;DOS interrupt

DOS_SETDTA	equ 1AH 			;Set DTA address
DOS_VER 	equ 30H 			;Get DOS version
DOS_OPEN	equ 3DH 			;Open file
DOS_CLOSE	equ 3EH 			;Close file
DOS_READ	equ 3FH 			;Read from file
DOS_WRITE	equ 40H 			;Write to file
DOS_EXIT	equ 4CH 			;Terminate process
DOS_FFIRST	equ 4EH 			;Find first file
DOS_FNEXT	equ 4FH 			;Find next file

FA_DIR		equ 10H 			;Directory attribute

; DOS Find First/Next structure

FF_BLK		struc
FF_RSVD 	db 15H dup(?)			;Reserved
FF_ATTR 	db ?				;File attribute
FF_TIME 	dw ?				;File time
FF_DATE 	dw ?				;File date
FF_SIZE 	dd ?				;File size
FF_NAME 	db 0DH dup(?)			;File name
FF_BLK		ends

; Memory management

PSPSIZ		equ   100H			;DOS PSP size

SUBDIR		equ    21H			;Search levels

BUFSIZ		equ   800H			;Work buffer size
CTLSIZ		equ   200H			;Control area size
SETSIZ		equ  1000H			;Set area size
LINSIZ		equ   400H			;Line buffer size
BLKSIZ		equ 0C000H			;Input buffer size
STKSIZ		equ  1000H			;Stack size

; Command line options: xg_opt[]

OPT_c		equ  0				;Count matches
OPT_h		equ  1				;No file names
OPT_l		equ  2				;File list
OPT_n		equ  3				;Number lines
OPT_s		equ  4				;Silent mode
OPT_v		equ  5				;Inverse mode
OPT_o		equ  6				;Errors to stdout
OPT_x		equ  7				;No magic
OPT_r		equ  8				;Recursive search
OPT_y		equ  9				;Case-insensitive
OPT_d		equ 10				;Alias for -r
OPT_i		equ 11				;Alias for -y

NUMOPT		equ 12				;Number of options

; Operators: rx_ops[]

RX_LIN		equ 0				;Line operators
RX_SET		equ 2				;Set operators
RX_REP		equ 4				;Repeat operators

NUMOPS		equ 8				;Number of operators

; Output format

FMT_LIN 	equ 80H 			;Line
FMT_VAL 	equ 40H 			;Number/count
FMT_NAM 	equ 20H 			;File name

; Return codes

RET_FAIL	equ 1				;No matches
RET_ERRS	equ 2				;Error(s) occurred

; Various

NOMAGIC 	equ 80H 			;Require \
CTYPES		equ 12				;Number of ctypes


_TEXT		segment


psp		db PSPSIZ dup(?)		;DOS PSP

start:		jmp short main
		dw OFS optstr - PSPSIZ		;Patch pointer

		dw 0803H, 0E01H, 0114H, 0C15H	;Signature
		dw 0011H			;Program number
		dw 0103H			;Version number
logo		db "xgrep  Version 1.03  "
		db "Copyright (c) 1994 Robert Nordier", 0
		db 1AH
		db "@(#)xgrep.s 1.3 94/06/17", 0


_TEXT		ends


_DATA		segment


; Command line options

optstr		db "chlnsvoxrydi"		;Option identifiers
xg_opt		db xg_opt - optstr dup(0)	;Options specified

; Regular expression operators

rx_ops		db '^'				;Beginning of line
		db '$'				;End of line
		db '.'				;Wildcard
		db '['				;Character class
		db '*'				;Repeat 0 or more
		db '+'				;Repeat 1 or more
		db '?'				;Repeat 0 or 1
		db '{' or NOMAGIC		;Repeat \{m,u\}

; Regular expression variables

rx_bol		db 0				;Beginning of line
rx_eol		db 0				;End of line
rx_chr		db 0AH, 0			;Literal for scan
rx_adj		dw 0				;Length adjustment
rx_ofs		dw 0				;Literal offset

; Miscellaneous variables

stkptr		dw 0				;Stack pointer
status		db RET_FAIL, 0			;Status for exit
namend		dw 0				;End of (file) name
number		dd 0				;Line number
count		dd 0				;Match count
format		dw 0				;Output format
eofile		db 0, 0 			;End of file
eodata		dw 0				;End of data

; Type definitions

alpha		label byte
		SETDEF 'A', 'Z'
		SETDEF 'a', 'z'
		dw 0

cntrl		label byte
		SETDEF 00H, 1FH
		SETDEF 7FH, 7FH
		dw 0

digit		label byte
		SETDEF '0', '9'
		dw 0

graph		label byte
		SETDEF '!', '~'
		dw 0

lower		label byte
		SETDEF 'a', 'z'
		dw 0

alnum		label byte
		SETDEF '0', '9'
		SETDEF 'A', 'Z'
		SETDEF 'a', 'z'
		dw 0

punct		label byte
		SETDEF '!', '/'
		SETDEF ':', '@'
		SETDEF '[', '`'
		SETDEF '{', '~'
		dw 0

space		label byte
		SETDEF 09H, 0DH
		SETDEF ' ', ' '
		dw 0

print		label byte
		SETDEF ' ', '~'
		dw 0

upper		label byte
		SETDEF 'A', 'Z'
		dw 0

xdigit		label byte
		SETDEF '0', '9'
		SETDEF 'A', 'F'
		SETDEF 'a', 'f'
		dw 0

ascii		label byte
		SETDEF 00H, 07FH
		dw 0

; Type definition table

ctype		dw OFS alpha			;a
		dw OFS cntrl			;c
		dw OFS digit			;d
		dw OFS graph			;g
		dw OFS lower			;l
		dw OFS alnum			;n
		dw OFS punct			;p
		dw OFS space			;s
		dw OFS print			;t
		dw OFS upper			;u
		dw OFS xdigit			;x
		dw OFS ascii			;z

typeid		db "zxutspnlgdca"		;Type identifiers

progid		db "XGREP", 5 dup(0)		;Program id for usage
errdev		dw STDERR			;Error device
errsep		db ": ", 0, 0			;Error message separator
crlf		db CR,LF, 0, 0			;DOS newline

; RE error lookup table

re_err		dw OFS e_re_0
		dw OFS e_re_1
		dw OFS e_re_2
		dw OFS e_re_3
		dw OFS e_re_4
		dw OFS e_re_5
		dw OFS e_re_6
		dw OFS e_re_7

; Error messages

e_xgrep 	db "xgrep", 0
e_nospc 	db "Not enough space", 0
e_badopt	db "Invalid option", 0
e_read		db "Read error", 0
e_write 	db "Write error", 0
e_find		db "Not found", 0
e_open		db "Can't open", 0
e_re		db "RE error", 0
e_re_0		db "NULL expression", 0
e_re_1		db "[  ] imbalance", 0
e_re_2		db "Range endpoint too large", 0
e_re_3		db "Bad number", 0
e_re_4		db "Range endpoint is zero", 0
e_re_5		db "More than 2 numbers given in \{  \}", 0
e_re_6		db "} expected after \", 0
e_re_7		db "First number exceeds second in \{  \}", 0

; Help messages

e_usage 	db "Usage: ", 0
e_syntax	db " [-chlnsvyorx] expression [file ...]", 0
e_help		db CR,LF
		db "Options:", CR,LF
		db HT,"-c  Line count only"
		db HT,"-r  Recursively search subdirectories", CR,LF
		db HT,"-h  No file names"
		db HT,"-s  Silent about inaccessible files", CR,LF
		db HT,"-l  File names only"
		db HT,"-v  Non-matching lines", CR,LF
		db HT,"-n  Number lines"
		db HT,"-x  No magic: all operators need \", CR,LF
		db HT,"-o  Errors to stdout"
		db HT,"-y  Case-insensitive match", CR,LF
		db CR,LF
		db "Syntax:", CR,LF
		db HT,"^    start of line     repeat:  *"
		db HT," 0 or more times", CR,LF
		db HT,"$    end of line", HT,HT, "+"
		db HT," 1 or more times", CR,LF
		db HT,".    wildcard", HT,HT,HT, "?"
		db HT," 0 or 1 times", CR,LF
		db HT,"[]   character class", HT,HT, "\{m\}"
		db HT," m times", CR,LF
		db HT,"[^]  inverse class", HT,HT, "\{m,\}"
		db HT," m or more times", CR,LF
		db HT,"\    quote next character", HT, "\{m,u\}"
		db HT," m to u times", CR,LF
		db CR,LF
		db "Classes:", CR,LF
		db HT,":a  alpha",  HT, ":c  cntrl"
		db HT,":d  digit",  HT, ":g  graph", CR,LF
		db HT,":l  lower",  HT, ":n  alnum"
		db HT,":p  punct",  HT, ":s  space", CR,LF
		db HT,":t  print",  HT, ":u  upper"
		db HT,":x  xdigit", HT, ":z  ascii", 0


_DATA		ends


_BSS		segment


xg_buf		db BUFSIZ dup(?)		;Work buffer

ff_sub		equ xg_buf[BUFSIZ - size FF_BLK * SUBDIR]
ff_dir		equ xg_buf[BUFSIZ - size FF_BLK]

rx_ctl		db CTLSIZ dup(?)		;Control area
rx_set		db SETSIZ dup(?)		;Set area
		db LINSIZ dup(?)		;Line buffer
rx_inp		db BLKSIZ dup(?)		;Input buffer

xg_end		label byte			;First free byte


_BSS		ends


_TEXT		segment


main		proc

; Startup

		cld				;Default
		mov stkptr,SP			;Save for errors
		cmp SP,OFS xg_end + STKSIZ - 2	;Enough memory?
		jb main06			;No
		xor AX,AX			;Clear
		mov DI,OFS rx_set		; out
		mov CX,SETSIZ shr 1		; set
		rep stosw			; space

; Parse command line options

		mov SI,OFS psp[80H]		;Get length of
		lodsb				; command tail
		add SI,AX			;Terminate
		mov [SI],AH			; with 0 and
		sub SI,AX			; point to start
		mov DX,'-/'			;Switch characters
		mov DI,OFS optstr		;Valid options
		mov CL,NUMOPT			;Number of options
main01: 	call getopt			;Parse option
		jz main09			;End of options
		jc main02			;Error
		mov xg_opt[BX],01H		;Set option
		jmp main01			;Till done

; Error: Invalid option

main02: 	mov DX,OFS e_badopt		;Error prefix
		xor AH,AH			;Save
		mov word ptr xg_buf,AX		; option
		mov AX,OFS xg_buf		;Point to it
		call error			;Output message
		call usage			;Output usage info
		jmp short main08		;Error exit

; Help: No expression specified

main03: 	call help			;Output help screen
		jmp short main08		;Error exit

; RE Error: NULL expression

main04: 	 xor AL,AL			;Error number

; RE error: generic

main05: 	mov DX,OFS e_re 		;Error prefix
		cbw				;Convert error
		xchg BX,AX			; number to
		shl BX,1			; index
		mov AX,re_err[BX]		;Error message
		jmp short main07		;Output and exit

; Error: Not enough space

main06: 	mov AX,OFS e_nospc		;Error message

; Error: generic

fatal:
		mov SP,stkptr			;Clear stack
		mov DX,OFS e_xgrep		;Error prefix
main07: 	call error			;Output message
main08: 	mov AL,RET_ERRS 		;Return code
		jmp main39			;Exit

; Process options

main09: 	mov AX,word ptr xg_opt[OPT_d]	;Options d and i are
		or word ptr xg_opt[OPT_r],AX	; aliases for r and y
		cmp xg_opt[OPT_o],CH		;Option o?
		je main10			;No
		mov byte ptr errdev,STDOUT	;Errors to stdout
main10: 	cmp xg_opt[OPT_x],CH		;Option x?
		je main12			;No
		mov DI,OFS rx_ops		;Operators
		mov CL,NUMOPS			;Byte count
main11: 	or byte ptr [DI],NOMAGIC	;Un-magic
		inc DI				; each
		loop main11			; operator

; Get and compile expression

main12: 	mov BP,OFS xg_buf		;Work buffer
		mov DI,BP			;Copy expression
		call getarg			; to buffer
		jc main03			;Nothing to copy
		cmp DI,BP			;Anything copied?
		jna main04			;No
		dec DI				;Last char
		push SI 			;Command line
		mov SI,BP			;Expression
		call xg_cmp			;Compile it
		pop SI				;Command line
		jc main05			;Error

; Determine output format

		mov DX,word ptr xg_opt[OPT_c]	;Options c and h
		mov BX,word ptr xg_opt[OPT_l]	;Options l and n
		mov AX,FMT_VAL shl 8 + FMT_VAL	;Initialize
		cmp DL,CH			;Option c?
		jne main15			;Yes
		cmp BL,CH			;Option l?
		je main13			;No
		cmp DH,CH			;Option h?
		jne main17			;Yes
		cmp BH,CH			;Option n?
		je main17			;No
main13: 	cmp BH,CH			;Option n?
		je main14			;No
		add AL,AH			;Adjust
main14: 	add AL,AH			; format
main15: 	cmp DH,CH			;Option h?
		jne main16			;Yes
		xor AH,AH			;Not complete
main16: 	mov format,AX			;Update format

; Use or close stdin

main17: 	call cmdarg			;File specified?
		jnz main18			;Yes
		call xg_ex			;Execute expression
		jmp main38			;Done
main18: 	xor BX,BX			;Handle
		mov AH,DOS_CLOSE		;DOS: Close
		int DOS_INT			; file

; Process a filespec from list

main19: 	dec SI				;Unget char
		mov DI,OFS xg_buf		;Buffer
		mov CL,0FFH			;Byte count
		call fncopy			;Copy filespec
		mov DI,BX			;Path pointer
		dec SI				;Unget char
		mov CX,SI			;Length of
		sub CX,DX			; file name
		push SI 			;Command line
		cmp byte ptr format[01H],CH	;Format complete?
		jne main26			;Yes
		test AH,AH			;Wildcards?
		jnz main21			;Yes
		cmp xg_opt[OPT_r],CH		;Recursive search?
		jne main20			;Yes
		call cmdarg			;To next argument
		jz main21			;Nothing
main20: 	inc AH				;Multiple files
main21: 	mov AL,byte ptr format		;Format specifier
		test AL,AL			;What type?
		jz main24			;List
		mov BX,word ptr xg_opt[OPT_l]	;Options l and n
		js main22			;Line
		cmp BH,CH			;Option n?
		jne main23			;Yes
main22: 	cmp BL,CH			;Option l?
		jne main24			;Yes
main23: 	test AH,AH			;Multiple files?
		jz main25			;No
main24: 	or AL,FMT_NAM			;Name of file
main25: 	mov AH,AL			;Format complete
		mov format,AX			;Update
main26: 	mov SI,DX			;File name argument
		mov DX,OFS xg_buf		;Full path
		mov BP,OFS ff_sub - size FF_BLK ;File find buffer

; Search directory

main27: 	add BP,size FF_BLK		;New DTA
		push DX 			;Save caller's
		mov DX,BP			;DTA
		mov AH,DOS_SETDTA		;DOS: Set
		int DOS_INT			; DTA address
		pop DX				;Restore caller's
		push BX 			;New path
		mov BX,DI			; pointer
		cmp xg_opt[OPT_r],CH		;Recursive search?
		je main31			;No
		cmp BP,OFS ff_dir		;Last DTA area?
		jnb main30			;Yes

; Look for subdirectories

		mov AX,'.*'			;Append
		stosw				; *.*
		xor AH,AH			; to
		stosw				; path
		push CX 			;Save caller's
		mov CL,FA_DIR			;Attributes
		mov AH,DOS_FFIRST		;DOS: Find
		int DOS_INT			; first match
		pop CX				;Restore caller's
		jc main30			;Not found
main28: 	test [BP.FF_ATTR],FA_DIR	;Directory?
		jz main29			;No
		cmp [BP.FF_NAME],'.'		;Current/parent?
		je main29			;Yes
		call ffcopy			;Get name
		dec DI				;Append
		mov AX,'\'			; \
		stosw				; to
		dec DI				; path
		jmp main27			;Up a level
main29: 	mov AH,DOS_FNEXT		;DOS: Find
		int DOS_INT			; next match
		jnc main28			;Found

; Look for files

main30: 	mov DI,BX			;Path pointer
		push SI 			;Save
		push CX 			; caller's
		rep movsb			;Copy string
		pop CX				;Restore
		pop SI				; caller's
		mov [DI],CH			;Terminate string
main31: 	push CX 			;Save caller's
		xor CX,CX			;Attributes
		mov AH,DOS_FFIRST		;DOS: Find
		int DOS_INT			; first match
		pop CX				;Restore caller's
		jc main34			;Not found
main32: 	call ffcopy			;Get name
		mov namend,DI			;End of name
		push BP 			;Save
		push SI 			; our
		push DX 			; registers
		push CX 			; in
		push BX 			; use
		mov AX,DOS_OPEN shl 8 + 0	;DOS: Open file
		int DOS_INT			; for reading
		jc main35			;Error
		call xg_ex			;Execute expression
		xor BX,BX			;Handle
		mov AH,DOS_CLOSE		;DOS: Close
		int DOS_INT			; file
main33: 	pop BX				;Restore
		pop CX				; the
		pop DX				; registers
		pop SI				; we
		pop BP				; saved
		mov AH,DOS_FNEXT		;DOS: Find
		int DOS_INT			; next match
		jnc main32			;Found
main34: 	pop BX				;Path pointer
		sub BP,size FF_BLK		;Previous DTA
		cmp BP,OFS ff_sub		;Valid?
		jb main36			;Done
		push DX 			;Save caller's
		mov DX,BP			;DTA
		mov AH,DOS_SETDTA		;DOS: Set
		int DOS_INT			; DTA address
		pop DX				;Restore caller's
		jmp main29			;Continue

; Handle inaccessible files

main35: 	mov status,RET_ERRS		;Flag error
		cmp xg_opt[OPT_s],CH		;Option s?
		jne main33			;Yes
		mov AX,OFS e_open		;Output
		call error			; warning
		jmp main33			;Continue

; Warn if file not found

main36: 	xor BP,BP			;Any file
		cmp namend,BP			; found?
		jne main37			;Yes
		mov status,RET_ERRS		;Flag error
		mov AX,OFS e_find		;Output
		call error			; warning

; Continue if more files

main37: 	pop SI				;Command line
		call cmdarg			;To next argument
		jz main38			;Nothing
		mov namend,BP			;Reset flag
		jmp main19			;Continue

; Exit

main38: 	mov AL,status			;Return code
main39: 	mov AH,DOS_EXIT 		;DOS: Terminate
		int DOS_INT			; process

main		endp


usage		proc

; Write usage message

		mov AH,DOS_VER			;DOS: Get
		int DOS_INT			; version
		cmp AL,03H			;At least 3.0?
		jb usage04			;No
		xor DI,DI			;Far pointer
		mov AX,word ptr psp[2CH]	; to
		mov ES,AX			; environment
		test AX,AX			;NULL?
		jz usage04			;Yes
		xor AL,AL			;Search char
		xor CX,CX			;Maximum
		dec CX				; value
usage01:	repne scasb			;Look
		scasb				; for
		jne usage01			; double 0
		inc DI				;Point to
		inc DI				; program name
		mov SI,DI			;Source
		mov DI,OFS xg_buf		;Destination
		push ES 			;Swap
		push DS 			; ES,DS
		pop ES				; for
		pop DS				; copy
		mov CX,00FFH			;Byte count
		call fncopy			;Copy name
		push ES 			;Restore
		pop DS				; DS
		mov SI,BX			;Source
		mov DI,OFS progid		;Destination
		mov CL,08H			;Count
usage02:	lodsb				;Copy
		cmp AL,'.'			; till
		je usage03			; 8 chars
		stosb				; moved or
		loop usage02			; extension
usage03:	mov [DI],CH			;Terminate string
usage04:	mov DX,OFS e_usage		;Output
		call ewrite			; "Usage: "
		mov DX,OFS progid		;Plus program
		call ewrite			; name
		mov DX,OFS e_syntax		;Plus syntax
		jmp short ewriteln		; string

usage		endp


help		proc

; Write help screen

		mov DX,offset logo		;Write
		call ewriteln			; logo,
		call usage			; usage,
		mov DX,OFS e_help		; and help
		jmp short ewriteln		; string

help		endp


error		proc

; Write error prefix: message

		push AX 			;Message
		test DX,DX			;Prefix?
		jz error01			;No
		call ewrite			;Write it
		mov DX,OFS errsep		;Separator
		call ewrite			;Write it
error01:	pop DX				;Message

error		endp


ewriteln	proc

; Write error string plus newline

		call ewrite			;Write string
		mov DX,OFS crlf 		; and newline

ewriteln	endp


ewrite		proc

; Write error string

		mov DI,DX			;String
		xor AL,AL			;Terminator
		xor CX,CX			;Maximum
		dec CX				; value
		repne scasb			;Find terminator
		add CX,+02H			;Derive string
		neg CX				; length
		mov BX,errdev			;Handle
		jmp write			;Write to file

ewrite		endp


getopt		proc

; Get and check a command line option

		test AH,AH			;In argument?
		jz getopt01			;No
		call cmdchr			;Get char
		jnz getopt02			;Part of argument
		dec SI				;Unget
getopt01:	call cmdopt			;Get switch argument
		jz getopt04			;End of options
getopt02:	push DI 			;Save
		push CX 			; caller's
		mov BX,CX			;Save it
		repne scasb			;Look for option?
		je getopt03			;Found
		mov CX,BX			;Force CY
getopt03:	dec BX				;Offset
		sub BX,CX			; or -1
		inc CX				;Force NZ
		pop CX				;Restore
		pop DI				; caller's
getopt04:	ret

getopt		endp


cmdopt		proc

; Get a command line option argument

		call cmdarg			;Get argument char
		cmp AL,DL			;Switch?
		je cmdopt01			;Yes
		cmp AL,DH			;Switch?
		jne cmdopt03			;No
cmdopt01:	mov AH,AL			;Which switch
		call cmdchr			;Get char
		jz cmdopt02			;Not part of argument
		cmp AL,AH			;Another switch?
		jne cmdopt05			;No
		call cmdchr			;Get char
		jz cmdopt03			;Not part of argument
		mov AL,AH			;Option is switch
		jmp short cmdopt04		;Join common code
cmdopt02:	dec SI				;Unget
cmdopt03:	xor AL,AL			;End of options
cmdopt04:	dec SI				;Unget
cmdopt05:	test AL,AL			;Set up flags
		ret

cmdopt		endp


getarg		proc

; Copy a command line argument

		call cmdarg			;Get argument char
		jz getarg03			;Nothing there
		mov DX,SI			;Save place
getarg01:	stosb				;Put char
		cmp AL,'"'			;Quote?
		je getarg04			;Yes
getarg02:	call cmdchr			;Get char
		jnz getarg01			;Part of argument?
		clc				;No error
getarg03:	dec SI				;Unget
		ret

; Handle quotes

getarg04:	mov BX,SI			;Save place
getarg05:	lodsb				;Get char
		cmp AL,'"'			;Quote?
		je getarg06			;Yes
		test AL,AL			;End of string?
		jnz getarg05			;No
		mov SI,BX			;Restore place
		jmp getarg02			;Continue
getarg06:	cmp BX,DX			;Anything before?
		jna getarg07			;No
		cmp [BX-02H],AL 		;Quote?
		je getarg08			;Yes
getarg07:	dec DI				;Exclude opening quote
getarg08:	dec SI				;Exclude terminator
		mov CX,SI			;Character
		sub CX,BX			; count
		mov SI,BX			;First char
		rep movsb			;Put chars
		inc SI				;Skip closing quote
		jmp getarg02			;Continue

getarg		endp


cmdarg		proc

; Find next command line argument

cmdarg00:	call cmdchr			;Get char
		jnc cmdarg00			;If whitespace
		ret

cmdarg		endp


fncopy		proc

; Copy a filename

		mov DX,SI			;Initialize
		mov BX,DI			; pointers
		xor AH,AH			;Clear wildcard count
fncopy01:	call cmdchr			;Get char
		jz fncopy08			;Whitespace/terminator
		cmp AL,'/'			;Ensure
		jne fncopy02			; MS-DOS
		mov AL,'\'			; separator
fncopy02:	cmp AL,'a'			;Convert
		jb fncopy03			; lower
		cmp AL,'z'			; to
		ja fncopy03			; upper
		and AL,0DFH			; case
fncopy03:	stosb				;Put char
		cmp AL,'*'			;Keep
		je fncopy04			; count
		cmp AL,'?'			; of
		jne fncopy05			; wildcard
fncopy04:	inc AH				; chars
fncopy05:	cmp AL,':'			;Keep
		je fncopy06			; track of
		cmp AL,'\'			; separators
		jne fncopy07			; in both
fncopy06:	mov DX,SI			; source and
		mov BX,DI			; destination
fncopy07:	loop fncopy01			;Till terminator
		dec DI				;Overflow
fncopy08:	mov AL,CH			;Terminate
		stosb				; filename
		ret

fncopy		endp


cmdchr		proc

; Get and test a command line character

		lodsb				;Get char
		cmp AL,' '			;Space?
		je cmdchr01			;Yes
		cmp AL,HT			;Horizontal tab?
		je cmdchr01			;Yes
		test AL,AL			;End of string?
		stc				;Not whitespace
cmdchr01:	ret

cmdchr		endp


ffcopy		proc

; Copy file name from FF_BLK

		mov DI,BX			;Path pointer
		push SI 			;Save caller's
		lea SI,[BP.FF_NAME]		;File name
ffcopy01:	lodsb				;Copy a
		stosb				; char
		test AL,AL			;Terminator?
		jnz ffcopy01			;No
		pop SI				;Restore caller's
		ret

ffcopy		endp


xg_cmp		proc

; Process EOL operator

		mov AL,[DI]			;Get last char
		test AL,AL			;Is ASCII?
		js xg_cmp05			;No
		xor AL,rx_ops[RX_LIN][01H]	;EOL char?
		jz xg_cmp01			;Yes
		cmp AL,NOMAGIC			;Operator if escaped?
		jne xg_cmp05			;No
xg_cmp01:	mov BX,DI			;Last char
xg_cmp02:	cmp BX,SI			;First char?
		jna xg_cmp03			;Yes
		dec BX				;To preceding
		cmp byte ptr [BX],'\'		;In escape sequence?
		je xg_cmp02			;Yes
		inc BX				;Undo
xg_cmp03:	sub BX,DI			;Set MSB of BL if
		ror BL,1			; EOL char escaped
		xor AL,BL			;As required?
		js xg_cmp05			;No
		test BL,BL			;Escaped?
		jns xg_cmp04			;No
		dec DI				;Past EOL
xg_cmp04:	dec DI				; operator
		inc rx_eol			;Flag it
xg_cmp05:	inc DI				;Terminate
		mov [DI],CH			; expression

; Process BOL operator

		mov BX,SI			;First char
		mov AX,OFS rx_ops[RX_LIN]	;Line operators
		mov CL,01H			;Count
		call expchr			;Get char/operator
		jnz xg_cmp06			;Not operator
		inc rx_bol			;Flag it
		mov BX,SI			;New first char
xg_cmp06:	mov SI,BX			;Expression

; Compile expression

		xor BP,BP			;Literal pointer
		mov DI,OFS rx_ctl		;Control buffer
		mov BX,OFS rx_set + 01H 	;Set marker
		call xg_exp			;Compile expression
		jc xg_cmp - 01H 		;Error

; Process set data

		dec DI				;Past
		dec DI				; terminator
		mov AX,DI			;Derive
		sub AX,OFS rx_ctl		; length
		jz xg_cmp10			;Zero
		mov CL,02H			;Divide by
		shr AX,CL			; bytes per entry
		add AL,07H			;For rounding
		inc CX				;Divide by
		shr AX,CL			; sets per page
		mov DX,AX			;Set pages
		mov CX,AX			; to process
		mov BX,OFS rx_set + LF		;Set position
		xor AL,AL			;Bit pattern
xg_cmp07:	mov [BX],AL			;Reset all
		inc BH				; newline
		loop xg_cmp07			; bits
		cmp xg_opt[OPT_y],CH		;Option y?
		je xg_cmp10			;No
		xor BP,BP			;Don't pre-scan
		mov SI,OFS rx_set + 'A' 	;First upper
		mov DI,OFS rx_set + 'a' 	;First lower
		mov BX,100H - ('z' - 'a' + 1)	;Page adjustment
xg_cmp08:	mov CL,'z' - 'a' + 1		;Byte count
xg_cmp09:	mov AL,[DI]			;Get lower
		or [SI],AL			;Merge with upper
		movsb				;Copy to lower
		loop xg_cmp09			;Till page done
		add SI,BX			;To next
		add DI,BX			; page
		dec DX				;Till all
		jnz xg_cmp08			; done

; Process control data

xg_cmp10:	xor BX,BX			;Offset to literal
		xor DX,DX			;Length of expression
		mov SI,OFS rx_ctl		;Control data
xg_cmp11:	cmp SI,BP			;At literal?
		je xg_cmp13			;Yes
xg_cmp12:	lodsw				;Maximum reps
		test AX,AX			;End of expression?
		jz xg_cmp14			;Yes
		add DX,AX			;Plus maximum reps
		lodsw				;Optional reps
		test AX,AX			;Conditional?
		jz xg_cmp11			;No
		xor BP,BP			;Don't pre-scan
		sub DX,AX			;Less optional reps
		jmp xg_cmp11			;Next item
xg_cmp13:	mov BX,DX			;Offset to literal
		jmp xg_cmp12			;Continue
xg_cmp14:	mov rx_adj,DX			;Length adjustment
		mov rx_ofs,BX			;Literal offset
		cmp rx_bol,AL			;BOL specified?
		mov AL,CR			;In case
		jne xg_cmp15			;Yes
		test BP,BP			;Literal pre-scan?
		jnz xg_cmp16			;Yes
		mov AL,LF			;No pre-scan
xg_cmp15:	mov rx_chr,AL			;Update
xg_cmp16:	ret

xg_cmp		endp


xg_set		proc

; Support routine: handle set operators for xg_exp

		jcxz xg_set01			;If '['
		call setall			;Wildcard
		jmp xg_exp02			;Continue
xg_set01:	lodsb				;Get first char
		mov DL,AL			;Save it
		cmp AL,'^'			;Inverse class?
		jne xg_set02			;No
		lodsb				;Get next char
xg_set02:	cmp AL,']'			;Literal ']'?
		je xg_set06			;Yes
xg_set03:	test AL,AL			;End of string?
		jz xg_set13			;Yes
		call istype			;Type?
		jnz xg_set06			;No
		inc SI				;Past type id
		push SI 			;Source pointer
		mov SI,CX			;Index for
		shl SI,1			; lookup
		mov SI,ctype[SI]		;Type definition
xg_set04:	lodsb				;Chars in
		mov CL,AL			; sub-type
		jcxz xg_set05			;Done
		lodsb				;Start char
		call setbit			;Add to set
		jmp xg_set04			;Loop
xg_set05:	pop SI				;Source pointer
		jmp short xg_set10		;Continue
xg_set06:	mov AH,AL			;Save char
		lodsb				;Get next
		cmp AL,'-'			;A range?
		jne xg_set08			;No
		lodsb				;Get next
		test AL,AL			;End of string?
		jz xg_set13			;Yes
		cmp AL,']'			;End of class?
		je xg_set07			;Yes
		call istype			;Type?
		jz xg_set07			;Yes
		xchg AH,AL			;Set up
		cmp AL,AH			;Valid range?
		jbe xg_set09			;Yes
		jmp short xg_set10		;Continue
xg_set07:	dec SI				;Unget
xg_set08:	dec SI				;Unget
		mov AL,AH			;Restore char
xg_set09:	call setchr			;Add to set
xg_set10:	lodsb				;Get char
		cmp AL,']'			;End of class?
		jne xg_set03			;No
		mov AL,DL			;Inverse
		sub AL,'^'			; class?
		jnz xg_set12			;No
		mov CX,0100H			;Bit count
		xchg BL,AL			;Get set marker
xg_set11:	xor [BX],AL			;Invert
		inc BX				; each
		loop xg_set11			; bit
		dec BX				;Restore BH
		xchg BL,AL			;Put set marker
xg_set12:	jmp short xg_exp02		;Continue
xg_set13:	inc AX				;RE error 1
		stc				;Flag error
		ret

xg_set		endp


xg_exp		proc

; Compile regular expression

xg_exp01:	mov AX,OFS rx_ops[RX_SET]	;Set operators
		mov CL,RX_SET			;Count
		call expchr			;Get char/operator
		jz xg_set			;Operator
		test AL,AL			;End of string?
		je xg_exp04			;Yes
		xchg BL,AL			;Get set marker
		or [BX],AL			;Set bit
		xchg BL,AL			;Put set marker
		test BP,BP			;First literal?
		jz xg_chr			;Yes
xg_exp02:	rol BL,1			;Bump set
		adc BH,0			; marker
		mov AX,OFS rx_ops[RX_REP]	;Repeat operators
		mov CL,RX_REP			;Count
		mov DX,SI			;Keep place
		call expchr			;Get char/operator
		mov AX,0001H			;Maximum reps
		jz xg_rep			;Operator
		mov SI,DX			;Restore place
		xor DX,DX			;Optional reps
xg_exp03:	stosw				;Maximum reps
		mov AX,DX			;Optional
		stosw				; reps
		jmp xg_exp01			;Continue
xg_exp04:	xor AX,AX			;End of
		stosw				; expression
		ret

xg_exp		endp


xg_chr		proc

; Support routine: handle first literal char for xg_exp

		mov rx_chr,AL			;Save char
		mov BP,DI			; and position
		jmp xg_exp02			;Continue

xg_chr		endp


xg_rep		proc

; Support routine: handle repeat operators for xg_exp

		jcxz xg_rep01			;If '\{'
		mov DX,AX			;Optional reps
		dec CX				;Done
		jz xg_exp03			; if '?'
		neg AX				;Maximum reps
		neg DX				;Optional reps
		dec CX				;Done
		jnz xg_exp03			; if '*'
		dec DX				;Optional reps
		jmp xg_exp03			;Is '+'
xg_rep01:	call strtob			;Get lo value
		jc xg_rep06			;Overflow
		jz xg_rep07			;Nothing
		mov CX,AX			;Save lo
		mov DX,AX			;Save as hi
		lodsb				;Get next
		cmp AL,','			;More?
		jne xg_rep03			;No
		call strtob			;Get hi value
		jc xg_rep06			;Overflow
		jnz xg_rep02			;Something
		dec AX				;Wrap for maximum
xg_rep02:	mov DX,AX			;Save hi
		lodsb				;Get next
		cmp AL,','			;More?
		je xg_rep09			;Yes
xg_rep03:	mov AH,rx_ops[RX_REP][03H]	;Retrieve operator
		test AH,AH			;Magic?
		jns xg_rep04			;Yes
		cmp AL,'\'			;Escape sequence?
		jne xg_rep07			;No
		lodsb				;Get next
xg_rep04:	cmp AL,'}'			;Is '}'?
		je xg_rep05			;Yes
		test AH,AH			;Magic?
		jns xg_rep07			;Yes
		jmp short xg_rep10		;No
xg_rep05:	mov AX,DX			;Maximum reps
		test AX,AX			;Zero?
		jz xg_rep08			;Yes
		sub DX,CX			;Get optional reps?
		jb xg_rep11			;If lo exceeds hi
		jmp xg_exp03			;Continue
xg_rep06:	mov AL,02H			;RE error 2
		SKIP2				;Skip two bytes
xg_rep07:	mov AL,03H			;RE error 3
		SKIP2				;Skip two bytes
xg_rep08:	mov AL,04H			;RE error 4
		SKIP2				;Skip two bytes
xg_rep09:	mov AL,05H			;RE error 5
		SKIP2				;Skip two bytes
xg_rep10:	mov AL,06H			;RE error 6
		SKIP2				;Skip two bytes
xg_rep11:	mov AL,07H			;RE error 7
		stc				;Flag error
		ret

xg_rep		endp


expchr		proc

; Get character or operator

		push DI 			;Save caller's
		mov DI,AX			;Lookup table
		xor CH,CH			;CL to CX
		lodsb				;Get char
		test AL,AL			;Is ASCII?
		js expchr02			;No
		mov AH,AL			;Save it
		cmp AL,'\'			;Escape sequence?
		jne expchr01			;No
		lodsb				;Get char
		test AL,AL			;Is ASCII?
		js expchr02			;No
		mov AH,AL			;Save it
		or AL,NOMAGIC			;Flag escaped
expchr01:	repne scasb			;Set ZF for match
		mov AL,AH			;Restore char
expchr02:	pop DI				;Restore caller's
		ret

expchr		endp


istype		proc

; Is this a defined type?

		cmp AL,':'			;Type prefix?
		jne istype01			;No
		mov AL,[SI]			;Type id
		mov CL,CTYPES			;Byte count
		push DI 			;Save caller's
		mov DI,OFS typeid		;Type ids
		repne scasb			;Set ZF for match
		pop DI				;Restore caller's
		mov AL,':'			;Restore char
istype01:	ret

istype		endp


setall		proc

; Set all bits

		mov AX,0FF00H			;Full range

setall		endp


setchr		proc

; Set a range of bits

		mov CL,AH			;Hi
		sub CL,AL			;Lo
		inc CX				;Bit count

setchr		endp


setbit		proc

; Set bits of set

		xchg BL,AL			;Get set marker
setbit01:	or [BX],AL			;Set
		inc BL				; CX
		loop setbit01			; bits
		xchg BL,AL			;Put set marker
		ret

setbit		endp


strtob		proc

; Convert string to byte

		push DX 			;Save
		push BX 			; caller's
		mov BX,SI			;First digit pointer
		mov DX,0A00H			;Radix, result
strtob01:	lodsb				;Get char
		sub AL,'0'			;Strip for digit
		jb strtob02			;Invalid
		cmp AL,'9' - '0'		;Digit?
		ja strtob02			;Yes
		xchg DL,AL			;Swap result, unit
		mul DH				;To next power
		jc strtob03			;Overflow
		add DL,AL			;Merge unit
		jc strtob03			;Overflow
		jmp strtob01			;Continue
strtob02:	dec SI				;Unget char
		mov AX,SI			;Set ZF if
		sub AX,BX			; nothing valid
		mov AL,DL			;Result
strtob03:	pop BX				;Restore
		pop DX				; caller's
		ret

strtob		endp


output		proc

; Output results of search

		mov AL,byte ptr format		;Get format
		test AL,FMT_NAM 		;Write name of file?
		jz output02			;No
		mov DX,OFS xg_buf		;Filespec
		mov BX,namend			;Point to
		dec BX				; terminator
		test AL,not FMT_NAM		;Just name of file?
		jz output01			;Yes
		mov byte ptr [BX],':'		;Append ':'
		inc BX				; to filename
output01:	call output08			;Write name of file
		mov AL,byte ptr format		;Restore AL
output02:	test AL,FMT_VAL 		;Write number?
		jz output05			;No
		mov AX,[BP]			;Get
		mov CX,[BP+02H] 		; number
		push DI 			;Save next line
		mov DI,OFS ff_sub - 01H 	;Work buffer
		mov BX,000AH			;Radix
output03:	xor DX,DX			;Zero for divide
		xchg CX,AX			;MSW to AX
		div BX				;Divide by radix
		xchg CX,AX			;LSW to AX
		div BX				;Divide by radix
		add DL,'0'			;ASCII bias
		dec DI				;Put
		mov [DI],DL			; digit
		mov DX,CX			;Anything
		or DX,AX			; left?
		jnz output03			;Yes
		mov DX,DI			;Start of number
		pop DI				;Restore next line
		mov BX,OFS ff_sub - 01H 	;Past number
		cmp byte ptr format,AL		;Write line?
		jge output04			;No
		mov byte ptr [BX],':'		;Append ':'
		inc BX				; to number
output04:	call output08			;Write number
		mov AL,byte ptr format		;Restore AL
output05:	test AL,FMT_LIN 		;Write line?
		jz output07			;No
		mov DX,SI			;Line
		mov BX,DI			;Next line
		dec BX				;To LF
		dec BX				;To preceding
		cmp BX,DX			;Past start?
		jb output06			;Yes
		cmp byte ptr [BX],LF		;Another LF?
		jne output06			;No
		dec BX				;Exclude LF
output06:	inc BX				;Past last char
		call output08			;Write line
output07:	mov DX,OFS crlf 		;Write
		mov BX,OFS crlf + 02H		; newline
output08:	mov CX,BX			;Bytes to
		sub CX,DX			; write
		jna write01			;Nothing
		mov BX,STDOUT			;Handle

output		endp


write		proc

; Write to file

		mov AH,DOS_WRITE		;DOS: Write
		int DOS_INT			; to file
		jc write_err			;Error
		cmp AX,CX			;All written?
		jb write_chk			;No
write01:	ret

write		endp


; Handle write errors


write_chk:	mov BX,DX			;First char
		add BX,AX			; not written
		cmp byte ptr [BX],EOF		;End of file?
		je write01			;Yes

write_err:	mov byte ptr errdev,STDERR	;Use stderr
		mov AX,OFS e_write		;Error message
		jmp fatal			;Output and exit

; Handle read errors


read_err:	mov AX,OFS e_read		;Error message
		jmp fatal			;Output and exit


xg_ex		proc

; Execute compiled expression

		xor AX,AX			;Zero
		mov word ptr number,AX		; line
		mov word ptr number[02H],AX	; number
		mov word ptr count,AX		; and match
		mov word ptr count[02H],AX	; count
		mov eofile,AL			;Clear end of file
		mov SI,OFS rx_inp		;Start of line
		mov eodata,SI			;End of data
		mov BX,rx_ofs			;Literal offset
		mov DL,xg_opt[OPT_v]		;Option v
		mov DH,rx_chr			;Literal

; Read data

xg_ex01:	push DX 			;Literal:option v
		mov DX,eodata			;End of data
		mov CX,OFS rx_inp + BLKSIZ	;End of buffer
		sub CX,DX			;End of data
		jnz xg_ex04			;Buffer not full
		mov CX,DX			;End of data
		sub CX,SI			;Start of line
		cmp CX,LINSIZ - 1		;Line too long?
		ja xg_ex05			;Yes
xg_ex02:	mov DX,OFS rx_inp		;Input buffer
		mov DI,DX			;Line buffer
		sub DI,CX			; pointer
		push DI 			;New start of line
		shr CX,1			;Move
		jnc xg_ex03			; to
		movsb				; line
xg_ex03:	rep movsw			; buffer
		pop SI				;Start of line
		mov CX,BLKSIZ			;Input buffer size
xg_ex04:	push BX 			;Literal offset
		xor BX,BX			;Handle: stdin
		mov AH,DOS_READ 		;DOS: Read
		int DOS_INT			; from file
		pop BX				;Literal offset
		jc read_err			;If error
		add DX,AX			;End of data
		cmp AX,BLKSIZ			;Full block read?
		je xg_ex12			;Yes
		cmp DX,SI			;Nothing there?
		je xg_ex11			;Yes
		mov DI,DX			;Point to
		dec DI				; last byte
		cmp byte ptr [DI],EOF		;DOS end of file?
		je xg_ex07			;Yes
		test AX,AX			;Nothing read?
		jz xg_ex08			;Yes
		jmp short xg_ex12		;Not end of file

; Handle line too long

xg_ex05:	call cutline			;Cut line
		jmp xg_ex02			;Continue
xg_ex06:	call cutline			;Cut line
		jmp short xg_ex19		;Continue

; If end of file

xg_ex07:	cmp DI,SI			;Nothing there?
		je xg_ex10			;Yes
		dec DI				;New last byte
xg_ex08:	cmp byte ptr [DI],LF		;Ends with LF?
		je xg_ex09			;Yes
		inc DI				;Append
		mov byte ptr [DI],LF		; LF
xg_ex09:	inc DI				;End of data
xg_ex10:	mov DX,DI			;End of data
xg_ex11:	inc eofile			;Flag end of file

; Set end of data

xg_ex12:	mov eodata,DX			;End of data
		pop DX				;Literal:option v
		mov DI,SI			;Start of line
		jmp short xg_ex17		;Continue

; Out of data

xg_ex13:	cmp eofile,CH			;End of file?
		je xg_ex01			;No

; Done

		cmp byte ptr format,CH		;Line output?
		jnl xg_ex15			;No
xg_ex14:	ret

; Count/list ouput

xg_ex15:	mov BP,OFS count		;Count
		test byte ptr format,FMT_VAL	;Output count?
		jnz xg_ex16			;Yes
		mov AX,[BP]			;Anything
		or AX,[BP+02H]			; found?
		jz xg_ex14			;No
xg_ex16:	jmp output			;Output results

; Next line

xg_ex17:	mov SI,DI			;Start of line
		mov CX,eodata			;End of data
		sub CX,DI			;Start of line
		jz xg_ex13			;Nothing there
		mov AL,LF			;Look for
		repne scasb			; LF
		jne xg_ex13			;Not found
		mov CX,DI			;Next line
		sub CX,SI			;Start of line
		dec CX				;Exclude LF
		jz xg_ex19			;Nothing else
		cmp byte ptr [DI-02H],CR	;Is CR before LF?
		jne xg_ex18			;No
		mov [DI-02H],AL 		;Replace with LF
		dec CX				;Exclude LF
xg_ex18:	cmp CX,LINSIZ - 1		;Line too long?
		jnb xg_ex06			;Yes

; Apply expression to line

xg_ex19:	inc CX				;Include LF
		push DI 			;Next line
		push SI 			;This line
		sub CX,rx_adj			;Derive steps
		jna xg_ex33			;Line too short
		mov AL,DH			;Literal
		cmp AL,CR			;BOL specified?
		je xg_ex26			;Yes
xg_ex20:	cmp AL,LF			;Literal?
		je xg_ex21			;No
		lea DI,[SI+BX]			;Line to DI
		repne scasb			;Scan for literal
		jne xg_ex34			;No match
		inc CX				;Back
		lea SI,[DI-1]			;Line to
		sub SI,BX			; SI

; Match set up

xg_ex21:	push SI 			;Line pointer
		push CX 			;Steps
		mov BP,SP			;Keep place
		mov DI,OFS rx_ctl		;Control data
		mov BX,OFS rx_set		;Set
		mov AH,01H			; marker

; Match compiled expression against line

xg_ex22:	mov CX,[DI]			;Maximum reps
		jcxz xg_ex28			;End of expression
xg_ex23:	lodsb				;Get char
		xlat				;To set pattern
		test AL,AH			;In set?
		jz xg_ex24			;No
		loop xg_ex23			;For maximum reps
		inc SI				;Keep last match
xg_ex24:	dec SI				;Back up
		sub CX,[DI+02H] 		;Get optional reps
		ja xg_ex31			;Required not done
		neg CX				;Optional reps
		rol AH,1			;Bump set
		adc BH,BL			; marker
		add DI,+04H			;To next
		jcxz xg_ex22			;No optional reps
xg_ex25:	push DI 			;Control pointer
		push BX 			;Set
		push AX 			; marker
		push SI 			;Line pointer
		push CX 			;Optional reps
		jmp xg_ex22			;Up a level

; Handle BOL operator

xg_ex26:	mov CX,0001			;One step
		jmp xg_ex21			;Continue

; Match failed

xg_ex27:	pop CX				;Optional reps
		pop SI				;Line pointer
		pop AX				;Set
		pop BX				; marker
		pop DI				;Control pointer
		dec SI				;Undo one
		dec CX				; rep
		jnz xg_ex25			;Still optional reps
		jmp xg_ex22			;Continue

; Matched succeeded

xg_ex28:	mov SP,BP			;Clear stack
		cmp rx_eol,CL			;EOL operator?
		jnz xg_ex30			;Yes
xg_ex29:	inc CX				;Set steps
		inc SP				;Clear CX
		inc SP				; from stack
		jmp short xg_ex32		;Done

; Handle EOL operator

xg_ex30:	cmp byte ptr [SI],LF		;End of line?
		je xg_ex29			;Yes

; Match clean up

xg_ex31:	cmp SP,BP			;Done?
		jb xg_ex27			;Down a level
		stc				;Flag not found
		pop CX				;Steps
xg_ex32:	pop SI				;Line pointer
		mov BX,rx_ofs			;Literal offset
		mov AL,DH			;Literal

; Deal with result

		jnc xg_ex34			;If found
		inc SI				;Step till found
		loop xg_ex20			; or done
xg_ex33:	xor CX,CX			;Flag no match
xg_ex34:	pop SI				;This line
		pop DI				;Next line
		inc word ptr number		;Line number LSW
		jz xg_ex39			;Wrapped
xg_ex35:	xor CL,DL			;Match?
		jnz xg_ex36			;Yes
		jmp xg_ex17			;Continue

; Match succeeded

xg_ex36:	and status,not RET_FAIL 	;Flag match found
		cmp byte ptr format,CH		;Line output?
		jnl xg_ex37			;No
		push DI 			;Next line
		push DX 			;Literal:option v
		push BX 			;Pointer adjustment
		mov BP,OFS number		;Line number
		call output			;Output line
		pop BX				;Pointer adjustment
		pop DX				;Literal:option v
		pop DI				;Next line
		jmp xg_ex17			;Continue

; Increment match count

xg_ex37:	inc word ptr count		;Bump LSW
		jnz xg_ex38			;No wrap
		inc word ptr count[02H] 	;Bump MSW
xg_ex38:	jmp xg_ex17			;Continue

; Handle wrapped line number

xg_ex39:	inc word ptr number[02H]	;Bump MSW
		jmp xg_ex35			;Continue


xg_ex		endp


cutline 	proc

; Cut line too long

		add SI,CX			;Plus old length
		push DX 			;Save
		push AX 			; caller's
		mov AX,CX			;Divide
		xor DX,DX			; length
		mov CX,LINSIZ - 1		; by maximum
		div CX				; line size
		mov CX,DX			;Remainder
		pop AX				;Restore
		pop DX				; caller's
		sub SI,CX			;Less new length
		ret

cutline 	endp


_TEXT		ends
		end start
