{ WC.PAS : Word Count - Counts lines, words and characters in textfile,
                        optionally counts long words and sentences and
                        computes "Fog-index" for text readability.

  Title   : WC
  Version : 2.4
  Date    : Feb 20, 2000
  Author  : J R Ferguson
  Language: Turbo Pascal v5.0
  Usage   : Refer procedure Help
  Download: http://hello.to/ferguson
            To compile this sourcefile you will need units from the Pascal
            library JRFPAS that can be found on the same Internet site.
  E-mail  : j.r.ferguson@iname.com
}

{$V- : Relaxed string var type checking }
{$B- : Short-circuit boolean expression evaluation }

{$DEFINE ASMLIB} { If defined: use assembler routine library }


program WC;


uses DefLib, ArgLib, StpLib, ChrLib;


const
  MAXFNM    = 79;           { Max filespec length (integrated environment) }
  DFLMSG    = 'CON';        { Default message output destination }
  INPBUFSIZ = 16 * 1024;    { Input buffer size in bytes }
  DFLWTHR   = 9;            { Default long word threshold }
  SentSep:StpTyp = '.?!;:';
  WordSep:StpTyp = ' ,.?!;:"'''#009#010#011#012#013#160#141;
  Hyphens:StpTyp = '-'#030#031;
  SkipSep:StpTyp = ' ,.?!;:"'''#009#010#011#012#013#160#141'-'#030#031;

  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRFNF    = 2;
  ERRCRE    = 3;
  ERRWRI    = 4;

  ERRMSG    : array[ERRFNF..ERRWRI] of StpTyp =
 ('File not found',
  'File creation error',
  'Write error'
 );

type
  InpBufTyp = array[1..INPBUFSIZ] of char; InpBufPtr = ^InpBufTyp;

var
  ErrCod    : integer;
  InpFnm,
  OutFnm    : StpTyp;
  Msg,
  Inp,
  Out       : Text;
  InpBuf    : InpBufPtr;
  InpOpn,
  OutOpn    : boolean;
  OptHlp    : boolean;
  OptFog    : boolean;   { Compute Fog-index }
  LwrdThr   : StpInd;    { Long word threshold }
  CurLine   : StpTyp;    { Current text line }
  CurChar   : char;      { Current character }
  LineInd   : StpInd;    { Current line index }
  WordLen   : StpInd;    { Current word length }
  Wcnt,                  { Word count }
  Xcnt,                  { Long word count }
  Scnt,                  { Sentence count }
  Lcnt,                  { Line count }
  Ccnt      : Longint;   { Character count }


{--- General routines ---}


procedure Help;
  procedure wl(s: StpTyp );  begin writeln(Msg,s) end;
  procedure wr(s: StpTyp );  begin write  (Msg,s) end;
  procedure wi(i: integer);  begin write  (Msg,i) end;
begin
wl('WC v2.4 - Word Count : Counts lines, words and characters in textfile,');
wl('                       optionally counts long words and sentences and');
wl('                       computes "fog-index" for text readability.');
wl('');
wl('Usage  : WC [[<]in] [>out] [/option[...] [...]]');
wl('');
wl('Options: H     Send this help text to (redirected) output.');
wl('         F[n]  Count sentences and words with length>n, compute fog-index.');
wr('               n=1..');wi(MaxStp);wr(', default ');wi(DFLWTHR);wl('.');
wl('Remarks:');
wl('- Wordstar document files are handled correctly.');
wl('- Hyphens are recognized and do not count to word length.');
wl('- Character count does not include line separators (CR,LF)');
wl('- Line count includes empty lines.');
wl('- Counters can hold a maximum value of 2147483647. No overflow check.');
wl('- Fog index (Robert Gunning,1952) is computed as 0.4 * (W/S + L*100/W)');
wl('  where W = word count');
wl('        S = sentence count');
wl('        L = long word count (Gunning: length >9).');
wl('  Long sentences and long words will increase the fog index.');
end;


function FogIndex(W,S,L: LongInt): real;
begin FogIndex:= 0.4 * (W / S + L*100.0 / W) end;


{--- Command line parsing routines ---}


function ReadUns(var arg: StpTyp): integer;
var c: char; n: integer;
begin
  n:= 0; c:= StpcRet(arg,1);
  while IsDigit(c) do begin
    n:= 10*n + (ord(c) - ord('0'));
    StpDel(arg,1,1);
    c:= StpcRet(arg,1);
  end;
  ReadUns:= n;
end;


procedure ReadOpt(var arg: StpTyp);
var nextopt: boolean; n: integer;
begin
  StpDel(arg,1,1);
  repeat
    if StpEmpty(arg) or (StpcRet(arg,1) = '/') then ErrCod:= ERRARG
    else begin
      nextopt:= false;
      while (ErrCod=ERROK) and not nextopt and not StpEmpty(arg) do
      case StpcGet(arg) of
        'H': OptHlp:= true;
        'F': begin
               OptFog:= true;
               if IsDigit(StpcRet(arg,1)) then begin
                 n:= ReadUns(arg);
                 if (n<1) or (n>MaxStp) then ErrCod:= ERRARG
                 else LwrdThr:= n;
               end; 
             end;
        '/': nextopt:= true;
        else ErrCod:= ERRARG;
      end;
    end;
  until (ErrCod <> ERROK) or not nextopt;
end;


procedure ReadArgs;
var i   : ArgInd;
    arg : StpTyp;
begin
  GetArgs; i:= 0;
  while (ErrCod = ERROK) and (i < ArgC) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    case StpcRet(arg,1) of
      '/' : ReadOpt(arg);

      {simulate MS-DOS redirectioning in Integrated Environment}
      '<' : if not StpEmpty(InpFnm) then ErrCod:= ERRARG
            else StpSub(InpFnm,arg,2,MAXFNM);
      '>' : StpSub(OutFnm,arg,2,MAXFNM);

      else  if not StpEmpty(InpFnm) then ErrCod:= ERRARG
            else StpSub(InpFnm,arg,1,MAXFNM);
    end;
  end;
  if OptHlp then if ErrCod = ERROK then ErrCod:= ERRARG else OptHlp:= false;
end;


{--- Low-level I/O routines ---}


procedure OpenMsg;
begin
  if OptHlp then Assign(Msg,OutFnm) else Assign(Msg,DFLMSG);
  rewrite(Msg);
end;


procedure CloseMsg;
begin Close(Msg) end;


procedure OpenInp;
begin
  Assign(Inp,InpFnm); new(InpBuf); SetTextBuf(Inp,InpBuf^);
  {$I-} reset(Inp); {$I+}
  if IOresult <> 0 then ErrCod:= ERRFNF else InpOpn:= true;
end;


procedure CloseInp;
begin
  dispose(InpBuf);
  {$I-} Close(Inp); {$I+}
  if IOresult = 0 then InpOpn:= false;
end;


procedure OpenOut;
begin
  Assign(Out,OutFnm); {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then ErrCod:= ERRCRE else OutOpn:= true;
end;


procedure CloseOut;
begin
  {$I-} Close(Out); {$I+}
  if IOresult = 0 then OutOpn:= false else begin
    if ErrCod = ERROK then ErrCod:= ERRWRI;
  end
end;


{--- Main processing routines ---}


{$IFDEF ASMLIB}

{$L WCLIB}
function  InpEof: boolean;  begin InpEof:= Eof(Inp) end;
procedure InpRead;          begin Readln(Inp,CurLine) end;

function GetWord: boolean; external;

{$ELSE}

function GetChar: boolean;
begin
  if LineInd < Length(CurLine) then begin
    Inc(LineInd); CurChar:= CurLine[LineInd]; GetChar:= true;
  end
  else begin
    if eof(Inp) then begin CurChar:= '.'; GetChar:= false; end
    else begin
      ReadLn(Inp,CurLine);
      Inc(Lcnt); Inc(Ccnt,Length(CurLine)); LineInd:= 0;
      if StpcPos(Hyphens,CurChar) = 0 then CurChar:= ' ';
      GetChar:= true;
    end;
  end;
end;


function GetWord: boolean;
  var InWord: boolean;
begin
  { skip to next word }
  InWord:= StpcPos(SkipSep,CurChar) = 0;
  while not InWord and GetChar do
    InWord:= StpcPos(SkipSep,CurChar) = 0;

  { skip word, counting length }
  if InWord then begin
    Inc(Wcnt); WordLen:= 1;
    while InWord and GetChar do begin
      InWord:= StpcPos(WordSep,CurChar) = 0;
      if InWord and (StpcPos(Hyphens,CurChar) = 0) then Inc(WordLen);
    end;
    while (CurChar = ' ') and GetChar do {nothing};
    if (StpcPos(SentSep,CurChar) <> 0) then Inc(Scnt);
    GetWord:= true;
  end
  else begin
    WordLen:= 0;
    GetWord:= false;
  end;
end;

{$ENDIF}


procedure MainProcess;
begin
  Wcnt:= 0; Xcnt:= 0; Scnt:= 0; Lcnt:= 0; Ccnt:= 0;
  CurChar:= ' '; CurLine:= ''; LineInd:= 0;
  if OptFog then begin
    while GetWord do
      if WordLen > LwrdThr then Inc(Xcnt);
  end
  else
    while GetWord
    do {nothing};

  WriteLn(Out,Lcnt:10,' line(s)');
  WriteLn(Out,Wcnt:10,' word(s)');
  WriteLn(Out,Ccnt:10,' char(s)');
  if OptFog then begin
    WriteLn(Out,Xcnt:10,' word(s) >',LwrdThr,' chars');
    WriteLn(Out,Scnt:10,' sentence(s)');
    if (Wcnt<>0) and (Scnt<>0) then
      Writeln('Fog Index = ',FogIndex(Wcnt,Scnt,Xcnt):10:2);
  end;
end;


{--- Main line ---}


procedure MainInit;
begin
  ErrCod:= ERROK; OptHlp:= false; OptFog:= false; LwrdThr:= DFLWTHR;
  StpCreate(InpFnm); InpOpn:= false;
  StpCreate(OutFnm); OutOpn:= false;
  ReadArgs;
  if ErrCod = ERROK then OpenInp;
  if ErrCod = ERROK then OpenOut;
end;


procedure MainTerm;
begin
  if InpOpn then CloseInp;
  if OutOpn then CloseOut;
  if ErrCod <> ERROK then begin
    OpenMsg;
    if (ErrCod=ERRARG) then Help
    else writeln(Msg,'WC: ',ERRMSG[ErrCod]);
    CloseMsg;
  end;
end;


begin { Main program }
  MainInit;
  if ErrCod = ERROK then MainProcess;
  MainTerm;
end.
