#
# Note: This monolithic script approach was used to make moving and using the
#       script as easy as possible.  Everything needed is in the source code
#       file TekDoc.pl.  The documentation may be generated into the directory
#       .\Doc by using:  perl TekDoc.pl -d Doc -f -l 1 TekDoc.pl
#       The documentation contains an example configuration file that may be
#       cut and pasted to create a new one.
#
#------------------------------------------------------------------------------
#
#`Node  Index/License
#`Title License
# There is no license.
#
# This script is hereby released into the PUBLIC DOMAIN.  Any use, modification,
# derivative work or other use of this script is EXPRESSLY GRANTED by the author.
# NO warrantee, guarantee, or fitness for any particular purpose is expressed or
# implied.  If you break it, you own both parts.
#
# This script was written because we needed the functionality, not as a Perl
# programming exercise.  It is not necessarily the most elegant or efficient
# way to implement this functionality, but it works.
#
# Written at:
#    GrayHill Systems, Inc.
#    by Todd R. Hill, todd.hill@grayhillsys.com
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node  Index
#`Title TekDoc HTML Documentation Generator
#
# The TekDoc utility extracts text from files to generate HTML documentation.
# An alphabetical index and hierarchical document map are also generated. A
# variety of options are available for controlling specific behaviors.
#
# The script uses a simple tag extraction approach for extracting and formatting
# text into HTML documentation.  It does not parse any specific programming
# language syntax.  Therefore, any text input file may be processed such as
# text files and C, C++, Perl or Java source code.
#
#`Index Usage
# USAGE
#   perl TekDoc.pl [-option(s)] pattern1 pattern2 ...
#
# The patterns may be directories, specific files, or wildcard patterns.  Each
# is processed and the results are combined as specified by the Node tags
# within the input files.  All files processed are combined into the alphabetical
# index and document map.
#
# Both
#`Tie Command Line Options
# and
#`Tie Configuration File Variables
# are used to control the
# behavior of the script.
#
# SEQUENCE
# The general process is:<BR>
# 1. Edit the configuration file to set default values for those variables that
# do not change often;<BR>
# 2. Place the various tags in your source text files;<BR>
# 3. Then run the script with the command line options set as needed.
#
# Formatting tags control the generation of the HTML documentation.  See
#`Tie Formatting Tags
# or
#`Tie Formatting Tags/Tag Summary
# for details.
#
# RETURNS
#  0 on success, 1 on warning(s), 2 on failure
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node  Command Line Options
#`Title Command Line Options
#`Index Order of precedence
# The order of precedence for controlling processing is:<BR>
# 1. Tags in the source text files;<BR>
# 2. Command line options;<BR>
# 3. Variables set in the configuration file.<BR>
#
# Not all configuration file variables have matching command line options.
# Option letters are case sensitive.
#
#`Table Syntax & Description
#`Row   --
#`Column            End of options.  Useful when the last option specified has
#                   optional data that has not been specified.  Use to separate
#                   the option from the first source pattern.
#
#                   Example:    perl TekDoc.pl -f -- file1.txt
#
#`Row   -b [0|1]
#`Column            Enable Bold Uppercase rule.
#                   The rule says to bold and break on any line from the source that
#                   is all uppercase letters or special characters (i.e. does
#                   not contain any lowercase letters, but contains at least one
#                   alpha-numeric character).  This is useful for
#                   highlighting paragraph headings.
#                   Option -b with no data enables the option (= 1).
#
#                   Default:    1 (enabled)
#
#                   Example:    -b 0
#
#                   See also:   BoldUpper variable
#
#`Row   -c cfg-name
#`Column            Set configuration filename.
#
#                   Default:    Name of the script with .cfg extension.
#
#                   Example:    -c MyProj.cfg
#
#`Row   -d directory
#`Column            Set documentation output directory.
#                   Can be relative or absolute path specification.
#                   Will be created if necessary.
#
#                   Default:    Doc, relative to current directory.
#
#                   Example:    -d Doc\HTML
#
#                   See also:   Directory variable
#
#`Row   -e "list"
#`Column            Set list of extensions.
#                   Separated by commas.
#                   Surrounded in double-quotes if more than one extension.
#
#                   Default:    c, cpp, cxx, h, hpp, hxx, pl, jav, java, txt
#
#                   Example:    -e "c,cpp,h,hpp"
#
#                   See also:   Extensions variable
#
#`Row   -f [0|1]
#`Column            Include source filename at top of node in HTML output file.
#                   Useful for identifying the source of the documentation.
#                   Option -f with no data enables the option (= 1).
#
#                   Default:    0, do not include.
#
#                   Example:    -f 0
#
#                   See also:   FileIncluded variable
#
#`Row   -k [0|1]
#`Column            Kill single-line comment characters.
#                   Removes known single-line comment character strings from the
#                   beginning of lines.  The comment strings are determined by
#                   association to the file extension. The known associations
#                   are for the default file extensions (see -e).
#
#                   Default:    1, kill comment characters.
#
#                   Example:    -k 0
#
#                   See also:   KillComments variable
#
#`Row   -l [0|1|2|3]
#`Column            Link mode. The following modes are available to control the
#                   way links are automatically generated for the bottom of
#                   applicable pages.  Forward links are formatted in their
#                   own paragraph with a bulleted list.  Backward links are
#                   generated below any forward links.
#
#                   0 = No links are added.
#
#                   1 = Include links ONLY for defined sub-nodes (e.g.
#                   ``Node My Node/My Node 2).  Also include a backward link
#                   to the parent node of the sub-node.  The node "Index" is
#                   not placed in the lists.  The assumption is Index is the
#                   same as the "Home" link in the navigation bar on the left.
#
#                   2 = Include links for all sub-nodes generated from the same
#                   source file.  No backward link is generated because all
#                   sub-nodes are listed.
#
#                   3 = Include links for all pages generated in this run,
#                   regardless of the source file.
#
#                   Default:    1
#
#                   Example:    -l 3
#
#                   See also:   LinkMode variable
#
#`Row   -o extension
#`Column            Output extension, with no dot.  Specifies the output file(s)
#                   extension for all output files and the links on the
#                   navigation bar on the left side of the generated pages.
#
#                   Default:    html
#
#                   Example:    -o htm
#
#                   See also:   OutExtension variable
#
#`Row   -r [0|1]
#`Column            Perform recursive processing on all subdirectories specified
#                   and encountered.  Option -r with no data enables the option (= 1).
#
#                   Default:    0, not recursive.
#
#                   Example:    -r
#
#                   See also:   Recursive variable
#
#`Row   -S [0-254]
#`Column            Define what is "Secure".  Any entries marked with a
#                   ``Security n value greater than or equal to this are marked
#                   with a block on the left side showing the security level to
#                   let readers know they are looking at secure information.  See
#                   -s below.
#
#                   Default:    1, to separate 0 and 1 levels.
#
#                   Example:    -S 3
#
#                   See also:   Secure variable
#
#`Row   -s [0-254]
#`Column            Set security level to be processed, inclusive.  See -S above.
#                   Any entry with a ``Security n value greater than this
#                   is not processed.
#
#                   Default:    0, only process first level.
#
#                   Example:    -s 5
#
#                   See also:   Security variable
#
#`EndTable
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node  Configuration File Variables
#`Title Configuration File Variables
# Settings made in the configuration file are overridden by any matching command
# line options.  Only those variables that can be specified on the command line
# are overridden, all other variables in the configuration file are used.  All
# variables have default values or behaviors if not specified at all (commented
# out).
#
# If the configuration file is not available, only the command line options are
# used.  Any variables that are not available as a command line parameter (such
# as Footer) use their default value.
#
# The default configuration filename is the name of the script with a .cfg
# extension.  If the script is renamed the default configuration filename will
# change as well.  A different configuration filename may be specified on the
# command line with the -c option.
#
#`Table Process Variables
#`Row   boldUpper = [0|1]
#`Column            See -b. Bold Uppercase rule.
#
#`Row   directory = [path]
#`Column            See -d. Documentation output directory.
#
#`Row   extensions = [list]
#`Column            See -e. Extensions to process. No quotes.
#
#`Row   fileIncluded = [0|1]
#`Column            See -f. Include source filename in output.
#
#`Row   footer = string
#`Column            Set default footer for the bottom of HTML pages.
#                   Used if the ``Footer tag is not found in a file.
#                   No default, and no command line equivalent.
#
#`Row   killComments = [0|1]
#`Column            See -k. Kill single-line comment character strings.
#
#`Row   linkMode = [0|1|2|3]
#`Column            See -l. Link mode.
#
#`Row   outExtension = extension
#`Column            See -o. Output file(s) extension.
#
#`Row   recursive = [0|1]
#`Column            See -r. Process directories recursively.
#
#`Row   security = [0-254]
#`Column            See -s. Default security level to process.
#
#`Row   secure = [0-254]
#`Column            See -S. Definition of "Secure Level".
#`EndTable
#
#`Table Mail and Logging Variables
#`Row   mailLevel = [0|1|2|3]
#`Column            Sets the level for when e-mail notification should be
#                   sent from the script.  Useful for fully automated processes.
#
#                   0 = Never send mail (other mail options ignored).<BR>
#                   1 = Send on error.<BR>
#                   2 = Send on error or warning.<BR>
#                   3 = Send always.
#
#                   Default: 2
#
#`Row   mailUser = [profile]
#`Column            Defines the e-mail profile name to be used to send a message.
#                   For Windows, this is the name of a profile from Control Panel,
#                   Mail.
#
#                   Default: none
#
#`Row   mailRecipient = [account]
#`Column            Defines the e-mail account to receive automated messages.
#
#                   Default: none
#
#`Row   log = [filename]
#`Column            The filename to use for logging.  The default is the name
#                   of the script with a .log extension.  If set with no value
#                   (e.g. log = ) the log is disabled entirely.
#
#                   Default: none
#
#`Row   logOverwrite = [0|1]
#`Column            Determines whether the log file should be overwritten (1),
#                   or appended (0).  Especially useful if mailLevel = 3.
#
#                   Default: 0
#
#`Row   writeNTlog = [0|1]
#`Column            Determines whether completion status entries are written
#                   to the Windows NT system log.  This feature only works
#                   on Windows NT and is ignored on other operating systems.
#
#                   Default: 0
#`EndTable
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node Configuration File Variables/Example
#`Title Configuration file example
# This is an example configuration file.  It is setup to use all the default
# values for the options.  This example may be cut & pasted to create a new
# configuration file.
#
#`Code
# #
# # TekDoc.cfg
# #
# # Used by TekDoc.pl to set process and mail variables.
# # Comments begin with a pound sign "#".
# #############################################################
#
# # Set the default output directory.
# # Overridden by the -d command line option.
# # May be relative or absolute path specification.
# # Will be created if necessary.
# # Default: Doc
# #directory = HTML
#
# # Set the default list of file extensions to be processed.
# # Overridden by the -e command line option.
# # No quotes, no periods.
# # Comma separated.
# # Default: c, cpp, cxx, h, hpp, hxx, pl, jav, java, txt
# #extensions = cpp, hpp, pl, java
#
# # Set recursive operation.
# # If enabled will recurse through all subdirectories of any source directory.
# # Overridden by the -r command line option.
# # Does not recurse any subdirectories if only specific source files are
# # specified on the command line, but no directories are specified.
# # Default: 0 (disabled)
# #recursive = 1
#
# # Set whether the source file name is included in the HTML output.
# # Overridden by the -f command line option.
# # Will insert the name of the source file for each output node.
# # Default: 0 (disabled)
# #fileIncluded = 1
#
# # Set the Bold Uppercase rule.
# # The rule says to bold any line from the source that
# # is all uppercase letters or special characters (i.e. does
# # not contain any lowercase letters, but does contain at least
# # one alpha-numeric character).  Useful for highlighting paragraph headings.
# # Default: 1 (enabled)
# #boldUpper = 1
#
# # Set the Kill Comments option.
# # If enabled, this option will strip leading single-line comments
# # from source file lines.  The comment character(s) are determined
# # by filename extension association for known extensions (c, cpp, etc.).
# # Default: 1 (enabled)
# #killComments = 0
#
# # Set default page footer.
# # Used if a footer tag is not specified in a source file.
# # Some useful special character entity names (without the space after the &):
# #     & copy;  Insert the (C) copyright symbol,
# #     & reg;   Insert the (R) registered trademark symbol.
# # Default: <none>
# #footer =
#
# # Set the Link Mode
# # The following modes are available to control the formatting
# # of links added at the bottom of applicable pages.
# # Modes
# #  0 = No links are added.
# #  1 = Include links at the bottom of pages ONLY for defined
# #      sub-nodes (e.g. ``Node My Node/My Node 2) in their own paragraph
# #      and bulleted list titled RELATED LINKS.
# #      Also include a link to the parent node of a sub-node in
# #      it's own paragraph and bulleted list titled BACK placed below
# #      any RELATED LINKS paragraph.
# #      The node "Index" is never listed in the RELATED LINKS or BACK paragraphs.
# #      The assumption is Index is the same as the "Home" link in the navigation
# #      bar on the left.
# #  2 = Include links for all sub-nodes generated from the same source file.
# #      No BACK paragraph is generated because all sub-nodes are listed.
# #  3 = Include links for all pages generated in this run, regardless of the
# #      source file.
# # Default: 1
# #linkMode = 3
#
# # Set the security level of nodes to be processed.
# # Any node designated with a ``Security n value greater than this default
# # will not be processed.
# # Overridden by the -s command line option.
# # Default: 0
# #security = 100
#
# # Set the definition of what security level is "secure".
# # Any nodes processed with a ``Security n value equal to or greater than
# # this value are marked in the HTML page as being secure.
# # See the -s and -S command line options.
# # Overridden by the -S command line option.
# # Default: 1
# #secure = 2
#
# # Set the mail level.
# # Sends a message from mailUser to mailRecipient with
# # the current log file attached.
# #  0 = never send mail (other mail options ignored)
# #  1 = send on error
# #  2 = send on error or warning
# #  3 = send always
# # Default: 2
# mailLevel = 0
#
# # Set the mail profile the message is sent from.
# # This is the name of the profile in Control Panel, Mail
# # to be used to send the message.
# # Default: no default
# #mailUser = Todd Hill
#
# # Set the mail recipient to receive messages.
# # May be multiple recipients separated by semicolons,
# # e.g. Buckaroo Banzai; Bill Gates, but be careful of
# # overrunning the 127 character command line restriction.
# # Default: no default
# #mailRecipient = Todd Hill
#
# # Set the log path and filename.
# # If set as "log =" with no value the log is disabled.
# # If commented out the default is used.
# # Default: Configuration filename with .log extension
# #log =
#
# # Overwrite log file flag.
# # Handy if mailLevel 3 is used.
# #  0 = False
# #  1 = True
# # Default: 0
# logOverwrite = 1
#
# # Write to the Windows NT event log flag.
# #  0 = False
# #  1 = True
# # Default: 0
# #WriteNTlog = 0
#`EndCode
#`End
#
#------------------------------------------------------------------------------
#
#`Node  100 Formatting Tags
#`Title Formatting Tags
# Formatting tags are placed in the source text files to control HTML page
# generation. Some tags are required for this script to recognize what content
# is to be extracted, and to identify those sections by name.
#
# Tags may be placed anywhere on a line, but all characters appearing before
# the tag are discarded.  So, it is generally best to place tags at the beginning
# of a line.  Tags are not case sensitive.  To place a tag name in the source
# text two tag trigger characters (``) are placed in a row before the tag word
# (e.g. ````Node).
#
# The following tags are available:
#
# ``Node and ``End mark the content to be extracted, with optional sequencing
# and sub-pages.
#
# ``Title gives a node a one line description used in the alphabetical index.
#
# ``Footer defines footer text for the node(s) in a source file.
#
# ``Security defines the security level of nodes(s) in a source file.
#
# ``Table, ``Row, ``Column and ``EndTable define the name and data for tables.
#
# ``Code and ``EndCode denote blocks of highlighted literal text.
#
# ``Tie and ``Link provide for hyperlinks.
#
# ``Index creates additional entries for the alphabetical index.
#
# See the
#`Tie Formatting Tags/Tag Summary
# page for a quick reference, or the detailed descriptions below.
#
#`Node  110 Formatting Tags
#`Index Node tag
# ``NODE, ``END TAGS
#
# A Node is similar in concept to an output filename, with the addition of a
# sub-page linkage hierarchy specification.  Node names may be any text, with
# whitespace, with the exception of 1) characters excluded by the local operating
# system for filenames, or 2) a number (see the node sequencing discussion below).
#
# The ``Node tag begins general text extraction, and must appear before other tags.
# At least one named node must be defined for the first node of each source text
# file.
#
# The ``End tag ends general text extraction.
#
#`Index Sub-page linkage
# Sub-page linkage is specified by placing a forward slash (/) between node names.
# Depending on the -l or linkMode option, forward and backward links may be
# generated at the bottom of output pages in a variety of ways.
#
# Different node names made be interspersed in the same or different input files.
# Nodes with the same name are placed in the same output file, and separated by
# an HTML horizontal rule (line).  Unnamed nodes are placed in the most recent
# named node's output file, without a horizontal rule.
#
#`Index Node sequencing
# Nodes may optionally include a sequence number to allow the user to control
# the order that nodes are placed in an output file.
#`Index Node naming rule
# This points out the node naming rule: if the first whitespace-delimited word
# of a node name is all numeric, it is taken as a sequence number and removed.
# Node sequence numbers may be 1 to 99,999.
#
# Node names are used in the generated forward and backward links at the bottom
# of output pages, in the document map, and in Tie tags.
#
# Bear in mind that node names are used as output filenames.  The total length
# of nodes + sub-nodes may not exceed the limit of the local operating system,
# e.g. Node One/Node Two/Node Three/Node Four/Node Five ... etc.
#
# A ``Node tag begins general text extraction.  The ``End tag ends extraction.
# See the Formatting Tags page for addition information.
#
# Output filenames are derived from node names by removing all whitespace and
# replacing any forward slashes (/) with underscores, e.g. "Index/Detailed
# Information" would become "Index_DetailedInformation".
#
# The output file extension is html by default.  The extension may be changed
# with the -o or outExtension options.
#
#`Index Home link on navigation bar
# * IMPORTANT Note *<BR>
# The "Home" link on the navigation bar on the left side of the generated pages
# is a hardcoded link to a page named "Index".  The script does not force the
# user to have a page named Index, but if one does not exist the "Home" link
# will not work.  This is not checked or enforced to allow the user to easily
# create an Index page with a different run of the script or a different HTML
# tool, without that file always being overwritten.
#
# SYNTAX
# ``Node [sequence] [name]
#
#`Tie Formatting Tags/Node tag examples
#
#`Node  Formatting Tags/Node tag examples
#`Title Node tag examples
#`Code
#
#``Node Index
#
#   Places the content following the tag in an output file named Index with
#   a filename extension of html or as specified by the -o or outExtension
#   options.
#
#
#``Node Index/Details
#
#   Places the content following the tag in an output page named Index_Details.
#   Depending on the -l or linkMode option, a link to the Details page may
#   be added to the bottom of the Index page.
#
#
#``Node 100 MyObject
#
#   Places the content in a page named MyObject, and assigns sequence
#   number 100 to the node.
#
#
#``Node 200 MyObject
#
#   Places the content after the 100 node content in a page named MyObject.
#
#
#``Node Index/Details/Example
#
#   Places the content in an output page named Index_Details_Example.
#   Depending on the -l or linkMode option, a link to the Example page may
#   be added to the bottom of the Details page, and a back link to the
#   Details page may be placed at the bottom of the Example page.
#`EndCode
#
#`Node  120 Formatting Tags
#`Index Title tag
# ``TITLE TAG
#
# Titles are simply strings used to describe a node's content.  The title of the
# first (possibly sequenced) node in an output file is used as the title of the
# HTML page.
#
# Titles are used in the alphabetical index page.  Nodes without titles are
# not listed in the index.  See the ``Index tag for another way to create index
# entries.
#
# SYNTAX
# ``Title title-text
#
#`Node  130 Formatting Tags
#`Index Footer tag
# ``FOOTER TAG
#
# A footer text line may be placed at the bottom of HTML pages.  If not specified
# the Footer variable in the configuration file will be used.  Any ``Footer
# tag in the source text overrides the Footer configuration variable. Handy for
# adding copyrights or other information at the bottom of pages.
#
# SYNTAX
# ``Footer footer-text
#
#`Node  140 Formatting Tags
#`Index Security tag
# ``SECURITY TAG
#
# Security is the simple notion that different information is intended for
# different audiences.  Some information may be for customers, while other
# information is for internal purposes only.  The security implementation has
# three parts.
#
# The first part is the definition of what security level denotes "secure"
# information.  This is defined using either the -S command line option, or the
# Secure configuration variable. Secure information extracted from source text
# is marked with a bar on the left side of the page showing it's security level.
#
# The second part is the definition of the security level for specific information.
# The ``Security tag is placed in the source text to set the security level of
# the node(s).  Security levels may be 0-254.
#
# The third part is the specification of the security level to be processed for
# a particular run of the script.  This is done with the -s command line option,
# or the Security configuration variable.  Only content marked with a ``Security
# tag level less than or equal to the process level will be extracted.
#
# SYNTAX
# ``Security [0-254]
#
#`Node  150 Formatting Tags
#`Index Table tag
#`Index Row tag
#`Index Column tag
#`Index EndTable tag
# ``TABLE, ``ROW, ``COLUMN, ``ENDTABLE TAGS
#
# Tables may be created using the ``Table, ``Row, ``Column and ``EndTable tags.
# A complete table specification looks like this:
#
#`Code
#
#``Table    My table name
#``Row      Row 1, column 1 data
#``Column   Row 1, column 2 data
#``Column ...
#
#``Row      Row 2, column 1 data
#``Column   Row 2, column 2 data
#``Column ...
#``EndTable
#`EndCode
#
# Table names are optional, and are bolded and left justified.  Row and Column
# data may be one or more lines of text.  Nested tables are not supported, but
# probably could be with some work.
#
# SYNTAX
# ``Table [table-name]<BR>
# ``Row row-data<BR>
# ``Column column-data<BR>
# ``EndTable
#
#`Node  160 Formatting Tags
#`Index Code tag
#`Index EndCode tag
# ``CODE, ``ENDCODE TAGS
#
# A code block is a section of content that will be highlighted in a black on
# white block.  Code block content is taken literally and not reformatted.
#
# SYNTAX
# ``Code<BR>
# ``EndCode
#
#`Node  170 Formatting Tags
#`Index Tie tag
# ``TIE TAG
#
# A Tie tag creates a link to a named node's output HTML page, and bookmark
# if available.  A Tie link is not generated if the named node exceeds
# the -s or Security option of this run of the script. In this case the node name
# is inserted as normal text in the output HTML page. The surrounding text is
# flowed with the ``Tie node-to-link text.  An optional sequence number may be
# used to tie to a specific sub-node within a page.
#
# SYNTAX
# ``Tie [sequence] node-to-link
#
#`Node  180 Formatting Tags
#`Index Link tag
# ``LINK TAG
#
# A Link tag creates an HTML link to any appropriate target.  Unlike ``Tie tags,
# a ``Link tag defines both the descriptive text and the HREF URL.  No checking
# is done to see that the referenced URL is valid.  The surrounding text is
# flowed with the ``Link description text.
#
# The description may be any text.  The URL may be any target the browser is
# capable of viewing or executing.
#
# SYNTAX
# ``Link description = URL
#
#`Node  190 Formatting Tags
#`Index Index tag
# ``INDEX TAG
#
# The Index tag creates an additional entry for the alphabetical index page.
# Any descriptive text may be used.  A bookmark is generated at the current
# location of the output HTML page.  The ``Index descriptive-text is ONLY
# used in the index page, and not included in the HTML output of the current
# node.
#
# SYNTAX
# ``Index descriptive-text
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node  Formatting Tags/Tag Summary
#`Title Tag Summary
#
#`Table
#`Row       SYNTAX
#`Column    REQUIRED?
#`Column    DESCRIPTION
#`Row       ``Node [sequence] [name]
#`Column    Required
#`Column    Begins general text extraction.  Must appear before other tags.
#           At least one named node must be defined for the first node of each
#           source text file.
#
#           Optional sequence number defines the ordering of nodes from multiple
#           locations into the HTML documentation.
#
#           Optional name defines a target output file for the content that
#           follows.  See ``NODE in
#`Tie 110 Formatting Tags
#`Row       ``End
#`Column    Required
#`Column    Denotes the end of general text extraction. See ``END in
#`Tie 110 Formatting Tags
#`Row       ``Title title-text
#`Column    Optional
#`Column    Creates an entry in the alphabetical index page with the Title text.
#           See ``TITLE in
#`Tie 120 Formatting Tags
#`Row       ``Footer footer-text
#`Column    Optional
#`Column    Defines a footer for all HTML pages to be generated from the
#           content extracted from that point on in that particular source file.
#           See ``FOOTER in
#`Tie 130 Formatting Tags
#`Row       ``Security security-level
#`Column    Optional
#`Column    Defines the security level (0-254) for the content that follows. See
#           ``SECURITY in
#`Tie 140 Formatting Tags
#`Row       ``Table [name]
#`Column    Optional
#`Column    Defines the beginning of a table sequence. See ``TABLE in
#`Tie 150 Formatting Tags
#`Row       ``Row row-data
#`Column    Required with ``Table
#`Column    Defines column 1 of a new row of data for the table. See ``ROW in
#`Tie 150 Formatting Tags
#`Row       ``Column column-data
#`Column    Optional
#`Column    Defines column 2 through n of the current row. See ``COLUMN in
#`Tie 150 Formatting Tags
#`Row       ``EndTable
#`Column    Required with ``Table
#`Column    Denotes the end of a table. See ``ENDTABLE in
#`Tie 150 Formatting Tags
#`Row       ``Code
#`Column    Optional
#`Column    Defines the beginning of a code block. Codes blocks are highlighted
#           and the text is literally extracted, not reformatted. See ``CODE in
#`Tie 160 Formatting Tags
#`Row       ``EndCode
#`Column    Required with ``Code
#`Column    Denotes the end of a code block. See ``ENDCODE in
#`Tie 160 Formatting Tags
#`Row       ``Tie node-name
#`Column    Optional
#`Column    Creates a link to a named ``Node. See ``TIE in
#`Tie 170 Formatting Tags
#`Row       ``Link description=URL
#`Column    Optional
#`Column    Creates a link with Description linked to the URL. See ``LINK in
#`Tie 180 Formatting Tags
#`Row       ``Index descriptive-text
#`Column    Optional
#`Column    Creates an entry in the alphabetical index page with the
#           descriptive text bookmarked to the current location. See ``INDEX in
#`Tie 190 Formatting Tags
#`EndTable
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node  100 Index/Notes and Caveats
#`Title Notes and Caveats
#`Index Orphan nodes
#   1.  It -IS- possible to create nodes (pages) that are "orphaned"; that is,
#       are not linked from anywhere.  It depends on the combination of named
#       and unnamed nodes, and your choice of the -l or linkMode option.
#
#   2.  Various combinations of tags in the source text have NOT been tested.
#       Things such as nested and overlapping begin/end tags, missing end tags,
#       and other strange usages may cause errors or unpredictable results.
#
#`Index E-mail component
#   3.  The e-mail component of this script has been written for Windows 95 and
#       Windows NT.  Some simple modifications would also allow other e-mail
#       systems to be used.  The e-mail subroutine is alert().
#
#`Index Generating TekDoc documentation
#   4.  This documentation was generated with TekDoc from the TekDoc.pl source
#       code using the default configuration file and the command line:
#       perl TekDoc.pl -f -l 3 TekDoc.pl, which is equivelent to:
#       perl TekDoc.pl -l 3 -f -- TekDoc.pl.  All other options and variables
#       were default values.
#
#`Index Embedding HTML tags
#   5.  It is possible to embed HTML tags directly in the source text processed
#       by the script.  Care must be taken to avoid conflicts with HTML sequences
#       generated by the script.  The most common thing to embed is a line break
#       < BR > (with no spaces).  HTML tags may also be embedded in the Footer
#       configuration variable.
#
#   6.  If HTML is generated, then node names are changed or sub-page are
#       reorganized, any old pages are left in the output directory.  The
#       script has no way of knowing about the changes, therefore it cannot
#       remove old pages.  The easiest method is to remove all pages before
#       regenerating the new pages.
#
#`Index Built-in rules
#`Node  200 Index/Notes and Caveats
# BUILT-IN RULES
#
#   1.  HTML inherently removes all whitespace in content, including newlines.
#       This script has some built-in rules to make formatting easier.  Hopefully
#       these rules make basic common sense.  Some have command-line options
#       and/or configuration file variables to control their behavior.
#
#   2,  Source text lines containing all uppercase letters, numbers, and special
#       characters only (no lowercase characters, but at least one alphabetic or
#       numeric character) are considered paragraph titles and are followed by a
#       line break.  These lines may also be bold, see the -b option and boldUpper
#       variable.
#
#`End
#
#------------------------------------------------------------------------------
#
#`Node  Index/History and Changes
#`Title History and Changes
#
#Version 1.0 - Original release, Perl 5.001
#
#Ideas for enhancements:<BR>
#1. When generating the Related Links at the bottom of pages (subroutine
#   addRelated), perform a pre-scan of the metaData array to calculate the
#   number of entries to add, given the -l linkMode option, and the longest
#   entry.  Then intelligently determine how many columns in an HTML table
#   to generate based on how long the data is (1 or 2 columns for long data,
#   3 or 4 columns for short data).  This would make the Related Links much
#   shorter for linkModes of 2 or 3 with many entries.
#
#2. Make the left navigation bar, the top page title, and the "body" all frames
#   so the left and top stay in place as the body is scrolled.
#
#3. And a command line option to process the content, but not generate any
#   output.  Instead, create the metaData array as usual, then delete all
#   the resulting output files.  Very handly for clearing out old content
#   pages from a large mixed output directory.
#
#------------------------------------------
#
#`End
#
#------------------------------------------------------------------------------

sub addBackLinks;                       # generate BACK paragraph
sub addIndex;                           # generate alphabetical index
sub addMap;                             # generate document map
sub addRelated;                         # generate Related Links paragraph
sub alert;                              # send mail
sub appendFile;                         # append a file to a file
sub byNode;                             # sort by node
sub byOutput;                           # sort by output
sub byTitle;                            # sort by title
sub bySubNode;                          # sort by sub-node
sub contentLength;                      # length of non-whitespace string
sub createPath;                         # create a directory path
sub dumpMeta;                           # dump the metaData array
sub dumpMetaRow;                        # dump a metaData row
sub fatal;                              # fatal error handler
sub findMeta;                           # find data in metaData array
sub isArg;                              # is string an option argument
sub isDirectory;                        # is path a directory
sub isNumeric;                          # is string all numbers
sub isUpper;                            # is string all upper case
sub logger;                             # runtime logger
sub logNT;                              # write to Windows NT log
sub maxSecurity;                        # maximum valid security within a node
sub minSecurity;                        # minimum security within a node
sub postProcess;                        # post-process metaData
sub preProcessCheck;                    # check status while pre-processing
sub preProcessFile;                     # pre-process a file
sub readConfig;                         # read & process configuration file
sub scanDir;                            # scan a directory for files
sub stripComment;                       # strip single-line comment
sub stripTag;                           # strips tag & whitespace
sub writeHtml;                          # write HTML to output file
sub writeHtmlEnd;                       # write ending HTML to output file
sub writeNavBar;                        # write the standard Navigation bar
sub writePd;                            # write the standard Public Domain block
sub writeTemp;                          # write to temporary 1st pass file


#------------------------------------------------------------------------INIT-
#
# Configuration variables, set default values if not specified in .cfg file
#

#
# Internal variables
#
$progName = 'TekDoc';                   # name of this program
$ver = '1.0';                           # version of this program

# define the default values for options,
# and flags for whether the value has been set from the command line
$optBoldUpper = 1;                      # bold lines that are all upper case
$optBoldUpperSet = 0;
$optExtensions = "c,cpp,cxx,h,hpp,hxx,pl,jav,java,txt";  # process extensions, no dots
$optExtensionsSet = 0;
$optFooter = "";                        # footer specified in configuration file
$optIncludeName = 0;                    # include source filename in output
$optIncludeNameSet = 0;
$optOutDir = "Doc";                     # output directory name
$optOutDirSet = 0;
$optOutExtension = "html";              # output file extension, no dot
$optOutExtensionSet = 0;
$optRecursive = 0;                      # operate on directories recursively
$optRecursiveSet = 0;
$optSecure = 1;                         # definition of what is secure or not
$optSecureSet = 0;
$optSecurity = 0;                       # security level to process
$optSecuritySet = 0;
$optStripComments = 1;                  # strip single-line comment characters
$optStripCommentsSet = 0;
$optLinkMode = 1;                       # include discrete sub-page links for source file
$optLinkModeSet = 0;

# define the tag (kw) trigger character or string which MUST BE different than
# the regular expression delimiter used in the script. Use something obscure,
# preferably not used in the source files.  Then define the complete list of
# tag words.
$kwTrigger  = chr( 96 );                # back-tick, use chr() to avoid doc warning
$kwNode     = "${kwTrigger}node";
$kwTitle    = "${kwTrigger}title";
$kwSecurity = "${kwTrigger}security";
$kwTable    = "${kwTrigger}table";
$kwRow      = "${kwTrigger}row";
$kwColumn   = "${kwTrigger}column";
$kwEndtable = "${kwTrigger}endtable";
$kwCode     = "${kwTrigger}code";
$kwEndcode  = "${kwTrigger}endcode";
$kwIndex    = "${kwTrigger}index";
$kwTie      = "${kwTrigger}tie";
$kwLink     = "${kwTrigger}link";
$kwFooter   = "${kwTrigger}footer";
$kwEnd      = "${kwTrigger}end";

# define symbolic index values for vector into 2-dimensional metaData array
$iNode      = 0;
$iSequence  = 1;                        # 1..n, 0 = unsequenced
$iTitle     = 2;
$iIndent    = 3;                        # 0..$metaEnd, -1 = index entry
$iSource    = 4;
$iOutput    = 5;
$iBookmark  = 6;
$iSecurity  = 7;                        # 0..254
$iFooter    = 8;
$iRefCount  = 9;

@metaData = ();
$metaEnd = -1;
$metaCurr = -1;

# known single-line comment characters for associated file extensions
$iSlExt     = 0;
$iSlStr     = 1;
$slComment[0][$iSlExt] = "c";           # C
$slComment[0][$iSlStr] = "//";
$slComment[1][$iSlExt] = "cpp";         # C++
$slComment[1][$iSlStr] = "//";
$slComment[2][$iSlExt] = "cxx";         # C++
$slComment[2][$iSlStr] = "//";
$slComment[3][$iSlExt] = "h";           # C/C++
$slComment[3][$iSlStr] = "//";
$slComment[4][$iSlExt] = "hpp";         # C++
$slComment[4][$iSlStr] = "//";
$slComment[5][$iSlExt] = "hxx";         # C++
$slComment[5][$iSlStr] = "//";
$slComment[6][$iSlExt] = "pl";          # Perl
$slComment[6][$iSlStr] = "#";
$slComment[7][$iSlExt] = "jav";         # Java
$slComment[7][$iSlStr] = "//";
$slComment[8][$iSlExt] = "java";        # Java
$slComment[8][$iSlStr] = "//";

# Internal links and default output page names (no extension)
$defHomeName    = "Index";              # default Home page file name
$defIndexName   = "${progName}_Index";  # default alphabetical Index page file name
$defMapName     = "${progName}_Map";    # default document map page file name

# Global working variables
$currNode = "";                         # current node name in metaData
$currSecurity = 0;                      # current source security value
$extracting = 0;                        # not extracting lines yet
$inCode = 0;                            # not formatting a code block
$justWroteParaBreak = 0;                # did not just write a paragraph break
$lastBookmark = 0;                      # bookmark counter
$lastMetaOut = -1;                      # last valid metaData output
$lc = 0;                                # source file line number read
$maxIndent = 0;                         # maximum indent in final output
$nodeChanged = 0;                       # current node just changed
$outPath = "";                          # current output path
outFh;                                  # define global file handle
$outputOpen = 0;                        # an output file has not been opened
$singleComment = 0;                     # last output line was an unstripped single-line comment

$keepTemps = 0;                         # internal, debug knob (0)
$useFirstTitle = 1;                     # internal, use first title found not last (1)

$ln = 0;                                # configuration file line number read
$srcCount = 0;                          # number of source on command line
$processed = 0;                         # files processed
$skipped = 0;                           # files skipped

# Temporary & Log files
$sendError = 0;                         # send an error message flag
$sendWarning = 0;                       # send a warning message flag
$logDisabled = 0;                       # log disabled flag

$tmpFile = "${progName}.tmp";           # temporary filename
unlink $tmpFile;                        # remove the tmp file
$logFile = $tmpFile;                    # log filename, use tmpFile to start
$writeNTlog = 0;                        # write completion event to NT log flag

# E-Mail
$mailUser = "";                         # mail profile used to send mail
$mailRecipient = "";                    # mail recipient for message
$mailLevel = 2;                         # default mail level
$mailLevelVal = $mailLevel;             # save mail level

# Operating system variables
$isDos = 0;
$isUnix = 0;
$isWindowsNT = 0;
if ($ENV{"OS"} eq "Windows_NT")         # check for a standard NT symbol
{
	$isWindowsNT = 1;
}
else
{
	if (length($ENV{"COMSPEC"}) > 0)	# check for DOS or Windows 95
	{
		$isDos = 1;
	}
	else
	{
		$isUnix = 1;
	}
}

if ($isUnix == 1)
{
	$slash = "/";
	$slashes = "/";
}
else
{
	$slash = "\\";
	$slashes = "\\\\";
}


#-----------------------------------------------------------------------------
# Startup
#

if ($ARGV[0] eq "-?")
{
    print "$progName $ver\n\n";

    print "Usage:\n";
    print "\tperl $progName.pl [options] source-file-or-directory ...\n\n";

    print "$progName generates HTML documentation from any text file.  A variety\n";
    print "of options are available to control HTML generation.  Please read the\n";
    print "documentation for complete details.\n\n";

    print "To generate the documentation for $progName into directory .\\Doc\n";
    print "use:\n";

    print "\tperl $progName.pl -f -l 1 -d Doc $progName.pl\n\n";
    exit;
};

# Print Banner
$nowstr = localtime();
$genStamp = $nowstr;                    # use same stamp for Index and Map pages

logger "\n% $progName $ver - Fired-up $nowstr\n";

# Setup the default configuration filename
$cfg = $0;                              # get full path to this script
$cfgSet = 0;
$cfg =~ s/\.pl/\.cfg/i;                 # substitute .pl for .cfg


#-----------------------------------------------------------------------------
# Process command line options
#
PROC_OPTS:
while (length($ARGV[0]) > 0)
{
    $arg = $ARGV[0];
    if (isArg $ARGV[0])
    {
        $_ = $arg;                      # set the match buffer

        SWITCH_OPTS:
        {
            /--/    && do               # end of options
                    {
                        shift;
                        last PROC_OPTS;
                    };
            /-b/    && do               # Bold-Uppercase rule
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optBoldUpper = $ARGV[0];
                            if ($optBoldUpper != 0)
                            {
                                $optBoldUpper = 1;
                            }
                            shift;
                        }
                        else
                        {
                            $optBoldUpper = 1;
                        }
                        $optBoldUpperSet = 1;
                        logger "= Setting bold upper:\t\t$optBoldUpper\n";
                        last SWITCH_OPTS;
                    };
            /-c/    && do               # configuration filename
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $cfg = $ARGV[0];
                            $cfgSet = 1;
                            shift;
                        }
                        else
                        {
                            fatal "! Configuration file not specified with -c option\n";
                        }
                        logger "= Setting configuration:\t$cfg\n";
                        last SWITCH_OPTS;
                    };
            /-d/    && do               # documentation output directory
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optOutDir = $ARGV[0];
                            shift;
                        }
                        else
                        {
                            fatal "! Documentation output directory not specified with -d option\n";
                        }
                        $optOutDirSet = 1;
                        logger "= Setting output directory:\t$optOutDir\n";
                        last SWITCH_OPTS;
                    };
            /-e/    && do               # extensions to process
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optExtensions = $ARGV[0];
                            shift;
                        }
                        else
                        {
                            fatal "! Extensions list not specified with -e option\n";
                        }
                        $optExtensionsSet = 1;
                        logger "= Setting extensions:\t\t$optExtensions\n";
                        last SWITCH_OPTS;
                    };
            /-f/    && do               # include filename of source in output
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optIncludeName = $ARGV[0];
                            if ($optIncludeName != 0)
                            {
                                $optIncludeName = 1;
                            }
                            shift;
                        }
                        else
                        {
                            $optIncludeName = 1;
                        }
                        $optIncludeNameSet = 1;
                        logger "= Setting include filename:\t$optIncludeName\n";
                        last SWITCH_OPTS;
                    };
            /-k/    && do               # kill single-line comment character strings
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optStripComments = $ARGV[0];
                            if ($optStripComments != 0)
                            {
                                $optStripComments = 1;
                            }
                            shift;
                        }
                        else
                        {
                            $optStripComments = 1;
                        }
                        $optStripCommentsSet = 1;
                        logger "= Setting kill comments:\t$optStripComments\n";
                        last SWITCH_OPTS;
                    };
            /-l/    && do               # link mode
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optLinkMode = $ARGV[0];
                            if ($optLinkMode < 0 || $optLinkMode > 3)
                            {
                                fatal "! Unknown link mode '$optLinkMode'\n";
                            }
                            shift;
                        }
                        else
                        {
                            fatal "! Missing or bad -l link mode\n";
                        }
                        $optLinkModeSet = 1;
                        logger "= Setting link mode:\t\t$optLinkMode\n";
                        last SWITCH_OPTS;
                    };
            /-o/    && do               # output file(s) extension
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optOutExtension = $ARGV[0];
                            shift;
                        }
                        else
                        {
                            fatal "! Output extension not specified with -o option\n";
                        }
                        $optOutExtensionSet = 1;
                        logger "= Setting output extension:\t$optOutExtension\n";
                        last SWITCH_OPTS;
                    };
            /-r/    && do               # operate recursively on directories
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optRecursive = $ARGV[0];
                            if ($optRecursive != 0)
                            {
                                $optRecursive = 1;
                            }
                            shift;
                        }
                        else
                        {
                            $optRecursive = 1;
                        }
                        $optRecursiveSet = 1;
                        logger "= Setting recursive:\t\t$optRecursive\n";
                        last SWITCH_OPTS;
                    };
            /-s/    && do               # security level to process
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optSecurity = $ARGV[0];
                            if ($optSecurity < 0 || $optSecurity > 254)
                            {
                                $optSecurity = 0;
                            }
                            shift;
                        }
                        else
                        {
                            fatal "! Security level to process not specified with -s option\n";
                        }
                        $optSecuritySet = 1;
                        logger "= Setting security:\t\t$optSecurity\n";
                        last SWITCH_OPTS;
                    };
            /-S/    && do               # what is "secure"
                    {
                        shift;
                        if (length($ARGV[0]) > 0 && (not isArg $ARGV[0]))
                        {
                            $optSecure = $ARGV[0];
                            if ($optSecure < 0 || $optSecure > 254)
                            {
                                $optSecure = 0;
                            }
                            shift;
                        }
                        else
                        {
                            fatal "! Secure definition not specified with -S option\n";
                        }
                        $optSecureSet = 1;
                        logger "= Setting secure:\t\t$optSecure\n";
                        last SWITCH_OPTS;
                    };
            fatal "! Unknown option '$arg'\n";
        }
    }
    else
    {
        last PROC_OPTS;
    }
}

#
# Assemble the 'sources' array
#
$srcCount = 0;

while (length($ARGV[0]) > 0)
{
    @source[$srcCount] = $ARGV[0];
    shift;
    ++$srcCount;
}


#-----------------------------------------------------------------------------
# Read the configuration filename
#

# Open the configuration file, process if available
if (open(config, $cfg) != 0)
{
    logger ". Opening configuration:\t$cfg\n";
    readConfig;
    close(config);
}
else
{
    # die if the file can't be opened, and the -c cfgFile option was used
    if ($cfgSet == 1)
    {
        fatal "! Cannot open configuration file $cfg for reading\n";
    }
}


#-----------------------------------------------------------------------------
# Setup the actual log file
#
# IMPORTANT NOTE:
# tmpFile is used as the log file to this point in the script to allow
# the log filename to be set within the configuration file while other
# data are being logged to the tmpFile to this point.
#

# Set logFile name as necessary
if ($logFileSet == 1)
{
    $logFile = $newLogFile;
}
else
{
    if ($logDisabled != 1)
    {
        # Formulate logFile name from the configuration name with .log extension
        $l = $cfg;
        $l =~ s/\.cfg/\.log/i;      # use filename of script with .log
        if ($l eq $cfg) { die "Configuration files must have a .cfg extension\n" };
        logger ". Logging to $l\n";
        $logFile = $l;
    }
}

# Copy temporary log to actual log file
if ($logDisabled != 1)
{
    if ($logOverwrite == 1)             # delete old log if overwrite requested
    {
        unlink $logFile;                # ignore any error
    }
    appendFile $tmpFile, $logFile, 0;   # copy tmp to log without displaying
}

unlink $tmpFile;                        # remove the tmp file

#
# check that we have some work to do
#
if ($srcCount < 1)
{
    logger "* No source(s) specified on command line, nothing to do\n";
    ++$sendWarning;
    goto DONE;
}


#-----------------------------------------------------------------------------
# Create the target documentation output directory
#

if (-e "$optOutDir" != 1)
{
    logger "+ Creating output directory: $optOutDir\n";
    createPath "$optOutDir";            # operates recursively on full path
}
else
{
    if (not isDirectory $optOutDir)     # it exists, so is it a directory?
    {
        fatal "! Output directory $optOutDir exists as a file, must be a directory\n";
    }
}


#-----------------------------------------------------------------------------
# Process each command line 'Source' argument
#
# This is the FIRST PASS process.  Extracts raw source input, creates
# metaData array of node and title information, and generates temporary
# output files to be formatted in the second pass later.  Each temporary
# output file has the Sequence value prepended to it's node name for
# later sorting for the second pass.
#

# Setup extensions array
(@extensions) = split(/,/, $optExtensions);
$extensionCount = $#extensions + 1;

#logger ". There are $extensionCount extensions in the list\n";

$i = 0;
while ($i < $extensionCount)
{
    @extensions[$i] =~ s/^[ \t]*//;     # trim leading whitespace
    @extensions[$i] =~ s/[ \t\n]*$//;   # trim trailing whitespace and newline
    #print "$i = '@extensions[$i]'\n";  #!D!
    ++$i;
}

# Separate the sources by file then directory, sort, then reassemble.
# This is done to adhere to the "files first, then directories" rule.
@sourceFiles = ();
@sourceDirs = ();

$i = $j = $k = 0;
while ($i < $srcCount)
{
    if (isDirectory @source[$i])
    {
        @sourceDirs[$j] = @source[$i];
        ++$j;
    }
    else
    {
        @sourceFiles[$k] = @source[$i];
        ++$k;
    }
    ++$i;
}

@sourceFiles = sort @sourceFiles;           # sort into order, just for pretty
@sourceDirs = sort @sourceDirs;

@source = (@sourceFiles, @sourceDirs);      # reassemble files first, then directories
@sourceFiles = ();                          # squash the individual arrays
@sourceDirs = ();

#
# Pre-process the content
#
logger "@ Pre-processing source files\n";
$currSource = 0;

while ($currSource < $srcCount)
{
    #print "\n. Source $currSource = @source[$currSource]\n\n";     #!D!

    scanDir @source[$currSource], "", "";   # <<<=== FIRST PASS work done here
    ++$currSource;
}

logger ". Total $processed pre-processed, $skipped skipped\n";

# make sure we got some output
if ( ($processed < 1) || ($metaEnd < 0) )
{
    logger "* No nodes processed from source(s), nothing more to do\n";
    ++$sendWarning;
    goto DONE;
}


#-----------------------------------------------------------------------------
# Post-process the temporary files, format each into final HTML output
#
# This is the SECOND PASS process.  Read and format each node (output file).
# The metaData array is sorted into Output and Sequence order.  Each numbered
# series of nodes is processed in ascending order.
#

logger "@ Post-processing temporary files\n";

postProcess;                                # <<<=== SECOND PASS work done here


#-----------------------------------------------------------------------------
# Generate the alphabetical Index HTML page
#
# All titles and index entries are written to the page.
#

logger "@ Generating alphabetical Index\n";

addIndex;


#-----------------------------------------------------------------------------
# Generate the Document Map page
#
# All unique node names are written to the page.  If there are nested sub-pages
# the relationship is shown in an indented hierarchy.
#

logger "@ Generating document Map\n";

addMap;


#-----------------------------------------------------------------------------
# Done
#

#dumpMeta();                                #!D!

goto DONE;                                  # <<<=== Entire process complete


#============================================================================#
#                                  SUBROUTINES                               #
#============================================================================#


#---------------------------------------------------------------SUB-BACKLINKS-
sub addBackLinks
{
    #
    # addBackLinks
    #
    # Generates the BACK paragraph in an output page.
    #
    # Returns: nothing
    #

    #-------# Sub-page back link (upward)
    if ($metaData[$metaCurr][$iIndent] > 0)
    {
        # do not link to excluded data
        if ($metaData[$metaCurr][$iSecurity] > $optSecurity)
        {
            return;
        }

        #print "node '$metaData[$metaCurr][$iNode]'  indent $metaData[$metaCurr][$iIndent]\n";      #!D!

        # formulate the link text
        (@segs) = split(/\//, $metaData[$metaCurr][$iNode]);
        $l = @segs[$metaData[$metaCurr][$iIndent] - 1];
        $l =~ s/^[ \t]//;
        $l =~ s/[ \t\n]$//;

        # do not back-link to the "Index", use the navigation bar
        if ($l !~ m/^index$/i)
        {
            #print "segs count $#segs\n";           #!D!

            # formulate the HREF
            #
            # concatenate node segments with a slash between
            # up to, but not including, the current node
            $i = 0;
            $h = "";
            while ($i < $metaData[$metaCurr][$iIndent])
            {
                if ($i > 0)
                    { $h = "$h/$segs[$i]"; }
                else
                    { $h = "$h$segs[$i]"; }
                ++$i;
            }
            $h = "$h.$optOutExtension";     # add the output extension
            $h =~ s/[ \t]//g;               # remove any embedded spaces
            $h =~ s/\//_/g;                 # replace any slashes with underscore

            #print "BACKLINK href=$h text=$l\n";    #!D!

            if ($justWroteParaBreak == 0)
                { writeHtml "\n<P></P>\n"; }

            writeHtml "\n<B><FONT SIZE=\"-1\" FACE=\"Arial\">BACK</FONT></B>\n";
            writeHtml "<UL>\n";
            writeHtml "\t<LI><A HREF=$h>$l</A>\n";
            writeHtml "</UL>\n";
        }
    }
} # sub addBackLinks


#----------------------------------------------------------------SUB-ADDINDEX-
sub addIndex
{
    #
    # addIndex;
    #
    # Generates the alphabetical index page.
    #
    # Returns: nothing
    #

    @metaData = sort byTitle @metaData;

    # open the output file for overwrite
    $outPath = "$optOutDir${slash}$defIndexName.$optOutExtension";
    open(outFh, ">$outPath") || fatal "! Cannot open alphabetical Index file $outPath for writing\n";

    #
    # generate the index content
    #

    #-------# DOCTYPE & HTML container
    writeHtml "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n";
    writeHtml "\n<HTML>\n";


    #-------# HEAD
    writeHtml "\n<HEAD>\n";
    writeHtml "\t<TITLE>Index</TITLE>\n";
    writePd;                                        # write Public Domain block
    writeHtml "</HEAD>\n";


    #-------# BODY - start
    writeHtml "\n<BODY BGCOLOR=SILVER>\n";          # background


    #-------# TABLE 1
    writeHtml "\n<TABLE VALIGN=TOP CELLPADDING=4 WIDTH=\"100%\">\n";

    # format Navigation Bar
    writeNavBar 2;


    #-------# Content is a table data entry
    writeHtml "\n<TD BGCOLOR=SILVER VALIGN=TOP WIDTH=\"100%\">\n";


    #-------# Title
    writeHtml "\n<FONT FACE=\"Arial\"><H4>Index</H4></FONT>\n";
    writeHtml "\n<HR WIDTH=\"100%\">\n";
    writeHtml "<CENTER>\n";
    writeHtml "\n<TABLE CELLPADDING=6 CELLSPACING=1 BORDER=1 BGCOLOR=MAROON>\n";
    writeHtml "<TR>\n";


    #-------# Generate the A B C... letter links
    $a = "A";
    $i = 0;
    $lastLetter = "";
    $letterEnd = 0;
    $letters = "";

    FMT_INDEX_LETTERS:
    while ( ($a <= "Z") && ($i <= $metaEnd) )
    {
        # CYA
        if (length($metaData[$i][$iNode]) < 1)
        {
            logger "* THIS SHOULD NEVER HAPPEN (1)\n";
            ++$sendWarning;
            ++$i;
            next FMT_INDEX_LETTERS;
        }

        # do not add entries with no title
        if (length($metaData[$i][$iTitle]) < 1)
        {
            ++$i;
            next FMT_INDEX_LETTERS;
        }

        # do not link to excluded data
        if ($metaData[$i][$iSecurity] > $optSecurity)
        {
            ++$i;
            next FMT_INDEX_LETTERS;
        }

        # do not add killed entries
        if (length($metaData[$i][$iOutput]) < 1)
        {
            ++$i;
            next FMT_INDEX_LETTERS;
        }

        # get the first character of the title
        $l = $metaData[$i][$iTitle];
        $c = substr($l, 0, 1);
        $c = uc $c;

        #print "c '$c'  a '$a'  lastLetter $lastLetter\n";      #!D!

        # do not repeat letters
        if ($c eq $lastLetter)
        {
            ++$i;
            #print "skipping\n";                    #!D!
            next FMT_INDEX_LETTERS;
        }

        $lastLetter = $c;

        # save it
        $letters = "$letters$lastLetter";
        ++$letterEnd;

        # flush out alphabet with non-link letter down to current letter
        while ( (ord($a) < ord($c)) && (ord($a) <= ord("Z")) )
        {
            #print "adding a '$a'  c '$c'\n";       #!D!
            writeHtml "\t<TD><FONT COLOR=WHITE>$a</FONT></TD>\n";
            $a = chr((ord($a) + 1));
        }

        $a = chr((ord($c) + 1));

        #print "refing c '$c'  a now '$a'\n";       #!D!

        writeHtml "\t<TD><A HREF=#$c><FONT COLOR=YELLOW><B>$c</B></FONT></A></TD>\n";

        ++$i;
    }

    # flush out any remaining alphabet
    while ( ord($a) <= ord("Z") )
    {
        #print "adding a '$a'  c '$c'\n";           #!D!
        writeHtml "\t<TD><FONT COLOR=WHITE>$a</FONT></TD>\n\n";
        $a = chr((ord($a) + 1));
    }

    writeHtml "</TR>\n";
    writeHtml "</TABLE>\n";
    writeHtml "</CENTER>\n";

    writeHtml "\n<HR WIDTH=\"100%\">\n\n";
    writeHtml "\tAlphabetical list of title and index entries.<BR><BR>\n";

    #print "letters '$letters'  letterEnd $letterEnd\n";        #!D!


    #-------# Generate the main portion of the index

    #print "MAIN INDEX\n";                          #!D!

    $i = 0;
    $j = 0;
    $lastLetter = "";
    $lc = 0;

    FMT_INDEX:
    while ($i <= $metaEnd)
    {
        # CYA
        if (length($metaData[$i][$iNode]) < 1)
        {
            logger "* THIS SHOULD NEVER HAPPEN (2)\n";
            ++$sendWarning;
            ++$i;
            next FMT_INDEX;
        }

        # do not add entries with no title
        if (length($metaData[$i][$iTitle]) < 1)
        {
            ++$i;
            next FMT_INDEX;
        }

        # do not link to excluded data
        if ($metaData[$i][$iSecurity] > $optSecurity)
        {
            ++$i;
            next FMT_INDEX;
        }

        # do not add killed entries
        if (length($metaData[$i][$iOutput]) < 1)
        {
            ++$i;
            next FMT_INDEX;
        }

        # make sure all previous letters of the alphabet have be written to this point
        $l = $metaData[$i][$iTitle];
        $c = substr($l, 0, 1);
        $lastLetter = $c;
        $a = substr($letters, $j, 1);

        #print "l '$l'  c $c  j $j  a $a\n";        #!D!
        while (($j < $letterEnd) && (ord($a) <=  ord($c)) )
        {
            #print "j $j\n";                        #!D!
            writeHtml "\n<A NAME=\"$a\"></A>\n";
            writeHtml "<H4>-- $a --</H4>\n";
            ++$j;
            $a = substr($letters, $j, 1);
        }

        # write the link
        if (length($metaData[$i][$iBookmark]) > 0)
            { $h = $metaData[$i][$iBookmark]; }
        else
            { $h = "$metaData[$i][$iOutput].$optOutExtension"; }

        writeHtml "\t&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<A HREF=$h>$metaData[$i][$iTitle]</A>\n";
        ++$metaData[$i][$iRefCount];

        if ($c eq $lastLetter)
            { writeHtml "<BR>\n"; }

        $lastLetter = $c;
        ++$i;
    }


    #-------# End TABLE 1
    writeHtml "\n<BR>\n";
    writeHtml "\n<HR WIDTH=\"100%\">\n";
    writeHtml "\n</TABLE>\n";


    #-------# Footer

    if (length($optFooter) > 0)
        { writeHtml "\t<DIV ALIGN=RIGHT>$optFooter</DIV>\n"; }

    writeHtml "\n<FONT FACE=\"Arial\" SIZE=\"-2\" COLOR=NAVY>\n";
    writeHtml "\t<DIV ALIGN=RIGHT><I>Generated:</I> $genStamp</DIV>\n";
    writeHtml "</FONT>\n";
    writeHtml "<BR>\n";


    #-------# End BODY
    writeHtml "\n</BODY>\n";


    close(outFh);
} # sub addIndex


#----------------------------------------------------------------SUB-ADDMAP-
sub addMap
{
    #
    # addMap;
    #
    # Generates the document map page.
    #
    # Returns: nothing
    #

    @metaData = sort byNode @metaData;      # sort the metaData by Node

    # open the output file for overwrite
    $outPath = "$optOutDir${slash}$defMapName.$optOutExtension";
    open(outFh, ">$outPath") || fatal "! Cannot open document Map file $outPath for writing\n";

    #
    # generate the map content
    #

    #-------# DOCTYPE & HTML container
    writeHtml "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n";
    writeHtml "\n<HTML>\n";


    #-------# HEAD
    writeHtml "\n<HEAD>\n";
    writeHtml "\t<TITLE>Document Map</TITLE>\n";
    writePd;                                # write Public Domain block
    writeHtml "</HEAD>\n";


    #-------# BODY - start
    writeHtml "\n<BODY BGCOLOR=SILVER>\n";  # background


    #-------# TABLE
    writeHtml "\n<TABLE VALIGN=TOP CELLPADDING=4 WIDTH=\"100%\">\n";

    # format the Navigation bar
    writeNavBar 3;


    #-------# Content is a table data entry
    writeHtml "\n<TD BGCOLOR=SILVER VALIGN=TOP WIDTH=\"100%\">\n";


    #-------# Title
    writeHtml "\n<FONT FACE=\"Arial\"><H4>Document Map</H4></FONT>\n";
    writeHtml "\n<HR WIDTH=\"100%\">\n";
    writeHtml "<CENTER>\n";
    writeHtml "\n<TABLE CELLPADDING=6 CELLSPACING=1 BORDER=1 BGCOLOR=MAROON>\n";
    writeHtml "<TR>\n";


    #-------# Generate the A B C... letter links
    $a = "A";
    $i = 0;
    $lastLetter = "";
    $letterEnd = 0;
    $letters = "";

    FMT_MAP_LETTERS:
    while ( ($a <= "Z") && ($i <= $metaEnd) )
    {
        # CYA
        if (length($metaData[$i][$iNode]) < 1)
        {
            logger "* THIS SHOULD NEVER HAPPEN (3)\n";
            ++$sendWarning;
            ++$i;
            next FMT_MAP_LETTERS;
        }

        # do not link to excluded data
        if ($metaData[$i][$iSecurity] > $optSecurity)
        {
            ++$i;
            next FMT_MAP_LETTERS;
        }

        # do not add killed entries
        if (length($metaData[$i][$iOutput]) < 1)
        {
            print "SKIPPING $i '$metaData[$i][$iNode]'\n";      #!D!
            ++$i;
            next FMT_MAP_LETTERS;
        }

        # get the first character of the Node
        $l = $metaData[$i][$iNode];
        $c = substr($l, 0, 1);
        $c = uc $c;

        #print "c '$c'  a '$a'  lastLetter $lastLetter\n";      #!D!

        # do not repeat letters
        if ($c eq $lastLetter)
        {
            ++$i;
            #print "skipping\n";                                #!D!
            next FMT_MAP_LETTERS;
        }

        $lastLetter = $c;

        # save it
        $letters = "$letters$lastLetter";
        ++$letterEnd;

        # flush out alphabet with non-link letter down to current letter
        while ( (ord($a) < ord($c)) && (ord($a) <= ord("Z")) )
        {
            #print "adding a '$a'  c '$c'\n";   #!D!
            writeHtml "\t<TD><FONT COLOR=WHITE>$a</FONT></TD>\n";
            $a = chr((ord($a) + 1));
        }

        $a = chr((ord($c) + 1));

        #print "refing c '$c'  a now '$a'\n";   #!D!

        writeHtml "\t<TD><A HREF=#$c><FONT COLOR=YELLOW><B>$c</B></FONT></A></TD>\n";

        ++$i;
    }

    # flush out any remaining alphabet
    while ( ord($a) <= ord("Z") )
    {
        #print "adding a '$a'  c '$c'\n";       #!D!
        writeHtml "\t<TD><FONT COLOR=WHITE>$a</FONT></TD>\n\n";
        $a = chr((ord($a) + 1));
    }

    writeHtml "</TR>\n";
    writeHtml "</TABLE>\n";
    writeHtml "</CENTER>\n";

    writeHtml "\n<HR WIDTH=\"100%\">\n\n";
    if ($maxIndent > 0)
        { writeHtml "\tAlphabetical list of document nodes indented to show the hierarchy.<BR><BR>\n"; }
    else
        { writeHtml "\tAlphabetical list of document nodes.<BR><BR>\n"; }

    #print "letters '$letters'  letterEnd $letterEnd\n";        #!D!


    #-------# Generate the main portion of the map BY NODE

    #print "MAIN MAP\n";                        #!D!
    $i = 0;
    $j = 0;
    $lastLetter = "";
    $lastNode = "";
    $lc = 0;

    FMT_MAP:
    while ($i <= $metaEnd)
    {
        # CYA
        if (length($metaData[$i][$iNode]) < 1)
        {
            logger "* THIS SHOULD NEVER HAPPEN (4)\n";
            ++$sendWarning;
            ++$i;
            next FMT_MAP;
        }

        # do not link to excluded data
        if ($metaData[$i][$iSecurity] > $optSecurity)
        {
            ++$i;
            next FMT_MAP;
        }

        # do not add killed entries
        if (length($metaData[$i][$iOutput]) < 1)
        {
            ++$i;
            next FMT_MAP;
        }

        # do not add index entries
        if ($metaData[$i][$iIndent] < 0)
        {
            ++$i;
            next FMT_MAP;
        }

        # do not add the same node again
        if ($metaData[$i][$iNode] eq $lastNode)
        {
            ++$i;
            next FMT_MAP;
        }
        $lastNode = $metaData[$i][$iNode];

        # make sure all previous letters of the alphabet have be written to this point
        $l = $metaData[$i][$iNode];
        $c = substr($l, 0, 1);
        $lastLetter = $c;
        $a = substr($letters, $j, 1);

        #print "l '$l'  c $c  j $j  a $a\n";    #!D!
        while (($j < $letterEnd) && (ord($a) <=  ord($c)) )
        {
            #print "j $j\n";                    #!D!
            writeHtml "\n<A NAME=\"$a\"></A>\n";
            writeHtml "<H4>-- $a --</H4>\n";
            ++$j;
            $a = substr($letters, $j, 1);
        }

        # write the link
        $h = "$metaData[$i][$iOutput].$optOutExtension";

        $ds = "";
        if ($metaData[$i][$iIndent] > 0)
        {
            $d = 0;
            while ($d < $metaData[$i][$iIndent])
            {
                $ds = "$ds. . ";
                ++$d;
            }
            $l = $metaData[$i][$iNode];
            $k = rindex($l, "/");
            $l = substr $l, $k + 1;             # get right-most node string
            writeHtml "\t&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;$ds<A HREF=$h>$l</A>\n";
        }
        else
            { writeHtml "\t&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;&nbsp;<A HREF=$h>$metaData[$i][$iNode]</A>\n"; }

        ++$metaData[$i][$iRefCount];

        if ($c eq $lastLetter)
            { writeHtml "<BR>\n"; }

        $lastLetter = $c;
        ++$i;
    }


    #-------# End TABLE
    writeHtml "\n<BR>\n";
    writeHtml "\n<HR WIDTH=\"100%\">\n";
    writeHtml "\n</TABLE>\n";


    #-------# Footer

    if (length($optFooter) > 0)
        { writeHtml "\t<DIV ALIGN=RIGHT>$optFooter</DIV>\n"; }

    writeHtml "\n<FONT FACE=\"Arial\" SIZE=\"-2\" COLOR=NAVY>\n";
    writeHtml "\t<DIV ALIGN=RIGHT><I>Generated:</I> $genStamp</DIV>\n";
    writeHtml "</FONT>\n";
    writeHtml "<BR>\n";


    #-------# End BODY
    writeHtml "\n</BODY>\n";


    close(outFh);
} # sub addMap


#--------------------------------------------------------------SUB-ADDRELATED-
sub addRelated
{
    #
    # addRelated
    #
    # Generates the Related Links paragraph in an output page.
    #
    # Returns:  nothing
    #

    my($i, $lastNode);

    # sort metaData by Sub-Node
    #
    # The metaData is sorted into another array, metaSort, because
    # metaData itself cannot be sorted while the post-processing is
    # in progress.  If it is sorted, new index entries being added to the
    # end of the metaData array will re-order the entries in mid-post-process
    # and the metaCurr current entry will not be correct or relevant.
    # Those new index entries are not used here anyway, so that does not
    # matter in this subroutine.
    @metaSort = sort bySubNode @metaData;

    #-------# Sub-page links (downward)
    $lastNode = "";
    $lc = 0;
    $i = 0;

    SUBPAGE_LINK:
    while ($i <= $metaEnd)
    {
        # do not link to "Index", it's on the navigation bar
        if ($metaSort[$i][$iNode] !~ m/^index$/i)
        {
            #print "COMPARE i '$metaData[$i][$iNode]'  curr '$metaSort[$metaCurr][$iNode]'\n";      #!D!

            # do not repeat the previous link
            if ($metaSort[$i][$iNode] eq $lastNode)
            {
                ++$i;
                next SUBPAGE_LINK;
            }

            # do not link index entries
            if ($metaSort[$i][$iIndent] < 0)
            {
                ++$i;
                next SUBPAGE_LINK;
            }

            # do not link to excluded data
            if ($metaSort[$i][$iSecurity] > $optSecurity)
            {
                ++$i;
                next SUBPAGE_LINK;
            }

            # do not link to self or same nodes
            if ( $metaSort[$i][$iNode] eq $metaData[$metaCurr][$iNode] )
            {
                ++$i;
                next SUBPAGE_LINK;
            }

            # if we are using modes 1 or 2 (which are local to the source file),
            # and they not are from the same sources,
            # then do not link

            if ( ($optLinkMode == 1) || ($optLinkMode == 2) )
            {
                #print "mode 1 or 2 compare\n";             #!D!
                if ($metaSort[$i][$iSource] ne $metaData[$metaCurr][$iSource])
                {
                    ++$i;
                    next SUBPAGE_LINK;
                }
            }

            # formulate the link text
            $l = $metaSort[$i][$iNode];
            $j = rindex($l, "/");
            $k = index($metaSort[$i][$iNode], $metaData[$metaCurr][$iNode]);

            #print " indent i $metaSort[$i][$iIndent]  curr $metaData[$metaCurr][$iIndent]  j $j  k $k\n";      #!D!

            # if using discrete sub-page links (optLinkMode == 1),      and
            # ( it is not indented 1 level from the current level,      or
            #   there is no "/" separator between nodes ($j),           or
            #   the current node is not the root path of the node ($k) )
            # then do not link

            if ( ($optLinkMode == 1) &&
                 ( ($metaSort[$i][$iIndent] - $metaData[$metaCurr][$iIndent] != 1) ||
                   ($j < 1) ||
                   ($k != 0)
                 )
               )
            {
                ++$i;
                next SUBPAGE_LINK;
            }

            $l = substr $l, $j + 1;                 # get right-most node string

            # formulate the HREF
            $h = "$metaSort[$i][$iOutput].$optOutExtension";
            ++$metaSort[$i][$iRefCount];

            #print "SUB-PAGE href=$h text=$l\n";    #!D!
            if ($lc == 0 )
            {
                if ($justWroteParaBreak == 0)
                {
                    writeHtml "\n<P></P>\n";
                    $justWroteParaBreak = 1;
                }
                writeHtml "\n<B><FONT SIZE=\"-1\" FACE=\"Arial\">RELATED LINKS</FONT></B>\n";
                writeHtml "<UL>\n";
            }

            $lastNode = $metaSort[$i][$iNode];

            writeHtml "\t<LI><A HREF=$h>$l</A>\n";
            ++$lc;
        }
        ++$i;
    }

    if ($lc > 0)
    {
        writeHtml "</UL>\n";
    }

    #getc();                                        #!D!

    @metaSort = ();                                 # squash the duplicate array

} # sub addRelated


#--------------------------------------------------------------SUB-APPENDFILE-
sub appendFile
{
    #
    # appendFile "fromFile", "toFile", display 0/1
    #
    # Returns line count.
    #

    my ($lc);

    $copyFrom = @_[0];
    $copyTo =   @_[1];
    $dsp =      @_[2];

    # Open the 'to' file
    open(fhTo, ">>$copyTo") || fatal "! Cannot open file $copyTo for appending\n";

    # Open the 'from' file
    open(fhFrom, $copyFrom) || fatal "! Cannot open file $copyFrom for reading\n";

    $lc = 0;
    while ($l = <fhFrom>)               # read a line
    {
        print fhTo $l;                  # write a line
        ++$lc;

        if ($dsp == 1)
        {
            print $l;                   # display on screen
        }
    }

    close(fhFrom);
    close(fhTo);
    return $lc;
} # sub appendFile


#------------------------------------------------------------------SUB-BYNODE-
sub byNode
{
    #
    # Used by sort() to compare Nodes in the metaData
    #
    my ($left, $right);

    $left = $$a[$iNode];                # pull out referenced nodes
    $right = $$b[$iNode];

    #print "  a '$left'\n";     #!D!
    #print "  b '$right'\n";    #!D!
    #getc();                    #!D!

    $left cmp $right;
} # sub byNode


#----------------------------------------------------------------SUB-BYOUTPUT-
sub byOutput
{
    #
    # Used by sort() to compare Output / Sequence in the metaData
    #
    my ($left, $right);

    $left = $$a[$iOutput];              # pull out referenced nodes
    $right = $$b[$iOutput];

    #print "  a '$left'\n";     #!D!
    #print "  b '$right'\n";    #!D!
    #getc();                    #!D!

    # if outputs are the same, sort by Sequence
    if ($left eq $right)
    {
        $left = $$a[$iSequence];
        $right = $$b[$iSequence];
        $left <=> $right;
        #print "SAME OUTPUT, using Sequence $left and $right\n";        #!D!
    }
    else
    {
        $left cmp $right;
    }
} # sub byOutput


#---------------------------------------------------------------SUB-BYSUBNODE-
sub bySubNode
{
    #
    # Used by sort() to compare Sub-Nodes in the metaData
    #
    my ($d, $i, $left, $right);

    $d = $$a[$iNode];
    $i = rindex($d, "/");               # returns -1 if not found
    $left = substr $d, $i + 1;          # get right-most node string

    $d = $$b[$iNode];
    $i = rindex($d, "/");               # returns -1 if not found
    $right = substr $d, $i + 1;         # get right-most node string

    #print "  a '$left'\n";     #!D!
    #print "  b '$right'\n";    #!D!
    #getc();                    #!D!

    $left cmp $right;
} # sub bySubNode


#-----------------------------------------------------------------SUB-BYTITLE-
sub byTitle
{
    #
    # Used by sort() to compare Title in the metaData
    #
    my ($left, $right);

    $left = $$a[$iTitle];               # pull out referenced nodes
    $right = $$b[$iTitle];

    #print "  a '$left'\n";     #!D!
    #print "  b '$right'\n";    #!D!
    #getc();                    #!D!

    $left cmp $right;
} # sub byTitle


#-----------------------------------------------------------SUB-CONTENTLENGTH-
sub contentLength
{
    #
    # contentLength "string";
    #
    # Returns: length of non-whitespace string
    #

    my($l);
    $l = @_[0];
    $l =~ s/^[ \t]//;
    $l =~ s/[ \t\n]$//;
    return length($l);
} # sub contentLength


#--------------------------------------------------------------SUB-CREATEPATH-
sub createPath
{
    #
    # Verifies that a target drive and each segment of a target path
    # exist on the target.  If not each segment will be created in turn.
    #
    # Argument may be relative or a fully-qualified path specification.
    #
    # Returns: nothing
    #

    $fpath = @_[0];
    #print "+ Attempting to create '$fpath'\n";     #!D!

    ($drive, $rest) = split(/:/,$fpath);
    #print "drive '$drive'  rest '$rest'\n";        #!D!

    if (length($drive) > 1 && length($rest) == 0)
    {
        $rest = $drive;
        $drive = "";
    }

    if (length($drive) != 1)
    {
        $build = "";
    }
    else
    {
        $build = "$drive:";             # Start with the drive:
        #print "if (-e $drive:$slash != 1)\n";      #!D!
        if (-e "$drive:$slash" != 1)
        {
            fatal "! Drive '$drive:' for output directory not found on system\n";
        }
    }

    # remember if fully-qualified
    if ($rest =~ m/^$slashes/)
    {
        $build = "$build$slash";
    }

    # Clean and check the rest of the path
    $rest =~ s/^$slashes//;             # trim leading slash
    $rest =~ s/^[ \t]*//;               # trim leading whitespace
    $rest =~ s/[ \t]*$//;               # trim trailing whitespace
    if (length($rest) < 1)
    {
        fatal "! No documentation output directory specified\n";
    }
    #print "rest after trim '$rest'\n"; #!D!

    @segs = split(/$slashes/, $rest);   # split path into segments

    $i = 0;
    foreach $seg (@segs)                # for each segment
    {
        if ($i == 0)
        {
            $build = "$build$seg";
        }
        else
        {
            $build = "$build$slash$seg";    # add on to partial path
        }

        ++$i;
        #print "$i $seg\n";             #!D!
        #print "build '$build'\n";      #!D!

        if (-e "$build" != 1)           # see if it exists
        {
            #print "making '$build'\n"; #!D!

            mkdir $build, 0666;         # if not create it
            if (-e "$build" != 1)       # make sure it got created
            {
                fatal "! Cannot create path: $build\n! Segment $i of: $fpath\n";
            }
        }
    }
} # sub createPath


#----------------------------------------------------------------SUB-DUMPMETA-
sub dumpMeta
{
    #
    # dumpMeta;
    #
    # Dumps metaData array to the screen.
    #
    # Returns:  nothing
    #

    my($c, $lc, $i);

    print "\nDump metaData (y/N)? ";
    $c = <STDIN>;
    if ($c =~ m/^y$/i)
    {
        $i = $metaEnd + 1;
        print "\n== metaData dump ($i) ==\n";
        $i = 0;
        $lc = 0;
        while ($i <= $metaEnd)
        {
            dumpMetaRow $i;
            ++$i;
            $lc += 11;
            if ($lc >= 44)
            {
                print "Press ENTER to continue - ";
                $c = <STDIN>;
                $lc = 0;
            }
        }
        print "\n";
    }
} # sub dumpMeta


#-------------------------------------------------------------SUB-DUMPMETAROW-
sub dumpMetaRow
{
    #
    # dumpMeta;
    #
    # Dumps a metaData row to the screen.
    #
    # Returns:  nothing
    #

    my($i);

    $i = @_[0];

    print "$i.\tNode:     $metaData[$i][$iNode]\n";
    print "\tSequence: $metaData[$i][$iSequence]\n";
    print "\tTitle:    $metaData[$i][$iTitle]\n";
    print "\tIndent:   $metaData[$i][$iIndent]\n";
    print "\tSource:   $metaData[$i][$iSource]\n";
    print "\tOutput:   $metaData[$i][$iOutput]\n";
    print "\tBookmark: $metaData[$i][$iBookmark]\n";
    print "\tSecurity: $metaData[$i][$iSecurity]\n";
    print "\tFooter:   $metaData[$i][$iFooter]\n";
    print "\tRefCount: $metaData[$i][$iRefCount]\n";
} # sub dumpMetaRow


#----------------------------------------------------------------SUB-FINDMETA-
sub findMeta
{
    #
    # findMeta column-number, string-to-find, direction
    #
    # column-number is the value of the second indices of the metaData array.
    # string-to-find is what to search for.
    # direction is 0 = backward, 1 = forward.
    #
    # This subroutine REQUIRES an EXACT MATCH, including length.
    #
    # Returns:  index of matching element (row)
    #           -1 if not found
    #

    my ($i);

    #print "METAEND $metaEnd\n";                    #!D!
    #print "Searching for @_[1]\n";

    if (@_[2] == 0)
        { $i = $metaEnd; }
    else
        { $i = 0; }

    while ( ($i >= 0) && ($i <= $metaEnd) )
    {
        #print "CHECKING $metaData[$i][@_[0]]\n";   #!D!

        if (length($metaData[$i][@_[0]]) == length(@_[1]))
        {
            if ($metaData[$i][@_[0]] eq @_[1])
            {
                #print "Found at $i '$metaData[$i][@_[0]]'\n";  #!D!
                return $i;
            }
        }
        if (@_[2] == 0)
            { --$i; }
        else
            { ++$i; }
    }
    return -1;
} # sub findMeta


#-------------------------------------------------------------------SUB-ISARG-
sub isArg
{
    #
    # isArg "string";
    #
    # Checks if first character in string is a dash "-".
    #
    # Returns: 1 = yes
    #          0 = no
    #

    if (@_[0] =~ m/^-/)
    {
        return 1;
    }
    else
    {
        return 0;
    }
} # sub isArg


#-------------------------------------------------------------SUB-ISDIRECTORY-
sub isDirectory
{
    #
    # isDirectory "path";
    #
    # This is a kludge to detect whether the path is
    # a file or directory.  The -d and -f operators do not
    # work under Microsoft's version of Perl 5.001.
    #
    # Returns: 1 = yes
    #          0 = no
    #

    my ($tempR, $tempFh);

    # This call fails if it is a file.
    $tempR = opendir tempFh, @_[0];
    if ($tempR eq 1)
    {
        close tempFh;
        return 1;
    }
    else
    {
        return 0;
    }
} # sub isDirectory


#---------------------------------------------------------------SUB-ISNUMERIC-
sub isNumeric
{
    #
    # isNumeric "string";
    #
    # Determines if the string is all numbers.
    #
    # CAUTION !!!!!!!!!!
    # This routine is probably character-set or code page specific to english.
    #
    # Returns:  0 if not all numbers,
    #           1 if all numbers
    #

    my($c, $i, $j, $l, $z);

    $l = @_[0];
    $i = 0;
    $j = length($l);
    while ($i < $j)
    {
        $c = substr $l, $i, 1;
        $z = -1;
        $z = index("1234567890", $c);
        if ($z < 0)
        {
            return 0;
        }
        ++$i;
    }
    return 1;
} # sub isNumeric


#-----------------------------------------------------------------SUB-ISUPPER-
sub isUpper
{
    #
    # isUpper "string";
    #
    # Determines if the string is all upper case.  It cannot contain any
    # lowercase letters, be must have at least one alpha-numeric character.
    #
    # CAUTION !!!!!!!!!!
    # This routine is probably character-set or code page specific to english.
    #
    # Returns:  0 if not all upper case,
    #           1 if all upper case
    #

    my($a, $c, $i, $j, $l, $z);
    $a = 0;
    $l = @_[0];
    $i = 0;
    $j = length($l);
    while ($i < $j)
    {
        $c = substr $l, $i, 1;
        $z = -1;
        # match against valid characters ( ' " \ $ and % are escaped)
        $z = index("ABCEDFGHIJKLMNOPQRSTUVWXYZ1234567890 \n\t,./<>?;\':\"[]{}``-=\\~!@#\$\%^&*()_+|", $c);
                   #0123456789.123456789.123456789.12345
        if ($z < 0)
        {
            return 0;
        }
        if ($z <= 35)       # 35 is position in literal string for index() above, 0 based
        {
            $a = 1;
        }
        ++$i;
    }
    return ($a == 0) ? 0 : 1;
} # sub isUpper


#-------------------------------------------------------------SUB-MAXSECURITY-
sub maxSecurity
{
    #
    # maxSecurity "node to search";
    #
    # Searches for the maximum VALID security within a node, within limits.
    #
    # Returns: Maximum security found for all matching nodes.
    #

    my($i, $sec);

    $i = 0;
    $sec = 0;

    while ($i <= $metaEnd)
    {
        if ($metaData[$i][$iNode] eq @_[0])
        {
            if ( ($metaData[$i][$iSecurity] <= $optSecurity) && ($metaData[$i][$iSecurity] > $sec) )
            {
                $sec = $metaData[$i][$iSecurity];
            }
        }
        ++$i;
    }

    return $sec;
} # sub maxSecurity


#-------------------------------------------------------------SUB-MINSECURITY-
sub minSecurity
{
    #
    # minSecurity "node to search";
    #
    # Searches for the minimum security within a node.
    #
    # Returns: Minimum security found for all matching nodes.
    #

    my($i, $sec);

    $i = 0;
    $sec = 254;

    while ($i <= $metaEnd)
    {
        if ($metaData[$i][$iNode] eq @_[0])
        {
            if ($metaData[$i][$iSecurity] < $sec)
            {
                $sec = $metaData[$i][$iSecurity];
            }
        }
        ++$i;
    }

    return ( ($sec == 254) ? 0 : $sec );
} # sub minSecurity

#-------------------------------------------------------------SUB-POSTPROCESS-
sub postProcess
{
    #
    # postProcess;
    #
    # Performs post-processing of the metaData.
    #
    # Returns: nothing
    #

    # sort metaData by Output (subsort by Sequence)
    @metaData = sort byOutput @metaData;

    #print "optOutExtension '$optOutExtension'\n";  #!D!

    $lastInclude = "";
    $lastOutPath = "";
    $lc = 0;
    $metaCurr = 0;
    $outputChanged = 0;
    $outputOpen = 0;

    #
    # go through the metaData array
    #
    FMT_FILE:
    while ($metaCurr <= $metaEnd)
    {
        # do not process Index entries
        if ($metaData[$metaCurr][$iIndent] < 0)
        {
            ++$metaCurr;
            next FMT_FILE;
        }

        $outPath = $metaData[$metaCurr][$iOutput];

        # do not process nodes with no output file,
        # there could be metaData nodes that exceed security
        # and were never filled-out.
        if (length($outPath) < 1)
        {
            ++$metaCurr;
            next FMT_FILE;
        }

        # do not reprocess the same file
        if ($outPath eq $lastOutPath)
        {
            # could be an unnumbered node from same source file,
            # 1 is the default sequence for an unnumbered node
            if ($metaData[$metaCurr][$iSequence] == 1)
            {
                ++$metaCurr;
                next FMT_FILE;
            }
            # otherwise, it's a higher sequence number of same node
        }
        else
        {
            # a different node, change the output file
            $outputChanged = 1;
        }

        # open, read and close the temporary file
        $inPath = sprintf "$optOutDir$slash%05d_%s.tmp", $metaData[$metaCurr][$iSequence], $metaData[$metaCurr][$iOutput];

        @lines = ();

        open(fhFrom, $inPath) || fatal "! Cannot open temporary file $inPath for reading\n";
        @lines = (<fhFrom>);                # read entire file into @lines array
        close(fhFrom);                      # close in temporary file

        if ($keepTemps == 0)
            { unlink $inPath; }             # delete the temporary file

        if ($#lines == 0)                   # if only 1 line (0) then there is only a node
        {
            ++$metaCurr;
            next FMT_FILE;                  # ...meaning, it's a phantom due to security
        }

        $lastOutPath = $outPath;

        logger "+ File: $optOutDir$slash$metaData[$metaCurr][$iOutput].tmp ($metaData[$metaCurr][$iSequence])\n";

        if ($metaData[$metaCurr][$iIndent] > $maxIndent)
        {
            $maxIndent = $metaData[$metaCurr][$iIndent];
        }

        # open and setup a new output file
        if ($outputChanged == 1)
        {
            if ($outputOpen == 1)           # write ending HTML to output file
            {
                writeHtmlEnd;
            }

            # formulate the new output path
            $outPath = "$optOutDir$slash$metaData[$metaCurr][$iOutput].$optOutExtension";
            #print "outPath $outPath\n";                            #!D!

            open(outFh, ">$outPath") || fatal "! Cannot open HTML output file $outPath for writing\n";
            $outputOpen = 1;
            $lastMetaOut = $metaCurr;


            #-------# DOCTYPE & HTML container
            writeHtml "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 3.2 Final//EN\">\n";
            writeHtml "\n<HTML>\n";


            #-------# HEAD
            writeHtml "\n<HEAD>\n";
            if (length($metaData[$metaCurr][$iTitle]) > 0)
                { writeHtml "\t<TITLE>$metaData[$metaCurr][$iTitle]</TITLE>\n"; }
            else
                { writeHtml "\t<TITLE>$metaData[$metaCurr][$iNode]</TITLE>\n"; }
            writePd;                                # write Public Domain block
            writeHtml "</HEAD>\n";


            #-------# BODY - start
            writeHtml "\n<BODY BGCOLOR=SILVER>\n";  # background


            #-------# TABLE 1
            writeHtml "\n<TABLE VALIGN=TOP CELLPADDING=4 WIDTH=\"100%\">\n";

            # format colored security table data, if appropriate
            $i = maxSecurity $metaData[$metaCurr][$iNode];
            if ($i >= $optSecure)
            {
                writeHtml "\n<TD VALIGN=TOP BGCOLOR=GREEN>\n";
                writeHtml "\t<FONT COLOR=WHITE>$i</FONT>\n";
                writeHtml "</TD>\n";
            }

            # format navigation bar
            if ($metaData[$metaCurr][$iNode] !~ m/^index$/i)
                { writeNavBar 0; }                  # do not skip anything
            else
                { writeNavBar 1; }                  # skip "Home" link

            #-------# Content is a table data entry
            writeHtml "\n<TD BGCOLOR=SILVER VALIGN=TOP WIDTH=\"100%\">\n";

            #-------# Title
            if (length($metaData[$metaCurr][$iTitle]) > 0)
                { writeHtml "\n<FONT FACE=\"Arial\"><H4>$metaData[$metaCurr][$iTitle]</H4></FONT>\n"; }
            else
            {
                $l = $metaData[$metaCurr][$iNode];
                $j = rindex($l, "/");
                $l = substr($l, $j + 1);
                writeHtml "\n<FONT FACE=\"Arial\"><H4>$l</H4></FONT>\n";
                logger "* No $kwTitle defined, using $kwNode value\n";
                ++$sendWarning;
            }

            # add a horizontal rule
            writeHtml "\n<HR WIDTH=\"100%\">\n";

            $lc = 0;            # $lc is changed while finishing files, reset it
            $outputChanged = 0;
            $lastInclude = "";
        }

        if ($lc > 2)
        {
            # add a bookmark
            if (length($metaData[$metaCurr][$iBookmark]) < 1)
            {
                # make up a bookmark and write it
                ++$lastBookmark;
                $b = "${progName}_${lastBookmark}";
                writeHtml "\n<A NAME=$b></A> ";
                $h = "$metaData[$metaCurr][$iOutput].$optOutExtension";
                $metaData[$metaCurr][$iBookmark] = "$h#$b";
            }

            # add a horizontal rule
            writeHtml "\n<HR WIDTH=\"100%\">\n";
        }

        #
        # Process the @lines array
        #
        $inTable = 0;                       # not formatting a table now
        $inTableData = 0;                   # not formatting table data now
        $inTableRow = 0;                    # not formatting a table row now
        $lc = 0;                            # reset line count

        FMT_LINE:
        foreach $l (@lines)
        {
            ++$lc;

            # match for the trigger string, but not two in a row (escaped)
            if ( ($l =~ m=$kwTrigger=i) && ($' !~ m=^${kwTrigger}=i) )
            {
                $i = index($l, $kwTrigger);
                $w = substr($l, $i);                # pull the tag
                ($w, $junk) = split(/[ \t]/, $w);   # parse on whitespace
                $w =~ s/[ \t\n]*$//;
                #print "Word is '$w'\n";            #!D!

                $_ = $w;                    # set match buffer

                SWITCH_POST:
                {                           # indentation is left 1 for room
                /$kwNode$/i     && do
                                {
                                    # do not add an empty node (due to security)
                                    if ($lines[$lc + 1] !~ m=^$kwNode=)
                                    {
                                        # we cheat and use the node tag to
                                        # carry the source filename from phase 1
                                        $l = stripTag $kwNode, $l;
                                        if ($lc > 2)
                                        {
                                            # add a bookmark
                                            if (length($metaData[$metaCurr][$iBookmark]) < 1)
                                            {
                                                # make up a bookmark and write it
                                                ++$lastBookmark;
                                                $b = "${progName}_${lastBookmark}";
                                                writeHtml "\n<A NAME=$b></A> ";
                                                $h = "$metaData[$metaCurr][$iOutput].$optOutExtension";
                                                $metaData[$metaCurr][$iBookmark] = "$h#$b";
                                            }
                                            # add a horizontal rule
                                            writeHtml "\n<HR WIDTH=\"100%\">\n";
                                        }

                                        # if this source name was written last, don't write it again
                                        if ( ($optIncludeName == 1) && ($l ne $lastInclude) )
                                        {
                                            writeHtml "\n<FONT FACE=\"Arial\" COLOR=NAVY SIZE=\"-2\">\n";
                                            writeHtml "\t<I>Source:</I> $l\n";
                                            writeHtml "</FONT>\n";

                                            writeHtml "\n<P></P>\n";
                                            $lastInclude = $l;
                                        }
                                    }
                                    next FMT_LINE;
                                };
                /$kwCode$/i     && do
                                {
                                    writeHtml "\n<TABLE>\n<TR>\n<TD BGCOLOR=WHITE><PRE>\n";
                                    $inCode = 1;
                                    next FMT_LINE;
                                };
                /$kwEndcode$/i  && do
                                {
                                    writeHtml "</PRE></TD></TR></TABLE>\n";
                                    $inCode = 0;
                                    next FMT_LINE;
                                };
                /$kwTie$/i      && do
                                {
                                    $n = stripTag $kwTie, $l;

                                    # see if we're looking for a specific sequence number
                                    ($s, @rest) = split(/[ \t]/, $n);
                                    if (isNumeric $s)
                                    {
                                        $n = substr($n, length($s));
                                        $n =~ s/^[ \t]*//;

                                    }
                                    else
                                    {
                                        $s = -1;
                                    }

                                    # formulate the link text
                                    $j = rindex($n, "/");
                                    $l = substr $n, $j + 1; # get right-most node string

                                    # find a matching node
                                    $i = 0;
                                    FMT_LINE_FND_TIE:
                                    while ($i <= $metaEnd)
                                    {
                                        if (length($metaData[$i][$iNode]) == length($n))
                                        {
                                            if ($metaData[$i][$iNode] eq $n)
                                            {
                                                # optionally match sequence too
                                                if ($s >= 0)
                                                {
                                                    if ($metaData[$i][$iSequence] != $s)
                                                    {
                                                        ++$i;
                                                        next FMT_LINE_FND_TIE;
                                                    }
                                                }
                                                last FMT_LINE_FND_TIE;
                                            }
                                        }
                                        ++$i;
                                    }

                                    # was anything found?
                                    if ($i > $metaEnd)
                                    {
                                        logger "* $kwTie '$n' could not be matched in node $metaData[$metaCurr][$iNode]\n";
                                        ++$sendWarning;
                                        writeHtml " $l ";       # add the plain text
                                        next FMT_LINE;
                                    }

                                    # does the target exceed security?
                                    $sec = minSecurity $n;
                                    if ($sec > $optSecurity)    # if there will be nothing in the
                                    {                           # referenced file,
                                        writeHtml " $l ";       # ..then just add the plain text
                                    }
                                    else
                                    {

                                        # find a valid output file to reference for the HREF
                                        if (length($metaData[$i][$iOutput]) < 1)
                                        {
                                            PP_FND_OUT:
                                            while ($i <= $metaEnd)
                                            {
                                                if ($metaData[$i][$iNode] eq $n &&
                                                    $metaData[$i][$iSecurity] == $sec)
                                                {
                                                    if (length($metaData[$i][$iOutput]) > 0)
                                                    {
                                                        last PP_FND_OUT;
                                                    }
                                                }
                                                ++$i;
                                            }
                                            if ($i > $metaEnd)
                                            {
                                                # logical problem: at this point there should have been
                                                # at least one matching valid node with an output.
                                                # But we can't find one.
                                                logger "* $kwTie '$n' could not find a valid page to link in node $metaData[$metaCurr][$iNode]\n";
                                                ++$sendWarning;
                                                writeHtml " $l ";   # add the plain text
                                                next FMT_LINE;
                                            }
                                        }

                                        # formulate the HREF
                                        if (length($metaData[$i][$iBookmark]) > 0)
                                            { $h = $metaData[$i][$iBookmark]; }
                                        else
                                            { $h = "$metaData[$i][$iOutput].$optOutExtension"; }
                                        ++$metaData[$i][$iRefCount];

                                        writeHtml " <A HREF=$h>$l</A> ";
                                    }
                                    next FMT_LINE;
                                };
                /$kwIndex$/i        && do
                                {
                                    $l = stripTag $kwIndex, $l;
                                    $l =~ s/^[ \t]*//;
                                    $l =~ s/[ \t\n]*$//;

                                    $h = "$metaData[$metaCurr][$iOutput].$optOutExtension";

                                    # make up a bookmark and write it
                                    ++$lastBookmark;
                                    $b = "${progName}_$lastBookmark";
                                    writeHtml "\n<A NAME=$b></A> ";

                                    # add the Index entry to metaData
                                    ++$metaEnd;
                                    $metaData[$metaEnd][$iNode] = $metaData[$metaCurr][$iNode];
                                    $metaData[$metaEnd][$iSequence] = 0;
                                    $metaData[$metaEnd][$iTitle] = $l;
                                    $metaData[$metaEnd][$iIndent] = -1;     # mark as Index entry
                                    $metaData[$metaEnd][$iSource] = $metaData[$metaCurr][$iSource];
                                    $metaData[$metaEnd][$iOutput] = $metaData[$metaCurr][$iOutput];
                                    $metaData[$metaEnd][$iSecurity] = $metaData[$metaCurr][$iSecurity];
                                    $metaData[$metaEnd][$iBookmark] = "$h#$b";
                                    $metaData[$metaEnd][$iRefCount] = 0;

                                    #print "INDEX '$l' at '$h#$b'\n";       #!D!
                                    next FMT_LINE;
                                };
                /$kwLink$/i     && do
                                {
                                    $l = stripTag $kwLink, $l;
                                    ($l, $h) = split(/=/, $l);  # split on the equals sign (=)
                                    $l =~ s/[ \t\n]*$//;        # clean up
                                    $h =~ s/^[ \t]*//;
                                    $h =~ s/[ \t\n]*$//;
                                    writeHtml " <A HREF=$h>$l</A> ";
                                    next FMT_LINE;
                                };
                /$kwTable$/i    && do
                                {
                                    if ($inTableData == 1)
                                    {
                                        writeHtml "</TD>\n";
                                        $inTableData = 0;
                                        logger "* Another $kwTable found, previous $kwTable missing $kwEndtable\n";
                                        ++$sendWarning;
                                    }
                                    if ($inTableRow == 1)
                                    {
                                        writeHtml "</TR>\n";
                                        $inTableRow = 0;
                                    }
                                    if ($inTable == 1)
                                    {
                                        writeHtml "</TABLE>\n";
                                    }
                                    $l = stripTag $kwTable, $l;
                                    writeHtml "\n<TABLE BORDER=1 CELLSPACING=1 CELLPADDING=6 WIDTH=\"100%\">\n";
                                    if (length($l) > 0)
                                    {
                                        writeHtml "\t<FONT SIZE=\"-1\" FACE=\"Arial\"><B>$l</B></FONT>\n";
                                    }
                                    $inTable = 1;
                                    next FMT_LINE;
                                };
                /$kwRow$/i      && do
                                {
                                    if ($inTableData == 1)
                                    {
                                        writeHtml "</TD>\n";
                                        $inTableData = 0;
                                    }
                                    if ($inTableRow == 1)
                                    {
                                        writeHtml "</TR>\n";
                                        $inTableRow = 0;
                                    }
                                    $l = stripTag $kwRow, $l;
                                    writeHtml "<TR>\n";
                                    writeHtml "<TD WIDTH=\"25%\">\n";
                                    writeHtml "\t$l\n";
                                    $inTableData = 1;
                                    $inTableRow = 1;
                                    next FMT_LINE;
                                };
                /$kwColumn$/i   && do
                                {
                                    $l = stripTag $kwColumn, $l;
                                    if ($inTableData == 1)
                                    {
                                        writeHtml "</TD>\n";
                                        $inTableData = 0;
                                    }
                                    writeHtml "<TD>\n";
                                    writeHtml "\t$l\n";
                                    $inTableData = 1;
                                    next FMT_LINE;
                                };
                /$kwEndtable$/i && do
                                {
                                    if ($inTableData == 1)
                                    {
                                        writeHtml "</TD>\n";
                                        $inTableData = 0;
                                    }
                                    if ($inTableRow == 1)
                                    {
                                        writeHtml "</TR>\n";
                                        $inTableRow = 0;
                                    }
                                    writeHtml "</TABLE>\n";
                                    $inTable = 0;
                                    next FMT_LINE;
                                };
                }
            }

            # handle an escaped trigger, and replace with a single trigger
            if ( ($l =~ m=$kwTrigger=i) && ($' =~ m=^${kwTrigger}=i) )
            {
                $l =~ s=${kwTrigger}${kwTrigger}=${kwTrigger}=g;
            }

            # setup to write the output line
            $justWroteParaBreak = 0;

            $l = stripComment $metaData[$metaCurr][$iSource], $l;

            if (contentLength($l) > 0)              # write content line
            {
                if ($inTableRow == 1)
                    { writeHtml "\t"; }             # indent
                if ($inCode == 1)
                {
                    $l =~ s/[\n]*$//;               # trim trailing newline only
                    writeHtml $l;                   # write preformatted content
                    writeHtml "\n";
                }
                else
                {
                    if ((isUpper "$l") && ($optBoldUpper != 0))
                    {
                        #print "BOLD ";             #!D!
                        $l =~ s/^[ \t]//;           # trim leading whitespace
                        $l =~ s/[ \t\n]*$//;        # trim trailing whitespace
                        writeHtml "\n<B><FONT SIZE=\"-1\" FACE=\"Arial\">";
                        writeHtml $l;
                        writeHtml "</FONT></B><BR>\n";
                    }
                    else
                    {
                        $l =~ s/^[ \t]//;           # trim leading whitespace
                        $l =~ s/[ \t\n]*$//;        # trim trailing whitespace
                        writeHtml $l;               # trimmed content line
                        if (($singleComment == 1) || (isUpper "$l"))
                            { writeHtml "<BR>"; }
                        writeHtml "\n";             # newline
                    }
                }
            }
            else
            {                                       # write paragraph tag pair
                if ($inCode == 0)
                {
                    if ($inTableRow == 1)
                        { writeHtml "\t"; }         # indent
                    else
                        { writeHtml "\n"; }
                    writeHtml "<P></P>\n";
                    $justWroteParaBreak = 1;        # just wrote a paragraph break
                }
                else
                {
                    writeHtml "\n";
                }
            }
        } # foreach $l

        #print "file done $metaCurr\n";             #!D!
        ++$metaCurr;

    } # foreach $inPath

    if ($outputOpen == 1)           # write ending HTML to output file
    {
        writeHtmlEnd;               # handles forward & back links,
    }

    #print "paused-"; getc();       #!D!
} # sub postProcess


#---------------------------------------------------------SUB-PREPROCESSCHECK-
sub preProcessCheck
{
    #
    # preProcessCheck "tag", metaFlag;
    #
    # Tag is the tag being processed (for error reporting).
    #
    # metaFlag is 0/1/2. If 1 checks that the metaData array is not empty
    # prior to inserting new data (for Title, Security, etc.).  If 2 this
    # is the only check performed.  If 0 this check is skipped.
    #
    # Used by preProcessFile() to check that we are extracting text prior to
    # writing to the temporary file.  The metaFlag option performs additional
    # checks.
    #
    # Returns:  0, check OK
    #           1, check failed & warning issued

    if (@_[1] != 0)
    {
        if ($metaEnd < 0)
        {
            fatal "! @_[0] found before a valid starting trigger\n! File: $filePath, line: $lc\n";
        }
        if (@_[1] == 2)
        {
            return 0;
        }
    }

    if ($extracting != 1)
    {
        logger "* @_[0] found before a valid starting trigger\n* File: $filePath, line: $lc\n";
        ++$sendWarning;
        return 1;
    }
    return 0;
}

#----------------------------------------------------------SUB-PREPROCESSFILE-
sub preProcessFile
{
    #
    # preProcessFile "filePath", "fileName";
    #
    # "filePath"            is the fully qualified path, filename and extension
    # "fileName"            is the filename and extension only.
    #
    # Performs the first pass processing of a raw source file out to a temporary
    # output file.
    #
    # Returns 1 if processed, 0 if skipped
    #

    $filePath = @_[0];                  # full path and filename
    $fileName = @_[1];                  # just the filename

    #logger ". File: $fileName\n";
    #logger ". File: $filePath\n";

    #
    # First determine if the filename extension is in the @extensions list
    #

    # get the filename extension
    (@parts) = split(/\./, $fileName);
    $ext = $parts[$#parts];             # only use the last <.something>

    # match the file extension against the extensions array
    $i = 0;
    $found = 0;

    MATCH_EXT:
    while ($i < $extensionCount)
    {
        if ($extensions[$i] =~ m/$ext/i)
        {
            $found = 1;
            last MATCH_EXT;
        }
        ++$i;
    }

    if ($found != 1)                    # extension not found, skip it
    {
        return 0;
    }

    #
    # Open the file & prepare to process the content
    #
    if (open(fhFrom, $filePath) != 0)
    {
        logger "+ File: $filePath\n";
    }
    else
    {
        logger "* Cannot open file $filePath for reading, skipping\n";
        ++$endWarning;
        return 0;
    }

    $currNode = "";
    $currSequence = 1;
    $currTitle = "";
    $currSecurity = 0;
    $currFooter = "";
    $extracting = 0;                    # not extracting lines yet
    $lc = 0;                            # reset line count
    $nodeChanged = 0;                   # node has not changed yet
    $outputOpen = 0;                    # an output file has not been opened
    $wasExtracting = 0;                 # haven't been extracting

    # read through the file, extracting document blocks
    EXTRACT:
    while ($l = <fhFrom>)
    {
        ++$lc;
        #print "\r$lc";                         #!D!

        $l =~ s/\n$//;                          # trim trailing newline

        # match for the trigger string, but not two in a row (escaped)
        if ( ($l =~ m=$kwTrigger=i) && ($' !~ m=^${kwTrigger}=i) )
        {
            #print " found:\t$l\n";             #!D!
            $i = index($l, $kwTrigger);
            $w = substr($l, $i);                # pull the tag
            ($w, $junk) = split(/[ \t]/, $w);   # parsed on whitespace
            #print "Word is '$w'\n";            #!D!

            $_ = $w;                            # set match buffer

            SWITCH_PRE:
            {                                   # indentation is left 1 for room
            /$kwNode$/i     && do
                            {
                                $extracting = 1;            # we're now extracting text
                                $nodeChanged = 1;           # the node has changed
                                $currSequence = 1;          # default is unsequenced (0)
                                $currTitle = "";            # reset the current title

                                $l =~ s=.*${kwNode}==i;     # strip front of line including tag
                                $l =~ s/^[ \t]*//;          # trim leading & trailing whitespace
                                $l =~ s/[ \t\n]*$//;
                                #print "NODE_a '$l'\n";     #!D!
                                if (length($l) > 0)
                                {
                                    ($n, @rest) = split(/[ \t]/, $l);
                                    if (isNumeric $n)
                                    {
                                        $currSequence = $n;
                                        $l = substr($l, length($n));
                                        $l =~ s/^[ \t]*//;  # trim leading whitespace
                                        if (length($l) < 1)
                                            { fatal "! $kwNode with sequence, but name missing\n! File: $filePath, line: $lc\n"; }
                                        if ($currSequence < 0 || $currSequence > 99999)
                                            { fatal "! $kwNode sequence number out of range (0-99999)\n! File: $filePath, line: $lc\n"; }
                                        #print "SEQUENCE $n\n";         #!D!
                                    }

                                    $currNode = $l;
                                    #print "NODE_b '$currNode'\n";      #!D!
                                    #getc();                            #!D!

                                    # Populate the metaData array
                                    ++$metaEnd;
                                    $metaCurr = $metaEnd;
                                    $metaData[$metaCurr][$iNode] = $currNode;
                                    $metaData[$metaCurr][$iSequence] = $currSequence;
                                    $metaData[$metaCurr][$iTitle] = "";
                                    # calculate the indent from the number of segments in the node
                                    (@segs) = split(/\//, $currNode);
                                    $metaData[$metaCurr][$iIndent] = $#segs;
                                    $metaData[$metaCurr][$iSource] = $filePath;
                                    $metaData[$metaCurr][$iOutput] = "";
                                    $metaData[$metaCurr][$iBookmark] = "";
                                    $metaData[$metaCurr][$iSecurity] = $currSecurity;
                                    $metaData[$metaCurr][$iFooter] = $currFooter;
                                    $metaData[$metaCurr][$iRefCount] = 0;
                                    #print "ADDING NODE $currNode from $filePath\n";     #!D!

                                    # we cheat and use the node to carry the source
                                    writeTemp "$kwNode $filePath";  # write a node tag to temporary
                                }
                                else                        # find LAST node with a matching source file
                                {                           # use the most recent node and output
                                    #print "LOOKING FOR $filePath\n";                   #!D!
                                    $i = findMeta $iSource, $filePath, 0;
                                    if ($i < 0)
                                        { fatal "! $kwNode with no name found before a named $kwNode\n! File: $filePath, line: $lc\n"; }
                                    $metaCurr = $i;
                                    $currNode = $metaData[$metaCurr][$iNode];
                                    $currSequence = $metaData[$metaCurr][$iSequence];
                                    $currSecurity = $metaData[$metaCurr][$iSecurity];
                                    $currFooter = $metaData[$metaCurr][$iFooter];
                                    #print "*USING NODE $currNode from $filePath\n";    #!D!
                                    # does -not- write node tag marker
                                }
                                next EXTRACT;
                            };
            /$kwEnd$/i      && do
                            {
                                preProcessCheck $w, 0;
                                $extracting = 0;
                                next EXTRACT;
                            };
            /$kwTitle$/i    && do
                            {
                                preProcessCheck $w, 1;
                                $l =~ s=.*${kwTitle}==i;    # strip front of line including tag
                                # only set the title if:
                                #   no title is defined,
                                #   or we are keeping the most recent title instead of the first
                                if ((length($metaData[$metaCurr][$iTitle]) < 1) || $useFirstTitle != 1)
                                {
                                    $currTitle = $l;
                                    $currTitle =~ s/^[ \t]*//;
                                    $currTitle =~ s/[ \t\n]*$//;
                                    if (length($currTitle) < 1)
                                    {
                                        fatal "! No title definition\n! File: $filePath, line: $lc\n";
                                    }
                                    #print "TITLE is '$currTitle'\n";       #!D!
                                    $metaData[$metaCurr][$iTitle] = $currTitle;
                                }
                                next EXTRACT;
                            };
            /$kwSecurity$/i && do
                            {
                                preProcessCheck $w, 1;
                                $l =~ s=.*${kwSecurity}==i;
                                $currSecurity = $l;
                                $currSecurity =~ s/^[ \t]*//;
                                $currSecurity =~ s/[ \t\n]*$//;
                                ($currSecurity, $rest) = split(/[ \t]/, $currSecurity);
                                if (length($currSecurity) < 1)
                                {
                                    fatal "! No security definition\n! File: $filePath, line: $lc\n";
                                }
                                if ($currSecurity < 0 || $currSecurity > 254)
                                {
                                    fatal "! 'Security' value out of range (0-254)\n! File: $filePath, line: $lc\n";
                                }
                                #print "SECURITY is '$currSecurity'\n";     #!D!
                                $metaData[$metaCurr][$iSecurity] = $currSecurity;
                                next EXTRACT;
                            };
            /$kwFooter$/i   && do
                            {
                                preProcessCheck $w, 1;
                                $l =~ s=.*${kwFooter}==i;   # strip front of line including tag
                                $l =~ s/^[ \t]*//;
                                $l =~ s/[ \t\n]*$//;
                                $currFooter = $l;
                                $metaData[$metaCurr][$iFooter] = $l;
                                next EXTRACT;
                            };
            /$kwCode$/i     && do
                            {
                                preProcessCheck $w, 2;
                                $wasExtracting = $extracting;
                                $extracting = 1;
                                last SWITCH_PRE;
                            };
            /$kwEndcode$/i  && do
                            {
                                if (preProcessCheck $w, 0)
                                {
                                    next EXTRACT;
                                }
                                # don't write tag line if we're not already open (node) and extracting (code)
                                if (($extracting == 1) && ($outputOpen == 1))
                                {
                                    writeTemp $l;
                                }
                                $extracting = $wasExtracting;     # reset flag to previous state
                                $wasExtracting = 0;
                                next EXTRACT;
                            };
            if (/$kwTie$/i      ||          # other known tags
                /$kwIndex$/i    ||
                /$kwLink$/i     ||
                /$kwTable$/i    ||
                /$kwRow$/i      ||
                /$kwColumn$/i   ||
                /$kwEndtable$/i )
                            {
                                if (preProcessCheck $w, 1)
                                {
                                    next EXTRACT;
                                }
                                last SWITCH_PRE;
                            };

            logger "* Unknown tag '$w' encountered\n* File: $filePath, line: $lc\n";
            ++$sendWarning;
            } # SWITCH_PRE
        }

        if ($extracting == 1)
        {
            writeTemp $l;               # write the output line; checks security; opens file if necessary
        }
    }

    if ($outputOpen == 1)
    {
        close(outFh);
        $outputOpen = 0;
    }
    close(fhFrom);

    return 1;
} # sub preProcessFile


#--------------------------------------------------------------SUB-READCONFIG-
sub readConfig
{
    #
    # readConfig;
    #
    # Reads and processes the configuration file.  The file must be opened
    # prior to calling this subroutine.
    #
    # Returns: nothing
    #

    $logFileSet = 0;
    $newLogFile = "";

    CFG:
    while ($line = <config>)                # read a line
    {
        ++$ln;                              # incr line number count

        if ($line !~ m/^[ \t]*#/)           # if not a comment line
        {
            $line =~ s/^[ \t]*//;           # trim leading whitespace
            $line =~ s/[ \t\n]*$//;         # trim trailing whitespace and newline

            if (length($line) > 0)          # if there is data
            {
                ($key, $value) = split(/=/,$line);  # split 'key = value'
                $key =~ s/[ \t]*$//;        # trim trailing key whitespace
                $value =~ s/^[ \t]*//;      # trim leading value whitespace
                #print "key='$key'\t\tvalue='$value'\n";        #!D!

                $_ = $key;                  # set the match buffer

                SWITCH:                     # match and set each file variable
                {
                    /boldupper/i    && do
                                    {
                                        if ($optBoldUpperSet == 0)
                                        {
                                            $optBoldUpper = $value;
                                            if (($optBoldUpper != 0) && ($optBoldUpper != 1))
                                            {
                                                fatal "! 'BoldUpper' value out of range (0/1) in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting bold upper:\t\t$optBoldUpper\n";
                                        }
                                        last SWITCH;
                                    };
                    /directory/i    && do
                                    {
                                        if ($optOutDirSet == 0)
                                        {
                                            $optOutDir = $value;
                                            if (length($optOutDir) < 1)
                                            {
                                                fatal "! 'Directory' value not specified in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting output directory:\t$optOutDir\n";
                                        }
                                        last SWITCH;
                                    };
                    /extensions/i   && do
                                    {
                                        if ($optExtensionsSet == 0)
                                        {
                                            $optExtensions = $value;
                                            if (length($optExtensions) < 1)
                                            {
                                                fatal "! 'Extensions' value not specified in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting extensions:\t\t$optExtensions\n";
                                        }
                                        last SWITCH;
                                    };
                    /fileincluded/i && do
                                    {
                                        if ($optIncludeNameSet == 0)
                                        {
                                            $optIncludeName = $value;
                                            if (($optIncludeName != 0) && ($optIncludeName != 1))
                                            {
                                                fatal "! 'FileIncluded' value out of range (0/1) in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting include filename:\t$optIncludeName\n";
                                        }
                                        last SWITCH;
                                    };
                    /footer/i       && do
                                    {
                                        $optFooter = $value;
                                        last SWITCH;
                                    };
                    /killComments/i && do
                                    {
                                        if ($optStripCommentsSet == 0)
                                        {
                                            $optStripComments = $value;
                                            if (($optStripComments != 0) && ($optStripComments != 1))
                                            {
                                                fatal "! 'StripComments' value out of range (0/1) in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting kill comments:\t$optStripComments\n";
                                        }
                                        last SWITCH;
                                    };
                    /linkmode/i     && do
                                    {
                                        if ($optLinkModeSet == 0)
                                        {
                                            if (length($value) < 1)
                                                { fatal "! Missing linkMode value in configuration $cfg, line $ln\n"; }
                                            $optLinkMode = $value;
                                            if ($optLinkMode < 0 || $optLinkMode > 3)
                                            {
                                                fatal "! Unknown link mode '$optLinkMode' in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting link mode:\t\t$optLinkMode\n";
                                        }
                                        last SWITCH;
                                    };
                    /^log$/i        && do
                                    {
                                        $newLogFile = $value;
                                        if (length($newLogFile) < 1)
                                        {
                                            $logDisabled = 1;
                                            logger "= Logging disabled\n";
                                        }
                                        else
                                        {
                                            $logFileSet = 1;
                                            logger "= Setting log:\t\t\t$newLogFile\n";
                                        }
                                        last SWITCH;
                                    };
                    /logoverwrite/i && do
                                    {
                                        $logOverwrite = $value;
                                        if ($logOverwrite != 0 && $logOverwrite != 1)
                                        {
                                            $logOverwrite = 0;
                                            ++$sendWarning;
                                            logger "* logOverwrite value out of range, using $logOverwrite\n";
                                        }
                                        logger "= Setting logOverwrite:\t\t$logOverwrite\n";
                                        last SWITCH;
                                    };
                    /outextension/i && do
                                    {
                                        if ($optOutExtensionSet == 0)
                                        {
                                            $optOutExtension = $value;
                                            if (length($optOutExtension) < 1)
                                            {
                                                fatal "! 'OutExtension' value not specified in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting output extension:\t$optOutExtension\n";
                                        }
                                        last SWITCH;
                                    };
                    /maillevel/i    && do
                                    {
                                        $mailLevel = $value;
                                        if ($mailLevel < 0 || $mailLevel > 4)
                                        {
                                            $mailLevel = $mailLevelVal;
                                            ++$sendWarning;
                                            logger "* mailLevel value out of range, using $mailLevel\n";
                                        }
                                        logger "= Setting mailLevel:\t\t$mailLevel\n";
                                        last SWITCH;
                                    };
                    /mailrecipient/i && do
                                    {
                                        $mailRecipient = $value;
                                        logger "= Setting mailRecipient:\t$mailRecipient\n";
                                        last SWITCH;
                                    };
                    /mailuser/i     && do
                                    {
                                        $mailUser = $value;
                                        logger "= Setting mailUser:\t\t$mailUser\n";
                                        last SWITCH;
                                    };
                    /recursive/i    && do
                                    {
                                        if ($optRecursiveSet == 0)
                                        {
                                            $optRecursive = $value;
                                            if (($optRecursive != 0) && ($optRecursive != 1))
                                            {
                                                fatal "! 'Recursive' value out of range (0/1) in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting recursive:\t\t$optRecursive\n";
                                        }
                                        last SWITCH;
                                    };
                    /secure/i       && do
                                    {
                                        if ($optSecureSet == 0)
                                        {
                                            $optSecure = $value;
                                            if ($optSecure < 0 || $optSecure > 254)
                                            {
                                                fatal "! 'Secure' value out of range (0-254) in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting secure:\t\t$optSecure\n";
                                        }
                                        last SWITCH;
                                    };
                    /security/i     && do
                                    {
                                        if ($optSecuritySet == 0)
                                        {
                                            $optSecurity = $value;
                                            if ($optSecurity < 0 || $optSecurity > 254)
                                            {
                                                fatal "! 'Security' value out of range (0-254) in configuration $cfg, line $ln\n";
                                            }
                                            logger "= Setting security:\t\t$optSecurity\n";
                                        }
                                        last SWITCH;
                                    };
                    /writentlog/i   && do
                                    {
                                        $writeNTlog = $value;
                                        if ($writeNTlog != 0 && $writeNTlog != 1)
                                        {
                                            $writeNTlog = 0;
                                            ++$sendWarning;
                                            logger "* writeNTlog value out of range, using $writeNTlog\n";
                                        }
                                        if ($isWindowsNT != 1)
                                        {
                                            $writeNTlog = 0;
                                        }
                                        logger "= Setting writeNTlog:\t\t$writeNTlog\n";
                                        last SWITCH;
                                    };
                    fatal "! Unknown variable in configuration file '$key'\n";
                }
            }
        }
    }
} # sub readConfig


#-----------------------------------------------------------------SUB-SCANDIR-
sub scanDir
{
    #
    # scanDir "dir", "previous subpath", "add directory"
    #
    # "dir"                 is the fully qualified directory to be opened
    # "previous subpath"    is the previous subpath to this point
    # "add directory"       is the new subdirectory below the previous subpath
    #
    # Scans a directory or file path, one entry at a time. Files first,
    # then subdirectories.
    #
    # Calls 'preProcessFile' for each file found.
    #
    # This subroutine is written to be recursive and reentrant.
    # The subpath is built-up as each subdirectory is recursed into.
    #
    # Use:  scanDir "realdir", "", "";  on the first call.  The other
    # arguments are only used when recursing.
    #

    my $dirName = @_[0];
    my ($r, $entry, $dh, $fr, $subPath, $prevEntry, $entryCount);

    $prevEntry = "";                    # previous directory entry
                                        # used in case of subdirectory deletion
    $entryCount = 0;                    # directory entry count

    #print "scanning $dirName\n";       #!D!

    $subPath = @_[1];                   # copy of the subpath, for recursion

    if (length(@_[2]) > 0)              # add to the subpath
    {
        if (length($subPath) < 1)
            { $subPath = @_[2]; }       # first one, do not add the "\"
        else
            { $subPath = "$subPath$slash@_[2]"; }
    }

    # Open the directory
    # Process all files first, recursively
    # Rewind the directory
    # Process all subdirectories
    $r = opendir dh, $dirName;
    if ($r eq 1)
    {
        logger ". Directory: $dirName\n";

        # Process all files
        while (($entry = readdir dh) ne "" )
        {
            if ($entry ne "." && $entry ne "..")
            {
                ++$entryCount;

                # only process files
                if (not isDirectory "$dirName$slash$entry")
                {
                    #logger ". File: $dirName$slash$entry\n";
                    $pors = preProcessFile "$dirName$slash$entry", "$entry";
                    if ($pors == 1)
                        { ++$processed; }
                    else
                        { ++$skipped; }
                }
            }
        }

        rewinddir dh;

        # Process all subdirectories, recursively
        if ($optRecursive)
        {
            while (($entry = readdir dh) ne "" )
            {
                if ($entry ne "." && $entry ne "..")
                {
                    # only process directories
                    if (isDirectory "$dirName$slash$entry")
                    {
                        close dh;

                        #print "! recurse $dirName$slash$entry\n";      #!D!
                        #getc;                                          #!D!

                        # Recurse into the next subdirectory
                        scanDir "$dirName$slash$entry", $subPath, $entry;

                        #print "! back to $dirName from $entry\n";      #!D!

                        # Reopen and reposition to current subdirectory entry.
                        # This forward logic works because the entry will not
                        # be the first entry, which is always "."
                        $t = opendir dh, $dirName;
                        if ($t eq 1)
                        {
                            my ($find);
                            FIND: while (($find = readdir dh) ne "")
                            {
                                if ($find eq $entry)
                                {
                                    last FIND;
                                }
                            }
                        }
                        else
                        {
                            fatal "! Cannot reopen $dirName to continue from recursion\n";
                        }
                    }
                }
                $prevEntry = $entry;
            }
        }

        close dh;                           # close the directory
    }
    else                                    # not a directory; process an individual file
    {
        if ($dirName ne "." && $dirName ne "..")
        {
            if (-e "$dirName" != 1)         # command line specified file not found
            {
                logger "* Cannot open file $dirName for processing, skipping\n";
                ++$sendWarning;
                ++$skipped;
                return 0;
            }
            else
            {
                my (@segs);

                $entryCount = 1;            # there's at least one entry

                (@segs) = split(/$slashes/, $dirName);
                $entry = $segs[$#segs];     # make entry name from last segment of path
                #logger ". File: $entry\n";

                $pors = preProcessFile "$dirName", "$entry";
                if ($pors == 1)
                    { ++$processed; }
                else
                    { ++$skipped; }
            }
        }
    }

    if ($entryCount == 0)
    {
        logger ". <directory is empty>\n";
    }

    return 0;
} # sub scanDir


#------------------------------------------------------------SUB-STRIPCOMMENT-
sub stripComment
{
    #
    # stripComment "filename", "line to trim"
    #
    # Strips any leading comment and trims leading & trailing whitespace,
    # depending on the value of the optStripComments.  Uses the slComment
    # array to do the file extension association to match the comment
    # string.
    #
    # Returns: Remainder of line
    #

    my($ext, $i, $l, @parts);

    # get the filename extension
    (@parts) = split(/\./, @_[0]);
    $ext = $parts[$#parts];             # only use the last <.something>

    $l = @_[1];
    $singleComment = 0;

    # match the filename extension against the slComment array
    $i = 0;
    SC_FND_EXT:
    while (defined($slComment[$i][$iSlExt]))
    {
        if ($slComment[$i][$iSlExt] =~ m/^${ext}$/i)
        {
            #print "MATCHED $slComment[$i][$iSlExt]\n";         #!D!
            last SC_FND_EXT;
        }
        ++$i;
    }

    if (not defined($slComment[$i][$iSlExt]))   # did not match
        { return $l; }

    #
    # The logic is arranged this way to be able to detect unstripped
    # single-line comments, as well as strip comments.
    # Unstripped single-line comments are handle specially during
    # HTML formatting.
    #
    if ($optStripComments == 1)                 # strip the comment
    {
        $l =~ s/^[ \t]*//;
        $l =~ s/[ \t\n]*$//;
        $l =~ s/^${slComment[$i][$iSlStr]}//;

        #print "STRIPPED $l\n";                 #!D!
        return $l;
    }
    else                                        # see if it is a one line comment
    {
        if ($l =~ m/^[ \t]*${slComment[$i][$iSlStr]}/)
        {
            #print "singleComment !\n";

            # !?! should this be used?  should it be an option?
            $singleComment = 1;                 # flag unstripped single-line comment
        }
        return $l;
    }
} # sub stripComment


#----------------------------------------------------------------SUB-STRIPTAG-
sub stripTag
{
    #
    # stripTag "tag to strip" "data line";
    #
    # Strips the leading tag and trims leading & trailing whitespace.
    #
    # Returns: Remainder of line
    #

    my( $i, $k, $l );

    $k = @_[0];
    $l = @_[1];
    $i = index($l, $kwTrigger);
    $l = substr($l, ($i + length($k)) );
    $l =~ s/^[ \t]*//;                  # trim leading whitespace
    $l =~ s/[ \t\n]*$//;                # trim trailing whitespace & newline
    $l =~ s=${kwTrigger}${kwTrigger}=${kwTrigger}=g;  # removed escaped triggers

    return $l;
} # sub stripTag


#---------------------------------------------------------------SUB-WRITEHTML-
sub writeHtml
{
    #
    # writeHtml "line to write";
    #
    # Trims trailing newline only.
    # Writes a single line to the output file handle outFh.
    #
    # Returns: nothing
    #

    print outFh @_[0];
} # sub writeHtml


#------------------------------------------------------------SUB-WRITEHTMLEND-
sub writeHtmlEnd
{
    #
    # writeHtmlEnd;
    #
    # Write HTML for the end of an output page.
    # Handles completion of missing table tags, generation of forward
    # and back links, and any footer.
    #
    # Returns: nothing
    #

    if ($lastMetaOut < 0)
    {
        fatal "* THIS SHOULD NEVER HAPPEN (5)\n";
        ++$sendWarning;
        return;
    }

    # this is a cheap & dirty way to solve the problem of metaCurr being changed
    # since the last valid output, while looking for more data for the current
    # output file.  It is done with lastMetaOut because one or more files may
    # be skipped before the next file starts to be processed.
    my($saveMeta);
    $saveMeta = $metaCurr;
    $metaCurr = $lastMetaOut;

    # A user CYA just in case they forgot something
    if ($inTableData == 1)
    {
        writeHtml "</TD>\n";
        $inTableData = 0;
    }
    if ($inTableRow == 1)
    {
        writeHtml "</TR>\n";
        $inTableRow = 0;
    }
    if ($inTable == 1)
    {
        writeHtml "</TABLE>\n";
        $inTable = 0;
    }

    # format the ending HTML output
    if ($justWroteParaBreak == 0)
        { writeHtml "\n<BR>\n"; }


    #-------# FORWARD & BACK LINKS
    if ($optLinkMode > 0)
    {
        #-------# Generate Related Links paragraph
        addRelated;

        if ($optLinkMode == 1)
        {
            #-------# Generate BACK paragraph
            addBackLinks;
        }
    }


    #-------# End TABLE 1
    writeHtml "\n<HR WIDTH=\"100%\">\n";

    writeHtml "\n</TABLE>\n";


    #-------# Footer
    if (length($metaData[$metaCurr][$iFooter]) > 0 || length($optFooter) > 0)
    {
        my ($f);
        if (length($metaData[$metaCurr][$iFooter]) > 0)
            { $f = $metaData[$metaCurr][$iFooter]; }
        else
            { $f = $optFooter; }
        writeHtml "\n<FONT FACE=\"Arial\" SIZE=\"-1\">\n";
        writeHtml "\t<DIV ALIGN=RIGHT>$f</DIV>\n";
        writeHtml "</FONT>\n";
        writeHtml "<BR>\n";
    }


    #-------# End BODY
    writeHtml "\n</BODY>\n";


    close(outFh);
    $outputOpen = 0;

    $metaCurr = $saveMeta;
} # sub writeHtmlEnd


#-------------------------------------------------------------SUB-WRITENAVBAR-
sub writeNavBar
{
    #
    # writeNavBar "skip value";
    #
    # skip value is:
    #   0 = do not skip anything
    #   1 = skip home (index) link
    #   2 = skip alphabetical index link
    #   3 = skip document map link
    #
    # Writes the standard Navigation Bar to the output
    #
    # CAUTION ! ! ! Must be called at the correct position in the Table 1 sequence.
    #
    # Returns: nothing
    #

    writeHtml "\n<TD VALIGN=TOP BGCOLOR=MAROON HEIGHT=\"100%\">\n";
    writeHtml "<TABLE HEIGHT=\"100%\">\n";
    writeHtml "\t<TD HEIGHT=\"100%\" VALIGN=TOP><B><FONT COLOR=WHITE>\n";
    writeHtml "\t$progName\n";
    writeHtml "\t</FONT><FONT FACE=\"Arial\"><P ALIGN=RIGHT>\n";
    if (@_[0] != 1)
        { writeHtml "\t<BR><BR><A HREF=$defHomeName.$optOutExtension><FONT COLOR=YELLOW>Home</FONT></A>\n"; }
    if (@_[0] != 3)
        { writeHtml "\t<BR><BR><A HREF=$defMapName.$optOutExtension><FONT COLOR=YELLOW>Map</FONT></A>\n"; }
    if (@_[0] != 2)
        { writeHtml "\t<BR><BR><A HREF=$defIndexName.$optOutExtension><FONT COLOR=YELLOW>Index</FONT></A>\n"; }
    writeHtml "\t</FONT></B></P>\n";
    writeHtml "</TD>\n";
    writeHtml "<TR>\n";
    writeHtml "<TD HEIGHT=\"100\">\n";
    writeHtml "</TD>\n";
    writeHtml "<TR>\n";
    writeHtml "<TD VALIGN=BOTTOM>\n";
    writeHtml "<HR>\n";

    writeHtml "\n<FONT COLOR=WHITE SIZE=-2>\n";
    writeHtml "\t$progName $ver<BR>\n\t<B><I>GrayHill&nbsp;Systems</B></I>\n\tTodd R. Hill<BR><BR>\n";
    writeHtml "</FONT>\n";
    writeHtml "</TD>\n";
    writeHtml "</TABLE>\n";

    writeHtml "</TD>\n";
} # sub writeNavBar


#-----------------------------------------------------------------SUB-WRITEPD-
sub writePd
{
    #
    # writePd;
    #
    # Writes the standard Public Domain meta block to the output
    #
    # Returns: nothing
    #

    writeHtml "\t<META HTTP-EQUIV=\"Content-Type\" CONTENT=\"text/html; charset=iso-8859-1\">\n";
    writeHtml "\t<META NAME=\"Generator\" CONTENT=\"$progName $ver\">\n";
    writeHtml "\t<META NAME=\"Description\" CONTENT=\"\n";
    writeHtml "\t\tThe script is hereby released into the PUBLIC DOMAIN.  Any use, modification,\n";
    writeHtml "\t\tderivative work or other use of this script is expressly granted by the author.\n";
    writeHtml "\t\tNO warrantee, guarantee, or fitness for any particular purpose is expressed or\n";
    writeHtml "\t\timplied.  If you break it, you own both parts.\n";
    writeHtml "\t\t$progName was written at:\n";
    writeHtml "\t\t\tGrayHill Systems, Inc.\n";
    writeHtml "\t\t\tby Todd R. Hill, todd.hill\@grayhillsys.com\">\n";
} # sub writePd


#----------------------------------------------------------------SUB-WRITETMP-
sub writeTemp
{
    #
    # writeTemp "line to write";
    #
    # Checks security before doing anything else.  Returns if security fails.
    # Writes a single line to the output file.
    # Opens the output file if it is not open, or if the node has changed.
    #
    # Returns: nothing
    #

    if ($currSecurity <= $optSecurity)
    {
        if ((not $outputOpen) || $nodeChanged)
        {
            my($found, $i, $outName);

            if (length($currNode) < 1)
            {
                fatal "! Extract process begun before $kwNode for output filename\n! File: $filePath, line: $lc\n";
            }

            if ($outputOpen == 1)
            {
                close outFh;
                $outputOpen = 0;
            }

            $outPath = $currNode;
            $outPath =~ s/[ \t]//g;     # remove any embedded spaces
            $outPath =~ s/\//_/g;       # replace any slashes with underscore

            #print "metaData output $outPath\n";            #!D!

            $outName = sprintf "$optOutDir$slash%05d_%s.tmp", $metaData[$metaCurr][$iSequence], $outPath;

            # see if we have already written to this node & sequence
            $found = 0;
            $i = 0;
            WT_FND_OUT:
            while ($i <= $metaEnd)
            {
                if (length($metaData[$i][$iOutput]) == length($outPath))
                {
                    if ($metaData[$i][$iOutput] eq $outPath &&
                        $metaData[$i][$iSequence] == $metaData[$metaCurr][$iSequence])
                    {
                        #print "Found at $i '$metaData[$i][@_[0]]'\n";  #!D!
                        $found = 1;
                        last WT_FND_OUT;
                    }
                }
                ++$i;
            }
            if ($found == 1)
            {
                #print "appending to node '$outName'\n";    #!D!

                # open in append mode
                open(outFh, ">>$outName") || fatal "! Cannot open file $outName for appending\n";
            }
            else
            {
                #print "writing new node  '$outName'\n";    #!D!

                # open in overwrite mode
                open(outFh, ">$outName") || fatal "! Cannot open file $outName for writing\n";
            }

            $metaData[$metaCurr][$iOutput] = $outPath;
            $nodeChanged = 0;
            $outputOpen = 1;
        }

        print outFh "@_[0]\n";      # write a line
    }
} # sub writeTemp


#============================================================================#
#                               ERROR HANDLING                               #
#============================================================================#


#-------------------------------------------------------------------SUB-ALERT-
sub alert
{
    #
    # alert "subject";
    #
    # Sends a message to the configured recipient, with the configured
    # user account, and attach the log file.
    #
    # Returns: nothing
    #

    if (length($mailUser) > 0 &&
        length($mailRecipient) > 0)
    {
        $subject = @_[0];
        $nowT = localtime();

        # Format the message body
        open(aMsg, ">$tmpFile");
        print aMsg "$progName $ver - Automated Message\n\n";

        print aMsg "$subject\n";
        print aMsg "$nowT\n\n";

        print aMsg "Configuration:\t$cfg\n";
        print aMsg "Log:\t\t$logFile\n";
        print aMsg "Log overwrite:\t";
        if ($logOverwrite == 1)
            { print aMsg "True\n\n"; }
        else
            { print aMsg "False\n\n"; }

        print aMsg "The following attachment contains the log file:\n";
        close(aMsg);

        # send the message, password ("X") is a required placeholder and not used
        print "\nSending notification message ... ";
        system "mapisend -u \"$mailUser\" -p X -r \"$mailRecipient\" -s \"$subject\" -t \"$tmpFile\" -f \"$logFile\"";

        unlink $tmpFile;
    }
    else
    {
        if ($mailLevel > 0)
        {
            logger ". Cannot send message, mailUser or mailRecipient not set\n";
        }
    }
} # sub alert


#-------------------------------------------------------------------SUB-FATAL-
sub fatal
{
    #
    # fatal "message";
    #
    # Logs the message, then goes to COMPLETION processing
    #
    # Returns: nothing
    #

    $sendError = 1;
    logger @_;
    close(config);

    #dumpMeta();                                #!D!

    goto DONE;
} # sub fatal


#------------------------------------------------------------------SUB-LOGGER-
sub logger
{
    #
    # logger "x message";
    #
    # x is a single character denoting the nature of the message:
    #   ! fatal error, and failure completion line
    #   * warning, and warning completion line
    #   . informational
    #   + something was added
    #   = a parameter was set
    #   % Beginning or end banner message
    #   ^ Success completion line
    #
    # message is the message to display / log.
    #
    # Returns: nothing
    #

    print @_;

    # Handle file logging
    if ($logDisabled == 0)
    {
        if (length($logFile) > 0)
        {
            open(alog, ">>$logFile");
            print alog @_;
            close(alog);                # always close to flush
        }
    }
} # sub logger


#-------------------------------------------------------------------SUB-LOGNT-
sub logNT
{
    #
    # logNT type, eventId, data, strings, ...
    #
    # type is one of these:
    #       0=success
    #       1=error
    #       2=warning
    #       4=information
    #       8=audit success
    #      16=audit failure
    #
    # eventId is an integer event code.
    #
    # data is the (binary) information logged.
    #
    # Uses $progName global for SOURCE of event
    #
    # Returns: nothing
    #

    $s = qx{ hostname };                # returns a single line
    $s =~ s/^[ \t]*//;                  # trim leading whitespace
    $s =~ s/[ \t\n]*$//;                # trim trailing whitespace and newline
    if (length($s) > 0)
    {
        $na = $#_;
        if ($na < 3) { return; }

        $ss = @_[3];
        $i = 4;
        while ($i <= $na)
        {
            $ss = "$ss, @_[$i]";
            ++$i;
        }

        # args: server source type category eventId reserved data strings ...
        # Not all Perl implementations may have this extension.
        NTWriteEventLog $s, $progName, @_[0], 0, @_[1], NULL, @_[2], $ss;
    }
} # sub logNT


#============================================================================#
#                                 COMPLETION                                 #
#============================================================================#


#------------------------------------------------------------------------DONE-
DONE:
$nowstr = localtime();
if ($sendError != 0)
{
    if ($sendWarning != 0)
    {
        logger "* $sendWarning Warning(s)\n";   # log warnings
    }
    logger "! Failed\n";
    logger "% $progName $ver - Done $nowstr\n";
    if ($mailLevel >= 1)
    {
        alert "$progName Failed";
    }
    if ($writeNTlog == 1)
    {
        logNT 1, 3, $logFile, "$progName Failed", "See log $logFile for details"
    }
    2;                                  # ERRORS
}
else
{
    if ($sendWarning != 0)
    {
        logger "* $sendWarning Warning(s)\n";
        logger "% $progName $ver - Done $nowstr\n";
        if ($mailLevel >= 2)
        {
            alert "$progName $sendWarning Warning(s)";
        }
        if ($writeNTlog == 1)
        {
            logNT 2, 4, $logFile, "$progName $sendWarning Warning(s)", "See log $logFile for details"
        }
        1;                              # WARNINGS
    }
    else
    {
        logger "^ Successful\n";
        logger "% $progName $ver - Done $nowstr\n";
        if ($mailLevel >= 3)
        {
            alert "$progName Successful";
        }
        if ($writeNTlog == 1)
        {
            logNT 4, 6, $logFile, "$progName Successful", "See log $logFile for details"
        }
        0;                              # SUCCESS
    }
}
print "\n";
