/* PROGRAM TABIN 5.1 converts spaces to tabs

usage : TABIN.exe Myfile.TXT
       - collapses spaces to tabs
       - leaves spaces alone inside " " and ' ' on single line.
       - does not know full C++ \ conventions.
       - trims off ^z Eof char
       - ensures each line ends in CrLf
       - single spaces are never converted to tabs.
*/
/*
Roedy Green
Canadian Mind Products
#208 - 525 Ninth Street
New Westminster BC Canada
V5H 2N6
tel:(604) 777-1804
mailto:roedy@mindprod.com
http://mindprod.com
*/

/*

compiled under TINY model for DOS under Borland C++ 5.01
no floating point needed.

Version 5.1 1998 November 8
- embed Barker address

Version 5.0 1996 October 25
- embed Quathiaski address.
- convert to C from Pascal
- new algorithm without max line length

Previous versions were written in Pascal
- embed new address and phone

The program convert spaces to tabs in the following sorts of files:
        .ASM .C .CMD .CPP .H .HPP .TXT

If you examine the files produced by TABIN, you may sometimes
see long strings of blanks.  You may think you have found a bug
in TABIN.  However, TABIN is not a compressing utility, it is a
tabbing utility.  A tab does not necessarily represent 8 spaces,
it means tab to the next tab stop which are spaced every 8
columns.  Only strings of blanks longer than 2 chars are
replaced by a tab.  Only strings of blanks that stretch all the
way to the next tab stop can be converted to a tab.

The moral of this tale is that if you use Tabin to compress your
source code, keep your columns aligned on columns 1, 9, 17, 25 etc.

Please report bugs and problems to:

Roedy Green
Canadian Mind Products
#208 - 525 Ninth Street
New Westminster BC Canada
V5H 2N6
tel:(604) 777-1804
mailto:roedy@mindprod.com
http://mindprod.com

Source and executables may be freely used for any purpose except military.

Possible future extensions:
There is a C library function to expand wildcards. All that
would be needed is loop in main to handle the multiple filenames.

*/

/* ==================================== */

#define Esc '\x1B'
#define TabSpacing 8

#include <stdlib.h>  /* exit, putenv, _splitpath */
#include <stdio.h>   /* fclose, fgetc, printf, remove, rename, setvbuf */
#include <string.h>  /* strcat, strcmp, strupr */
#include <conio.h>   /* getch */
#include <ctype.h>   /* toupper */
/* ==================================== */

/* use all global variables and no parameter passing for simplicity. */

static FILE *Before;
/* input file containing no ^Z chars, except possibly at the end */

static FILE *After;
/* output file with  spaces collapsed to tabs */

static char * BFilename;
/* name of file we will convert */

static char * AFilename;
/* name of the temporary output file */

static int col = 0; /* 1-based column we have written */
/* 0 = no non-blank chars on line yet */

static int pendingSpaces = 0;  /* spaces to convert to tabs/spaces */

static int insideQuote   = 0;  /* 1 if inside ' ' */

static int insideDQuote  = 0;  /* 1 if inside " " */

/* ==================================== */

/* function prototypes, lowest level first */

void Honk (void);
void Die (void);
void OpenAFile (void);
void OpenBFile (void);
void SafeFilename (void);
void Banner (void);
void HandlePendingSpaces (void);
int  main (int argc,  char *argv[]);

/* ==================================== */

int main( int argc,  char *argv[] )
/* main TABIN */
{
  int c; /* process char by char */

  if ( _osmajor < 3 )
    {
    Banner();
    printf("Oops!  DOS 3.0 or later needed\n");
    Die();
    }
  if ( argc != 2 /*  0=Tabin.Exe 1=MyFile.Txt  */ )
    {
    Banner();
    printf("Oops!  usage:  TABIN  Myfile.TXT\n");
    Die();
    }
  BFilename = *++argv; /*  want first arg, not progname  */
  OpenBFile();    /* Open input "before" file. */
  /* Make sure file exists before */
  /* song and dance about extension. */
  SafeFilename(); /* make sure filename has sane extension */
  OpenAFile();    /* open output "after" file */
  printf("Converting spaces to tabs in %s ...\n",BFilename);

  while ( (c=fgetc(Before)) != EOF )
    {
    switch ( c )
      {
      case ' ':
        /* save up spaces to be handled by tabs */
        ++pendingSpaces;
        break;
      case '\'':
        HandlePendingSpaces();
        /* toggle state if not inside " " */
        if ( !insideDQuote )insideQuote= (~insideQuote) & 1;
        fputc(c,After);
        col++;
        break;
      case '\"':
        HandlePendingSpaces();
        /* toggle state if not inside ' ' */
        if ( !insideQuote ) insideDQuote= (~insideDQuote) & 1;
        fputc(c,After);
        col++;
        break;
      case '\t':
        /* effectively expand tab to 1 to 8 spaces */
        pendingSpaces += (TabSpacing - ((col+pendingSpaces) % TabSpacing));
        break;
      case '\n':
        /* ignore pending/trailing spaces */
        fputc('\n',After);
        /* runtime library expands to CrLf */
        /* Assume quotes never span more than one line. */
        /* This way a stray apostrophe won't throw us off. */
        col = 0;
        pendingSpaces = 0;
        insideQuote= 0;
        insideDQuote= 0;
        break;
      default:
        HandlePendingSpaces();
        fputc(c,After);
        col++;
        break;
      } /* end switch */
    } /* end while */
  /* Rename output to input */
  fclose (Before); fclose (After);
  remove (BFilename);
  rename (AFilename, BFilename);
  return (0);
} /* main TABIN */

/* ==================================== */
void HandlePendingSpaces (void)
{
  int i; /* local loop counter */
  int effect; /* net effect of tab */
  if ( insideQuote | insideDQuote )
    { /* don't convert to tab, put out as raw spaces */
    for ( i=0; i<pendingSpaces; i++ )
      {
      fputc(' ',After);
      }
    col += pendingSpaces;
    pendingSpaces = 0;
    } else
    { /* convert to tabs/spaces */
    while ( (effect = (TabSpacing - (col % TabSpacing))) <= pendingSpaces )
      {
      /* effect will always be in range 1..8 */
      /* So can't get in here unless pendingSpaces > 0 */
      if ( effect == 1 ) fputc(' ', After);
      else fputc('\t',After);
      col += effect;
      pendingSpaces -= effect;
      }
    /* handle remaining spaces after the generated tabs */
    for ( i=0; i<pendingSpaces; i++ )
      {
      fputc(' ',After);
      }
    col += pendingSpaces;
    pendingSpaces = 0;

    } /* end else */

} /* HandlePendingSpaces */

/* ==================================== */
void Banner(void)
{
  /* display copyright banner.  Usually not displayed, just embedded. */

  printf("\n Tabin 5.1 ۲\n"
         "\nFreeware to convert spaces to tabs."
         "\nCopyright (c) 1990,1998 Canadian Mind Products"
         "\n" "#208 - 525 Ninth Street, New Westminster, BC Canada V3M 5T9"
         "\n" "tel:(604) 777-1804   mailto:roedy@mindprod.com   http://mindprod.com"
         "\n" "May be used freely for non-military use only"
         "\n\n");

} /* Banner */
/* ==================================== */

void SafeFilename(void)
{
  /* Ensure appropriate file name extensions.
     good =.ASM .PAS .etc - done without prompt
      bad =.EXE .COM .OBJ - abort
  warning =.DOC & others
  */
  static const char  * GoodExtensions  [] =
  { ".C", ".CPP",  ".H", ".HPP", ".RH", ".IH", ".TXT",
    ".ASM",".PAS",".BAT",".CTL",".CMD",
    ".LST",".MAC",".TXT",".ANS",
    ".USE",".KEY",".HTM",".SQL",".INI",0};
    /* 0 is just end marker */

    static const char * BadExtensions [] =
    { ".EXE",".COM",".OBJ",".DLL",0};

    int Response; /* Y or N, yes Virginia, int, C is weird */

    char Extension[_MAX_EXT];

    int i; /* local loop counter */

    _splitpath(BFilename,
               NULL /* drive */, NULL /* dir */, NULL /* name */, Extension);

    strupr(Extension); /* convert to upper case for compare */

    for ( i=0 ; GoodExtensions[i]; i++ )
      {
      if ( strcmp(Extension,GoodExtensions[i])==0 )
        { /* match, it is Good */
        return;
        }
      }
    for ( i=0 ; BadExtensions[i] ; i++ )
      {
      if ( strcmp(Extension,BadExtensions[i])==0 )
        { /* match, it is bad */
        Banner();
        printf("Oops!  Tabin cannot be used on EXE COM or OBJ files "
               "such as %s\n",
               BFilename);
        Die();
        }
      }
    /* just give a warning */
    printf("Warning!\n"     /* new line to give room for long filename */
           "Tabin is not usually used on %s files such as %s\n",
           Extension,BFilename);
    printf("Do you want to continue and insert the tabs anyway?"
           "  (Y)es (N)o  ");
    while ( 1 ) /* loop forever till user enters Y or N */
      {
      Honk();
      Response = getch();
      /* not echoed because user might hit tab or Enter */
      /* and mess up the screen */
      Response = toupper(Response);
      /* toupper is a macro, so needs simple argument */
      switch ( Response )
        {
        case 'Y':
          printf("Yes\n");
          return;
        case 'N':
          printf("No\n");
          /* fallthru */
        case Esc :
          printf("\nTabin aborted\n");
          Die();
          /* others, keep looping */
        }
      }
  }  /* SafeFileName */

/* ==================================== */

  void OpenBFile(void)
  { /* open input Before file as read-only text */
    if ( ( Before = fopen( BFilename, "rt" ) ) == NULL )
      {
      Banner();
      printf("Oops!  Cannot find file %s\n",BFilename);
      Die();
      }
    setvbuf(Before,NULL,_IOFBF,40*512);
  } /* OpenBFile */

/* ==================================== */

  void OpenAFile(void)
  {
    /* create a uniquely named temporary file to hold converted text */
    /* It must be in the same directory as the BFilename since we will */
    /* rename it later. */

    char drive[_MAX_DRIVE]; /* with colon */
    char dir[_MAX_DIR];     /* with lea/trail \ */
    char name[_MAX_FNAME];
    char ext[_MAX_EXT];     /* with lead . */

    char filepath[_MAX_DRIVE + _MAX_DIR];

    _splitpath(BFilename, drive, dir, name, ext);

    strcpy(filepath, drive);
    strcat(filepath, dir);

    /* Force to current directory if empty */
    if ( strcmp(filepath, "") == 0 )
      strcpy(filepath, ".");
    else
      filepath[strlen(filepath) - 1] = 0;  /* chop trail \ */

    /* stop tempnam from using SET TMP directory instead of filepath */
    /* Sets TMP just for this and any "children" */
    /* processes -- doesn't change parent's TMP */
    putenv("TMP=");

    if ( (AFilename = tempnam(filepath, "TMP")) == NULL )
      {
      printf("Oops!  Cannot create the temporary work file\n\a");
      exit(1);
      }
    if ( (After = fopen(AFilename, "wt")) == NULL )
      {
      printf("Oops! Cannot open work file %s\n\a", AFilename);
      exit(1);
      }

    setvbuf(After,NULL,_IOFBF,40*512);
  } /* end OpenAFile */

/* ==================================== */

  void Die (void)
  {
    Honk();
    fclose (Before);
    fclose (After);
    exit(1);   /* exit with errorlevel = 1 */
  } /* Die */

/* ==================================== */

  void Honk (void)
  {
    /* make a noise */
    printf("\a");
  }

/* =================================== */

/* -30- */
