**********
* FoxPro editor for small HP fonts (about 64K
* file size limit--typically up to 8 points).
* Program needs VGA 50-line mode.  A mouse is
* also very helpful.
*
* Load option on startup expects font file (e.g.,
* 12K+ HPTINY.FNT as extracted with /F option).
* On normal save, file shrinks some when embedded
* leading zeroes are discarded.
*
* Editor includes option to save in packed raster
* format for HPTINY.ASM (small landscape font
* expected with specific characteristics).  The
* HPTINY.ASM equates must be synched with result
* of packed save.
*
* Load also expects to find the initial HP escape
* sequence specifying font ID.  This is not really
* a proper part of the font, but it is usually
* included in font files along with corresponding
* tail sequence to simplify download to printer--
* just a COPY /B to device PRN.
*
* Writing of program relied heavily on chapter on
* soft font creation in an old HP Laserjet II
* Technical Reference Manual (part# 33440-90905).
* Program is not guaranteed to be error-free, but
* it worked fine for the HPTINY font.  There is no
* on-line help here.  Refer to the HP manual or
* else infer what is going on from the source
* code below.
*
* Note that the original HP II requires separate
* portrait and landscape soft fonts for printing.
* All models since HP IIP will rotate fonts, so
* that either a portrait or landscape font file
* download will print in either orientation.
* Newer HP models also include internal scalable
* fonts, so there is less need for soft fonts.
*
* The HPTINY font is a reasonable approximation of
* PC-8 Letter Gothic, but with ASCII 7-15 excluded
* to save space in COM file.
*
* Source here is 50 characters wide to demo a
* 5-column printout by HPTINY.COM (4 pages).
* HPTINY scans file to determine longest line
* and total pages so the width need not be
* specified.
*
* 12 Feb 93 -- CRH.
* 03 Nov 93 -- Include packed font output.
* 18 Jun 95 -- Include big character text dump.
**********
priv lLand, lNew, nSvOrient, cFile, nPick,;
     nCellHgt, nCellWid, nBaseRow, nBaseCol

set talk off
set bell off
set safety off
set deleted on
set display to vga50

store .f. to lLand, lNew
cFile = ""
store 0 to nCellHgt, nCellWid, nBaseRow,;
           nBaseCol, nSvOrient, nPick

define window hdr;
  from 5, 6 to 42, 42;
  title " HP Font Header ";
  system float color scheme 10 shadow
if loadfont()
  read cycle when chknew()
endif
rele wind hdr
set display to cga
close data

**********
* Handle edit/save/dump/quit.
**********
func pushvalid
priv cAns

do case
  case nPick = 1
    do chkorient
    do setparms
    define window box;
      from 1, 73-5-16-nCellWid;
      to max(36, 5+nCellHgt), 74;
      title " Edit Font Characters ";
      system float color scheme 10 shadow
    set cursor off
    do editfont
    set cursor on
    rele wind box
    do chknew

  case nPick = 2
    do chkorient
    do setparms
    do savefont
    do packfont

  case nPick = 3
    do chkorient
    do setparms
    do dumpfont

  case nPick = 4
    clear read
endcase
return .t.

**********
* Check if orientation changed and rotate
* font characters, if so.
**********
proc chkorient
priv nTop, nLft, nHgt, nWid

if nSvOrient = header.orient
  return
endif
nSvOrient = header.orient

select font
scan
  wait wind nowait ;
    "Rotating character: "+str(ordinal,3)
  nTop = top
  nLft = lft
  nWid = wid
  nHgt = hgt
  if header.orient = 1
    repl lft   with -nTop,;
         top   with  nLft+nWid,;
         wid   with  nHgt,;
         hgt   with  nWid,;
         bytes with  nWid * int((nHgt+7)/8),;
         data  with  rotleft(nHgt, nWid, data)
  else
    repl lft   with  nTop-nHgt,;
         top   with -nLft,;
         wid   with  nHgt,;
         hgt   with  nWid,;
         bytes with  nWid * int((nHgt+7)/8),;
         data  with  rotright(nHgt, nWid, data)
  endif
endscan
wait clear

**********
* Set lNew in case edit deletes all characters.
**********
func chknew

go top in font
lNew = eof("font")

**********
* Save font to downloadable file.
**********
proc savefont
priv nHndl, cText, n

cFile = putfile("Save font as...", cFile)
if empty(cFile)
  return
endif
nHndl = fcreate(cFile)
if nHndl = -1
  wait wind "File open error"
  return
endif
wait wind nowait "Writing header..."

sele header
cText = chr(27)+"*c"+ltrim(str(fontid))+"D"+;
        chr(27)+")s"+;
          ltrim(str(64+len(trim(comment))))+"W"+;
        chr(0)+chr(64)+;
        chr(0)+chr(fonttype)+;
        chr(0)+chr(0)+;
        normint(baseline)+;
        normint(cellwid)+;
        normint(cellhgt)+;
        chr(orient)+chr(spacing)+;
        normint(symset)+;
        normint(pitch)+;
        normint(height)+;
        normint(xheight)+;
        signsint(widtype)+chr(style)+;
        signsint(weight)+chr(typeface)+;
        chr(0)+chr(serif)+;
        chr(0)+chr(0)+;
        signsint(udist)+chr(uthick)+;
        normint(theight)+;
        normint(twidth)+;
        chr(0)+chr(0)+;
        chr(0)+chr(0)+;
        chr(pitchext)+chr(heightext)+;
        chr(0)+chr(0)+;
        chr(0)+chr(0)+;
        chr(0)+chr(0)+;
        fontname+trim(comment)

sele font
scan
  wait wind nowait ;
    "Writing character: "+str(ordinal,3)
  cText = cText+;
          chr(27)+"*c"+ltrim(str(ordinal))+"E"+;
          chr(27)+"(s"+ltrim(str(16+bytes))+"W"+;
          chr(4)+chr(0)+;
          chr(14)+chr(1)+;
          chr(iif(lLand,1,0))+chr(0)+;
          signint(lft)+;
          signint(top)+;
          normint(wid)+;
          normint(hgt)+;
          signint(xdelta)+;
          left(data,bytes)
endscan

cText = cText+chr(27)+"*c"+;
        ltrim(str(header.fontid))+"d5F"

n = fwrite(nHndl,cText)
=fclose(nHndl)
if n # len(cText)
  wait wind "File write error"
  erase &cFile
else
  wait wind nowait "Font file written"
endif

**********
* Save in special format for HPTINY.ASM.  Expect
* landscape font with specific characteristics.
**********
proc packfont
priv nHndl, cText, n

if not lLand or header.spacing # 0
  return
endif
wait wind ;
    "Packed save to PACKFONT? (Y/N)" to cText
if not cText $ "Yy"
  return
endif
nHndl = fcreate("PACKFONT")
if nHndl = -1
  wait wind "File open error"
  return
endif
wait wind nowait "Writing packed font..."

cText = ""
sele font
for n = 0 to 255
  wait wind nowait ;
    "Writing character: "+str(ordinal,3)
  locate for char = chr(n)
  if not found()
    cText = cText+;
    chr(0)+;
    chr(0)
  else
    if top > 14 or top < 7 or;
       top-hgt > 7 or top-hgt < 0
      wait wind "Range error:"+str(ordinal,4)+;
                " "+char+str(top,5)+str(hgt,5)
    endif

    cText = cText+;
      chr(8*(lft+header.baseline)+(top-7))+;
      chr(8*wid+(top-hgt))+;
      left(data,bytes)
  endif
next

cText = cText+;
        chr(27)+"*c5F"

sele header  && Next specifies ID 2
cText = cText +;
        chr(27)+"*c2D"+;
        chr(27)+")s64W"+;
        chr(0)+chr(64)+;
        chr(0)+chr(fonttype)+;
        chr(0)+chr(0)+;
        normint(baseline)+;
        normint(cellwid)+;
        normint(cellhgt)+;
        chr(orient)+chr(spacing)+;
        normint(symset)+;
        normint(pitch)+;
        normint(height)+;
        normint(xheight)+;
        signsint(widtype)+chr(style)+;
        signsint(weight)+chr(typeface)+;
        chr(0)+chr(serif)+;
        chr(0)+chr(0)+;
        signsint(udist)+chr(uthick)+;
        normint(theight)+;
        normint(twidth)+;
        chr(0)+chr(0)+;
        chr(0)+chr(0)+;
        chr(pitchext)+chr(heightext)+;
        chr(0)+chr(0)+;
        chr(0)+chr(0)+;
        chr(0)+chr(0)+;
        fontname

sele font
go top
cText = cText +;
        chr(27)+"*c000E"+;
        chr(27)+"(s000W"+;
        chr(4)+chr(0)+;
        chr(14)+chr(1)+;
        chr(1)+chr(0)+;
        signint(0)+;
        signint(0)+;
        normint(0)+;
        normint(0)+;
        signint(xdelta)

n = fwrite(nHndl,cText)
=fclose(nHndl)
if n # len(cText)
  wait wind "File write error"
  erase PACKFONT
else
  wait wind nowait "Packed font file written"
endif

**********
func signsint
para n

return chr(iif(n < 0, n+256, n))

**********
func signint
para n

return normint(iif(n < 0, n+256*256, n))

**********
func normint
para n

return chr(int(n/256)) + chr(n%256)

**********
* Dump as large characters (for HPTINY print).
**********
proc dumpfont
priv aDot, aLine, nEven, nPerRow, nPerCol,;
     n, nOrd, cTmp, cName

cName = "BIGCHAR.TXT"
cName = putfile("Dump font as...", cName)
if empty(cName)
  return
endif

nEven   = 2*int((nCellHgt+1)/2)
nPerRow = int(128/(nCellWid+1))
nPerCol = int(90/(nEven/2+1))
dimen aDot[nEven,nCellWid], aLine[nEven/2+1]
store "" to aLine

set alte to &cName
set alte on
set console off
sele font
for nOrd = 0 to 255
  wait wind nowait ;
    "Dumping big characters: "+str(nOrd,3)
  locate for char = chr(nOrd)
  store .f. to aDot
  if found()
    do filldot
  endif
  cTmp = str(nOrd,3)+":"+str(hgt,3)+"x"+str(wid,3)
  cTmp = padc(strtran(cTmp, " ", "0"), nCellWid+1)
  aLine[1] = aLine[1] + cTmp
  for n = 1 to nEven/2
    aLine[n+1] = aLine[n+1] + getpair(2*n-1) + " "
  next
  if (nOrd+1) % nPerRow = 0 or nOrd = 255
    for n = 1 to alen(aLine)
      ?? aLine[n]
      ?
    next
    store "" to aLine
    if (nOrd+1) % (nPerRow * nPerCol) = 0
      for n = nPerCol*(nEven/2+1) + 1 to 90
        ?
      next
    endif
  endif
next
set console on
close alte

wait wind to cTmp;
  "Print big characters with HPTINY? (Y/N)"
if cTmp $ "Yy"
  run hptiny &cName /w128
endif

**********
* Join row pair using half-block resolution.
**********
func getpair
para nOdd
priv cRet, n

cRet = ""
for n = 1 to nCellWid
  do case
    case not aDot[nOdd,n] and not aDot[nOdd+1,n]
      cRet = cRet + " "
    case     aDot[nOdd,n] and not aDot[nOdd+1,n]
      cRet = cRet + ""
    case not aDot[nOdd,n] and     aDot[nOdd+1,n]
      cRet = cRet + ""
    case     aDot[nOdd,n] and     aDot[nOdd+1,n]
      cRet = cRet + ""
  endcase
next
return cRet

**********
* Set global parms from font data.
**********
proc setparms
priv aVal, cMsg, nBaseDiff

cMsg = ""
lLand = (header.orient # 0)
calculate max(top), min(top-hgt),;
          min(lft), max(lft+wid) to array aVal
go top
if lLand
  nCellWid = aVal[1] - aVal[2]
  nBaseCol = -aVal[2]
  nCellHgt = aVal[4] - aVal[3]
  nBaseRow = aVal[4]
else
  nCellWid = aVal[4] - aVal[3]
  nBaseCol = aVal[3]
  nCellHgt = aVal[1] - aVal[2]
  nBaseRow = -aVal[2]
endif
if lNew
  nCellHgt = header.cellhgt
  nBaseRow = nCellHgt - header.baseline
  nCellWid = header.cellwid
  nBaseCol = 0
endif

nBaseDiff = nCellHgt - nBaseRow
if nCellHgt < header.cellhgt
  nCellHgt = header.cellhgt
endif
if nCellHgt > header.cellhgt
  sele header
  repl header.cellhgt with nCellHgt
  show get header.cellhgt
  sele font
  cMsg = cMsg + ", cell height"
endif

if nCellWid < header.cellwid
  nBaseCol = nBaseCol + ;
    int((header.cellwid-nCellWid)/2)
  nCellWid = header.cellwid
endif
if nCellWid > header.cellwid
  sele header
  repl header.cellwid with nCellWid
  sele font
  show get header.cellwid
  cMsg = cMsg + ", cell width"
endif

if nBaseRow < nCellHgt - header.baseline
  nBaseRow = nCellHgt - header.baseline
endif
if nBaseDiff > nCellHgt - nBaseRow
  nBaseRow = nCellHgt - nBaseDiff
endif
if nBaseRow # nCellHgt - header.baseline
  sele header
  repl header.baseline with nCellHgt - nBaseRow
  show get header.baseline
  sele font
  cMsg = cMsg + ", baseline"
endif

if not empty(cMsg)
  wait wind "Font header adjusted:"+ subs(cMsg,2)
endif

**********
* Display say/get of header data, postponing read.
**********
proc showhdr

activate wind hdr

@ 1, 1 say "    Font ID" get header.fontid   ;
         size 1,5 valid hdrvalid()
@ 2, 1 say "  Font Type" get header.fonttype ;
         size 1,5 valid hdrvalid()
@ 3, 1 say "   Baseline" get header.baseline ;
         size 1,5 valid hdrvalid()
@ 4, 1 say " Cell Width" get header.cellwid  ;
         size 1,5 valid hdrvalid()
@ 5, 1 say "Cell Height" get header.cellhgt  ;
         size 1,5 valid hdrvalid()
@ 6, 1 say "Orientation" get header.orient   ;
         size 1,5 valid hdrvalid()
@ 7, 1 say "    Spacing" get header.spacing  ;
         size 1,5 valid hdrvalid()
@ 8, 1 say " Symbol Set" get header.symset   ;
         size 1,5 valid hdrvalid()
@ 9, 1 say "      Pitch" get header.pitch    ;
         size 1,5 valid hdrvalid()
@10, 1 say "     Height" get header.height   ;
         size 1,5 valid hdrvalid()
@11, 1 say "   X-Height" get header.xheight  ;
         size 1,5 valid hdrvalid()
@12, 1 say " Width Type" get header.widtype  ;
         size 1,5 valid hdrvalid()
@13, 1 say "      Style" get header.style    ;
         size 1,5 valid hdrvalid()
@14, 1 say "Font Weight" get header.weight   ;
         size 1,5 valid hdrvalid()
@15, 1 say "   Typeface" get header.typeface ;
         size 1,5 valid hdrvalid()
@16, 1 say "      Serif" get header.serif    ;
         size 1,5 valid hdrvalid()
@17, 1 say " UndLn Dist" get header.udist    ;
         size 1,5 valid hdrvalid()
@18, 1 say "UndLn Thick" get header.uthick   ;
         size 1,5 valid hdrvalid()
@19, 1 say "Text Height" get header.theight  ;
         size 1,5 valid hdrvalid()
@20, 1 say " Text Width" get header.twidth   ;
         size 1,5 valid hdrvalid()
@21, 1 say "  Pitch Ext" get header.pitchext ;
         size 1,5 valid hdrvalid()
@22, 1 say " Height Ext" get header.heightext;
         size 1,5 valid hdrvalid()
@23, 1 say "  Font Name" get header.fontname ;
         size 1,16

@25, 1 say "* = ignored by LaserJet series II"

@27, 0 say padc(" Comment ",35,"")

@29, 1 edit header.comment size 4,33

@34, 4 get nPick func "*NH Edit;Save;Dump;Quit";
                 size 1,6;
                 valid pushvalid()

do hdrshow

**********
* Display header text--some text varies.
**********
func hdrshow

@ 1,19 say "arbitrary"
@ 2,19 say ftypetxt()
@ 3,19 say "dots"
@ 4,19 say "dots"
@ 5,19 say "dots"
@ 6,19 say orienttxt()
@ 7,19 say spctxt()
@ 8,19 say symtxt()
@ 9,19 say "quarter dots"
@10,19 say "quarter dots"
@11,18 say "*quarter dots"
@12,18 say "*"+widtypetxt()
@13,19 say styletxt()
@14,19 say weighttxt()
@15,19 say facetxt()
@16,18 say "*"+seriftxt()
@17,19 say "dots"
@18,18 say "*dots"
@19,18 say "*quarter dots"
@20,18 say "*quarter dots"
@21,19 say "1/1024 dots"
@22,19 say "1/1024 dots"

**********
* Validate some header data.
**********
func hdrvalid

do case
  case _curobj=objnum(header.fontid)
    if not header.fontid > 0
      wait wind nowait ;
        "Font ID must be greater than 0"
      return 0
    endif
  case _curobj=objnum(header.fonttype)
    if not between(header.fonttype,0,2)
      wait wind nowait "Type must be 0, 1, or 2"
      return 0
    endif
  case _curobj=objnum(header.baseline)
    if not between(header.baseline,0,;
                   header.cellhgt-1)
      wait wind nowait ;
        "Baseline must be 0 to "+;
        "cell height less one"
      return 0
    endif
  case _curobj=objnum(header.cellwid)
    if not between(header.cellwid,1,40)
      wait wind nowait ;
        "Cell width must be 1 to 40 "+;
        "(for this program)"
      return 0
    endif
  case _curobj=objnum(header.cellhgt)
    if not between(header.cellhgt,1,80)
      wait wind nowait ;
        "Cell height must 1 to 80 "+;
        "(for this program)"
      return 0
    endif
  case _curobj=objnum(header.orient)
    if not between(header.orient,0,1)
      wait wind nowait ;
        "Orientation must be 0 or 1"
      return 0
    endif
  case _curobj=objnum(header.spacing)
    if not between(header.spacing,0,1)
      wait wind nowait ;
        "Spacing must be 0 or 1"
      return 0
    endif
endcase
do hdrshow
return

**********
* Following functions return text equivalents
* for some HP header data.  See HP tech
* reference guide for details.
**********

**********
func ftypetxt
priv n
n = header.fonttype
do case
  case n = 0
    return "low ASCII only"
  case n = 1
    return "high/low ASCII"
  case n = 2
    return "full PC-8 font"
  otherwise
    return "unknown       "
endcase

**********
func orienttxt
priv n
n = header.orient

do case
  case n = 0
    return "portrait "
  case n = 1
    return "landscape"
  otherwise
    return "unknown  "
endcase

**********
func spctxt
priv n
n = header.spacing

do case
  case n = 0
    return "fixed       "
  case n = 1
    return "proportional"
  otherwise
    return "unknown     "
endcase

**********
func symtxt
priv n
n = header.symset

do case
  case n = 2
    return "HP line draw"
  case n = 277
    return "HP roman-8  "
  case n = 341
    return "HP PC-8     "
  otherwise
    return "other       "
endcase

**********
func styletxt
priv n
n = header.style

do case
  case n = 0
    return "upright"
  case n = 1
    return "italics"
  otherwise
    return "unknown"
endcase

**********
func widtypetxt
priv n
n = header.widtype

do case
  case n = -2
    return "condensed"
  case n = -1
    return "semi-cond"
  case n = 0
    return "normal   "
  case n = 1
    return "semi-expn"
  case n = 2
    return "expanded "
  otherwise
    return "unknown  "
endcase

**********
func weighttxt
priv n
n = header.weight

do case
  case n < 0
    return "light "
  case n > 0
    return "bold  "
  otherwise
    return "normal"
endcase

**********
func facetxt
priv n
n = header.typeface
do case
  case n = 0
    return "line printer "
  case n = 3
    return "courier      "
  case n = 4
    return "helvetica    "
  case n = 5
    return "times roman  "
  case n = 6
    return "letter gothic"
  case n = 8
    return "prestige     "
  case n = 11
    return "presentations"
  otherwise
    return "other        "
endcase

**********
func seriftxt
priv n
n = header.serif
do case
  case n = 0
    return "square        "
  case n = 1
    return "rounded       "
  case n = 2
    return "serif line    "
  case n = 3
    return "serif triangle"
  case n = 4
    return "serif swath   "
  case n = 5
    return "serif block   "
  case n = 6
    return "serif bracket "
  case n = 7
    return "round bracket "
  case n = 8
    return "flair stroke  "
  otherwise
    return "unknown       "
endcase

**********
* Edit font.  Region is 1/2/3 for character
* list/options/edit box.  Tab between them.
* See also F5/F6/F7 below.
**********
proc editfont
priv nKey, nDelta, cChar, cPrev, aDot,;
     aGrid, lRead, lGrid, lKey,;
     nRegion, aPtr, aPrev, aWid, aHgt,;
     lRedraw, cMenu, nRow, nCol

dimen aDot[nCellHgt,nCellWid],;
      aGrid[nCellHgt,nCellWid],;
      aPtr[3], aPrev[3], aWid[3], aHgt[3]

cMenu = "NextPrevDel CopySaveGridQuit"
store 0 to aPtr, aPrev
store " " to aGrid, cChar, cPrev
store .t. to aDot, lRead, lGrid, lRedraw, lKey
nRegion = 1
aWid[1] = 16
aWid[2] = 1
aWid[3] = nCellWid
aHgt[1] = 16
aHgt[2] = len(cMenu)/4
aHgt[3] = nCellHgt

activate window box

do init
do while .t.
  if lRead
    lRead = .f.
    lReDraw = .t.
    cPrev = cChar
    cChar = char
    nDelta = xdelta
    do filldot
  endif
  do dispchr
  nKey = inkey(0,"MH")
  lKey = .t.
  do case
    case nKey = 151  && Mouse click
      lKey = .f.
      if mousechk(mrow(),mcol(),mrow(""),mcol(""))
        exit
      endif
    case nKey = 9   && Tab
      nRegion = nRegion % 3 + 1
    case nKey = 15  && Shifttab
      nRegion = (nRegion-2) % 3 + 1
    case nKey = 19  && Left
      aPrev[nRegion] = aPtr[nRegion]
      aPtr[nRegion]  = ;
        (aPtr[nRegion]-1) %;
        (aWid[nRegion]*aHgt[nRegion])
    case nKey = 4   && Right
      aPrev[nRegion] = aPtr[nRegion]
      aPtr[nRegion]  =;
        (aPtr[nRegion]+1) %;
        (aWid[nRegion]*aHgt[nRegion])
    case nKey = 5   && Up
      aPrev[nRegion] = aPtr[nRegion]
      aPtr[nRegion]  =;
        (aPtr[nRegion]-aWid[nRegion]) %;
        (aWid[nRegion]*aHgt[nRegion])
    case nKey = 24  && Down
      aPrev[nRegion] = aPtr[nRegion]
      aPtr[nRegion]  =;
        (aPtr[nRegion]+aWid[nRegion]) %;
        (aWid[nRegion]*aHgt[nRegion])
    case nKey = 27
      exit
    case nKey = 18  && PgUp
      do shift with -1,0
    case nKey = 3   && PgDn
      do shift with +1,0
    case nKey = 1   && Home
      do shift with  0,-1
    case nKey = 6   && End
      do shift with  0,+1
    case nKey = 13
      do case
        case nRegion = 1
          do seekchr with chr(aPtr[1])
        case nRegion = 3
          nRow = int(aPtr[3]/nCellWid)+1
          nCol = aPtr[3]%nCellWid+1
          aDot[nRow,nCol] = not aDot[nRow,nCol]
          lRedraw = .t.
        case nRegion = 2
          do case
            case aPtr[2] = 0
              do nextchr
            case aPtr[2] = 1
              do prevchr
            case aPtr[2] = 2
              do delchr
            case aPtr[2] = 3
              do copy
            case aPtr[2] = 4
              do savechr
            case aPtr[2] = 5
              lGrid = not lGrid
              lRedraw = .t.
            case aPtr[2] = 6
              exit
        endcase
      endcase
    case nKey = -6  && F7 -- ctrl-F7 is "d"
      wait wind nowait ;
        "Move window then press Enter"
      do while .t.
        nKey = at(chr(inkey(0,"MH")%256),;
                  chr(19)+chr(5)+chr(4)+chr(24))
        if nKey = 0
          exit
        endif
        move window box by;
          iif(nKey%2 = 0, nKey-3, 0),;
          iif(nKey%2 = 1, nKey-2, 0)
      enddo
    case nKey = -5  && F6 -- shortcut
      do savechr
    case nKey = -4 and header.spacing # 0  && F5
      do eddelta
    case nKey >= 32 and nKey <= 255
      nRegion = 1
      aPrev[1] = aPtr[1]
      aPtr[1] = nKey
      do seekchr with chr(nKey)
    otherwise

  endcase
enddo

**********
* Initialize character edit screen.
**********
proc init
priv n, cLft, cRht, cLine, nRow, nCol, c, nBase

nBase = nCellHgt-nBaseRow+1

@ 0,0 to 17,17
cLine = ""
for n = 1 to 256
  cLine = cLine + chr(n-1)
  if n % 16 = 0
    @ int(n/16), 1 say cLine
    cLine = ""
  endif
next

for nRow = 1 to nCellHgt
  for nCol = 1 to nCellWid
    do case
      case (nRow-nBase) % 4 = 0 and nCol % 4 = 0
        c = iif(nRow = nBase,"","")
      case (nRow-nBase) % 4 = 0
        c = iif(nRow = nBase,"","")
      case nCol % 4 = 0
        c = ""
      otherwise
        c = " "
    endcase
    aGrid[nRow,nCol] = c
  next
next

@ 0,18 say ""+left(repl("",20),nCellWid)+""
for n = 1 to nCellHgt
  store "" to cLft,cRht
  if (n - nBase) % 4 = 0
    cLft = iif(n = nBase,"","")
    cRht = iif(n = nBase,"","")
  endif
  @ n,18 say cLft + space(nCellWid) + cRht
next
@ n,18 say ""+left(repl("",20),nCellWid)+""
@ n+1,18 say "" +space(nCellWid)+""
if nCellWid >= 8
  @ n+1,19 say "HOME"
  @ n+1,18+nCellWid-2 say "END"
endif
if nCellHgt >=9
  @ 1,20+nCellWid say "P"
  @ 2,20+nCellWid say "G"
  @ 3,20+nCellWid say "U"
  @ 4,20+nCellWid say "P"
  @ nCellHgt-3,20+nCellWid say "P"
  @ nCellHgt-2,20+nCellWid say "G"
  @ nCellHgt-1,20+nCellWid say "D"
  @ nCellHgt  ,20+nCellWid say "N"
endif

for n = 0 to len(cMenu)/4 - 1
  @ 20 + 2*n, 6 say ;
     "<" + subs(cMenu,4*n+1,4) + ">"
next

**********
* Fill aDot array for current character.
**********
proc filldot
priv cData, cStr, nRow, nCol,;
     nHgt, nWid, nLft, nTop, nLen

aDot  = .f.   && Global in editfont
if lLand
  cData = data
  nHgt  = wid
  nWid  = hgt
  nLft  = nBaseCol+top
  nTop  = nCellHgt+1-nBaseRow+lft
  nLen  = int((nHgt+7)/8)

  for nCol = 0 to nWid - 1
    cStr = subs(cData,1 + nLen*nCol, nLen)
    for nRow = 0 to nHgt - 1
      aDot[nTop+nRow, nLft-nCol] =;
        (bit(cStr,nRow)=1)
    next
  next
else
  cData = data
  nHgt  = hgt
  nWid  = wid
  nLft  = lft-nBaseCol+1
  nTop  = nCellHgt+1-nBaseRow-top
  nLen  = int((nWid+7)/8)

  for nRow = 0 to nHgt - 1
    cStr = subs(cData,1 + nLen*nRow, nLen)
    for nCol = 0 to nWid - 1
      aDot[nTop+nRow, nLft+nCol] =;
        (bit(cStr,nCol)=1)
    next
  next
endif

**********
* Rotate passed character data left 90 degrees.
**********
func rotleft
para nHgt, nWid, cData
priv nRow, nCol, cNew, nVal, nRowBytes, cStr

nRowBytes = int((nWid+7)/8)
cNew = ""
for nCol = nWid-1 to 0 step -1
  nVal = 0
  for nRow = 0 to nHgt-1
    cStr = subs(cData,nRowBytes*nRow+1,nRowBytes)
    nVal = nVal + int(2**(7-(nRow%8))) *;
                  bit(cStr,nCol)
    if nRow % 8 = 7
      cNew = cNew + chr(nVal)
      nVal = 0
    endif
  next
  if nHgt % 8 # 0
    cNew = cNew + chr(nVal)
  endif
next
return cNew

**********
* Rotate passed character data right 90 degrees.
**********
func rotright
para nHgt, nWid, cData
priv nRow, nCol, cNew, nVal, nRowBytes, cStr

nRowBytes = int((nWid+7)/8)
cNew = ""
for nCol = 0 to nWid-1
  nVal = 0
  for nRow = nHgt-1 to 0 step -1
    cStr = subs(cData,nRowBytes*nRow+1,nRowBytes)
    nVal = nVal + int(2**(7-((nHgt-1-nRow)%8))) *;
                  bit(cStr,nCol)
    if (nHgt-1-nRow) % 8 = 7
      cNew = cNew + chr(nVal)
      nVal = 0
    endif
  next
  if nHgt % 8 # 0
    cNew = cNew + chr(nVal)
  endif
next
return cNew

**********
* Return bit 0/1 in string, given zero-based
* position.
**********
func bit
para cStr, nPos

return int(asc(subs(cStr, int(nPos/8)+1))/;
           int(2 ** (7-(nPos % 8)))) % 2

**********
* Locate character in cursor, appending new stub
* if not found and if append confirmed.
**********
proc seekchr
para c
priv cAns, cSave

cSave = char
locate for char = c
if not found()
  wait wind "Character "+ltrim(str(asc(c)))+;
            " not found--add? (Y/N)" to cAns
  if cAns $ "Yy"
    append blank
    repl ordinal with asc(c),;
         char    with c,;
         xdelta  with header.twidth
    aDot = .f.
    aDot[nCellHgt/2,nCellWid/2] = .t.
    do savechr
  else
    locate for char = cSave
  endif
endif
lRead = .t.

**********
* Save character back to cursor from aDot array.
**********
proc savechr
priv cData, nTop, nLft, nHgt, nWid

cData = ""
store 0 to nTop, nLft, nHgt, nWid
do squeeze with nTop, nLft, nHgt, nWid, cData
if lLand
  repl lft    with nTop-1-nCellHgt+nBaseRow,;
       top    with nLft+nWid-1-nBaseCol,;
       wid    with nHgt,;
       hgt    with nWid,;
       xdelta with nDelta,;
       bytes  with nWid * int((nHgt+7)/8),;
       data   with cData
else
  repl lft    with nLft-1+nBaseCol,;
       top    with -(nTop-1-nCellHgt+nBaseRow),;
       wid    with nWid,;
       hgt    with nHgt,;
       xdelta with nDelta,;
       bytes  with nHgt * int((nWid+7)/8),;
       data   with cData
endif
wait wind "Character saved..." timeout 1
return

**********
* Return top/left/height/width/data of squeezed
* aDot array.  Empty data allowed.  ASCII 32 not
* needed in a fixed-space font, but may be
* included in proportional font to set xdelta.
**********
proc squeeze
para nTop, nLft, nHgt, nWid, cData
priv nRow, nCol, n, nVal

nWid = nCellWid
do while nWid > 0 and not coldirty(nWid)
  nWid = nWid - 1
enddo
nLft = 1
do while nLft < nCellWid and not coldirty(nLft)
  nLft = nLft + 1
  nWid = nWid - 1
enddo
nHgt = nCellHgt
do while nHgt > 0 and not rowdirty(nHgt)
  nHgt = nHgt - 1
enddo
nTop = 1
do while nTop < nCellHgt and not rowdirty(nTop)
  nTop = nTop + 1
  nHgt = nHgt - 1
enddo
if nHgt <= 0  && No data, e.g., ASCII 32 or 255
  store 1 to nHgt, nWid
  nTop = int(nCellHgt/2)
  nLft = int(nCellWid/2)
endif
cData = ""
if lLand
  for nCol = nLft + nWid - 1 to nLft step -1
    n = 7      && Bit pointer
    nVal = 0   && Accumulate 8 bits
    for nRow = nTop to nTop + nHgt - 1
      nVal = nVal + iif(aDot[nRow,nCol], 2**n, 0)
      n = n - 1
      if n < 0
        cData = cData + chr(nVal)
        n = 7
        nVal = 0
      endif
    next
    if n < 7
      cData = cData + chr(nVal)
    endif
  next
else
  for nRow = nTop to nTop + nHgt - 1
    n = 7      && Bit pointer
    nVal = 0   && Accumulate 8 bits
    for nCol = nLft to nLft + nWid - 1
      nVal = nVal + iif(aDot[nRow,nCol], 2**7, 0)
      n = n - 1
      if n < 0
        cData = cData + chr(nVal)
        n = 7
        nVal = 0
      endif
    next
    if n < 7
      cData = cData + chr(nVal)
    endif
  next
endif

**********
* Return true if nCol dirty.
**********
func coldirty
para nCol
priv nRow

for nRow = 1 to nCellHgt
  if aDot[nRow,nCol]
    return .t.
  endif
next
return .f.

**********
* Return true if nRow dirty.
**********
func rowdirty
para nRow
priv nCol

for nCol = 1 to nCellWid
  if aDot[nRow,nCol]
    return .t.
  endif
next
return .f.

**********
* Refresh display as flagged.
**********
proc dispchr
priv nRow, nCol, cLine, n, cColor

if lRedraw
  lRedraw = .f.
  for nRow = 1 to nCellHgt
    cLine = ""
    for nCol = 1 to nCellWid
      cLine = cLine +;
        iif(aDot[nRow, nCol],"",;
            iif(lGrid,aGrid[nRow,nCol]," "))
    next
    @ nRow, 19 say cLine
  next
  @ 18,1  say padr(ltrim(str(asc(cChar))),3)
  @ 18,16 say cChar
endif
if header.spacing # 0
  cColor = scheme(10,5)
  @ 18,7 say str(nDelta,4) color &cColor
endif

cColor = scheme(10,iif(nRegion=1,2,1))
n = aPrev[1]
@ int(n/16)+1, n%16+1 say chr(n)
n = aPtr[1]
@ int(n/16)+1, n%16+1 say chr[n] color &cColor

cColor = scheme(10,iif(nRegion=2,2,1))
n = aPrev[2]
@ 20 + 2*n,6 say "<" + subs(cMenu,4*n+1,4) + ">"
n = aPtr[2]
@ 20 + 2*n,6 say "<" + subs(cMenu,4*n+1,4) + ">";
  color &cColor

cColor = scheme(10,iif(nRegion=3 and lKey,2,1))
n = aPrev[3]
nRow = int(n/nCellWid)+1
nCol = n%nCellWid+1
@ nRow,nCol+18 say;
    iif(aDot[nRow, nCol],"",;
        iif(lGrid,aGrid[nRow,nCol]," "))
n = aPtr[3]
nRow = int(n/nCellWid)+1
nCol = n%nCellWid+1
@ nRow,nCol+18 say;
    iif(aDot[nRow, nCol],"",;
        iif(lGrid,aGrid[nRow,nCol]," "));
        color &cColor

**********
* Edit quarter dot width (proportional font only).
**********
proc eddelta

@ 18,7 get nDelta size 1,4
set cursor on
read
set cursor off

**********
* Handle mouse click, returning true if done.
**********
func mousechk
para nRow, nCol, nScrRow, nScrCol
priv lDot

do case
  case (nRow = -1 or nCol = -1) and ;
        mwindow("box")
    do while mdown()
      nRow = mrow("")
      nCol = mcol("")
      move window box by ;
        nRow-nScrRow, nCol-nScrCol
      nScrRow = nRow
      nScrCol = nCol
    enddo
  case between(nRow,1,nCellHgt) and;
       between(nCol,19,18+nCellWid)
    nRegion = 3
    aPrev[3] = aPtr[3]
    aPtr[3] = (nRow-1)*nCellWid + nCol-19
    lDot = not aDot[nRow,nCol-18]
    aDot[nRow,nCol-18] = lDot
    do while mdown()
      nRow = mrow()
      nCol = mcol()
      if not between(nRow,1,nCellHgt) or;
         not between(nCol,19,18+nCellWid)
        exit
      endif
      aDot[nRow,nCol-18] = lDot
    enddo
    lRedraw = .t.
  case between(nRow,1,16) and;
       between(nCol,1,16)
    nRegion = 1
    aPrev[1] = aPtr[1]
    aPtr[1] = (nRow-1)*16 + nCol-1
    do seekchr with chr(16*(nRow-1)+nCol-1)
  case between(nRow,20,20+len(cMenu)/2-2) and;
       between(nCol,6,11) and;
       nRow % 2 = 0
    nRegion = 2
    aPrev[2] = aPtr[2]
    aPtr[2] = int(nRow/2-10)
    do case
      case nRow = 20  && Next
        do nextchr
      case nRow = 22  && Prev
        do prevchr
      case nRow = 24  && Delete
        do delchr
      case nRow = 26  && Copy
        do copy
      case nRow = 28  && Save
        do savechr
      case nRow = 30  && Grid
        lGrid = not lGrid
        lRedraw = .t.
      case nRow = 32  && Quit
        return .t.
    endcase
  case nRow = 0          and nCol = 20+nCellWid
    do shift with -1, 0
  case nRow = nCellHgt+1 and nCol = 20+nCellWid
    do shift with +1, 0
  case nRow = nCellHgt+2 and nCol = 18
    do shift with  0, -1
  case nRow = nCellHgt+2 and nCol = 19+nCellWid
    do shift with  0, +1
  case header.spacing # 0 and;
       nRow = 18 and;
       between(nCol,8,11)
    do eddelta
  otherwise
    * Ignore
endcase
return .f.

**********
* Handle vertical/horizontal character shift,
* adjusting aDot array and flagging refresh.
**********
func shift
para nRowShf, nColShf
priv nRow, nCol

do while .t.
  if nRowShf < 0 and not rowdirty(1)
    for nCol = 1 to nCellWid
      for nRow = 1 to nCellHgt-1
        aDot[nRow,nCol] = aDot[nRow+1,nCol]
      next
      aDot[nRow,nCol] = .f.
    next
    lRedraw = .t.
  endif
  if nRowShf > 0 and not rowdirty(nCellHgt)
    for nCol = 1 to nCellWid
      for nRow = nCellHgt to 2 step -1
        aDot[nRow,nCol] = aDot[nRow-1,nCol]
      next
      aDot[nRow,nCol] = .f.
    next
    lRedraw = .t.
  endif
  if nColShf < 0 and not coldirty(1)
    for nRow = 1 to nCellHgt
      for nCol = 1 to nCellWid-1
        aDot[nRow,nCol] = aDot[nRow,nCol+1]
      next
      aDot[nRow,nCol] = .f.
    next
    lRedraw = .t.
  endif
  if nColShf > 0 and not coldirty(nCellWid)
    for nRow = 1 to nCellHgt
      for nCol = nCellWid to 2 step -1
        aDot[nRow,nCol] = aDot[nRow,nCol-1]
      next
      aDot[nRow,nCol] = .f.
    next
    lRedraw = .t.
  endif
  do dispchr
  if not mdown()
    exit
  endif
enddo

**********
func nextchr

if not eof()
  skip
endif
lRead = .t.
if eof()
  go bott
endif
aPrev[1] = aPtr[1]
aPtr[1] = asc(char)

**********
func prevchr

if not bof()
  skip -1
endif
lRead = .t.
if bof()
  go top
endif
aPrev[1] = aPtr[1]
aPtr[1] = asc(char)

**********
func delchr
priv cAns

wait wind "Delete character? (Y/N)" to cAns
if cAns $ "Yy"
  delete
  do nextchr
endif

**********
func copy
priv cAns

if not eof()
  wait wind ;
    "Copy character "+ltrim(str(asc(cPrev)))+;
    " over current one? (Y/N)" to cAns
  if cAns $ "Yy"
    scatter memvar memo
    seek cPrev
    if found()
      scatter memvar memo
    endif
    seek cChar
    m.char = cChar
    m.ordinal = asc(cChar)
    gather memvar memo
    lRead = .t.
  endif
endif

**********
* Load font file (requires font ID prefix) to
* single record header cursor and multi-record
* font cursor.  Return false on error.
**********
func loadfont
priv nHndl, cText, nFontID, nHdrWid

cFile = getfile("","Font file...","Load",1)
if empty(cFile)
  return .f.
endif
lNew = "Untitled" $ cFile
if not lNew
  nHndl = fopen(cFile)
  if nHndl = -1
    wait wind "Unable to open file"
    return .f.
  endif
  wait wind nowait "Reading file..."
  cText = fread(nHndl, 65000)
  =fclose(nHndl)
  nFontID = getval(chr(27)+"*c", "D", .t.)
  if nFontID = -1
    wait wind "Font ID missing"
    return .f.
  endif
  nHdrWid = getval(chr(27)+")s","W")
  if nHdrWid = -1
    wait wind "Font header size missing"
    return .f.
  endif
  if nHdrWid < 64 or len(cText) < nHdrWid
    wait wind "Font header too small"
    return .f.
  endif
endif

sele 0
create cursor header;
  (fontid    n(5),;
   fonttype  n(3),;
   baseline  n(5),;
   cellwid   n(5),;
   cellhgt   n(5),;
   orient    n(3),;
   spacing   n(1),;
   symset    n(5),;
   pitch     n(5),;
   height    n(5),;
   xheight   n(5),;
   widtype   n(3),;
   style     n(3),;
   weight    n(3),;
   typeface  n(3),;
   serif     n(3),;
   udist     n(3),;
   uthick    n(3),;
   theight   n(5),;
   twidth    n(5),;
   pitchext  n(3),;
   heightext n(3),;
   fontname  c(16),;
   comment   m)

if lNew
  do newhdr
else
  do readhdr
endif
nSvOrient = orient  && For change detection
do showhdr

sele 0
create cursor font;
  (ordinal n(3),;
   char    c(1),;
   lft     n(3),;
   top     n(3),;
   wid     n(3),;
   hgt     n(3),;
   xdelta  n(3),;
   bytes   n(3),;
   data    m)

index on char tag char

if not lNew
  do readfont
endif

return .t.

**********
* Create default font header record.
**********
proc newhdr

append blank
repl fontid    with 1,;
     fonttype  with 2,;
     baseline  with 22,;
     cellwid   with 14,;
     cellhgt   with 28,;
     orient    with 0,;
     spacing   with 0,;
     symset    with 341,;
     pitch     with 48,;
     height    with 100,;
     xheight   with 48,;
     widtype   with 0,;
     style     with 0,;
     weight    with 0,;
     typeface  with 0,;
     serif     with 0,;
     udist     with -5,;
     uthick    with 2,;
     theight   with 120,;
     twidth    with 48,;
     pitchext  with 0,;
     heightext with 0,;
     fontname  with "Untitled"

**********
* Create header record from beginning of cText.
**********
proc readhdr

append blank
repl fontid    with nFontID,;
     fonttype  with asc(subs(cText,4)),;
     baseline  with word(subs(cText,7)),;
     cellwid   with word(subs(cText,9)),;
     cellhgt   with word(subs(cText,11)),;
     orient    with asc(subs(cText,13)),;
     spacing   with asc(subs(cText,14)),;
     symset    with word(subs(cText,15)),;
     pitch     with word(subs(cText,17)),;
     height    with word(subs(cText,19)),;
     xheight   with word(subs(cText,21)),;
     widtype   with signbyte(subs(cText,23)),;
     style     with asc(subs(cText,24)),;
     weight    with signbyte(subs(cText,25)),;
     typeface  with asc(subs(cText,26)),;
     serif     with asc(subs(cText,28)),;
     udist     with signbyte(subs(cText,31)),;
     uthick    with asc(subs(cText,32)),;
     theight   with word(subs(cText,33)),;
     twidth    with word(subs(cText,35)),;
     pitchext  with asc(subs(cText,41)),;
     heightext with asc(subs(cText,42)),;
     fontname  with subs(cText,49,16),;
     comment   with subs(cText,65,nHdrWid-64)

cText = subs(cText,1+nHdrWid)

**********
* Fill font cursor from rest of cText.
**********
proc readfont
priv nOrd, nWid, cTemp, nCnt

nCnt = 0
do while .t.
  nOrd = getval(chr(27)+"*c","E")
  if nOrd < 0
    exit
  endif
  nWid = getval(chr(27)+"(s","W")
  if nWid < 0 or len(cText) < nWid
    exit
  endif
  cTemp = left(cText, nWid)
  cText = subs(cText, nWid+1)
  nCnt  = nCnt + 1

  wait wind nowait ;
    "Reading character:"+str(nOrd,4)

  appe blank
  repl ordinal with nOrd,;
       char    with chr(nOrd),;
       lft     with signword(subs(cTemp,7)),;
       top     with signword(subs(cTemp,9)),;
       wid     with signword(subs(cTemp,11)),;
       hgt     with signword(subs(cTemp,13)),;
       xdelta  with signword(subs(cTemp,15)),;
       bytes   with nWid - 16,;
       data    with subs(cTemp,17)
enddo
wait wind ltrim(str(nCnt)) + " characters read" ;
  timeout 1

**********
func signbyte
para cByte
priv n

n = asc(cByte)
return iif(n < 128, n, n - 256)

**********
func signword
para cTwoByte
priv n

n = asc(cTwoByte)*256 + asc(subs(cTwoByte,2))
return iif(n < 256*128, n, n - 256*256)

**********
func word
para cTwoByte

return asc(cTwoByte)*256 + asc(subs(cTwoByte,2))

**********
* Scan cText, returning value between markers.
* cText also shrinks.
**********
func getval
para cBegMark, cEndMark, lLoop
priv n, nRet

nRet = -1  && Anticipate error
do while len(cText) > 0
  n = at(cBegMark, cText)
  if n = 0
    exit
  endif
  cText = subs(cText, n+len(cBegMark))
  n = val(cText)
  do while isdigit(cText)
    cText = subs(cText,2)
  enddo
  if cText = cEndMark
    cText = subs(cText,1+len(cEndMark))
    nRet = n
    exit
  endif
  if not lLoop
    exit
  endif
enddo
return nRet

* EOF
