;  File: HPTINY.ASM    System: HP Utility    Version: 2.0   Date: 07-22-95  ;

;-----------------------------------------------------------------------------
; Utility prints text files to HP LaserJet II or later model in compact format.
; HP.DOC (which excludes initial syntax summary in HPTINY.DOC) is adjusted so
; that so that compressed COM file is 8K.  HP.FNT is packed raster version of
; HPTINY.FNT (see MKFONT.PRG).  PKLXTRA shrinks PKLITE decoder by 220 bytes.
;
; Create COM file with:  MASM    HPTINY,,NUL,NUL /DDOCLEN=n (size of HP.DOC)
;                        LINK    HPTINY,,NUL,,
;                        EXE2BIN HPTINY HPTINY.COM
;                        COPY /B HPTINY.COM+HP.DOC+HP.FNT
;                        PKLITE  HPTINY.COM                 (optional)
;                        PKLXTRA HPTINY.COM                 (optional)
;
; This ASM source assembles to about 1.9K/1.4K code/data bytes.  The appended
; HP.DOC/HP.FNT files are an additional 3.6K/5.8K data bytes---CRH.
;
; Characters printable by utility:
;
;     ͻ
;                   !"#$%&'()*+,-./0123456789:;<=>? 
;      @ABCDEFGHIJKLMNOPQRSTUVWXYZ[\]^_`abcdefghijklmnopqrstuvwxyz{|}~ 
;       
;       
;     ͼ
;-----------------------------------------------------------------------------
; 25 Apr 93 - V1.0 Initial release, supporting text widths 80 or 127
; 21 Nov 93 - V1.1 Allows text widths 1-255
;           -      Text width specified or computed
;           -      Continuous print now default (/K replaces /C)
;           -      12x24 dot font replaced 12x25 font (90 lines per column)
;           -      Font reconstituted from raster data
;           -      Font to file /F added
; 23 Sep 94 -      Bug fix: high tab density caused problems
; 22 Jul 95 - V2.0 Wrapped print /R added
;           -      Block print /B added (DOE request for PHONE.TXT file)
;           -      Two-sided print /2 replaces alternate margin /A
;           -      LF treated as CRLF for Unix compatibility (and wrap)
;           -      FF treated as CRLF (previously ignored)
;           -      Null treated as space (previously ignored)
;           -      Handle up to 999 pages (previously 255)
;           -      Start page /S and end page /E replace /P single page
;           -      Vertical centering /C added
;           -      Right column margin extended for bad HP clones
;           -      Font ID changed to 2 on /F
;           -      Null print /Z added
;           -      Print to disk /D added
;           -      Replaced DOS critical error handler
;           -      Proportional spacing /P added (for file prints or with /F)
;           -      Multiple file specs allowed (previously only one file)
;           -      HPTINY.PRN excluded from file specs
;           -      Display each file name when printed
;           -      Omit unneeded change to raw mode during download
;           -      Column eject option /J added
;           -      Beep-abort on invalid switch (previously ignored)
;           -      Removed ASCII 7-15, 27 from font and enlarged characters
;-----------------------------------------------------------------------------
DATALEN    EQU   5964                  ; Size of appended HP.FNT data
BASELINE   EQU   22                    ; Font baseline distance from left
TOPMIN     EQU   7                     ; Minimum Top value
TAILLEN    EQU   5                     ; Font tail length
FNTHDRLEN  EQU   75                    ; Font header length
CHRHDRLEN  EQU   30                    ; Character header length
PACKLEN    EQU   DATALEN - TAILLEN - FNTHDRLEN - CHRHDRLEN
MOVE       EQU   xchg                  ; Saves byte on some AX moves
NL         EQU   <13,10>               ; New line for info message
EOS        EQU   ';'                   ; End of string (not in filenames)
KILL       EQU   0C3h                  ; Opcode for near ret stub instruction
STDPRN     EQU   4                     ; Handle for standard print device
TOPMARMASK EQU   301h                  ; Xor mask to flip '02' and '11'
LINEMAX    EQU   90                    ; Lines per column
PAGEMAX    EQU   999                   ; Max pages (under 32768/26) per file
LOWSIZE    EQU   OFFSET SeekTbl - OFFSET HpTiny + 256  ; 4K
SEEKSIZE   EQU   0800h                 ; Room for 4 bytes per even page
INSIZE     EQU   2000h                 ; Input file buffer size
OUTSIZE    EQU   2800h                 ; Output file buffer size with slack
MINSTACK   EQU   0800h                 ; Font expanded into low K of stack
MINSIZE    EQU   LOWSIZE + SEEKSIZE + INSIZE + OUTSIZE + MINSTACK  ; 26K

GOTCHR     EQU   1                     ; Flag for WrapChk--assume low bit
INWORD     EQU   2                     ; Flag for WrapChk
NULLPRN    EQU   1                     ; /Z switch--assume low byte
CENTERBOX  EQU   2                     ; /C switch--ditto
HLPTOFILE  EQU   4                     ; /? switch--ditto
FNTTOFILE  EQU   8                     ; /F switch--ditto
PRNTOFILE  EQU   16                    ; /D switch--ditto, clear bit in 0C3h
BLKFLAG    EQU   32                    ; /B switch--assume next five
WRAPFLAG   EQU   64                    ; /R switch    consecutive
WIDFLAG    EQU   128                   ; /W switch--assume low byte
TABFLAG    EQU   256                   ; /T switch--also FFONLY
ENDPAGE    EQU   512                   ; /E switch--also ODDPASS
STARTPAGE  EQU   1024                  ; /S switch
KEYPRN     EQU   2048                  ; /K switch
TWOSIDES   EQU   4096                  ; /2 switch
PROPFLAG   EQU   8192                  ; /P switch--assume high byte
FFEJECT    EQU   16384                 ; /J switch
FFONLY     EQU   256                   ; Flag for PrnFoot to print just FF
ODDPASS    EQU   512                   ; Flag for NextBuf to fill SeekTbl
EVENPASS   EQU   32768                 ; Flag for PrnToTop--assume high bit
NUMSWITCH  EQU   15                    ; Count of switches, besides /N

BEEPDIFF   EQU   LOW (OFFSET BeepMsg - OFFSET BeepMsg)
BACKDIFF   EQU   LOW (OFFSET BackMsg - OFFSET BeepMsg)
DUMPDIFF   EQU   LOW (OFFSET DumpMsg - OFFSET BeepMsg)
DNLDDIFF   EQU   LOW (OFFSET DnldMsg - OFFSET BeepMsg)
READDIFF   EQU   LOW (OFFSET ReadMsg - OFFSET BeepMsg)
PRNTDIFF   EQU   LOW (OFFSET PrntMsg - OFFSET BeepMsg)
WAITDIFF   EQU   LOW (OFFSET WaitMsg - OFFSET BeepMsg)
EVENDIFF   EQU   LOW (OFFSET EvenMsg - OFFSET BeepMsg)
DONEDIFF   EQU   LOW (OFFSET DoneMsg - OFFSET BeepMsg)
CRITDIFF   EQU   LOW (OFFSET CritMsg - OFFSET BeepMsg)
INFODIFF   EQU   LOW (OFFSET InfoMsg - OFFSET BeepMsg)
NAMEDIFF   EQU   LOW (OFFSET NameBuf - OFFSET BeepMsg)

Code_Seg   SEGMENT
           ASSUME CS:Code_Seg,DS:Code_Seg,ES:Code_Seg

           ORG   5Ch                   ; Free PSP area through 7Fh
SaveBX     LABEL WORD                  ; Register checkpoints saved here
           ORG   5Eh                   ;   during wrap handling--see WrapChk
SaveCX     LABEL WORD
           ORG   60h
SaveDI     LABEL WORD
           ORG   62h
SaveSI     LABEL WORD
           ORG   64h
LineCnt    LABEL BYTE                  ; Line counter in PrnBuf

           ORG   100h
;-----------------------------------------------------------------------------
; Install error handler, set DTA, display message, and check RAM.  Assume DOS
; 2.0+.  No direct BIOS calls are made by program.  Printing done via STDPRN.
;-----------------------------------------------------------------------------
HpTiny:    mov   dx,OFFSET CritErr
           mov   ax,2524h
           int   21h                   ; Set own critical error handler
           mov   dx,OFFSET DTABuf
           mov   ax,1A00h OR INFODIFF  ; AL preserved as DispApnd input
           int   21h                   ; Move DTA buffer away from 80h
           call  DispApnd              ; Display InfoMsg, zeroing AL
           mov   BYTE PTR Ten+1,al     ; Null InfoMsg terminator
           cmp   sp,MINSIZE            ; Check space requirements
           jnc   Start                 ; Enough room?  Ahead, else abort
;-----------------------------------------------------------------------------
; Beep/Display routines.  DispAL erases current line text first.  DispApnd
; appends to current text.  Both use short AL input for message.  On exit, AH
; is 8 in case key wait call follows, AL is zeroed for other post-call use, and
; DX points to string (for OpenNext).  Flags are preserved (for CritErr).
;-----------------------------------------------------------------------------
Beep:      mov   al,BEEPDIFF
DispAL:    push  ax
           mov   al,BACKDIFF
           call  DispApnd              ; CR/Tabs/CR erase current text
           pop   ax
DispApnd:  cld                         ; Fix for program in first call and
           pushf                       ;   also needed if critical error
           mov   ah,0
           add   ax,OFFSET BeepMsg     ; Adjust with base offset
           push  si
           push  ax                    ; Save string start
           MOVE  si,ax
           lodsb                       ; Assume first character not EOS
charloop:  MOVE  dx,ax
           mov   ah,2
           int   21h                   ; DOS display character DL
           lodsb
           cmp   al,EOS                ; Check for end of string--not $, since
           jne   charloop              ;   that is legal filename character

           pop   dx                    ; Output start of string
           pop   si
           mov   ax,800h               ; In case DOS key wait, etc., afterward
           popf
dispout:   ret                         ; Return or exit via PSP
;-----------------------------------------------------------------------------
; Initialization then main loop.  Program generally has long memory of register
; contents to minimize code size.  Most byte/word data is stored in instruction
; operands for same reason.  BP register holds global flags throughout (see
; equates above).
;
; Changes to code are visible to current Intel processors (through Pentium)
; provided that break in sequential execution occurs (call/ret/jmp/branch/int)
; before new code is reached or provided that changed code is further away than
; size of pre-fetch queue (16 or so bytes).  Break causes flush of queue.
;
; For this program, at least one call instruction is executed after each change
; before new code is reached.  P6 chip will do some new things (e.g., guessing
; on branches and parallel provisional execution), so above assumptions may
; no longer hold.  But call instruction should be sufficient to cause flush.
;
; This is programming for fun, so no flames about code modification please...
;-----------------------------------------------------------------------------
Start:     call  GetFlags              ; Returns flags in BP and zeroes CX
           je    Beep                  ; Bad switch?  Beep and abort

           mov   ax,bp                 ; For next few AH/AL tests
           test  ah,PROPFLAG SHR 8     ; Check for /P (high--others low byte)
           je    notp                  ; No?  Ahead

           inc   BYTE PTR FontHdr+24   ; Bump 0 to 1 for proportional spacing
           inc   WORD PTR StubKill     ; Call PropSpc instead of StubLoc later
notp:      test  al,CENTERBOX          ; Check /C for vertical centering
           je    notc                  ; No?  Ahead

           mov   BYTE PTR TopMar+1,'7' ; "07" top margin replaces "02"
           mov   WORD PTR XorMask,cx   ; Zero /2 xor mask to keep margin fixed
notc:      mov   dx,OFFSET HelpFile    ; Anticipate help to file
           test  al,HLPTOFILE          ; /?
           jne   istofile              ; Yes?  Ahead

           mov   dx,OFFSET FontFile    ; Anticipate font to file
           test  al,FNTTOFILE          ; /F
           jne   istofile              ; Yes?  Ahead

           test  al,NULLPRN            ; Postpone til after help/font tests
           je    notz                  ; Not /Z?  Ahead

           mov   al,KILL               ; Also clears possible /D flag
           mov   BYTE PTR KillPrn,al   ; Disable Print call with ret stub
notz:      mov   dx,OFFSET PrnFile     ; Anticipate print to file
           test  al,PRNTOFILE          ; /D
           je    toname                ; No?  Ahead

istofile:  mov   ah,3Ch                ; Create open, CX zero from GetFlags
           call  Int21h                ; May abort internally
           mov   WORD PTR OutHandle,ax ; Replace STDPRN

           mov   ax,bp                 ; For next two tests
           test  al,HLPTOFILE          ; Test if help to file
           je    chkfnt                ; No?  Ahead

           mov   al,DUMPDIFF
           call  DispAL                ; Display dump message
           mov   dx,OFFSET InfoMsg     ; DOC file includes initial display
           mov   cx,OFFSET FontData - OFFSET InfoMsg
           jmp   SHORT dumpexit        ; Write to file and exit, closing
                                       ;   with int 20h in PSP
chkfnt:    test  al,FNTTOFILE          ; Test if font to file
           jne   todnld                ; Yes?  Ahead, else print to file

toname:    call  PrepName              ; Get spec, find first, set DOS stamp
           je    dispout               ; No files?  Quit--need this test
                                       ;   before any download attempted
           dec   BYTE PTR FontHdr + 3  ; Decrement font ID to 1
todnld:    mov   al,DNLDDIFF           ; ID stays 2 if to file
           call  DispAL                ; Display message (AX is 800h afterward)
           call  MakeFont              ; Make font, setting DX/CX/AX for next
           test  al,FNTTOFILE          ; Test if want download to FNT file
           jne   dumpexit              ; Yes?  Then write to file and exit,
                                       ;   closing with int 20h in PSP
           MOVE  bp,ax                 ; Restore flags to BP
           call  Print                 ; Download to printer/file
fileloop:  call  OpenNext              ; Open next file and adjust footer
           je    findnext              ; HPTINY.PRN?  Skip

           mov   al,READDIFF
           call  DispAL                ; Display read message
           call  ScanFile              ; Set DI/MaxCol and zero AX/CX/DX
           call  SetSeek               ; Zero SeekHi/SeekLo from AX/DX
           call  PrepWid               ; Prepare width/page/footer parameters
           test  ax,ax                 ; Page total was returned
           je    nextfile              ; None?  Ahead (must avoid PrnNewPg and
                                       ;   LastPass, if zero)
           call  CheckTwo              ; Adjust margin/ODDPASS flag, as needed
           call  PrnNewPg              ; Cursor to top of first column
mainloop:  call  NextBuf               ; Get next buffer, returning DI/CX
           jcxz  tidyup                ; No data?  Done, but need eject

           call  PrnBuf                ; Print buffer at DI/CX
           call  ChkMax                ; Check page count
           jbe   mainloop              ; Equal/under total?  Loop

donemain:  call  CheckTwo              ; Also adjusts margin/clears ODDPASS
           je    nextfile              ; Not two passes?  Ahead

           call  LastPass              ; Handle even page print
nextfile:  mov   ah,3Eh
           call  Read21h               ; Close file
findnext:  mov   ah,4Fh                ; DOS find next
           int   21h
           jnc   fileloop              ; Another file?  Loop

           call  PrepName              ; Check for another file spec
           jne   fileloop              ; Yes?  Loop, else CX zero for next

           mov   dx,OFFSET KillFont    ; Reset HP and kill font
           mov   cl,OFFSET BeepMsg - OFFSET KillFont
dumpexit:  call  Print
           mov   al,DONEDIFF           ; Announce done
           jmp   DispApnd              ; Exit via PSP, closing any open files

tidyup:    call  PrnFoot               ; Print footer/box then eject
           jmp   SHORT donemain
;-----------------------------------------------------------------------------
; Critical error handler to beep-abort on everything but PRN error.  On PRN
; error, allow retry with custom message.  Erase afterward to acknowledge key,
; since next program message may take awhile (e.g. on initial download).
;
; Command-line redirection eats beep/retry messages (and rest of messages).
;-----------------------------------------------------------------------------
CritErr:   push  ds                    ; Using only DS/DX/AX here, with
           push  dx                    ;   AL set to flag value
           push  bp                    ; BP:SI points to device header
           pop   ds                    ; Need both AX and [SI+4] high bits set
           and   ax,[si+4]             ;   for a character mode device
           or    ax,1                  ; Interposed only to force inequality
           jns   notchar               ; Not high bit set?  Then not character
                                       ;   device, so ahead with inequality
           mov   ax,[si+10]            ; Else check start of device name
           add   ax,[si+12]            ;   for "PRN "
           sub   ax,'RP' + ' N'        ; Equality here should be close enough
notchar:   push  cs                    ; Set DS for message access, with
           pop   ds                    ;   display calls also using DX/AX
           call  Beep                  ; Announce error, setting AL to 0
           jne   termout               ; Not PRN error?  Flag terminate
                                       ;   (equality flag was preserved)
           mov   al,CRITDIFF           ; Retry message
           call  DispApnd              ; Display message, setting AH to 8
           int   21h                   ; Wait for key press
           and   al,NOT 20h            ; Force upper case
           cmp   al,'Y'                ; Compare, with equality if y or Y
           mov   al,BACKDIFF
           call  DispApnd              ; Erase message, setting AL to 0
           je    retryout              ; User wants retry?  Bump AL to 1
                                       ;   (equality flag was preserved)
termout:   inc   ax                    ; Terminate flag is 2 in AL
retryout:  inc   ax                    ; Retry flag is 1 in AL
           pop   dx
           pop   ds
           iret
;-----------------------------------------------------------------------------
; Handle backwards print of even pages for two-sided print, with initial dummy
; eject if odd page total.  Assume at least one page total.
;
; For partial two-sided prints, /S should be odd and /E even for full sheets,
; though this is not enforced.  ByteCnt zero reset at page start is reason /B
; ignores all LFs/FFs (other than those inserted) and disallows /J.
;-----------------------------------------------------------------------------
LastPass:  mov   al,EVENDIFF
           call  DispAL                ; Display refeed prompt, setting AH to 8
           int   21h                   ; Wait for any key

           mov   ax,WORD PTR PageTot
           test  al,1                  ; Check if even total pages
           je    evenloop              ; Yes?  Ahead

           inc   ax                    ; Else bump to dummy even page
           mov   WORD PTR PageCnt,ax
           or    bp,FFONLY             ; Flag FF only
lastpg:    call  PrnFoot               ; Print last even page and/or eject
           and   bp,NOT (FFONLY OR EVENPASS)  ; Kill flags
           jmp   SHORT nextpg

evenloop:  or    bp,EVENPASS           ; Flag even pass--turned off when page
           mov   WORD PTR PageCnt,ax   ;   done (unless short last page)
           shl   ax,1                  ; Double word table entries
           add   ax,OFFSET SeekTbl
           MOVE  si,ax                 ; Point SI to desired SeekTbl entry
           lodsw
           MOVE  dx,ax
           lodsw
           MOVE  cx,ax
           call  SeekBOF               ; Seek to page start CX:DX
           call  SetSome               ; Reset LineCnt/ColPos/ColFrac/ByteCnt
           call  PrnNewPg              ; Cursor to top of first column
eploop:    call  NextBuf               ; Get next buffer, returning DI/CX
           jcxz  lastpg                ; No data?  Then end of last even page

           call  PrnBuf                ; Print buffer at DI/CX with early out
           test  bp,bp                 ;   (clearing EVENPASS) if page done
           js    eploop                ; Not early out?  Loop--EVENPASS
                                       ;   assumed high bit
nextpg:    mov   ax,WORD PTR PageCnt   ; Already have AX sometimes
           dec   ax
           dec   ax
           jne   evenloop              ; Done?  Fall harmlessly...
;-----------------------------------------------------------------------------
; Copy short name from DTABuf to print area footer, ending with null.  CX set
; to 14 less length (including null) and DI advanced.  CopyDI entry uses DI
; input for destination.  Both calls from OpenNext.
;-----------------------------------------------------------------------------
CopyShort: mov   di,OFFSET ShortName   ; Footer area name location
CopyDI:    mov   si,OFFSET DTABuf+30   ; ASCIIZ name in DTA buffer
           mov   cx,14                 ; Extra 1 avoids bump after 2nd call
csloop:    lodsb                       ;   in OpenNext
           stosb
           test  al,al
           loopne csloop               ; Copy til null reached

csout:     ret
;-----------------------------------------------------------------------------
; Copy file specification to NameBuf, make ASCIIZ, execute DOS find first,
; fix short name pointer used by OpenNext, and set DOS stamp in footer.
; If no matching files, beep and continue to next spec.  Return equality to
; indicate done with all file specs (and zero CX if done).
;
; To allow multiple file specs, routine patched so that start point advances
; in command-line (with new length set in previous byte) for subsequent calls
; or for loop within this call.
;-----------------------------------------------------------------------------
pnloop:    call  Beep                  ; Announce bad file or spec

NextName   =     $ + 1                 ; Adjusted for next call or loop
PrepName:  mov   di,80h
           mov   al,[di]               ; Command-line length 0-127
           scasb                       ; Sets equality, advancing DI
           cbw                         ; Zero AH
           MOVE  cx,ax
           mov   ax,4E20h              ; DOS find first high, space low
           repe  scasb                 ; If CX was zero, equality passes
           je    csout                 ; End of command-line?  Out

           dec   di                    ; Point back to name start
           inc   cx                    ; Adjust CX (still excludes CR)
           cmp   BYTE PTR [di],'/'     ; Check if hit first switch
           je    csout                 ; Switch?  Then file specs done

           mov   si,di                 ; Ready SI for copy
           repne scasb                 ; Scan past name, looking for space

           push  cx                    ; Save remaining length
           mov   cx,di
           sub   cx,si                 ; CX now copy length
           mov   di,OFFSET NameBuf     ; File spec destination
           mov   dx,di                 ; For find first and backloop comparsion
           rep   movsb                 ; Copy file spec, zeroing CX
           mov   [di],cl               ; Make spec ASCIIZ (ok if after space)
           int   21h                   ; DOS find first (AH/DX/CX ready),
           pop   ax                    ;   setting carry if no matching files

           dec   si                    ; Must postpone store til copy done
           mov   [si],al               ; Adjust length for next call/loop
           mov   WORD PTR NextName,si  ; Adjust start for next call/loop
           jc    pnloop                ; No files?  Loop

backloop:  mov   al,[di]               ; Back up to find start of short name
           cmp   al,'\'
           je    gotstart              ; Backslash?  Exit loop
           cmp   al,':'
           je    gotstart              ; Colon?  Exit loop
           dec   di
           cmp   di,dx                 ; DX still NameBuf offset
           jnb   backloop              ; Not before start of full name?  Loop

gotstart:  inc   di                    ; To start of short name
           mov   WORD PTR ShortLoc,di  ; Save in OpenNext and fall...
;-----------------------------------------------------------------------------
; Put DOS date/time stamp at right in footer line.
;-----------------------------------------------------------------------------
DosStamp:  mov   di,OFFSET DosDtTm
           mov   ah,2Ch
           int   21h                   ; CH/CL is DOS hours/minutes
           MOVE  ax,cx                 ; Convert to packed time in AX
           mov   cl,2
           shl   al,cl
           inc   cx
           shl   ax,cl
           push  ax                    ; Packed time to stack
           mov   ah,2Ah
           int   21h                   ; CX/DH/DL is DOS year/month/day
           MOVE  ax,cx                 ; Convert to packed date in DX
           sub   ax,1980
           xchg  ah,dh
           mov   cl,4
           shl   ah,cl
           mov   cl,7
           ror   ax,cl
           or    dx,ax                 ; Time on stack, date in DX, DI ready
           jmp   SHORT unpack          ; Ahead into FileStamp
;-----------------------------------------------------------------------------
; Open next file for read and put short name and date/time stamp at left in
; footer line.  Also display file name.  Call only after successful find first
; or find next.  Abort with equality if file name is HPTINY.PRN.
;
; Repeated file name is permitted for multiple print copies.
;-----------------------------------------------------------------------------
ShortLoc   =     $ + 1                 ; Set by PrepName
OpenNext:  mov   di,0                  ; Offset of short name in NameBuf
           push  di
           call  CopyDI                ; Copy up to and including null
           mov   ax,EOS SHL 8 OR 10    ; Linefeed then string terminator
           stosw                       ; Follows null
           pop   di

           mov   si,OFFSET PrnFile     ; Guard against print of /D file
           mov   cl,10                 ; CH is zero
           repe  cmpsb
           je    onout                 ; HPTINY.PRN?  Out with equality

           mov   al,NAMEDIFF           ; Display filename, setting DX to
           call  DispAL                ;   name start and zeroing AL

           mov   ah,3Dh                ; Open read-only (DX/AL ready)
           call  Int21h                ; May abort internally
           mov   WORD PTR InHandle,ax  ; Save file handle for reads/etc.
           MOVE  bx,ax                 ; To BX for FileStamp

           call  CopyShort             ; Copy short name up to and including
                                       ;   null to footer area
           dec   di                    ; Back up DI (CX ok)
           mov   al,' '                ; Blank through 13th position, pointing
           rep   stosb                 ;   DI to FileDtTm, then fall...
;-----------------------------------------------------------------------------
; Put file date/time stamp in footer line.  Input handle BX and offset DI.
;-----------------------------------------------------------------------------
FileStamp: mov   ax,5700h              ; Could get info from DTABuf also
           int   21h                   ; Output CX/DX is time/date
           push  cx                    ; Save time
unpack:    mov   cx,0F05h              ; DI/DX ready, with time on stack
           call  StoDgts               ; Month
           mov   cx,1F00h
           call  StoDgts               ; Day
           mov   al,dh
           shr   al,1                  ; AL now year offset from 1980
           add   al,80
subloop:   sub   al,100                ; Modulo 100 loop
           jnc   subloop

           add   al,100
           call  StoAam
           pop   dx                    ; Restore time
           mov   cx,1F0Bh
           call  StoDgts               ; Hour 0-23
           mov   cx,3F05h              ; Fall, storing minutes
StoDgts:   mov   ax,dx                 ; Not MOVE
           shr   ax,cl
           and   al,ch
StoAam:    aam
           or    ax,3030h              ; To ASCII
           xchg  al,ah
           stosw                       ; Store two digits
           inc   di                    ; Skip separator--sets inequality
onout:     ret                         ; Return inequality if storing stamp
;-----------------------------------------------------------------------------
; Force command-line to upper case through /N, set BP to flags, and zero CX.
; Also prepare footer note, handle some flags, and tweak XlatTbl.  Abort with
; equality if bad switch character found.
;-----------------------------------------------------------------------------
GetFlags:  mov   si,80h
           lodsb                       ; Command-line size 0-127
           cbw                         ; Zero AH
           MOVE  cx,ax
           mov   di,si
           push  di
           inc   cx                    ; In case was zero--will scan final CR
upperloop: lodsb                       ; Force upper case command-line for
           cmp   al,'a'                ;   name display later and for flags
           jb    notlower
           cmp   al,'z'
           ja    notlower

           and   al,NOT 20h            ; Force upper case
notlower:  stosb                       ; Initial AL is space/Cr--AH irrelevant
           cmp   ax,'/N'               ; Test for command-line note
           mov   ah,al                 ; Save previous byte
           loopne upperloop            ; Exit on CX zero or /N

           MOVE  ax,di
           jne   nonote                ; Not /N?  Ahead

           mov   WORD PTR NoteLoc,ax   ; Else save CR-terminated note offset
           dec   ax                    ; Back up to avoid bad switch error
                                       ;   since N not in ParmTbl
nonote:    MOVE  cx,ax
           pop   di                    ; Restore 81h
           sub   cx,di                 ; Scan for switches through CR or /N
           xor   bp,bp                 ; Accumulate parmeter switches in BP
           jmp   SHORT flagchk         ; Enter loop

dgtloop:   cbw                         ; Zero AH, AL is 0-9
           xchg  [si],ax               ; Parm values initially zero
           mul   WORD PTR Ten          ; Multiply by 10, zeroing DH for later
           add   [si],ax               ; Value accumulated back to table
dgtenter:  inc   di                    ; Point to next expected digit
           dec   cx                    ; Keep CX in synch
           mov   al,[di]               ; Next command-line character
           sub   al,'0'
           jb    saveflag              ; Not digit?  Exit digit loop

           cmp   al,9
           jbe   dgtloop               ; Digit?  Loop

saveflag:  or    bp,bx                 ; Save switch in BP
flagchk:   mov   al,'/'                ; Look for more switches
           repne scasb                 ; CR at end (or / if /N)
           jcxz  flagtidy              ; End command-line?  Exit below

           mov   bx,1 SHL (NUMSWITCH-1); Largest bit value to test in BX
           mov   si,OFFSET ParmTbl     ; Search for match to legal parm
swtchlp:   lodsb
           cmp   al,[di]
           je    dgtenter              ; Match?  Enter digit loop

           lodsw                       ; Skip SI past parm value
           shr   bx,1                  ; Halve to next lower bit value
           jne   swtchlp               ; Not zero?  Loop, else fall harmlessly,
                                       ;   returning equality...
;-----------------------------------------------------------------------------
; Restore position registers from last checkpoint.  Called by NextBuf/WrapChk.
;-----------------------------------------------------------------------------
RestPos:   mov   si,OFFSET SaveBX
           lodsw
           MOVE  bx,ax
           lodsw
           MOVE  cx,ax
           lodsw
           MOVE  di,ax
           lodsw
           MOVE  si,ax
           ret                         ; Shared exit point
;-----------------------------------------------------------------------------
; GetFlags tail does miscellaneous adjustments, exitting with inequality.
;-----------------------------------------------------------------------------
flagtidy:  mov   si,OFFSET ParmTbl+19  ; Offset of Tab value
           dec   WORD PTR [si-3]       ; Decrement /E value--see PageTest
           lodsw                       ; Ignore AH
           and   al,3Fh                ; Cap tab at 63
           je    tabdflt               ; Zero or no /T?  Leave default 8

           mov   BYTE PTR TabSize,al   ; Else adjust TabSize--TABFLAG/ENDPAGE
                                       ;   cleared next for later reuse
tabdflt:   and   bp,NOT (WIDFLAG OR WRAPFLAG OR BLKFLAG OR TABFLAG OR ENDPAGE)
           mov   dl,WIDFLAG            ; DH zero if digit loop entered
           lodsb                       ; Skip W switch
           lodsw                       ; /W, /R, /B mutually exclusive, so
           test  al,al                 ;   insure at most one set
           jne   setwid                ; Non-zero /W?  Ahead

           lodsb                       ; Skip R switch
           lodsw                       ; Ignore AH
           shr   dx,1
           and   al,7Fh                ; Wrap max is 127
           jne   setwid                ; Non-zero /R?  Ahead

           lodsb                       ; Skip B switch
           lodsw                       ; Ignore AH
           shr   dx,1
           test  al,al
           je    clrwid                ; Zero /B?  Ahead

           mov   XlatTbl[10],cl        ; If /B, zero all LFs and insure none
           and   bp,NOT FFEJECT        ;   added by /J--see LastPass and below
setwid:    or    bp,dx                 ; Set one of three flags
clrwid:    xchg  BYTE PTR CmdWid,al    ; For PrepWid and NextBuf, with CR to AL
           dec   cx                    ; -1 to CX
           mov   di,WORD PTR NoteLoc
           mov   si,di                 ; SI for StripBuf
           repne scasb                 ; Scan through trailing CR
           not   cx                    ; CX now note length, including CR
           call  StripBuf              ; Insure no form feeds, etc.
           mov   al,[bx+10]            ; Ready change of XlatTbl entry for FF
           test  bp,FFEJECT            ; Check for /J
           je    notj                  ; No?  Ahead

           mov   al,12                 ; Else allow FFs
notj:      mov   [bx+12],al            ; 0 if /B, else 12 if /J, else 10
           MOVE  ax,cx                 ; Note length (CX was preserved)
           mov   BYTE PTR NoteLen,al
           neg   al                    ; 179 is page width at 16.66 pitch,
           add   al,179                ;   adjusted for CR and box position
           shr   al,1                  ; Now start offset for note
           mov   bx,OFFSET NoteNum+5   ; End of note position in page footer
;-----------------------------------------------------------------------------
; Store AL as 3 ASCII digits leftward, starting at offset BX.  AXtoASC entry
; for AX input.  AXCXAsc entry for AX/CX input.  AL/AX/SI/DI preserved, CX
; zeroed, and inequality set.
;-----------------------------------------------------------------------------
ALtoAsc:   mov   ah,0                  ; Not cbw due to MakeFont calls
AXtoAsc:   mov   cx,3
AXCXAsc:   push  ax
ascloop:   cwd                         ; Assume AX under 32768
           div   WORD PTR Ten          ; Divide by 10 (LF low, zero high)
           or    dl,'0'                ; Make remainder ASCII digit
           mov   [bx],dl               ; Store and move left
           dec   bx                    ; Forces inequality also
           loop  ascloop               ; Loop for CX digits

           pop   ax
           ret                         ; CX zero and inequality on exit
;-----------------------------------------------------------------------------
; Adjust program data to account for width.  On entry, assume AX/CX/DX zero and
; input DI holds column count from ScanFile.  Return page total AX.
;
; Because of poor HP emulation by Okidata, right margin is one larger than
; needed for HP with fixed space font.  For proportional spacing option, the
; extra column is needed anyway.  Okidatas also truncate bottom of odd pages
; in /2 print (so /C switch was added).
;-----------------------------------------------------------------------------
PrepWid:   mov   al,BYTE PTR CmdWid    ; Zero or value from /W, /R, /B
           test  al,al                 ; Input AH zero
           jne   notzwid               ; Not zero?  Ahead

           mov   al,BYTE PTR MaxCol    ; Else use ScanFile computed value
notzwid:   push  ax                    ; See above on Okidata for next
           mov   WORD PTR ChrWid,ax    ; PrnToTop add value for right margin
           dec   ax                    ; Decrement for 9/7 round
           pop   bx
           mov   cl,7
           shl   ax,cl                 ; Adjust to 9/7 bit integer fraction
           push  ax                    ; Save for later
           inc   bx                    ; Bump twice to insure room for
           inc   bx                    ;   space at each end of line
           mov   ax,260                ; Total character width of box
           push  ax                    ; Save--input DX zero for next division
           div   bx                    ; Get quotient AL as columns per page
           cmp   al,26                 ; AH zero since BX in range 3-257
           jb    colok                 ; Less than 26?  Ahead

           mov   al,26                 ; Else cap at 26
colok:     mov   BYTE PTR ColTot,al    ; Fix column total for PrnFoot
           mov   BYTE PTR ColCtr,al    ; Initialize column pointer for ODDPASS
           xor   al,'A' - 1            ; Force to uppercase letter
           mov   BYTE PTR AlphaCol,al  ; Fix max column letter for PrnFoot
           xor   al,'A' - 1            ; Back to 1-26
           MOVE  bx,ax                 ; Divisor for next three divisions
           pop   ax                    ; 260 again
           cwd                         ; Zero DX for next division
           shl   ax,cl                 ; Adjust to 9/7 bit integer/fraction
           div   bx                    ; Divide to get column width plus slack
           mov   WORD PTR ColWid,ax    ; For PrnToTop
           pop   dx                    ; 9/7 bit ChrWid (less one for round)
           sub   ax,dx                 ; Subtract to get slack space
           shr   ax,1                  ; Halve
           cwd                         ; Zero DX for next division
           add   ax,3 SHL 7            ; Add 3-character left edge indent
           mov   WORD PTR ColIndent,ax ; For PrnToTop
           mov   ax,3120 SHL 4         ; Dot width of box with nibble fraction
           div   bx                    ; Divide by column count
           mov   WORD PTR DotWid,ax    ; Dot width of column with fraction
                                       ;   for PrnFoot
           call  SetSome               ; Set LineCnt-Ctr/ColPos-Frac/ByteCnt/DX
           MOVE  ax,di                 ; Total column count
           dec   ax                    ; Add BX less one so that division
           add   ax,bx                 ;   gives pages
           div   bx                    ; DX zero from SetSome
           cmp   ax,PAGEMAX
           jbe   totok                 ; Cap page total

           mov   ax,PAGEMAX
totok:     mov   WORD PTR PageTot,ax   ; Save total page count for main loop
           mov   bx,OFFSET PrntMsg+18  ; End of page total in message
           call  AXtoAsc               ; Include in display
           mov   bx,OFFSET FootNums+6  ; End of page total in page footer
           jmp   SHORT AXtoAsc         ; Include in footer, zeroing CX
;-----------------------------------------------------------------------------
; Check if two-sided print.  If so, flip margin and ODDPASS flag.  On exit,
; equality means not two-sided print.  If /C, margin flip is disabled, but
; xor still forces inequality.
;-----------------------------------------------------------------------------
CheckTwo:  test  bp,TWOSIDES
           je    ctout

           xor   bp,ODDPASS
XorMask    =     $ + 4                       ; /C zeros mask, so no 7 flip
           xor   WORD PTR TopMar,TOPMARMASK  ; Flip margin, 2 or 11 lines
ctout:     ret
;-----------------------------------------------------------------------------
; Scan file to get total column count DI and to set ColMax (in NextBuf).  Also
; initialize some variables.  On exit, AX/CX/DX zeroed.
;
; Scan mimics main print loop, but with printing omitted.
;-----------------------------------------------------------------------------
ScanFile:  call  SetSome               ; Also zeroes AX/DX
           inc   ax
           mov   WORD PTR PageCnt,ax   ; Set PageCnt/PageCtr/MaxCol to 1
           mov   WORD PTR PageCtr,ax
           mov   BYTE PTR MaxCol,al
           MOVE  di,ax                 ; DI counts columns--initially 1 to
                                       ;   anticipate last dirty column
scanoutlp: push  di                    ; Save column count
           push  dx                    ; Save line count/dirty flag DL/DH
           call  NextBuf               ; Fill OutBuf, returning DI/CX
           mov   si,di
           pop   dx
           pop   di
           jcxz  scanexit              ; No bytes?  EOF, so done

scaninlp:  lodsb                       ; Get next OutBuf byte
           or    dh,al                 ; Update dirty flag
           cmp   al,10                 ; Look for next line feed
           jne   scancont              ; Not LF?  Continue

           inc   dx                    ; Update line count
           cmp   dl,LINEMAX            ; Check for last line in column
           jne   scancont              ; Not last line?  Continue

           xor   dx,dx                 ; Reset line count/dirty flag
           inc   di                    ; Bump column count
           jns   scancont              ; Not at cap?  Ahead, else lock DI
                                       ;   at 32767 to force PageTot to
           dec   di                    ;   to PAGEMAX cap later
scancont:  loop  scaninlp              ; Not done with this OutBuf?  Loop
           jmp   SHORT scanoutlp       ; Back for next buffer

scanexit:  test  dx,dx                 ; Check if last column is dirty
           jne   nodec                 ; Yes?  Ahead

           dec   di                    ; Else undo initial 1 setting
nodec:     xor   dx,dx                 ; Both CX and DX zero for rewind...
;-----------------------------------------------------------------------------
; Seek CX:DX from BOF in file stream, setting DX:AX to current file position.
; Read21h entry for file reads and Int21h entry for file open/file write/print.
; On successful ScanFile fall through, AX/CX/DX all zeroed.
;-----------------------------------------------------------------------------
SeekBOF:   mov   ax,4200h              ; Seek from BOF
InHandle   =     $ + 1                 ; Set by read open
Read21h:   mov   bx,0
Int21h:    int   21h
abortchk:  jnc   ctout                 ; Ok?  Out

Abort:     call  Beep                  ; Else beep and abort, with DOS
           int   20h                   ;   closing any open files
;-----------------------------------------------------------------------------
; Print box lines/page footer and eject, just eject, or skip all, as flagged.
;-----------------------------------------------------------------------------
PrnFoot:   call  PageTest              ; /S, /E, ODDPASS check
           jne   ctout                 ; No print this page?  Out

           test  bp,KEYPRN             ; Check for /K
           je    skipwait              ; Keypress wait not flagged?  Skip wait

           mov   al,WAITDIFF
           call  DispAL                ; Display prompt for keypress
           int   21h                   ; Wait for any key (AH is 8)

skipwait:  mov   dx,OFFSET FormFeed    ; Anticipate FF print
           mov   cx,1
           test  bp,FFONLY             ; Test if last even page FF flagged
           jne   Print                 ; Yes?  Just print FF

ColTot     =     $ + 1                 ; PrepWid adjusts page column total
           mov   cl,0                  ; Typically 3 -- CH is zero
           mov   ax,36 SHL 4           ; Always indent 3 characters
vertloop:  push  cx
           push  ax
           mov   cl,4                  ; Also input to AXCXAsc (CH zero)
           shr   ax,cl                 ; Truncate fractional dot
           mov   bx,OFFSET VertBar + 8
           call  AXCXAsc               ; Set horizontal dot position
           mov   dx,OFFSET VertBar     ; CX was zeroed
           mov   cl,OFFSET PageNote - OFFSET VertBar
           call  Print                 ; Draw vertical bar
           pop   ax
DotWid     =     $ + 1                 ; PrepWid adjusts to column dot width,
           add   ax,0                  ;   with nibble fraction--typically
           pop   cx                    ;   (3120 / 3) SHL 4
           loop  vertloop

           mov   dx,OFFSET PageNote
           mov   cl,OFFSET PageFoot - OFFSET PageNote
           call  Print
NoteLoc    =     $ + 1                 ; GetFlags adjusts offset, if /N
           mov   dx,OFFSET DfltNote
NoteLen    =     $ + 1                 ; GetFlags sets note length
           mov   cl,0                  ; CH is zero
           call  Print
           mov   dx,OFFSET PageFoot
           mov   cl,OFFSET Reset - OFFSET PageFoot
;-----------------------------------------------------------------------------
; Print CX bytes at offset DX.  Disk full check made in case /?, /F , or /D.
;-----------------------------------------------------------------------------
KillPrn    =     $                     ; Stubbed by /Z to ret instruction
Print:     mov   ah,40h                ; Write to file/device
OutHandle  =     $ + 1                 ; Adjusted if /?, /F, or /D
           mov   bx,STDPRN             ; Default to device PRN
           call  Int21h                ; May abort internally
           cmp   ax,cx                 ; Set carry on short write
           jmp   SHORT abortchk        ; Return or abort, depending on carry
;-----------------------------------------------------------------------------
; Print one buffer at DI containing CX bytes.  Entry at PrnBuf.  On EVENPASS
; of two-sided print, PrnToTop may force exit from PrnBuf.  Same occurs in
; main loop if page count exceeds total pages (i.e., big file with over
; PAGEMAX pages or file for which last page is completely full).
;-----------------------------------------------------------------------------
endline:   dec   LineCnt
           jne   nextline              ; Not to column end yet?  Ahead

           mov   al,LINEMAX
           mov   LineCnt,al            ; Else reset line count down value
           push  cx                    ; Save count of bytes remaining
           call  PrnSome               ; Print column of data then cursor to
           call  PrnToTop              ;   top of next column (preserving DI)
           pop   cx                    ; Restore byte count
PrnBuf:    mov   dx,di                 ; Save start offset for next print
nextline:  jcxz  PrnSome               ; End of buffer?  Print and exit

           mov   al,10                 ; Look for next line feed
           repne scasb                 ; Advances DI, decrementing CX
           je    endline               ; EOL?  Back, else print and exit
;-----------------------------------------------------------------------------
; Print data from DX through DI-1, if flagged, else do nothing.
;-----------------------------------------------------------------------------
PrnSome:   call  SetCX                 ; Set CX to DI less DX--zero ok
PrnIf:     call  PageTest              ; Returns equality if ok to print
           je    Print                 ; Ok?  Print, else fall harmlessly...
;-----------------------------------------------------------------------------
; Return equality if ok to print on current page by checking /S, /E, and
; ODDPASS.  /Z accounted for elsewhere.  /E value was decremented to second
; last page earlier (so that a default zero becomes 65535).
;-----------------------------------------------------------------------------
PageCnt    =     $ + 1                 ; Zeroed by ScanFile
PageTest:  mov   ax,0
           cmp   WORD PTR ParmTbl+13,ax; /S value--5th value in table
           ja    testout               ; Not to start?  No print

           dec   ax                    ; Synch with decremented /E value
           cmp   WORD PTR ParmTbl+16,ax; /E value--6th value in table
           jb    testout               ; Past end?  No print

           test  bp,ODDPASS            ; Only set if two-pass print
           je    testout               ; Not odd pages only?  Can print

           test  al,1                  ; Sets equality if PageCnt odd
testout:   ret
;-----------------------------------------------------------------------------
; Advance to new column, adjusting page data and printing footer if new page.
;
; If EVENPASS flagged, force exit from higher level PrnBuf after one page.
; Same exit can occur from ChkMax test, so be careful that PrnNewPg entry
; never triggers this exit (since call not inside PrnBuf).
;-----------------------------------------------------------------------------
PrnToTop:  mov   al,PrntMsg+28         ; Column letter in message
           inc   ax
AlphaCol   =     $ + 1                 ; PrepWid sets maximum letter
           cmp   al,0
           jbe   samepage              ; Same page?  Ahead

           call  PrnFoot               ; Else print footer
           test  bp,bp                 ; Test if on even pass (one page print)
           js    killout               ; Yes?  Kick out from PrnBuf--EVENPASS
                                       ;   assumed high bit
           inc   WORD PTR PageCnt      ; Bump page count
PrnNewPg:  mov   dx,OFFSET Reset
           mov   cx,OFFSET ColTop - OFFSET Reset
           call  PrnIf                 ; Print if on page for printing
           call  ChkMax                ; Set AX, checking if past PageTot
           ja    killout               ; Past?  Kick out from PrnBuf

           mov   bx,OFFSET PrntMsg+11  ; End of page number in message
           call  AXtoAsc               ; Set ASCII display page from AX
           mov   bx,OFFSET FootNums+2  ; End of page number in page footer
           call  AXtoAsc               ; Set ASCII print page from AX
           mov   al,'A'                ; Reset display column
samepage:  mov   PrntMsg+28,al         ; Column letter in message
           push  ax
           mov   al,PRNTDIFF
           call  DispApnd              ; Display progress
           pop   ax

           sub   al,'A'
           cbw                         ; AX now 0-25 binary column
ColWid     =     $ + 1                 ; PrepWid adjusts to 9/7 bit value
           mov   bx,0                  ; Column width typically 86 SHL 7
           mul   bx                    ; Result still fits AX (DX zeroed)
ColIndent  =     $ + 1                 ; PrepWid sets 9/7 bit column indent
           add   ax,0                  ; Skip 3 character box indent plus
           mov   cl,7                  ;   column indent from PrepWid
           shr   ax,cl                 ; Truncate fraction
           mov   bx,OFFSET Margins+5   ; End of left margin in page footer
           call  AXtoAsc
ChrWid     =     $ + 1                 ; PrepWid adjusts character width, e.g.,
           add   ax,0                  ;   80--includes proportional slack byte
           mov   bx,OFFSET Margins+9   ; End of right margin in page footer
           call  AXtoAsc               ; Also zeroes CH
           mov   dx,OFFSET ColTop
           mov   cl,OFFSET KillFont - OFFSET ColTop
           jmp   SHORT PrnIf           ; Print if on page for printing

killout:   and   bp,NOT EVENPASS       ; Flag done with page (if on last pass)
           pop   ax                    ; Kill return address
           pop   ax                    ; Kill pushed CX in PrnBuf and fall
                                       ;   harmlessly...
;-----------------------------------------------------------------------------
; Set AX to page count and compare to total pages.
;-----------------------------------------------------------------------------
ChkMax:    mov   ax,WORD PTR PageCnt
PageTot    =     $ + 1                 ; Set by PrepWid
           cmp   ax,0
           ret
;-----------------------------------------------------------------------------
; Reset LineCnt/LineCtr and rezero ColPos/ColFrac/ByteCnt, also zeroing AX/DX.
; Entry SetCol uses AX/DL to set only ColPos/ColFrac/ByteCnt.
;-----------------------------------------------------------------------------
SetSome:   mov   al,LINEMAX
           mov   LineCnt,al
           mov   BYTE PTR LineCtr,al
           xor   ax,ax
           cwd
SetCol:    mov   WORD PTR ColPos,ax
           mov   BYTE PTR ByteCnt,dl
           ret
;-----------------------------------------------------------------------------
; Strip (null out) illegal characters at SI and translate 0/10/12 specially.
; CX (assumed non-zero) bytes tested.  On exit, BX points to XlatTbl.
;
; ASCII 26 is a discretionary null-out (EOF mark not usually wanted in text
; printouts).  The others would require a special HP transparent print, if
; present in font.  So ASCII 7-15 and 27 have been omitted from font.  FFs are
; nulled for footer note, but usually translated to LFs in file (later seen as
; CRLFs).  Exceptions are /B, when both LFs and FFs are nulled after footer
; note has been handled, and /J, which allows form feeds.
;-----------------------------------------------------------------------------
StripBuf:  mov   di,si                 ; Synch DI/SI
           mov   bx,OFFSET XlatTbl     ; Also exit value
           push  si                    ; Preserve SI/CX
           push  cx
striplp:   lodsb
           cmp   al,27
           ja    nochange              ; 28-255?  No change

           xlatb                       ; Else null 7/11/14/15/26/27, blank 0,
nochange:  stosb                       ;   translate 10 to 0/10, and translate
           loop  striplp               ;   12 to 0/10/12

           pop   cx
           pop   si                    ; Fall harmlessly...
;-----------------------------------------------------------------------------
; Return clear carry if AL is space character (8/13/32/255), else set carry.
;-----------------------------------------------------------------------------
IsSpc:     cmp   al,' '
           je    isout
           cmp   al,255
           je    isout
           cmp   al,13
           je    isout
           cmp   al,8
           je    isout
           stc
isout:     ret
;-----------------------------------------------------------------------------
; Store AL to DI repeatedly, bumping AH til zero.  ColPos updated by dot width
; of AL, saving prior position for WrapChk.  MaxCol updated as needed.
;-----------------------------------------------------------------------------
BumpSto:   push  ax
           mov   WORD PTR PrevBX,bx    ; For WrapChk

           push  bx
           mov   bx,OFFSET WidTbl      ; Table of dot widths
           xlatb                       ; Width of character plus interspace
           pop   bx

           add   al,bh                 ; Add in previous 0-11 fraction
           dec   bx                    ; Offset first BL bump
bploop:    inc   bx                    ; Bump integer column position BL
           sub   al,12
           jnb   bploop                ; Fraction 12 or over?  Adjust BL

           add   al,12                 ; Back to final 0-11 value
           mov   bh,al                 ; Update column fraction
           pop   ax

           stosb                       ; Store character to OutBuf
           inc   ah                    ; Bump negative counter (usually 255)
           jne   BumpSto               ; Loop as needed

MaxCol     =     $ + 2                 ; Initialized by ScanFile to 1
           cmp   bl,0                  ; Compare current column to max so far
           jbe   nomax                 ; Less equal?  Out

           call  IsSpc                 ; Includes 255 check
           jnc   nomax                 ; Space?  Skip max update

           mov   BYTE PTR MaxCol,bl    ; Else update maximum column
nomax:     ret
;-----------------------------------------------------------------------------
; Prepare next output buffer, setting CX to byte count and DI to buffer start.
; MaxCol also updated throughout in BumpSto (for reference in PrepWid) and
; SeekLo/SeekHi updated at end (in case this is ODDPASS).
;
; After read/strip, data is expanded to OutBuf (expansion due to Tabs/CRLFs).
; There may be at most one tab or column eject spillover into bottom of stack
; (179 bytes maximum spill).  At end, a backward seek adjusts file pointer for
; unprocessed InBuf bytes or for /R wrap backup to word/line start checkpoint.
; OutBuf slack makes first unlikely.
;
; For /B prints, ByteCnt is insured to be zero at page starts by earlier
; nulling of all LFs/FFs (other than those inserted here).  This simplifies
; setting known ColPos/ByteCnt page start points in LastPass if /B.
;-----------------------------------------------------------------------------
putff:     call  StoCrLf               ; Treat form feed as column eject
           mov   al,BYTE PTR LineCtr
           cmp   al,LINEMAX
           jne   putff                 ; Not reset yet?  Loop
           jmp   SHORT toignore        ; Can skip wrap checking

NextBuf:   mov   dx,OFFSET InBuf
           mov   si,dx                 ; Ready SI for StripBuf call
           mov   cx,INSIZE
           mov   ah,3Fh
           call  Read21h               ; Read to InBuf, returning size AX
           MOVE  cx,ax
           jcxz  nomax                 ; No data?  Done (DI irrelevant)

           call  StripBuf              ; Input SI/CX preserved

           mov   di,OFFSET OutBuf
           push  di                    ; Save till NextBuf exit

ColPos     =     $ + 1                 ; ColPos/ColFrac/ByteCnt zeroed by
ColFrac    =     $ + 2                 ;   ScanFile/PrepWid/LastPass
           mov   bx,0                  ; High byte is fraction 0-11
ByteCnt    =     $ + 1
CmdWid     =     $ + 2                 ; Set by GetFlags (replacing CR)
           mov   dx,0D00h              ; DH fixed during loop
           call  SaveChk               ; In case /R
exploop:   cmp   di,OFFSET OutBuf + OUTSIZE
           ja    expout                ; Past output buffer?  Out

           test  bp,BLKFLAG            ; Check if blocking
           je    noblock               ; No /B?  Ahead

           cmp   dl,dh                 ; Compare count to max
           inc   dl                    ; Update count in case not at max
           jc    noblock               ; Not there yet?  Ahead

           mov   dl,0                  ; Rezero count
           inc   cx                    ; Counter loop decrement
putcrlf:   call  StoCrLf               ; Insert CRLF into output stream
toignore:  jmp   SHORT ignore          ; SaveChk done, so can skip WrapChk

noblock:   lodsb
           cmp   al,12                 ; Only allowed if /J
           je    putff                 ; Form feed?  Eject to next column

           mov   ah,255                ; Anticipate only one bump (also is
           cmp   al,0                  ;   12-dot space character)
           je    ignore                ; Null?  Ignore

           cmp   al,8
           jne   chklf                 ; Not backspace?  Ahead

           xchg  al,ah                 ; Force HP to recognize 12-dot space
           stosw                       ;   as last printed character, then back
           xchg  al,ah                 ;   up 12 dots to original position
           test  bl,bl                 ; Test if near print column start
           jne   midcol                ; No?  Ahead

toleft:    mov   bx,1                  ; Else rig BX zero after next
midcol:    dec   bx                    ; Backup one integer column
           stosb                       ; Store to OutBuf
           jmp   SHORT wrchk

chklf:     cmp   al,10
           je    putcrlf               ; Line feed?  Insert both CR and LF
                                       ;   (for wrap and Unix)
           cmp   al,13
           je    toleft                ; Carriage return?  Back

           cmp   al,9
           jne   stochr                ; Not tab?  Handle normally

           mov   ax,'*' SHL 8 OR 27    ; Store 27,"*p-ddX" where dd is
           stosw                       ;   00 to 11 (always 00 if fixed
           mov   ax,"-p"               ;   spacing) to back up to integral
           stosw                       ;   column
           mov   al,bh                 ; Current column fraction
           call  StoAam                ; Store as two ASCII digits
           dec   di                    ; Undo extra DI bump in StoAam
           mov   al,'X'
           stosb
           mov   bh,0                  ; Synch BH--BX now on column boundary
           mov   ax,bx                 ; Divide integer column position by
           div   BYTE PTR TabSize      ;   tab size to get remainder
TabSize    =     $ + 2                 ; Tab size may be adjusted by GetFlags
           sub   ah,8                  ; Negative number of spaces to advance
           mov   al,255                ; Use instead of space for 12-dot width
stochr:    call  BumpSto               ; Store AL one or more (-AH) times
wrchk:     shr   bp,1                  ; Discard GOTCHR flag (assumed low bit)
           call  IsSpc                 ; Clear carry on space, etc.
           rcl   bp,1                  ; Reinsert GOTCHR flag
           mov   ax,bp                 ; Test AL in WrapChk for GOTCHR/INWORD
           test  al,WRAPFLAG
           je    ignore                ; Not wrapping?  Bypass next

           dec   cx                    ; Synch DI/SI/CX/BX for convenience
           call  WrapChk               ; Handle word wrap
           inc   cx                    ; Offset next loop instruction
ignore:    loop  exploop

expout:    test  bp,WRAPFLAG           ; If wrapping, back up to word start
           je    wrok                  ;   unless final pass

           cmp   si,OFFSET InBuf + INSIZE
           je    wrback                ; Full buffer processed?  Can back up
           jcxz  wrok                  ; Short buffer fully processed?  Then
                                       ;   last one, so skip backup
wrback:    call  RestPos               ; Else back up to last checkpoint
wrok:      MOVE  ax,bx
           call  SetCol                ; Update ColPos/ByteCnt from AX/DL
           MOVE  ax,cx                 ; Seek back CX bytes
           neg   ax
           cwd                         ; 0 or -1 to DX
           xchg  dx,ax                 ; AX:DX now relative seek offset
           MOVE  cx,ax                 ; Now CX:DX, zero or small negative
           mov   ax,4201h              ; Seek from current offset
           call  Read21h               ; Returns absolute offset DX:AX
           mov   cx,di
           pop   di                    ; Start of OutBuf
           sub   cx,di                 ; OutBuf byte count to CX and fall...
;-----------------------------------------------------------------------------
; Save DX:AX to local copy of file pointer (in StoCrLf).
;-----------------------------------------------------------------------------
SetSeek:   mov   WORD PTR SeekLo,ax    ; Update StoCrLf copy of file pointer
           mov   WORD PTR SeekHi,dx    ;   for use in next call to NextBuf--
           ret                         ;   insure PrepWid rezeroes these
;-----------------------------------------------------------------------------
; This is WrapChk branch that falls into StoCrlf.  Branch is reached (with
; position backed up one character) when column position passes wrap boundary
; and non-space character found.
;-----------------------------------------------------------------------------
inscrlf:   test  al,INWORD             ; Test if this is start of word
           je    StoCrLf               ; Yes?  Force CRLF before character

           mov   ax,WORD PTR SaveBX    ; Can destroy AX now
           test  ax,ax                 ; See if checkpoint was line start
           je    StoCrLf               ; Yes?  Must split word

           call  RestPos               ; Else restore position to word start
;-----------------------------------------------------------------------------
; Store CRLF, leaving CR in AL, and zero column position.  Also update LineCtr.
; At end, save line start check point info in case /R.  If ODDPASS, populate
; SeekTbl entries with even page offsets in input file.  The LineCtr update is
; always made in case /J form feed handling.
;
; Cannot fill SeekTbl (entry 0 ignored) during ScanFile, since ColTot not
; known yet if width is computed value.  PageCtr/LineCtr are separate from
; PageCnt/LineCnt, since they are out of synch.
;-----------------------------------------------------------------------------
StoCrLf:   xor   bx,bx                 ; Rezero ColPos/ColFrac

LineCtr    =     $ + 1                 ; Initialized by SetSome to LINEMAX
           mov   ax,0                  ; LineCtr high byte always zero
           dec   ax                    ; AX free for use, since updated below
           jne   updtline              ; Not new column yet?  Ahead

           test  bp,ODDPASS            ; Test if /2 odd page pass
           je    resetline             ; No?  Skip column/page/seek check

ColCtr     =     $ + 1                 ; Initialized by PrepWid to ColTot
           mov   al,0                  ; AH zero from LineCtr high
           dec   ax
           jne   updtcol

PageCtr    =     $ + 1                 ; Initialized by ScanFile to 1
           mov   ax,0
           inc   ax
           cmp   ax,PAGEMAX            ; Test if at page limit
           ja    resetboth             ; Yes?  Let page counter stick there

           mov   WORD PTR PageCtr,ax
           test  al,1                  ; Test if start of odd page
           jne   resetboth             ; Yes?  Skip table save

           push  di
           shl   ax,1                  ; Table entry is 2 words
           add   ax,OFFSET SeekTbl
           MOVE  di,ax
           mov   ax,si
           sub   ax,OFFSET InBuf
SeekLo     =     $ + 1                 ; Zeroed by PrepWid and updated
           add   ax,0                  ;   at NextBuf tail
           stosw
SeekHi     =     $ + 1                 ; Zeroed by PrepWid and updated
           mov   ax,0                  ;   at NextBuf tail
           adc   ax,bx                 ; BX is zero from top of call
           stosw
           pop   di

resetboth: mov   al,BYTE PTR ColTot
updtcol:   mov   BYTE PTR ColCtr,al    ; Store reset or decremented column

resetline: mov   al,LINEMAX
updtline:  mov   BYTE PTR LineCtr,al   ; Store reset or decremented line

           mov   ax,0A0Dh              ; Insert CRLF in DI output stream
           stosw                       ; Fall in case /R...
;-----------------------------------------------------------------------------
; Save position registers and flag not in word.
;-----------------------------------------------------------------------------
SaveChk:   mov   WORD PTR SaveBX,bx    ; Save checkpoint (word or line start)
           mov   WORD PTR SaveCX,cx
           mov   WORD PTR SaveDI,di
           mov   WORD PTR SaveSI,si
inspace:   and   bp,NOT INWORD         ; Flag not in word
           ret
;-----------------------------------------------------------------------------
; Handle wrap checking at expand loop tail.  Assumes DI/SI/CX/BX in synch and
; assumes AL holds GOTCHR/INWORD flags.
;
; Maximum wrap value is under 255 to allow slack zone for word handling across
; boundary.  Maximum could be larger than 127, but note safety check below and
; possibility of tab near boundary.  This is also reason for small tab maximum.
;-----------------------------------------------------------------------------
WrapChk:   test  bx,bx                 ; Check if at line start
           je    SaveChk               ; Yes?  Just save checkpoint--not
                                       ;   redundant if overprinting
           cmp   bl,207                ; Safety check--worst case is tab=52
           ja    StoCrLf               ; Over safety margin?  Force CRLF

           test  al,GOTCHR             ; Check for space, etc.
           je    inspace               ; Yes?  Insure in space flagged

           dec   di                    ; Else back up one character to
           dec   si                    ;   word start (leaving BX as is)
           inc   cx

           cmp   bl,dh                 ; Compare column position to wrap size
toins:     ja    inscrlf               ; Over?  Back to force CRLF
           jb    chkword               ; Under?  Ahead

           cmp   bh,3                  ; Else check fraction 0-11
           ja    toins                 ; 4-11?  Back to force CRLF--need to
                                       ;   allow some spill in case /R1
chkword:   test  al,INWORD
           jne   skipsave              ; In word?  Just restore position

           push  bx
PrevBX     =     $ + 1                 ; Set by BumpPos
           mov   bx,0                  ; Complete back up with BX adjustment
           call  SaveChk               ; Save checkpoint
           pop   bx

skipsave:  inc   di                    ; Restore position
           inc   si
           dec   cx
           or    bp,INWORD             ; Flag in word (undoing SaveChk)
           ret
;-----------------------------------------------------------------------------
; Construct font from appended packed font data, returning offset/size DX/CX
; and returning BP flags in AX (must restore to BP afterward).  Also fill
; WidTbl with dot widths of characters (includes interspacing).
;
; This relies on equates, which must be in synch with font data.  Assume input
; AX (800h) is at least 256 but small enough so there is no overrun of help
; text when WidTbl initialized.  HP.DOC is actually about 3.6K, so no problem.
;
; Two bytes precede each character's raster data.  If word is zero, no data
; follows for that ordinal (0-255), so ignore.  If non-zero, the word holds
; Lft/Top/Wid/Hgt values for the character and int((Wid+7)/8) * Hgt raster
; bytes follow.  The font tail, font header, and template character header are
; at the end of the packed font data in that order.
;
; Because font is expanded from end of raster data, minimum RAM is about 13K
; beyond (total 25K plus small stack).  So I/O buffer sizes were adjusted to
; make 26K plus environment the minimum RAM.  This is inconsequential from
; command prompt, but allows spawning utility (int 4Bh) with little free RAM.
;
; During Start, font ID in font header is tweaked if not /F, spacing byte
; there is tweaked if /P, and StubLoc call is switched to PropSpc if /P.
;-----------------------------------------------------------------------------
MakeFont:  mov   di,OFFSET WidTbl      ; Anticipate fixed spacing
           MOVE  cx,ax                 ; At least 256, but smallish
           mov   al,12                 ; Fixed space character dot width
           rep   stosb                 ; Also zeroes CH

           mov   si,OFFSET FontData    ; Read from packed data
           mov   di,OFFSET ChrHdr      ; Write from character header template
           push  bp                    ; Save flags
           mov   bp,0FF00h             ; Track ordinal in BP low byte
mfloop:    lodsw                       ; Fetch dimensions of next character
           test  ax,ax
           je    skipchr               ; Zero word?  Skip this ordinal

           push  si                    ; Save raster data pointer til loop end

           push  ax                    ; Save dimensions a moment
           mov   si,OFFSET ChrHdr      ; Copy first character header forward
           mov   cl,CHRHDRLEN          ;   (or over itself initially)
           rep   movsb                 ; CH zero
           mov   bx,di
           sub   bx,CHRHDRLEN - 5      ; Point to end of ASCII ordinal
           mov   ax,bp                 ; Ignore AH
           call  ALtoASC               ; Store ASCII ordinal
           pop   dx                    ; Restore dimensions

           sub   di,10                 ; Backup DI to Lft location
           mov   al,dl                 ; Need high 5 bits of DL for Lft
           mov   cl,3
           shr   al,cl
           sub   al,BASELINE           ; Adjust by baseline value
           cbw
           xchg  al,ah                 ; To non-Intel order
           stosw                       ; Store Lft value (possibly negative)
           mov   al,dl                 ; Need low 3 bits of DL for Top
           and   al,7
           add   al,TOPMIN             ; Adjust from zero offset
           scasb                       ; Skip high zero
           stosb                       ; Store Top value
           mov   dl,al                 ; Save Top in DL also
           mov   al,dh                 ; Need high 5 bits of DH for Wid
           shr   al,cl
           scasb                       ; Skip high zero
           stosb                       ; Store Wid value
           add   al,7                  ; Convert Wid DH from bit count
           shr   al,cl                 ;   to byte count
           xchg  al,dh                 ; Need low 3 bits of DH for Hgt,
           and   al,7                  ;   saving Wid byte count in DH
           neg   al                    ; Negate and add Top to get Hgt
           add   al,dl
           scasb                       ; Skip high zero
           stosb                       ; Store Hgt value
           scasw                       ; Advance DI past xdelta word
StubKill   =     $ + 1                 ; /P bumps call to PropSpc
           call  StubLoc               ; Do nothing if fixed spacing
           mul   dh                    ; AX now data bytes with AH zero

           push  ax                    ; Save count
           add   al,16                 ; Add in character header length
           add   bx,10                 ; Point BX to end of ASCII header size
           call  ALtoASC               ; Store header size as ASCII
           pop   cx                    ; Restore byte count

           pop   si                    ; Restore packed data pointer
           rep   movsb                 ; Copy data after character header

skipchr:   inc   bp                    ; Bump ordinal value
           jne   mfloop                ; Not done?  Loop

           pop   ax                    ; Restore flags to AX, not BP
           mov   cl,TAILLEN
           rep   movsb                 ; Copy tail data after font
           mov   dx,si                 ; Point DX to font and fall...
;-----------------------------------------------------------------------------
; Set CX to DI less DX.  Also called by PrnBuf.
;-----------------------------------------------------------------------------
SetCX:     mov   cx,di
           sub   cx,dx
StubLoc:   ret                         ; PropSpc must follow immediately
;-----------------------------------------------------------------------------
; This is proportional spacing patch adjustment to font characters, with
; characters adjusted leftward, xdeltas changed from default 48, and WidTbl
; entries changed from default 12.  High ASCII block/bar characters (176-223)
; retain fixed spacing.  Input AL/DI is preserved.
;
; ASCII 32 is present in raster data specifically for proportional spacing
; (character could be omitted in fixed space font).  Its xdelta is set to 10
; dots here.  ASCII 255 is absent from font, so it has spacing that matches
; pitch (12 dots).  This is required for tab/backspace handling.  Characters
; j/i/l and f/r get additional special adjustments below.
;-----------------------------------------------------------------------------
PropSpc:   push  ax
           mov   ax,bp                 ; AL is ordinal til end
           mov   ah,HIGH OFFSET WidTbl ; WidTbl is 256-aligned
           mov   si,ax                 ; Point SI to WidTbl entry

           mov   cl,48                 ; Anticipate fixed spacing
           cmp   al,''
           jb    notfixed              ; Not high graphic?  Proportional
           cmp   al,''
           jbe   isfixed               ; High graphic?  Force xdelta to 48

notfixed:  pop   cx                    ; Input AL to CL
           push  cx

           cmp   al,'j'                ; Handle j/i/l characters specially,
           je    isjil                 ;   forcing leftward (actually downward
           cmp   al,'i'                ;   since landscape) one dot
           je    isjil
           cmp   al,'l'
           jne   notjil

isjil:     dec   cx
notjil:    mov   [di-7],cl             ; Adjust Top

           pop   cx                    ; Refresh input AL in CL
           push  cx

           cmp   al,'f'                ; Handle f/r specially, reducing
           je    isfr                  ;   inter-character spacing by a dot
           cmp   al,'r'
           je    isfr
           cmp   al,' '                ; Handle space specially
           jne   notfr

           mov   cl,8                  ; 10-dot width for space
notfr:     inc   cx                    ; 2-dot spacing between characters
isfr:      inc   cx                    ; 1-dot spacing for f and r
           mov   [si],cl               ; Save width plus spacing to WidTbl
           shl   cl,1                  ; Multiply by 4, since xdelta is
           shl   cl,1                  ;   expressed in quarter dots
isfixed:   dec   di                    ; Point back to xdelta
           MOVE  ax,cx                 ; AL now computed (or fixed space 48)
           stosb                       ; Adjust xdelta, restoring DI
           pop   ax                    ; Restore input AL
           ret
;-----------------------------------------------------------------------------
; Program data, other than help/font data appended to COM file, some PSP data,
; and word/byte data embedded in code operands.  Buffers overlap appended data.
;
; ParmTbl word values follow each switch, with some (those with ?) ignored.
; XlatTbl entries 10 and 12 may change.  Spaces in the HP escape sequences and
; in PrntMsg are filled as needed.  TopMar may change.  InfoMsg terminator is
; nulled after display for DOC dump and for use as Ten high byte.  NameBuf is
; 128+ bytes and DTABuf is 43 bytes (both inside syntax summary).
;
; Offset of SeekTbl is 1000h for total space calculation.
;-----------------------------------------------------------------------------
ParmTbl    DB "J",?,?,"P",?,?,"2",?,?,"K",?,?,"S",0,0
           DB "E",0,0,"T",0,0,"W",0,0,"R",0,0,"B",0,0
           DB "D",?,?,"F",?,?,"?",?,?,"C",?,?,"Z",?,?

XlatTbl    DB 32,1,2,3,4,5,6,0,8,9,10,0,0,13,0,0
           DB 16,17,18,19,20,21,22,23,24,25,0,0

VertBar    DB 27,"*p9y    X"           ; Vertical/horizontal dot offset
           DB 27,"*c2a2186bP"          ; Vertical bar--7.3 inch, 300 dpi
PageNote   DB 27,"*p9y36X"             ; Draw box--36 is 3-character indent
           DB 27,"*c3120a2bP"          ; Top--10.4 inch bar--12*260=3120
           DB 27,"*p2194y36X"
           DB 27,"*c3120a2bP"          ; Bottom--same
           DB 27,"*p9y3155X"           ; 3179 is right edge maximum--other
           DB 27,"*c2a2186bP"          ;   vertical bars handled separately
           DB 27,"(10U"                ; PC-8 symbol set
           DB 27,"(sp16.66h8.5vsbT"    ; Fixed spacing/16.66 pitch font/etc.
           DB 27,"9"                   ; Clear horizontal margins
NoteNum    DB 27,"&a   c92R"           ; Position print cursor
PageFoot   DB 27,"*p60X"               ; Reposition horizontally (dots)
ShortName  DB "             "
FileDtTm   DB "  /  /     :  "
           DB 27,"*p2650X"             ; Reposition horizontally (dots)
DosDtTm    DB "  /  /     :   PAGE "
FootNums   DB "   /   "
FormFeed   DB 12                       ; Eject
Reset      DB 27,"E"                   ; HP reset
           DB 27,"&s1C"                ; Disable line wrap
           DB 27,"&l1ouzl3.84c"        ; Landscape/no registration-skip/line
TopMar     DB    "02e93F"              ;   spacing 1/48"/top margin (2/11)/lpp
ColTop     DB 27,"(1X"                 ; Select downloaded font (ID 1)
           DB 27,"&k4.8hG"             ; Column spacing 1/120"/normal CRLFs
           DB 27,"9"                   ; Clear horizontal margins
Margins    DB 27,"&a   l   m1R"        ; Reset margins/position to row 1
           DB 13                       ; Position to left margin
KillFont   DB 27,"E"                   ; HP reset
           DB 27,"*c1d2F"              ; Delete downloaded font (ID 1)

BeepMsg    DB 7                        ; Insure at most 255 bytes from here
BackMsg    DB 13,9,9,9,9,9,13,EOS      ;   to start of NameBuf
DumpMsg    DB 10,"Extracting",EOS
DnLdMsg    DB 10,"Downloading",EOS
ReadMsg    DB "Reading",EOS
PrntMsg    DB 13,"On page     of    , column  ",EOS
WaitMsg    DB "Press key to eject",EOS
EvenMsg    DB "Refeed sheets and press key",EOS
DoneMsg    DB "...done",EOS
CritMsg    DB "Check printer.  Retry?",EOS

InfoMsg    DB NL,9,"   "
DfltNote   DB "HPTINY V2.0 1995    Print Utility for Programmers",NL
NameBuf    DB NL,"Syntax:    "
           DB      "HPTINY /?",9,9,9,"Help to "
HelpFile   DB "HPTINY.DOC",0,NL
           DB 9,"   HPTINY /F [/P]",9,9,"Font to "
FontFile   DB "HPTINY.FNT",0,NL
           DB 9,"   HPTINY files [[/switch]...]",9,"Prints files",NL
           DB NL,"Switches:  "
           DB      "Wn",9," = Width, 1-255",9,9,"Default computed",NL
           DB 9,"   Rn",9," = Wrap, 1-127",9,9,"E.g., if paragraph format",NL
           DB 9,"   Bn",9," = Block, 1-255", 9,9,"E.g., if no CRLFs",NL
           DB 9,"   Tn",9," = Tab size, 1-63",9,"Default 8",NL
           DB 9,"   Sn",9," = Start page, 1-999",9,"Default 1",NL
           DB 9,"   En",9," = End page, 1-999",9,"Default 999",NL
           DB 9,"   K",9," = Keypress to eject",9,"For single sheet feeds",NL
           DB 9,"   C",9," = Center print box",9,"Default for 3-ring binder",NL
           DB 9,"   2",9," = Two-sided print",9,"Odd then even pages",NL
           DB 9,"   Z",9," = Null print",9,9,"For page count display",NL
           DB 9,"   D",9," = Print to disk",9,"Overwrites "
PrnFile    DB "HPTINY.PRN",0,NL
DTABuf     DB 9,"   P",9," = Proportional font",9,"Default fixed space",NL
           DB 9,"   J",9," = Column eject on FF",9,"Default CRLF on FF",NL
           DB 9,"   Ntext = Footer note",9,9,"Default utility logo",NL
           DB NL
           DB "Utility prints text files to LaserJet II or later model in "
           DB "a compact newspaper-",NL
           DB "style format, typically 270 lines per page in 3 columns.  "
           DB "See DOC file.",13
Ten        DB 10,EOS                   ; Multiplier/divisor after EOS nulled

SLACK      =     LOW (OFFSET $ - OFFSET HpTiny)
SLACK      =     LOW (256 - SLACK)     ; 256-align WidTbl--see PropSpc

WidTbl     =     $ + SLACK             ; Holds character dot widths
SeekTbl    =     $ + SLACK + 256       ; Holds 4-byte even page seek offsets
InBuf      =     $ + SLACK + 256 + SEEKSIZE
OutBuf     =     $ + SLACK + 256 + SEEKSIZE + INSIZE

FontData   =     $ + DOCLEN            ; Packed font data follows appended
FontTail   =     $ + DOCLEN + PACKLEN  ;   help text--DOCLEN passed to MASM
FontHdr    =     $ + DOCLEN + PACKLEN + TAILLEN
ChrHdr     =     $ + DOCLEN + PACKLEN + TAILLEN + FNTHDRLEN

Code_Seg   ENDS
           END HpTiny
