/*
 * FPLOT - fpcmd.c
 *
 * Copyright (C) 1997, 1998   Michael C. Ring
 *
 * Permission to use, copy, and distribute this software and its
 * documentation for any purpose with or without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and 
 * that both that copyright notice and this permission notice appear 
 * in supporting documentation.
 *
 * Permission to modify the software is granted, but not the right to
 * distribute the modified code.  Modifications are to be distributed 
 * as patches to released version.
 *  
 * This software is provided "as is" without express or implied warranty.
 */

/*
 *      fpcmd.c
 *
 *      $Log: fpcmd.c $
 *      Revision 1.8  1998/04/03 20:30:50  mring
 *      detect and exit if data file has no data in it
 *
 *      Revision 1.7  1998/04/03 20:22:03  mring
 *      minor updates to exit-codes
 *
 *      Revision 1.6  1998/03/15 23:25:10  mring
 *      changed call to fread
 *
 *      Revision 1.5  1998/03/07 04:53:54  mring
 *      detect X axis data < 0 on a LOG plot
 *
 *      Revision 1.4  1998/03/06 00:24:32  mring
 *      added some comments
 *
 *      Revision 1.3  1998/03/05 03:43:08  mring
 *      update for 0.5a
 *
 *      Revision 1.2  1997/12/06 23:19:26  mring
 *      added a few comments
 *
 *      Revision 1.1  1997/12/06 22:57:48  mring
 *      Initial revision
 *
 */

#include "fplot.h"
#include "glob.h"

/* RCS TRACER */
static  char    
RCS_ID[] = "$Id: fpcmd.c 1.8 1998/04/03 20:30:50 mring Exp $";


/***********************************************************************/
plot_raw_file()
{
generate_plot(RAW_FILE,0);
}
/***********************************************************************/
plot_command_file()
{
int     cmd_flag, count, eof_flag;

if ((g_tmp_ptr = fopen(g_tmp_file,"r")) == NULL)
  {
   fprintf(stderr,
           "ERROR : Could not open temporary file \'%s\'\n",g_tmp_file);
   exit_program(60);
  }

eof_flag = FALSE;
                                              /* loop thru all commands */
while (fgets(g_input_string,MAX_LINE_LENGTH,g_tmp_ptr))
  {
   cmd_flag = FALSE;
   filter_input_string(g_input_string);

   count = parse_line_into_fields(g_input_string);

   if (count >= 1)               /* only process a command if there is one */
     {
      lowercase(g_buf[1]);       /* convert field 1 to lowercase */

      if (g_buf[1][0] != '#')    /* comments begin with a '#' */
        {
         if (strcmp(g_buf[1],"set") == 0)   /* we have a 'set' command */
           {
            cmd_flag = TRUE;
            parse_set_command(count);
           }
   
         if (strcmp(g_buf[1],"reset") == 0) /* we have a 'reset' command */
           {
            cmd_flag = TRUE;
            parse_reset_command(count);
           }
   
         if (strcmp(g_buf[1],"data") == 0)  /* we have a 'data' command */
           {
            cmd_flag = TRUE;
            eof_flag = get_data_from_command_file();

            strcpy(g_buf[1],"plot");        /* simulate this command  */
            strcpy(g_buf[2],g_tmp_file2);   /* plot this data file    */

            count = 2;                      /* say we have 2 fields   */
            generate_plot(COMMAND_FILE,count);

            g_buf[1][0] = '\0';		    /* must erase the command */

            if (eof_flag)
              break;
           }

         if (strcmp(g_buf[1],"plot") == 0)  /* we have a 'plot' command */
           {
            cmd_flag = TRUE;
            generate_plot(COMMAND_FILE,count);
           }

         if (cmd_flag == FALSE)
           {
            fprintf(stderr,"ERROR : Unknown command \'%s\'\n",g_buf[1]);
            exit_program(24);
           }
        }
     }
  }

fclose(g_tmp_ptr);
g_tmp_ptr = NULL;
}
/***********************************************************************/
parse_reset_command(fieldcount)
int     fieldcount;
{
fieldcount = reset_field_count(fieldcount);

if (fieldcount != 2)
  {
   fprintf(stderr,"ERROR : Invalid \'reset\' command\n");
   exit_program(40);
  }

lowercase(g_buf[2]);

if (strposition(g_buf[2],"defa") != 0)       /* accept this as 'defaults' */
  {
   fprintf(stderr,"ERROR : Unknown \'reset\' command\n");
   exit_program(42);
  }

/* all is good with 'reset defaults', so do it */

init_globals();
}
/***********************************************************************/
get_data_from_command_file()
{
int	fieldcount, ret;
char    *buf;

ret = FALSE;      	   /* assume we will not be at EOF */

buf = memory_allocate((MAX_LINE_LENGTH + 4) * sizeof(char));

if ((g_tmp_ptr2 = fopen(g_tmp_file2,"w")) == NULL)
  {
   fprintf(stderr,
           "ERROR : Could not open temporary file \'%s\'\n",g_tmp_file2);
   exit_program(60);
  }

while (1)
  {
   if (fgets(buf,MAX_LINE_LENGTH,g_tmp_ptr) == NULL)  
     {
      ret = TRUE;
      break;
     }

   replace_crlf(buf);
   filter_input_string(buf);
   fieldcount = parse_line_into_fields(buf);
   fieldcount = reset_field_count(fieldcount);

   if (fieldcount > 0)                  /* look for ending 'data' command */
     {
      lowercase(g_buf[1]);
      if (strcmp(g_buf[1],"data") == 0)
        break;
     }

   if (fieldcount > 0)                  /* a valid data line, so store it */
     fprintf(g_tmp_ptr2,"%s\n",buf);
  }

fclose(g_tmp_ptr2);
g_tmp_ptr2 = NULL;

memory_free(buf);
return(ret);
}
/***********************************************************************/
generate_plot(caller,fieldcount)
int     caller, fieldcount;
{
if (caller == COMMAND_FILE)
  {
   fieldcount = reset_field_count(fieldcount);
   
   if (fieldcount != 2)
     {
      fprintf(stderr,"ERROR : Invalid \'plot\' command\n");
      exit_program(50);
     }
   
   strcpy(g_plot_file,g_buf[2]);
  }

if (caller == RAW_FILE)
  {
   strcpy(g_plot_file,g_tmp_file);
  }

/*
 *   determine what we are plotting, number of Y ranges, which fields
 *   to use for which Y range, etc.
 *
 *   also convert the input ASCII data to a binary format in temp
 *   file 'g_tmp_file3'.
 */

g_num_y_ranges = auto_parse_field_defs(g_plot_file);

/*
 *   if plotting more than 2 ranges, force the Y2 range to be the
 *   same scaling as Y1
 */

if (g_num_y_ranges > 2)
  {
   g_force_y2_y1 = TRUE;
  }

/*
 *   now check for auto-ranging on X and Y ranges
 */

auto_scale_axis();

/*
 *   now do the plotting ...
 */

if (g_outputformat == FORMAT_POSTSCRIPT)
  output_postscript_plot();

if (g_outputformat == FORMAT_HPGL2)
  output_hpgl2_plot();

if (g_outputformat == FORMAT_EPSON_LQ)
  output_epson_lq_plot();

if (g_outputformat == FORMAT_EPSON_FX)
  output_epson_fx_plot();
}
/***********************************************************************/
invalid_set_command(fieldcount,code)
int     fieldcount, code;
{
int     i;

fprintf(stderr,"ERROR : Invalid \'set\' command\n");

for (i=1; i <= fieldcount; i++)
  {
   fprintf(stderr,"%s ",g_buf[i]);
  }

fprintf(stderr,"\n");
exit_program(code);
}
/***********************************************************************/
/*
 *      this function will read the data file to determine what
 *      to plot
 *
 *      it also parses the ASCII data and re-stores the data in
 *      binary format in g_tmp_file3
 *
 *      function returns : the number of Y ranges to plot
 */
auto_parse_field_defs(filename)
char    *filename;
{
int     i, k, count, num_y, fieldcount, num_y_array[14];
long    local_skip, m, samples;
double  xdata, ydata, sum, *dp;
FILE    *ptr;
        
if ((ptr = fopen(filename,"r")) == NULL)
  {
   fprintf(stderr,"Could not open plot file \'%s\'\n",filename);
   exit_program(10);
  }

if ((g_tmp_ptr3 = fopen(g_tmp_file3,"wb")) == NULL)
  {
   fclose(ptr);
   fprintf(stderr,"Could not create temp file \'%s\'\n",g_tmp_file3);
   exit_program(60);
  }

setvbuf(g_tmp_ptr3,g_big_buffer,_IOFBF,BIG_BUF_SIZE);

dp = (double *)memory_allocate(16 * sizeof(double));

for (i=0; i < 14; i++)
  num_y_array[i] = 0;

num_y = 0;

if (g_field_def_y1 > 0)   num_y_array[++num_y] = g_field_def_y1;
if (g_field_def_y2 > 0)   num_y_array[++num_y] = g_field_def_y2;
if (g_field_def_y3 > 0)   num_y_array[++num_y] = g_field_def_y3;
if (g_field_def_y4 > 0)   num_y_array[++num_y] = g_field_def_y4;
if (g_field_def_y5 > 0)   num_y_array[++num_y] = g_field_def_y5;
if (g_field_def_y6 > 0)   num_y_array[++num_y] = g_field_def_y6;
if (g_field_def_y7 > 0)   num_y_array[++num_y] = g_field_def_y7;
if (g_field_def_y8 > 0)   num_y_array[++num_y] = g_field_def_y8;
if (g_field_def_y9 > 0)   num_y_array[++num_y] = g_field_def_y9;
if (g_field_def_y10 > 0)  num_y_array[++num_y] = g_field_def_y10;
if (g_field_def_y11 > 0)  num_y_array[++num_y] = g_field_def_y11;
if (g_field_def_y12 > 0)  num_y_array[++num_y] = g_field_def_y12;

if (num_y == 0)           /* no Y ranges specified, sample the data     */
  {                       /* file to figure it out.                     */
   if (g_field_def_x > 0) /* if you specify an X field, must do Y also. */
     {
      fclose(ptr);
      fprintf(stderr,"ERROR : No Y ranges specified in command file\n");
      exit_program(26);
     }

   sum   = 0.0;
   count = 0;

   for (i=1; i <= 20; i++)         /* read the first 20 lines of the file */
     {
      if (fgets(g_input_string,MAX_LINE_LENGTH,ptr) == NULL)  break;

      count++;
      filter_input_string(g_input_string);

      fieldcount = parse_line_into_fields(g_input_string);
      fieldcount = reset_field_count(fieldcount);

      sum += (double)fieldcount;
     }

   if (count == 0)
     {
      fclose(ptr);
      fprintf(stderr,"Data file has no data, it is empty\n");
      exit_program(21);
     }

   rewind(ptr);

   sum /= count;
   i = (int)(sum + 1.0E-3);
   if (fabs(sum - i) > 0.08 || i == 0)
     {
      fclose(ptr);
      fprintf(stderr,"Unable to determine which fields to plot\n");
      exit_program(12);
     }

   num_y = i - 1;                    /* first field is X data */
   if (num_y > 12)  num_y = 12;      /* set 12 Y ranges max   */

   if (num_y == 0)           /* only 1 field, plot single Y range */
      num_y_array[++num_y] = 1;
   else                      /* multiple fields, first is X, rest are Y */   
     {
      g_field_def_x = 1;     /* field 1 is X data */

      for (i=1; i <= num_y; i++)
        num_y_array[i] = i + 1;
     }
  }

local_skip            = 0L;
samples               = 0L;  
g_num_samples_to_plot = 0L;

if (g_usesamples)            /* using a limited set of the data ?? */
  {                          /* scan by initial samples            */
   for (m=1; m < g_usesamples_low; m++)
     {
      samples++;

      if (fgets(g_input_string,MAX_LINE_LENGTH,ptr) == NULL)
        {
         fclose(ptr);
         fprintf(stderr,
     "Fewer samples in data file than specified with \'set usesamples\'\n");
         exit_program(15);
        }
     }
  }

while (fgets(g_input_string,MAX_LINE_LENGTH,ptr))
  {
   samples++;

   if (local_skip == 0)
     {
      filter_input_string(g_input_string);
   
      fieldcount = parse_line_into_fields(g_input_string);
      fieldcount = reset_field_count(fieldcount);
   
      if (g_field_def_x > 0)
        {
         if (g_field_def_x > fieldcount)
           {
            fclose(ptr);
   
            fprintf(stderr,
                    "Invalid field in data file, field %d specified for X,\n",
                    g_field_def_x);
   
            fprintf(stderr,"file only has %d fields : line # %ld\n",
                            fieldcount,samples);
            exit_program(16);
	   }
   
         xdata = atof(g_buf[g_field_def_x]) * g_xmultiplier + g_xoffset;

         if (g_xaxis_scale == X_AXIS_SCALE_LOG)
	   {
	    if (xdata <= 0.0)
	      {
               fclose(ptr);
               fprintf(stderr,
               "ERROR! : There is X axis data < 0 on a LOG plot\n");
               exit_program(18);
	      }
	   }
        }
      else
        {
         xdata = samples * g_xmultiplier + g_xoffset;
        }
   
      dp[0] = xdata;                 /* store X data sample */
   
      for (i=1; i <= num_y; i++)
        {
         k = num_y_array[i];

         if (k > fieldcount)
           {
            fclose(ptr);
   
            fprintf(stderr,
            "Invalid field in data file, field %d specified for Y-%d,\n",k,i);
   
            fprintf(stderr,"file only has %d fields : line # %ld\n",
                            fieldcount,samples);
            exit_program(16);
	   }
   
         if (i == 2)
           ydata = atof(g_buf[k]) * g_y2multiplier + g_y2offset;
         else
           ydata = atof(g_buf[k]) * g_y1multiplier + g_y1offset;
   
         dp[i] = ydata;
        }

      g_num_samples_to_plot++;
      fwrite((char *)dp,sizeof(double),(num_y + 1),g_tmp_ptr3);
     }

   if (local_skip == 0)
     {
      if (g_skip_factor > 0)
        {
         local_skip = g_skip_factor;
        }
     }
   else
     local_skip--;

   if (g_usesamples == TRUE && samples == g_usesamples_high)
     break;
  }

fclose(ptr);
fclose(g_tmp_ptr3);
g_tmp_ptr3 = NULL;
memory_free(dp);

if (g_num_samples_to_plot == 0L)
  {
   fprintf(stderr,"Data file has no data, it is empty\n");
   exit_program(21);
  }

return(num_y);
}
/***********************************************************************/

/*
 *             Logic for various auto-ranging combinations
 *
 *                         T=TRUE,  F=FALSE
 *
 * Y1_AUTO_RANGE Y2_AUTO_RANGE FORCE_Y2_Y1            DO WHAT
 * -------------------------------------------------------------------------
 *      T             T             T        auto Y1 Y2, combine into 1
 *
 *      T             T             F        auto Y1 Y2, leave separate
 *
 *      T             F             T        auto Y1 Y2, combine into 1
 *                                                       ignore Y2=set
 *
 *      T             F             F        auto Y1, set Y2, leave separate
 *
 *      F             T             T        set Y1, Y2=Y1, ignore Y2=Auto
 *
 *      F             T             F        set Y1, auto Y2, leave separate
 *
 *      F             F             T        set Y1, Y2=Y1, ignore Y2=set
 *
 *      F             F             F        set Y1, set Y2, leave separate
 *
 */

/***********************************************************************/
auto_scale_axis()
{
int     x_axis_specified, y1_axis_specified, y2_axis_specified, nr, k;
double	*dp, a1, a2, a3, lo, hi, x_data, y_data, x_max, x_min, 
	y1_max, y1_min, y2_max, y2_min;

x_axis_specified  = FALSE;
y1_axis_specified = FALSE;
y2_axis_specified = FALSE;

x_min  =  1.0E200;
x_max  = -1.0E200;
y1_min =  1.0E200;
y1_max = -1.0E200;
y2_min =  1.0E200;
y2_max = -1.0E200;

g_x_divisions  = 16;
g_y1_divisions = 10;
g_y2_divisions = 10;
                    /* no need to auto-range Y2 if only plotting 1 range */
if (g_num_y_ranges == 1)
  {
   y2_axis_specified = TRUE;
  }

if (g_xrange_auto == FALSE)
  {
   x_axis_specified = TRUE;
   x_min = g_xrange_low;
   x_max = g_xrange_high;

   lo = x_min;
   hi = x_max;

   g_true_x_min = lo;
   g_true_x_max = hi;

   scale_values(TRUE,&lo,&hi,&g_x_divisions);
  }

if (g_y1range_auto == FALSE)
  {
   y1_axis_specified = TRUE;
   y1_min = g_y1range_low;
   y1_max = g_y1range_high;

   lo = y1_min;
   hi = y1_max;
   scale_values(TRUE,&lo,&hi,&g_y1_divisions);
  }

if (g_force_y2_y1 == TRUE && y1_axis_specified == TRUE)
  {
   y2_axis_specified = TRUE;
   y2_min = y1_min;
   y2_max = y1_max;
   g_y2_divisions = g_y1_divisions;
  }

if (g_force_y2_y1 == FALSE && g_y2range_auto == FALSE)
  {
   y2_axis_specified = TRUE;
   y2_min = g_y2range_low;
   y2_max = g_y2range_high;

   lo = y2_min;
   hi = y2_max;
   scale_values(TRUE,&lo,&hi,&g_y2_divisions);
  }

/* all 3 axis's were specified, return now */

if (x_axis_specified && y1_axis_specified && y2_axis_specified)
  {
   g_graph_x_min  = x_min;
   g_graph_x_max  = x_max;
   g_graph_y1_min = y1_min;
   g_graph_y1_max = y1_max;
   g_graph_y2_min = y2_min;
   g_graph_y2_max = y2_max;
   return(0);
  }

if ((g_tmp_ptr3 = fopen(g_tmp_file3,"rb")) == NULL)
  {
   fprintf(stderr,"Could not open plot file \'%s\'\n",g_tmp_file3);
   exit_program(60);
  }

setvbuf(g_tmp_ptr3,g_big_buffer,_IOFBF,BIG_BUF_SIZE);

dp = (double *)memory_allocate(16 * sizeof(double));

nr = g_num_y_ranges + 1;		/* num records */

while (1)
  {
   if (fread((char *)dp,sizeof(double),nr,g_tmp_ptr3) < nr) 
     break;

   if (x_axis_specified == FALSE)
     {
      x_data = dp[0];
      if (x_data < x_min)  x_min = x_data;
      if (x_data > x_max)  x_max = x_data;
     }

   for (k=1; k <= g_num_y_ranges; k++)
     {
      y_data = dp[k];

      if (k == 2)
        {
         if (y2_axis_specified == FALSE)
           {
            if (y_data < y2_min)  y2_min = y_data;
            if (y_data > y2_max)  y2_max = y_data;
           }
        }
      else
        {
         if (y1_axis_specified == FALSE)
           {
            if (y_data < y1_min)  y1_min = y_data;
            if (y_data > y1_max)  y1_max = y_data;
           }
        }
     }
  }

fclose(g_tmp_ptr3);
g_tmp_ptr3 = NULL;

if (g_force_y2_y1 == TRUE && y1_axis_specified == FALSE)
  {
   if (y2_min < y1_min)  y1_min = y2_min;
   if (y2_max > y1_max)  y1_max = y2_max;

   y2_min = y1_min;
   y2_max = y1_max;
  }

g_true_x_min = x_min;
g_true_x_max = x_max;

if (x_axis_specified == FALSE)
  scale_values(FALSE,&x_min,&x_max,&g_x_divisions);

if (y1_axis_specified == FALSE)
  scale_values(FALSE,&y1_min,&y1_max,&g_y1_divisions);

if (y2_axis_specified == FALSE)
  scale_values(FALSE,&y2_min,&y2_max,&g_y2_divisions);

/* 
 *    If we are plotting 2 Y ranges and the scaling is not
 *    the same, check to see they are 'close enough'.    
 *    If they are 'close enough' and neither range has been
 *    'set' yet, then force them to be the same.
 */

if (g_num_y_ranges == 2 && g_force_y2_y1 == FALSE)
  {
   if (y1_axis_specified == FALSE && y2_axis_specified == FALSE)
     {
      a1 = y1_max - y1_min;
      a2 = y2_max - y2_min;
      a3 = (a1 - a2) / (a1 + a2);
      
      if (fabs(a3) <= 0.15)
        {
         a3 = (y1_max - y2_max) / (a1 + a2);
   
         if (fabs(a3) <= 0.15)
           {
            g_force_y2_y1 = TRUE;
   
	    if (y1_min < y2_min)  
	      y2_min = y1_min;
   
	    if (y1_max > y2_max)  
	      y2_max = y1_max;
   
            scale_values(FALSE,&y2_min,&y2_max,&g_y2_divisions);

            y1_min = y2_min;
            y1_max = y2_max;
            g_y1_divisions = g_y2_divisions;
	   }
        }
     }
  }

/*
 *      If we are plotting 2 ranges, adjust the Y1 or Y2 divisions
 *      so they are the same. Then we can label every division for
 *      both the Y1 and Y2 ranges. We can only do this if at least
 *      one of the ranges is auto-range. If both ranges are set by
 *      a set command, we will assume that is what the user wants.
 */

if (g_num_y_ranges == 2 && g_force_y2_y1 == FALSE)
  {
   if (g_y1_divisions != g_y2_divisions)    /* not already the same ?   */
     {
      if (y1_axis_specified == FALSE && y2_axis_specified == FALSE)
        {
         if (g_y1_divisions < g_y2_divisions)	/* move Y1 */
	   {
            rescale_divisions(g_y2_divisions,g_y1_divisions,&y1_min,&y1_max);
            g_y1_divisions = g_y2_divisions;
	   }
         else					/* move Y2 */
	   {
            rescale_divisions(g_y1_divisions,g_y2_divisions,&y2_min,&y2_max);
            g_y2_divisions = g_y1_divisions;
	   }
        }

      /*  we can only move Y2 divisions since Y1 is being set by the user */

      if (y1_axis_specified == TRUE && y2_axis_specified == FALSE)
        {
         rescale_divisions(g_y1_divisions,g_y2_divisions,&y2_min,&y2_max);
         g_y2_divisions = g_y1_divisions;
        }

      /*  we can only move Y1 divisions since Y2 is being set by the user */

      if (y1_axis_specified == FALSE && y2_axis_specified == TRUE)
        {
         rescale_divisions(g_y2_divisions,g_y1_divisions,&y1_min,&y1_max);
         g_y1_divisions = g_y2_divisions;
        }
     }
  }
                                                        
g_graph_x_min  = x_min;
g_graph_x_max  = x_max;
g_graph_y1_min = y1_min;
g_graph_y1_max = y1_max;
g_graph_y2_min = y2_min;
g_graph_y2_max = y2_max;

memory_free(dp);
}
/***********************************************************************/
/* 
   parse a line delimited by spaces, tabs or commas. 
   build an array of individual strings in global 'g_buf'. 

   chars between double quotes are considered part of one field. 
   spaces, tabs, and commas will be included in the field (everything 
   up to the closing double quote).

   function returns number of 'fields' converted.  (>= 1)

   OR

   function returns 0 if line is blank, i.e. line only consists of spaces, 
   tabs, commas, or newline.

   function returns -2 if max number of strings exceeded. (80 strings)

   function returns -3 if the length of any field is too long (256 char each) 
*/ 
parse_line_into_fields(s)
char    *s; 
{ 
int     quote_flag, k, ct; 
char	*p, ch; 

p          = s; 
ct         = 0; 
quote_flag = FALSE;
 
while (1) 
  { 
   ch = *p;
   if (ch == '\0' || ch == 0x0D || ch == '\n')  return(ct); 

   if (ch == ' ' || ch == ',' || ch == '\t') 
      p++; 
   else 
     { 
      if (++ct == 80)  return(-2); 

      k = -1; 

      if (ch == '\"')  /* found double quote */
	{
         quote_flag = TRUE;
	 p++;
	}

      while (1)          /* build the string for each entry */ 
	{ 
	 if (++k == 256)  return(-3);

	 ch = *p++; 
	 g_buf[ct][k] = ch;
 
	 if (ch == '\0' || ch == 0x0D || ch == '\n') 
	   { 
	    g_buf[ct][k] = '\0'; 
	    return(ct); 
	   } 

         if (quote_flag == FALSE)
	   {
	    if (ch == ' ' || ch == ',' || ch == '\t')  break; 
	   }
         else
	   {
	    if (ch == '\"')   /* found closing double quote */
	      {
	       quote_flag = FALSE;
	       break;
	      }
	   }
	} 
 
      g_buf[ct][k] = '\0';
     } 
  } 
} 
/***********************************************************************/ 
/*
 *      if a field begins with a '#', it's the start
 *      of a comment.
 */
reset_field_count(count)
int     count;
{
int     k;

for (k=1; k <= count; k++)
  {
   if (g_buf[k][0] == '#')
     return (k-1);
  }

return (count);
}
/***********************************************************************/ 
