{ FILTER.PAS : Multi-purpose text filter

  Title   : FILTER
  Language: Turbo Pascal v7.0
  Version : 4.0
  Date    : Mar 24,2000
  Author  : J.R. Ferguson, Amsterdam, The Netherlands
  Usage   : Refer procedures Help0, Help1 and Help2
  Download: http://hello.to/ferguson (DOS Filters)
            To compile this sourcefile, you will need some units from the
            Pascal library JRFPAS that can be found at the same site.
  E-mail  : j.r.ferguson@iname.com
}

{$V-}
{$R+}
{$B-}

{$DEFINE ASMLIB}  { if defined: use assembler routine library FILTLIB }
{$UNDEF  CLRHEAP} { if defined: clear heap at end of processing }

program Filter;
uses DefLib, ConLib, ArgLib, StpLib, ChrLib, NumLib;

  { P.M. Constants/types marked ASM are also declared in FILTLIB.ASM }
const
  PROGNAME  = 'FILTER';      { Program name }
  PROGVER   = '4.0';         { Program version }
  PROGTTL   = 'Multi-purpose text filter.';
  CPYRIGHT  = '(c) JR Ferguson, 1994-2000';
  MAXFNM    = 79;            { Max filespec length }
  DFLMSG    = 'CON';         { Default message output destination }
  INPBUFSIZ = 16*1024; {ASM} { Input buffer size in bytes }
  OUTBUFSIZ = 16*1024;       { Output buffer size in bytes }
  MAXTXT    = MaxStp;  {ASM} { Max line text string length }
  MAXSEP    = 2;       {ASM} { Max line separator string length }

  { Error codes and messages: }
  RCOK      = 0;
  ERRARG    = 1;
  ERRFNF    = 2;
  ERRCRE    = 3;
  ERRREA    = 4;
  ERRWRI    = 5;
  ERRMEM    = 6;

  ErrMsg    : array[ERRARG..ERRMEM] of StpTyp =
 ('Command syntax error. Type "'+PROGNAME+' /?" for help.',
  'File not found',
  'File creation error',
  'Read error',
  'Write error',
  'Out of memory'
 );

  EofCode      = $1A; {ASM}
  FixedBlank   = $0F; {ASM}
  HiddenHyphen = $1E; {ASM}
  ActiveHyphen = $1F; {ASM}
  SoftCR       = $8D; {ASM}
  SkipCode     = -1;  {ASM}


type
  InpBufInd = 0..INPBUFSIZ;
  InpBufTyp = array[1..INPBUFSIZ] of byte;
  OutbufTyp = array[1..OUTBUFSIZ] of char;
  SwitchTyp = (neutral,off,on);
  TxtInd    = 0..MAXTXT;
  TxtStp    = string[MAXTXT];
  SepInd    = 0..MAXSEP;
  SepStp    = string[MAXSEP];
  LineTyp   = record sep: SepStp; txt: TxtStp; end; {ASM}
  LinePtr   = ^LineTyp;
  LnumTyp   = Longint;
  LnodPtr   = ^LnodRec;
  LnodRec   = record inh: LinePtr; nxt: LnodPtr end;
  LbufInd   = integer;
  LbufTyp   = record
                cnt       : LbufInd;
                head,tail : LnodPtr;
              end;
  RangePtr  = ^RangeRec;
  RangeRec  = record
               low, high : integer;
               next      : RangePtr;
              end;
  OptTxtRec = record
                InclSw   : SwitchTyp;
                CmpUpp   : boolean;
                CmpCol   : integer;
                CmpAny   : boolean;
                CmpTxt   : StpTyp;
              end;

var
  rc        : integer;
  InpFnm,
  OutFnm    : StpTyp;
  Msg       : Text;
  Inp       : File;
  Out       : Text;
  InpBuf    : InpBufTyp;   {ASM}
  OutBuf    : OutBufTyp;
  InpInd,                  {ASM}
  InpTop    : InpBufInd;   {ASM}
  InpEof,
  InpOpn,
  OutOpn    : boolean;
  LineBuf   : LbufTyp;

  OptAfter  : OptTxtRec; NeedATxt : boolean;
  OptBefore : OptTxtRec; NeedBTxt : boolean;
  OptGet    : OptTxtRec; NeedGTxt : boolean;
  OptHlpB   : boolean; {Batch help}
  OptHlpS   : boolean; {Screen help}
  OptParit  : boolean;
  OptUpper  : SwitchTyp;
  OptOntWS  : boolean;
  OptRZero  : SwitchTyp; {ASM}
  OptExpand : SwitchTyp; ExpandWidth: integer;
  OptFill   : boolean; FillWidth: integer; FillSkip: integer;
  OptNumber : boolean; NumWidth : integer;
  OptTrim   : boolean;
  OptReturn : SwitchTyp;
  OptLfeed  : SwitchTyp;
  OptCopy   : boolean; CopyLen: integer; CopySrc: integer; CopyDst: integer;
  OptMove   : boolean; MoveLen: integer; MoveSrc: integer; MoveDst: integer;
  OptOver   : boolean; OverLen: integer; OverSrc: integer; OverDst: integer;
  OptDel    : boolean; DelLen : integer; DelPos : integer;
  OptIns    : boolean; InsLen : integer; InsPos : integer;
  OptSkip   : boolean; SkipRng: RangePtr;
  OptRev    : boolean; RevLen : integer; RevPos : integer;
  OptXtr    : boolean; XtrRng : RangePtr;
  OptRem    : boolean; RemCnt : integer;
  Dummy     : integer;
  Spaces    : TxtStp;

  CurChar   : integer; {ASM}
  SaveSta   : boolean; {ASM}
  CurLine   : LineTyp; {ASM}
  LineNum   : LnumTyp;
  PreTxtOpt : boolean; { Need line processing before text search }
  PstTxtOpt : boolean; { Need line processing after  text search }
  XlateTbl  : array[byte] of integer; {ASM} { Character translation table }
  InTxtRng  : boolean;
  EndTxtRng : boolean;


{--- General routines ---}


procedure wr(s: StpTyp); begin write  (Msg,s) end;
procedure wl(s: StpTyp); begin writeln(Msg,s) end;


procedure Help0;
begin
wl(PROGNAME+' v'+PROGVER+' - '+PROGTTL+' '+CPYRIGHT);
wl('usage : '+PROGNAME+' [[<]in] [>out] [/option[...]] [...]] [txtopt [...]]');
end;


procedure Help1;
{ not used yet: K, Q, Y }
begin
wl('option: C[n,s,d]   Copy n characters from position s to d.');
wl('        D[n,p]     Delete n characters at position p.');
wl('        E[+/-][n]  Expand tabs ([+]) or replace spacegroups by tabs (-),');
wl('                   where n [8] is tab field length.');
wl('        F[n,m]     Fill nonblank lines with dots to width n [70],');
wl('                   skipping first m [0] lines. Implies /T.');
wl('        H          Send this help text to (redirected) output.');
wl('        ?          Send this help text to screen (page by page)');
wl('        I[n,p]     Insert n spaces at position p.');
wl('        J[+/-]     Add/remove Carriage Return before Line Feed [+].');
wl('        L[+/-]     Add/remove Line Feed after Carriage Return [+].');
wl('        M[n,s,d]   Move n characters from position s to d.');
wl('        N[n]       Number lines, use n [4] digits,');
wl('        O[n,s,d]   Overwrite n chars from position s to d.');
wl('        P          Reset parity bit. Implied by /W.');
wl('        R[n]       Remove n trailing lines after processing /S and /X.');
wl('        S[n,m]     Skip m lines starting at line n.');
wl('        T          Trim trailing blanks. Implied by /F.');
wl('        U[+/-]     Convert to upper/lower case [+].');
wl('        V[n,s]     Reverse n [all] characters starting at position s.');
wl('        W          Wordstar document ==> ASCII textfile. Implies /P.');
wl('        X[n,m]     Extract m lines starting at line n.');
end;

procedure Help2;
begin
wl('        Z[+]       Remove NULLs. Z+: also ANSI screen control sequences.');
wl('');
wl('txtopt: /A[+/-][I][p][*] text   Include lines after  specified text only.');
wl('        /B[+/-][I][p][*] text   Include lines before specified text only.');
wl('        /G[+/-][I][p][*] text   Include lines with the specified text only.');
wl('');
wl('        + : Include matching line (default for /G).');
wl('        - : Do not include the matching line (default for /A and /B).');
wl('        I : Ignore upper/lower case.');
wl('        p : Search for text starting at column p. Default p=1.');
wl('        * : Text may be found at any column at or after p.');
wl('');
wl('        These suboptions may appear in any order.');
wl('');
wl('        The text to search for is a separate command argument, immediately');
wl('        following the /A, /B or /G argument. If it contains any spaces or');
wl('        tabs, it must be surrounded by single or double quotes.');
wl('');
wl('The three filter action groups below are processed in the specified order:');
wl('  1. option  E,J,L,P,R,S,T,U,W,X and Z');
wl('  2. txtopt  A,B,G');
wl('  3. option  C,D,F,I,M,N,O,V');
end;


function WaitFor(Prompt: StpTyp): boolean;
var c: char; ok: boolean;
begin
  wr(Prompt+'? [Yn]: ');
  repeat c:= UppKey until c in ['Y','N',chr(AsciiCR)];
  ok:= c <> 'N';
  if ok then wl('Y') else wl('N');
  WaitFor:= ok;
end;


procedure Help;
begin
  Help0;
  Help1;
  if OptHlpB or WaitFor('More') then Help2;
end;


{$F+} function HeapFunc(Size: word): integer; {$F-}
{ Make new() return a nil pointer when the heap is full }
begin HeapFunc:= 1 end;


{$ifdef ASMLIB}
{$L FILTLIB}
{$endif}


{--- Integer range handling routines ---}


procedure InitRange(var r: RangePtr);
begin r:= nil end;


{$ifdef CLRHEAP}
procedure DisposeRange(r: RangePtr);
begin if r<>nil then begin
  DisposeRange(r^.next);
  Dispose(r);
end end;
{$endif}


procedure InsertRange(var r: RangePtr; l,h: integer);
var r1: RangePtr;
begin
  new(r1);
  if r1=nil then rc:= ERRMEM else begin
    with r1^ do begin low:= l; high:= h; next:= r end;
    r:= r1;
  end;
end;


function InRange(r: RangePtr; i: integer): boolean;
begin
  if r=nil then InRange:= false
  else if (i >= r^.low) and (i <= r^.high) then InRange:= true
  else InRange:= InRange(r^.next,i);
end;


{--- Command line parsing routines ---}


procedure ReadSwitch(var arg: StpTyp; var sw: SwitchTyp; dfl: SwitchTyp);
begin
  case StpcRet(arg,1) of
    '+' : begin StpDel(arg,1,1); sw:= on; end;
    '-' : begin StpDel(arg,1,1); sw:= off end;
    else  sw:= dfl;
  end;
end;


function ReadUns(var arg: StpTyp): integer;
var
  n: integer;
  c: char;
begin
  n:= 0;
  c:= StpcRet(arg,1);
  while IsDigit(c) do begin
    n:= 10*n + (ord(c) - ord('0'));
    StpDel(arg,1,1);
    c:= StpcRet(arg,1);
  end;
  ReadUns:= n;
end;


procedure ReadNumOpt(var arg      : StpTyp;     { Command arg to parse }
                     var option   : boolean;    { Value for option     }
                         MaxNr    : integer;    { Max nr of parameters }
                     var p1       : integer;    { Value     for parm 1 }
                         m1       : integer;    { Max value for parm 1 }
                     var p2       : integer;    { Value     for parm 2 }
                         m2       : integer;    { Max value for parm 2 }
                     var p3       : integer;    { Value     for parm 3 }
                         m3       : integer);   { Max value for parm 3 }
begin
  option:= true;
  if MaxNr>3 then MaxNr:= 3;
  if (MaxNr >= 1) and IsDigit(StpcRet(arg,1)) then begin
    p1:= ReadUns(arg);
    if p1 > m1 then rc:= ERRARG
  end;
  if (rc=RCOK) and (MaxNr >= 2) and (StpcRet(arg,1) = ',') then begin
    StpDel(arg,1,1);
    if IsDigit(StpcRet(arg,1)) then begin
      p2:= ReadUns(arg);
      if p2 > m2 then rc:= ERRARG
    end;
  end;
  if (rc=RCOK) and (MaxNr >= 3) and (StpcRet(arg,1) = ',') then begin
    StpDel(arg,1,1);
    if IsDigit(StpcRet(arg,1)) then begin
      p3:= ReadUns(arg);
      if p3 > m3 then rc:= ERRARG;
   end;
  end;
end;


procedure ReadTxtOpt(var arg: StpTyp; var TxtOpt: OptTxtRec; DflIncl: SwitchTyp);
var c: char; xInclSw, xCmpUpp, xCmpCol, xCmpAny, xCmpTxt: boolean;
begin with TxtOpt do begin
  InclSw:= DflIncl; xInclSw:= false;
  CmpUpp:= false;  xCmpUpp:= false;
  CmpCol:= 1;      xCmpCol:= false;
  CmpAny:= false;  xCmpAny:= false;
  CmpTxt:= '';     xCmpTxt:= false;
  c:= StpcRet(arg,1);
  while (rc = RCOK) and (c <> #0) do begin
    case c of
      '0'..'9': if xCmpCol then rc:= ERRARG
                else begin CmpCol:= ReadUns(arg); xCmpCol:= true; end;
      '-'     : if xInclSw then rc:= ERRARG
                else begin InclSw:= off; xInclSw:= true; StpDel(arg,1,1); end;
      '+'     : if xInclSw then rc:= ERRARG
                else begin InclSw:= on ; xInclSw:= true; StpDel(arg,1,1); end;
      'I'     : if xCmpUpp then rc:= ERRARG
                else begin CmpUpp:= true ; xCmpUpp:= true; StpDel(arg,1,1); end;
      '*'     : if xCmpAny then rc:= ERRARG
                else begin CmpAny:= true ; xCmpAny:= true; StpDel(arg,1,1); end;
      else      rc:= ERRARG;
    end;
    c:= StpcRet(arg,1);
  end;
end; end;


procedure ReadTxtParm(var arg: StpTyp; var TxtOpt: OptTxtRec);
begin
  with TxtOpt do begin
    CmpTxt:= arg;
    if CmpUpp then StpUpp(CmpTxt);
  end;
  arg:= '';
end;


procedure ReadOpt(var arg: StpTyp);
var nextopt: boolean; SkipBeg, SkipCnt, XtrBeg, XtrCnt: integer;
begin
  StpDel(arg,1,1);
  repeat
    if StpEmpty(arg) or (StpcRet(arg,1) = '/') then rc:= ERRARG
    else begin
      nextopt:= false;
      while (rc=RCOK) and not nextopt and not StpEmpty(arg) do begin
        SkipBeg:= 1; SkipCnt:= 1;
        XtrBeg := 1; XtrCnt := 1;
        case StpcGet(arg) of
          'A' : begin ReadTxtOpt(arg,OptAfter ,Off); NeedATxt:= true; end;
          'B' : begin ReadTxtOpt(arg,OptBefore,Off); NeedBTxt:= true; end;
          'C' : ReadNumOpt(arg,OptCopy,3,
                  CopyLen,MaxStp, CopySrc,MaxStp, CopyDst,MaxStp);
          'D' : ReadNumOpt(arg,OptDel,2,
                  DelLen,MaxStp, DelPos,MaxStp, Dummy,0);
          'E' : begin
                  ReadSwitch(arg,OptExpand,on);
                  if IsDigit(StpcRet(arg,1)) then begin
                    ExpandWidth:= ReadUns(arg);
                    if ExpandWidth = 0 then OptExpand:= neutral;
                    if ExpandWidth > MaxStp then rc:= ERRARG
                  end;
                end;
          'F' : ReadNumOpt(arg,OptFill,2,
                  FillWidth,MaxStp, FillSkip,MaxInt, Dummy,0);
          'G' : begin ReadTxtOpt(arg,OptGet   ,On ); NeedGTxt:= true; end;
          'H' : begin OptHlpB:= true; arg:= ''; end;
          'I' : ReadNumOpt(arg,OptIns,2,
                  InsLen,MaxStp, InsPos,MaxStp, Dummy,0);
          'J' : ReadSwitch(arg,OptReturn,on);
          'L' : ReadSwitch(arg,OptLfeed,on);
          'M' : ReadNumOpt(arg,OptMove,3,
                  MoveLen,MaxStp, MoveSrc,MaxStp, MoveDst,MaxStp);
          'N' : ReadNumOpt(arg,OptNumber,1,
                  NumWidth,MaxInt, Dummy,0, Dummy,0);
          'O' : ReadNumOpt(arg,OptOver,3,
                  OverLen,MaxStp, OverSrc,MaxStp, OverDst,MaxStp);
          'P' : OptParit := true;
          'R' : begin
                  ReadNumOpt(arg,OptRem,1,
                    RemCnt,MaxInt, Dummy,0, Dummy,0);
                  if rc=RCOK then
                    if RemCnt = 0 then OptRem:= false;
                end;
          'S' : begin
                  ReadNumOpt(arg,OptSkip,2,
                    SkipBeg,MaxInt, SkipCnt,MaxInt, Dummy,0);
                  if rc=RCOK then
                    InsertRange(SkipRng,SkipBeg,SkipBeg+SkipCnt-1);
                end;
          'T' : OptTrim:= true;
          'U' : ReadSwitch(arg,OptUpper,on);
          'V' : begin
                  ReadNumOpt(arg,OptRev,2,
                    RevLen,MaxStp, RevPos,MaxStp, Dummy,0);
                  if (RevLen=0) or (RevLen=1)
                  then OptRev:= false
                  else if RevPos=0 then RevPos:= 1;
                end;
          'W' : OptOntWS := true;
          'X' : begin
                  ReadNumOpt(arg,OptXtr,2,
                    XtrBeg,MaxInt, XtrCnt,MaxInt, Dummy,0);
                  if rc=RCOK then
                    InsertRange(XtrRng,XtrBeg,XtrBeg+XtrCnt-1);
                end;
          'Z' : ReadSwitch(arg,OptRZero,off);
          '?' : begin OptHlpS:= true; arg:= ''; end;
          '/' : nextopt:= true;
          else rc:= ERRARG;
        end;
      end;
    end;
  until (rc <> RCOK) or not nextopt;
end;


procedure ReadArgs;
var i,f : ArgInd;
    arg : StpTyp;
begin
  GetArgs; i:= 0; f:= 0; NeedATxt:= false; NeedBTxt:= false;
  while (rc = RCOK) and (i < ArgC) and not OptHlpS do begin
    Inc(i); StpCpy(arg,ArgV[i]);
    if      NeedATxt then begin
      ReadTxtParm(arg,OptAfter) ; NeedATxt:= false;
    end
    else if NeedBTxt then begin
      ReadTxtParm(arg,OptBefore); NeedBTxt:= false;
    end
    else if NeedGTxt then begin
      ReadTxtParm(arg,OptGet)   ; NeedGTxt:= false;
    end
    else begin
      StpUpp(arg);
      case StpcRet(arg,1) of
        '/': ReadOpt(arg);
        '<': StpSub(InpFnm,arg,2,MAXFNM); { emulate DOS redirection for IDE }
        '>': StpSub(OutFnm,arg,2,MAXFNM); { emulate DOS redirection for IDE }
        else case f of
               0  : begin StpNCpy(InpFnm,arg,MAXFNM); Inc(f); end;
               else rc:= ERRARG;
             end;
      end;
    end;
  end;
  if NeedATxt or NeedBTxt or NeedGTxt then rc:= ERRARG;
end;


{--- Line buffer (FIFO queue) handling routines  ---}


procedure LbufInit;
begin with LineBuf do begin head:= nil; tail:= nil; cnt:= 0 end end;


function LbufPut: boolean;
var np: LnodPtr;
begin with LineBuf, CurLine do begin
  new(np);
  if np=nil then LbufPut:= false else begin
    np^.nxt:= nil; GetMem(np^.inh, SizeOf(SepStp)+succ(Length(txt)));
    if np^.inh=nil then LbufPut:= false else begin
      Move(CurLine, np^.inh^, SizeOf(SepStp)+succ(Length(txt)));
      if cnt=0 then begin head:= np; tail:= np end
      else begin tail^.nxt:= np; tail:= np end;
      inc(cnt); LbufPut:= true;
    end;
  end;
end end;


function LbufGet: boolean;
var np: LnodPtr;
begin with LineBuf do begin
  if cnt < 1 then LbufGet:= false
  else begin
    Move(head^.inh^, CurLine, SizeOf(SepStp)+succ(Length(head^.inh^.txt)));
    FreeMem(head^.inh, SizeOf(SepStp)+succ(Length(head^.inh^.txt)));
    np:= head^.nxt; dispose(head); head:= np;
    dec(cnt); LbufGet:= true;
  end;
end end;


{$ifdef CLRHEAP}
procedure LbufDone;
begin while LbufGet do {nothing}; end;
{$endif}


{--- I/O routines ---}


procedure OpenMsg;
begin
  if OptHlpB then Assign(Msg,OutFnm) else Assign(Msg,DFLMSG);
  rewrite(Msg);
end;


procedure CloseMsg;
begin Close(Msg) end;


procedure OpenInp;
begin if not InpOpn then begin
  Assign(Inp,InpFnm);
  {$I-} reset(Inp,1); {$I+}
  if IOresult <> 0 then rc:= ERRFNF else begin
    InpOpn:= true; SaveSta:= false; InpEof:= false;
    InpInd:= 0; InpTop:= 0;
  end;
end; end;


procedure CloseInp;
begin if InpOpn then begin
  {$I-} Close(Inp); {$I+}
  if IOresult = 0 then InpOpn:= false;
end; end;


procedure OpenOut;
begin if not OutOpn then begin
  Assign(Out,OutFnm);
  SetTextBuf(Out,OutBuf);
  {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then rc:= ERRCRE else OutOpn:= true;
end; end;


procedure CloseOut;
begin if OutOpn then begin
  {$I-} Close(Out); {$I+}
  if IOresult = 0 then OutOpn:= false else begin
    if rc = RCOK then rc:= ERRWRI;
  end;
end; end;


function GetBuf: boolean;        {ASM}
begin
  if InpEof then GetBuf:= false
  else begin
    {$I-} BlockRead(Inp, InpBuf, SizeOf(InpBuf), InpTop); {$I+}
    if IOresult <> 0 then begin rc:= ERRREA; GetBuf:= false; end
    else begin
      InpInd:= 0;
      if InpTop = 0 then begin InpEof:= true; GetBuf:= false; end
      else GetBuf:= true;
    end;
  end;
end;


procedure InitXlateTbl;
var b,i: byte;
begin
  for b:= 0 to 255 do begin
    if (    ((OptRZero<>neutral) and (b = $00         ))
         or ( OptOntWS           and (b = HiddenHyphen))
       )
    then XlateTbl[b]:= SkipCode
    else begin
      i:= b;
      if OptOntWS then case b of
        ActiveHyphen : i:= ord('-');
        FixedBlank   : i:= ord(' ');
        else           i:= b and $7F;
      end
      else if OptParit then i:= b and $7F;
      case OptUpper of
        neutral : XlateTbl[b]:= i;
        on      : XlateTbl[b]:= ord(ToUpper(chr(i)));
        off     : XlateTbl[b]:= ord(ToLower(chr(i)));
      end;
    end;
  end;
end;


{$ifdef ASMLIB}

function GetLine: boolean;
external;

{$ELSE}

function GetLine: boolean;
var EndLine: boolean; InAnsi: 0..2;

  function GetChar: boolean;
  begin
    if      SaveSta then begin SaveSta:= false; GetChar:= true; end
    else if InpEof  then GetChar:= false
    else begin
      if (InpInd >= InpTop) and (not GetBuf) then GetChar:= false
      else begin
	Inc(InpInd); CurChar:= XlateTbl[InpBuf[InpInd]];
	if InpBuf[InpInd] = EofCode then begin
          InpEof:= true; GetChar:= false;
	end
	else GetChar:= true;
      end;
    end;
  end;

begin {GetLine}
  if GetChar then with CurLine do begin
    txt:= ''; sep:= ''; EndLine:= false; InAnsi:= 0;
    repeat
      if OptRZero = on then case InAnsi of
        0: if CurChar = AsciiESC then begin
             Inc(InAnsi);
             CurChar:= SkipCode;
           end;
        1: if CurChar = ord('[') then begin
             Inc(InAnsi);
             CurChar:= SkipCode;
           end
           else begin
             InAnsi:= 0;
             txt:= txt + chr(AsciiESC);
           end;
        2: begin
             if IsAlpha(chr(CurChar)) then InAnsi:= 0;
             CurChar:= SkipCode;
           end;
      end;
      case CurChar of
        SkipCode: { skip } ;
        SoftCR,
        AsciiCR : begin
                    EndLine:= true; InAnsi:= 0;
                    sep:= chr(CurChar);
                    if GetChar then begin
                      if CurChar = AsciiLF then sep:= sep + chr(CurChar)
                                           else SaveSta:= true;
                    end;
                  end;
        AsciiLF,
        AsciiFF : begin
                    EndLine:= true; InAnsi:= 0;
                    sep:= chr(CurChar);
                  end;
        else      txt:= txt + chr(CurChar);
      end;
    until EndLine or not GetChar;
    GetLine:= true;
  end
  else GetLine:= false;
end;

{$endif}


{--- Filter processing routines ---}


function HasTxt(TxtOpt: OptTxtRec): boolean;
var s: StpTyp;
begin with TxtOpt, CurLine do begin
  StpSub(s,txt,CmpCol,MaxStp); if CmpUpp then StpUpp(s);
  if CmpAny then HasTxt:= StpPos(s,CmpTxt) >= 1
            else HasTxt:= StpPos(s,CmpTxt)  = 1;
end; end;

procedure FillLine;
var i: TxtInd;
begin with CurLine do begin
  i:= StpLen(txt);
  if i>0 then begin
    if i<FillWidth then begin StpcCat(txt,' '); inc(i); end;
    while i<FillWidth do begin
      if Odd(FillWidth-i) then StpcCat(txt,' ') else StpcCat(txt,'.');
      inc(i);
    end;
  end;
end end;

procedure CopyChars(s,d,n: StpInd);
var t: LineTyp;
begin with CurLine do begin
  StpSub(t.txt,txt,s,n);
  StpFill(t.txt,' ',n);
  StpFill(txt,' ',IMax(d-1,0));
  StpIns(txt,t.txt,d);
end end;


procedure MoveChars(s,d,n: StpInd);
var t: LineTyp;
begin with CurLine do begin
  StpSub(t.txt,txt,s,n);
  StpFill(t.txt,' ',n);
  StpDel(txt,s,n);
  if d>s+n then dec(d,n);
  StpFill(txt,' ',IMax(d-1,0));
  StpIns(txt,t.txt,d);
end end;


procedure Overwrite(s,d,n: StpInd);
var t: LineTyp;
begin with CurLine do begin
  StpSub(t.txt,txt,s,n);
  StpFill(t.txt,' ',n);
  StpDel(txt,d,n);
  StpFill(txt,' ',IMax(d-1,0));
  StpIns(txt,t.txt,d);
end end;

procedure Reverse(n: integer; s: StpInd);
var t: LineTyp;
begin with CurLine do begin
  if n=-1 then n:= StpLen(txt)+1-s;
  if n>1 then begin
    StpSub(t.txt,txt,s,n);
    StpFill(t.txt,' ',n);
    StpRev(t.txt);
    StpDel(txt,s,n);
    StpIns(txt,t.txt,s);
  end;
end end;


procedure PreProcessTxt;
begin with CurLine do begin
  case OptExpand of
    neutral : {do nothing};
    on      : StpDetab(txt,txt,ExpandWidth);
    off     : StpEntab(txt,txt,ExpandWidth);
  end;
  if OptTrim then StpRTS(txt);
end end;


procedure PostProcessTxt;
begin with CurLine do begin
  if OptFill then if LineNum>FillSkip then FillLine;
  if OptCopy then CopyChars(CopySrc,CopyDst,CopyLen);
  if OptMove then MoveChars(MoveSrc,MoveDst,MoveLen);
  if OptOver then OverWrite(OverSrc,OverDst,OverLen);
  if OptDel  then StpDel(txt,DelPos,DelLen);
  if OptIns  then StpNIns(txt,Spaces,InsPos,InsLen);
  if OptRev  then Reverse(RevLen,RevPos);
end end;


procedure PutLine;
var k : SepInd;
begin with CurLine do begin
  if PreTxtOpt then PreProcessTxt;
  if not InTxtRng then case OptAfter.InclSw of
    neutral : InTxtRng:= true;
    on      : if HasTxt(OptAfter) then InTxtRng:= true;
    off     : if HasTxt(OptAfter) then OptAfter.InclSw:= neutral;
  end;
  if InTxtRng then case OptBefore.InclSw of
    neutral : { do nothing };
    on      : if HasTxt(OptBefore) then EndTxtRng:= true;
    off     : if HasTxt(OptBefore) then begin
                InTxtRng:= false; EndTxtRng:= true;
              end;
  end;
  if InTxtRng and
     (( OptGet.InclSw=neutral                        )  or
      ((OptGet.InclSw=On     ) and     HasTxt(OptGet))  or
      ((OptGet.InclSw=Off    ) and not HasTxt(OptGet)))
  then begin
    if PstTxtOpt then PostProcessTxt;
    case OptReturn of
      neutral : { do nothing };
      on      : if sep = chr(AsciiLF) then sep:= chr(AsciiCR) + chr(AsciiLF);
      off     : if sep = chr(AsciiCR) + chr(AsciiLF) then sep:= chr(AsciiLF);
    end;
    case OptLFeed of
      neutral : { do nothing };
      on      : if sep = chr(AsciiCR) then sep:= chr(AsciiCR) + chr(AsciiLF);
      off     : if sep = chr(AsciiCR) + chr(AsciiLF) then sep:= chr(AsciiCR);
    end;
    if OptNumber then begin
      if OptRem then write(Out,LineNum-RemCnt:NumWidth,': ')
                else write(Out,LineNum:NumWidth,': ');
    end;
    write(Out,txt,sep);
  end;
end end;


procedure BufPutLine;
begin
  if not LbufPut then rc:= ERRMEM
  else begin
    if LineBuf.cnt > RemCnt then if LbufGet then PutLine;
  end;
end;


{--- Main line ---}


procedure MainInit;
begin
  rc:= RCOK;
  HeapError:= @HeapFunc; {install function to catch out-of-heap condition}
  { set defaults }
  OptAfter.InclSw  := neutral;
  OptBefore.InclSw := neutral;
  OptGet.InclSw    := neutral;
  OptHlpB   := false;
  OptHlpS   := false;
  OptParit  := false;
  OptUpper  := neutral;
  OptOntWS  := false;
  OptRZero  := neutral;
  OptExpand := neutral; ExpandWidth:= 8;
  OptFill   := false; FillWidth := 70;  FillSkip := 0;
  OptNumber := false; NumWidth  := 4;
  OptTrim   := false;
  OptReturn := neutral;
  OptLfeed  := neutral;
  OptCopy   := false; Copylen := 1; CopySrc := 1; CopyDst := 1;
  OptMove   := false; MoveLen := 1; MoveSrc := 1; MoveDst := 1;
  OptOver   := false; OverLen := 1; OverSrc := 1; OverDst := 1;
  OptDel    := false; DelLen  := 1; DelPos  := 1;
  OptIns    := false; InsLen  := 1; InsPos  := 1;
  OptSkip   := false; InitRange(SkipRng);
  OptRev    := false; RevLen  :=-1; RevPos  := 1;
  OptXtr    := false; InitRange(XtrRng);
  OptRem    := false; RemCnt:= 1;

  FillChar(Spaces,MAXTXT+1,' '); Spaces[0]:= chr(MAXTXT);
  LineNum := 0;
  LbufInit;
  StpCreate(InpFnm); InpOpn:= false;
  StpCreate(OutFnm); OutOpn:= false;
  ReadArgs;
  if not (OptHlpS or OptHlpB) then begin
    if OptOntWS then OptParit:= true;
    if OptFill  then OptTrim := true;
    InTxtRng := OptAfter.InclSw = neutral;
    EndTxtRng:= false;
    PreTxtOpt:= (OptExpand<>neutral) or OptTrim;
    PstTxtOpt:= OptFill or OptCopy or OptMove or OptOver or OptDel or OptIns or OptRev;
    if (rc=RCOK) then begin InitXlateTbl; OpenInp; end;
    if (rc=RCOK) then OpenOut;
  end;
end;


procedure MainTerm;
begin
  CloseInp; CloseOut;
{$ifdef CLRHEAP}
  DisposeRange(SkipRng);
  DisposeRange(XtrRng);
  LbufDone;
{$endif}
  if rc <> RCOK then begin
    OptHlpB:= false;
    OpenMsg;
    writeln(Msg,PROGNAME+': ',ErrMsg[rc]);
    CloseMsg;
  end;
end;

procedure MainProcess;
begin
  while (rc=RCOK) and (not EndTxtRng) and GetLine do begin
    Inc(LineNum);
    if not (   ( OptSkip and     InRange(SkipRng,LineNum) )
            or ( OptXtr  and not InRange(XtrRng ,LineNum) )
           )
    then begin
      if OptRem then BufPutLine else PutLine;
    end;
  end;
end;


begin { Main program }
  MainInit;
  if (rc=RCOK) then begin
    if OptHlpS or OptHlpB then begin
      OpenMsg;
      Help;
      CloseMsg;
    end
    else MainProcess;
  end;
  MainTerm;
end.
