{ DB3SCN.PAS : dBase III Screen Program or Format Generator

  Title   : DB3SCN
  Language: Turbo Pascal v4.0 or higher
  Version : 1.4
  Date    : Feb 16, 2000
  Author  : J R Ferguson
  Download: http://hello.to/ferguson
  E-mail  : j.r.ferguson@iname.com
  Usage   : Refer procedure Help
  Remarks : To compile this program you will need some units from the
            author's JRFPAS Pascal routine library that can be found
            on the author's internet website.

  This program and its source code may be copied and used freely without
  charge, but only for non-commercial purposes. The author can not be held
  responsible for any damage or loss of data that may be caused by the use
  of this software.


  Like SED for dBase II, with the following differences:
  - ">var!pict" issues "@r,c SAY var PICTURE pict" (not USING).
  - Embedded commands are supported only on separate lines, starting with
    "[". The ending "]" is optional.
  - Picture symbol definitions must precede their use. Symbol letters are
    not case-sensitive.
  - Command characters "*", "[", "!" etc. may be redefined by any
    non-conflicting punctuation character.
  - Command lines, comment lines, picture symbol definitions and command
    character redefinitions do not add to the current line count.
  - Video attribute control is not supported.
}

{$V-}
{$R+}

{$UNDEF OUTBUFHEAP}   { UNDEF to work around a BP 7.0 bug resulting in
                        erroneous file output }

program DB3SCN;


uses DefLib, ArgLib, StpLib, StfLib, ChrLib, TimLib, CvtLib;


const
{ Default values: }
  DFLINPEXT = 'SCN';    { Input file extension }
  DFLCMDEXT = 'PRG';    { Command File extension }
  DFLFMTEXT = 'FMT';    { Format File extension }
  DFLCOMCMT = '*';      { Comment }
  DFLCOMECB = '[';      { Embedded command begin }
  DFLCOMECE = ']';      { Embedded command end }
  DFLCOMPIC = '!';      { Picture symbol }
  DFLCOMGET = '<';      { Get command }
  DFLCOMPUT = '>';      { Put command }
  DFLCOMCHG = '/';      { Command character change }

  { Option defaults: }
  DFLFMT    = false;

  MAXOUTLEN = 60;
  MAXFSPEC  = 79;
  IOBUFSIZ  = 4096;

  { Warning codes and messages: }
  WRNQUO    = 0;
  WRNUNK    = 1;
  WRNDEF    = 2;
  WRNGET    = 3;
  WRNPUT    = 4;
  WRNCHG    = 5;

  WRNMSG    : array[WRNQUO..WRNCHG] of StpTyp =
  ('No ending " in picture string : ',
   'Unkwown symbol : ',
   'Symbol already defined : ',
   'No name for GET command : ',
   'No name for SAY command : ',
   'Conflicting command redefinition : '
  );

  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRINP    = 2;
  ERROUT    = 3;

  ERRMSG    : array[ERRINP..ERROUT] of StpTyp =
 ('File not found : ',
  'Can''t open output : '
 );


type
  IOBufTyp  = array[1..IOBUFSIZ] of char;
  IOBufPtr  = ^IOBufTyp;
  ComTyp    = (ComCmt, ComEcb, ComEce, ComPic, ComGet, ComPut, ComChg);
  ComInd    = ComCmt..ComChg;

var
  InpFspec,
  OutFspec  : StpTyp;
  InpFvar,
  OutFvar   : Text;
  InpBuf    : IOBufPtr;
{$IFDEF OUTBUFHEAP}
  OutBuf    : IOBufPtr;
{$ELSE}
  OutBuf    : IOBufTyp;
{$ENDIF}
  InpOpen,
  OutOpen   : boolean;
  ComChr    : array[ComInd] of char;
  PictSym   : array['@'..'Z'] of StpTyp;
  ErrCod    : integer;
  OptFmt    : boolean;

  InpRow    : integer;
  OutRow    : integer;
  OutCol    : integer;
  InpLine   : StpTyp;
  OutLine   : StpTyp;
  Name      : StpTyp;
  Pict      : StpTyp;
  PictSkip  : StpInd;


procedure Help;
  procedure ws(s: StpTyp); begin write  (s); end;
  procedure wl(s: StpTyp); begin writeln(s); end;
begin
wl('DB3SCN v1.4: dBase III Screen Program or Format Generator');
ws('usage  : DB3SCN inpfile[.ext] [outfile][.ext] [/F]');
wl('');
wl('inpfile required       , default ext=SCN');
wl('outfile default=inpfile, default ext=PRG (FMT when /F option is given)');
wl('');
wl('Recognized internal commands starting in column 1:');
wl('  * any text  : Comment line.');
wl('  [ command ] : embedded command, ending "]" is optional.');
wl('  !x pict     : define symbol x ('''',''a''..''z'') to represent pict.');
wl('  /cd         : change command char c to d (punctuation only)');
wl('');
wl('Recognized internal commands starting in any column:');
wl('  <var        : generates "@r,c GET var"');
wl('  >var        : generates "@r,c SAY var"');
wl('  <var!pict   : generates "@r,c GET var PICTURE pict"');
wl('  >var!pict   : generates "@r,c SAY var PICTURE pict"');
wl('                pict may be a predefined symbol');
end;


{--- general routines ---}


function TwoDig(i: integer): StpTyp;
var tmp: StpTyp;
begin ItoABL(i,tmp,10,2); TwoDig:= tmp end;


procedure Warning(i: integer; txt: StpTyp);
begin
  writeln(chr(AsciiBEL),'>> Line ',InpRow:2,' - ',WRNMSG[i],txt);
end;


procedure Detab(var str: StpTyp);
var tmp: StpTyp;
begin
  tmp:= str;
  StpDetab(str,tmp,8);
end;


{--- command line parsing ---}


procedure ConstructFilenames;
  var inpfile, inpext, outfile, outext: StpTyp;
begin
  inpfile:= StfBefore(InpFspec,'.'); inpext:= StfAfter(InpFspec,'.');
  outfile:= StfBefore(OutFspec,'.'); outext:= StfAfter(OutFspec,'.');
  if StpEmpty(inpfile) then ErrCod:= ERRARG
  else begin
    if StpEmpty(inpext)  then StpCpy(inpext,DFLINPEXT);
    if StpEmpty(outfile) then StpCpy(outfile,inpfile);
    if StpEmpty(outext)  then begin
      if OptFmt then StpCpy(outext,DFLFMTEXT) else StpCpy(outext,DFLCMDEXT);
    end;
    InpFspec:= inpfile + '.' + inpext;
    OutFspec:= outfile + '.' + outext;
  end;
end;


procedure ReadOpt(arg: StpTyp);
begin
  StpDel(arg,1,1);
  while (ErrCod=ERROK) and not StpEmpty(arg) do case ToUpper(StpcGet(arg)) of
    'F' : OptFmt := not DFLFMT;
    else ErrCod:= ERRARG;
  end;
end;


procedure ReadArgs;
var i,f : ArgInd;
    arg : StpTyp;
begin
  StpCreate(InpFspec); StpCreate(OutFspec);
  GetArgs;
  i:= 0; f:= 0;
  while (i < ArgC) and (ErrCod = ERROK) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    case StpcRet(arg,1) of
      '/' : ReadOpt(arg);
      '<',
      '>' : { nothing };
      else  case f of
              0  : begin f:= 1; StpNCpy(InpFspec,arg,MAXFSPEC); end;
              1  : begin f:= 2; StpNCpy(OutFspec,arg,MAXFSPEC); end;
              else ErrCod:= ERRARG;
            end;
    end;
  end;
  if ErrCod=ERROK then ConstructFilenames;
end;


{--- I/O routines ---}


procedure OpenInp;
begin
  StpNCpy(InpFspec,InpFspec,MAXFSPEC); Assign(InpFvar,InpFspec);
  new(InpBuf); SetTextBuf(InpFvar,InpBuf^); {$I-} reset(InpFvar); {$I+}
  if IOresult <> 0 then ErrCod:= ERRINP else InpOpen:= true;
end;


procedure OpenOut;
begin
  StpNCpy(OutFspec,OutFspec,MAXFSPEC); Assign(OutFvar,OutFspec);
{$IFDEF OUTBUFHEAP}
  new(OutBuf); SetTextBuf(OutFvar,OutBuf^);
{$ELSE}
  SetTextBuf(OutFvar,OutBuf);
{$ENDIF}
  {$I-} rewrite(OutFvar); {$I+}
  if IOresult <> 0 then ErrCod:= ERROUT else OutOpen:= true;
end;


{--- Process File ---}


procedure WriteOutLine;
var tmp: StpTyp; len: StpInd;
begin
  while not StpEmpty(Outline) do begin
    while IsSpace(StpcRet(OutLine,1)) do begin
      StpDel(OutLine,1,1); Inc(OutCol);
    end;
    StpNCpy(tmp,OutLine,MAXOUTLEN); len:= StpLen(tmp); StpRTS(tmp);
    if not StpEmpty(tmp) then
      writeln(OutFvar,'@ ',OutRow:2,',',OutCol:2,' SAY "',tmp,'"');
    StpDel(OutLine,1,len); Inc(OutCol,len);
  end;
end;


procedure WriteOutTitle;
  var datum: TimDateRec; tijd: TimTimeRec; fnm: StpTyp;
begin with datum,tijd do begin
  TimGetTime(tijd); TimGetDate(datum);
  StpCpy(fnm,OutFspec);
  StpDel(fnm,1,StpcPos(OutFspec,':'));
  StpDel(fnm,1,StpcRPos(OutFspec,'\'));
  writeln(OutFvar,'* ',fnm,' [DB3SCN ',
                  year,'-',TwoDig(month),'-',TwoDig(day),' ',
                  TwoDig(hours),':',TwoDig(minutes),']');
end end;


function NxtChr: char;
begin
  StpDel(InpLine,1,1); NxtChr:= StpcRet(InpLine,1);
end;


procedure ParsePictString;
{ Extract possibly quoted picture string Pict from InpLine }
var c: char;
begin
  StpCreate(Pict); StpcCat(Pict,'"');
  c:= StpcRet(InpLine,1);
  if c='"' then begin  { parse quoted string }
    c:= NxtChr;
    while (c<>'"') and (c<>chr(0)) do begin StpcCat(Pict,c); c:= NxtChr end;
    if c='"' then StpDel(InpLine,1,1)
    else Warning(WRNQUO,Pict);
  end
  else begin           { parse up to first white space }
    while (c<>chr(0)) and not IsSpace(c) do begin
      StpcCat(Pict,c); c:= NxtChr;
    end;
  end;
  StpcCat(Pict,'"');
end;


function ParseName: boolean;
var c: char;
begin
  c:= StpcRet(InpLine,2);
  if IsSpace(c) or (c=chr(0)) or (c=ComChr[ComPic]) then ParseName:= false
  else begin
    StpDel(InpLine,1,1);
    StpCreate(Name);
    repeat StpcCat(Name,c); c:= NxtChr;
    until IsSpace(c) or (c=ComChr[ComPic]) or (c=chr(0));
    ParseName:= true;
  end;
end;


function ParsePict: boolean;
var c: char;
begin
  c:= StpcRet(InpLine,1);
  if c= ComChr[ComPic] then begin
    c:= ToUpper(StpcRet(InpLine,2));
    if c='"' then begin
      StpDel(InpLine,1,1); ParsePictString; PictSkip:= 1 + StpLen(Pict);
      ParsePict:= true;
    end
    else if (c=' ') then begin
      StpCpy(Pict,PictSym['@']);
      if StpEmpty(Pict) then Warning(WRNUNK,StfNCpy(InpLine,1));
      StpDel(InpLine,1,1); PictSkip:= 1;
      ParsePict:= true;
    end
    else if IsUpper(c) then begin
      StpCpy(Pict,PictSym[c]);
      if StpEmpty(Pict) then Warning(WRNUNK,StfNCpy(InpLine,2));
      StpDel(InpLine,1,2); PictSkip:= 2;
      ParsePict:= true;
    end
    else ParsePict:= false;
  end
  else ParsePict:= false;
end;


function ParseGet: boolean;
begin
  if StpcRet(InpLine,1) <> ComChr[ComGet] then ParseGet:= false
  else begin
    if not ParseName then begin
      Warning(WRNGET,InpLine);
      ParseGet:= false;
    end
    else begin
      WriteOutLine;
      Inc(OutCol);
      write(OutFvar,'@ ',OutRow:2,',',OutCol:2,' GET ',Name);
      Inc(OutCol,StpLen(Name));
      if ParsePict then begin
        write(OutFvar,' PICTURE ',Pict);
        Inc(OutCol,PictSkip);
      end;
      writeln(OutFvar);
      ParseGet:= true;
    end;
  end;
end;


function ParsePut: boolean;
begin
  if StpcRet(InpLine,1) <> ComChr[ComPut] then ParsePut:= false
  else begin
    if not ParseName then begin
      Warning(WRNPUT,InpLine);
      ParsePut:= false;
    end
    else begin
      WriteOutLine;
      Inc(OutCol);
      write(OutFvar,'@ ',OutRow:2,',',OutCol:2,' SAY ',Name);
      Inc(OutCol,StpLen(Name));
      if ParsePict then begin
        write(OutFvar,' PICTURE ',Pict);
        Inc(OutCol,PictSkip);
      end;
      writeln(OutFvar);
      ParsePut:= true;
    end;
  end;
end;


procedure ParseLine;
var c: char;
begin
  StpCreate(OutLine); OutCol:= 0;
  c:= StpcRet(InpLine,1);
  while c<>chr(0) do begin
    if      ParseGet then c:= StpcRet(InpLine,1)
    else if ParsePut then c:= StpcRet(InpLine,1)
    else begin
      StpcCat(OutLine,c);
      c:= NxtChr;
    end;
  end;
  WriteOutLine; Inc(OutRow);
end;


procedure ParseComSym;
var c: char;
begin
  c:= ToUpper(StpcRet(InpLine,2));
  if c=' ' then begin
    if not StpEmpty(PictSym['@']) then Warning(WRNDEF,StfNCpy(InpLine,1));
    StpDel(InpLine,1,1); StpRLS(InpLine); ParsePictString;
    PictSym['@']:= Pict;
  end
  else if IsUpper(c) then begin
    if not StpEmpty(PictSym[c]) then Warning(WRNDEF,StfNCpy(InpLine,2));
    StpDel(InpLine,1,2); StpRLS(InpLine); ParsePictString;
    PictSym[c]:= Pict;
  end
  else ParseLine;
end;


procedure ParseComChg;
var i,k: ComInd; c1, c2: char; found, conflict: boolean;
begin
  c1:= StpcRet(InpLine,2); c2:= StpcRet(InpLine,3);
  if IsPunct(c1) and IsPunct(c2) then begin
    found:= false; conflict:= false;
    for i:= ComCmt to ComChg do begin
      if      ComChr[i] = c1 then begin k:= i; found:= true end
      else if ComChr[i] = c2 then conflict:= true;
    end;
    if conflict then Warning(WRNCHG,StfNCpy(InpLine,3))
    else begin
      if found then ComChr[k]:= c2 else ParseLine;
    end;
  end
  else ParseLine;
end;


procedure ConvertLine;
var c: char;
begin
  c:=StpcRet(InpLine,1);
  if      c=ComChr[ComCmt] then writeln(OutFvar,InpLine)
  else if c=ComChr[ComEcb] then
    writeln(OutFvar,StfBefore(StfDel(InpLine,1,1),ComChr[ComEce]))
  else if c=ComChr[ComPic] then ParseComSym
  else if c=ComChr[ComChg] then ParseComChg
  else ParseLine;
end;


procedure Process;
begin
  writeln('DB3SCN: ',InpFspec,' ==> ',OutFspec);
  InpRow:= 0; OutRow:= 0;
  WriteOutTitle;
  while not eof(InpFvar) do begin
    ReadLn(InpFvar,InpLine); Detab(InpLine); Inc(InpRow);
    ConvertLine;
  end;
end;


{--- Main line ---}


procedure MainInit;
var SymChr: '@'..'Z';
begin
  OptFmt:= DFLFMT;
  ComChr[ComCmt]:= DFLCOMCMT;
  ComChr[ComEcb]:= DFLCOMECB;
  ComChr[ComEce]:= DFLCOMECE;
  ComChr[ComPic]:= DFLCOMPIC;
  ComChr[ComGet]:= DFLCOMGET;
  ComChr[ComPut]:= DFLCOMPUT;
  ComChr[ComChg]:= DFLCOMCHG;
  for SymChr:= '@' to 'Z' do StpCreate(PictSym[SymChr]);
  ErrCod:= ERROK; InpOpen:= false; OutOpen:= false;
  ReadArgs;
  if ErrCod = ERROK then begin
    OpenInp;
    if ErrCod = ERROK then OpenOut;
  end;
end;


procedure MainExit;
begin
  if InpOpen then Close(InpFvar);
  if OutOpen then Close(OutFvar);
  if ErrCod = ERROK then begin
    dispose(InpBuf);
{$IFDEF OUTBUFHEAP}
    dispose(OutBuf);
{$ENDIF}
  end
  else begin
    if ErrCod=ERRARG then Help
    else begin
      write(ERRMSG[ErrCod]);
      if      ErrCod = ERRINP then write(InpFspec)
      else if ErrCod = ERROUT then write(OutFspec);
      writeln;
    end;
  end;
end;


begin { Main program }
  MainInit;
  if ErrCod = ERROK then Process;
  MainExit;
end.
