#include "poweroff.h"

HINSTANCE hInst;
HWND g_hWnd=(HWND)0;
UINT g_tid=0,g_time_tid=0;

void Error(HWND hWnd,char *str)
{
  MessageBox(hWnd,str,NULL,MB_OK|MB_ICONERROR);
}

void DisplayInformation(char *str)
{
  MessageBox(NULL,str,NULL,MB_OK|MB_ICONINFORMATION);
}

void WriteEvent(WORD event_type,char *msg)
{
	HANDLE event_log;
  OSVERSIONINFO windowsversion;

  windowsversion.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
  if (!GetVersionEx(&windowsversion))
  {
    Error(NULL,"Failed to get windows version");
    return;
  }
  if (windowsversion.dwPlatformId==VER_PLATFORM_WIN32_NT)
  {
  	event_log=OpenEventLog(NULL,"Poweroff");
		if (event_log==NULL)
		{
			Error(NULL,"Failed to open event log");
			return;
		}
		if (!ReportEvent(event_log,event_type,0,0,NULL,1,0,&msg,NULL))
		{
			Error(NULL,"Failed to write to event log");
		  return;
		}
		CloseEventLog(event_log);
	}
}

void DisplayLastError(HWND hWnd)
{
  LPVOID lpMsgBuf;
  
  FormatMessage( 
    FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
    NULL,
    GetLastError(),
    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
    (LPTSTR) &lpMsgBuf,    0,    NULL );
  Error( hWnd, lpMsgBuf);
  WriteEvent(EVENTLOG_ERROR_TYPE,lpMsgBuf);
  LocalFree( lpMsgBuf );
}

char *GetModeStr(UINT mode)
{
  switch (mode)
	{
	  case EWX_LOGOFF:
      return "Initiating LOGOFF";
		case EWX_POWEROFF:
			return "Initiating POWEROFF";
		case EWX_REBOOT:
			return "Initiating REBOOT";
		case EWX_SHUTDOWN:
			return "Initiating SHUTDOWN";
	  case EWX_LOGOFF|EWX_FORCE:
      return "Initiating FORCE LOGOFF";
		case EWX_POWEROFF|EWX_FORCE:
			return "Initiating FORCE POWEROFF";
		case EWX_REBOOT|EWX_FORCE:
			return "Initiating FORCE REBOOT";
		case EWX_SHUTDOWN|EWX_FORCE:
			return "Initiating FORCE SHUTDOWN";
		default:
			return "Initiating UNKNOWN action";
	}
}

BOOL FAR PASCAL WarningProc(HWND hWnd, unsigned message,WORD wParam, LONG lParam)
{
  static DWORD time_left;

  switch (message) 
  {
    case WM_INITDIALOG:
      {
        char tmpstr[100];
        WarningParams *params;
 
        params=(WarningParams*)lParam;
        time_left=params->timeout;
        sprintf(tmpstr,"Time left: %d seconds",params->timeout);
        SetDlgItemText(hWnd,IDC_SHUTDOWN_TIME,tmpstr);
        if (params->message!=NULL && params->message[0]!='\0')
          SetDlgItemText(hWnd,IDC_SHUTDOWN_TEXT,params->message);
        else
          SetDlgItemText(hWnd,IDC_SHUTDOWN_TEXT,"Please save all your work and close all applications");
        EnableWindow(GetDlgItem(hWnd,IDC_SHUTDOWN_TEXT),FALSE);
        EnableWindow(GetDlgItem(hWnd,IDC_SHUTDOWN_TEXT),TRUE);
        SetDlgItemText(hWnd,IDC_SHUTDOWN_INFO,GetModeStr(params->mode));
        SetTimer(hWnd,TIME_TIMER,1000,NULL);
      }      
      return TRUE;
    case WM_COMMAND:
      switch (LOWORD(wParam))
      {
        case IDC_DO_IT_NOW:
          EndDialog(hWnd,TRUE);
          break;
      }
      break;
    case WM_TIMER:
      time_left--;
      if (time_left<=0)
        EndDialog(hWnd,TRUE);
      else
      {
        char tmpstr[100];

        sprintf(tmpstr,"Time left: %d seconds",time_left);
        SetDlgItemText(hWnd,IDC_SHUTDOWN_TIME,tmpstr);
      }
      return TRUE;
  	default:
 	    break;
  }
  return FALSE;
}

int PowerOff(HWND hWnd,UINT mode,int force,char *computer,DWORD timeout,char *message)
{
  HANDLE current_thread,token;
  TOKEN_PRIVILEGES privs;
  LUID luid;
  OSVERSIONINFO windowsversion;
  char logstr[50+MAX_PATH];
  FARPROC warningproc;
  WarningParams params;

  windowsversion.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
  if (!GetVersionEx(&windowsversion))
  {
    DisplayLastError(hWnd);
    return -1;
  }
  if (windowsversion.dwPlatformId==VER_PLATFORM_WIN32_NT)
  {
    current_thread=GetCurrentProcess();
    if (!OpenProcessToken(current_thread,TOKEN_ADJUST_PRIVILEGES,&token))
    {
      DisplayLastError(hWnd);
      return -1;
    }
    if (!LookupPrivilegeValue(NULL,SE_SHUTDOWN_NAME,&luid))
    {
      DisplayLastError(hWnd);
      return -1;
    }
    privs.PrivilegeCount=1;
    privs.Privileges[0].Attributes=SE_PRIVILEGE_ENABLED;
    privs.Privileges[0].Luid=luid;
    if (!AdjustTokenPrivileges(token,FALSE,&privs,0,NULL,NULL))
    {
      DisplayLastError(hWnd);
      return -1;
    }
  }
  else if (windowsversion.dwPlatformId==VER_PLATFORM_WIN32_WINDOWS && mode==EWX_POWEROFF)
    mode=EWX_SHUTDOWN;
  if (force)
    mode|=EWX_FORCE;
  if (computer==NULL || computer[0]=='\0')
  {
    if (timeout!=0)
    {
	    warningproc=MakeProcInstance((FARPROC)WarningProc,hInst);
      params.message=message;
      params.timeout=timeout;
      params.mode=mode;
	    DialogBoxParam(hInst,MAKEINTRESOURCE(IDD_WARNING),hWnd,(DLGPROC)warningproc,(LPARAM)&params);
	    FreeProcInstance(warningproc);
    }
    WriteEvent(EVENTLOG_INFORMATION_TYPE,GetModeStr(mode));
    if (!ExitWindowsEx(mode,0))
    {
      DisplayLastError(hWnd);
      return -1;
    }
  }
  else
  {
    sprintf(logstr,"%s on remote computer %s",GetModeStr(mode),computer);
    WriteEvent(EVENTLOG_INFORMATION_TYPE,logstr);
    if (!InitiateSystemShutdown(computer,message,timeout,force,(mode&EWX_REBOOT)?1:0))
    {
      DisplayLastError(hWnd);
      return -1;
    }
    if (g_hWnd!=0)
      DisplayInformation("Remote computer has accepted the request");
  }
  return 0;
}

long FAR PASCAL MainWndProc(HWND hWnd, unsigned message,WORD wParam, LONG lParam)
{
  switch (message) {
    case WM_CREATE: 
	    break;
    case WM_PAINT:
	    break;
    case WM_COMMAND:
	    break;
    case WM_QUIT:
    case WM_DESTROY:
	    PostQuitMessage(0);
	    break;
    default:
	    return (DefWindowProc(hWnd, message, wParam, lParam));
  }
  return 0;
}

void BrowseComputers(HWND hWnd)
{
  BROWSEINFO bi;
  char buffer[MAX_PATH];
  ITEMIDLIST *ppidl;
  LPMALLOC malc;

  SHGetMalloc(&malc);
  SHGetSpecialFolderLocation(hWnd,CSIDL_NETWORK,&ppidl);
  bi.hwndOwner=hWnd;
  bi.pidlRoot=ppidl;
  bi.pszDisplayName=buffer;
  bi.lpszTitle="Select remote computer";
  bi.ulFlags=BIF_BROWSEFORCOMPUTER;
  bi.lpfn=NULL;
  bi.lParam=0;
  if (SHBrowseForFolder(&bi)!=NULL)
    SetDlgItemText(hWnd,IDC_COMPUTER,buffer);
  malc->lpVtbl->Free(malc,ppidl);
}

void EnableComputerFields(HWND hWnd)
{
  OSVERSIONINFO windowsversion;

  windowsversion.dwOSVersionInfoSize=sizeof(OSVERSIONINFO);
  if (!GetVersionEx(&windowsversion))
  {
    DisplayLastError(hWnd);
  }
  if (windowsversion.dwPlatformId!=VER_PLATFORM_WIN32_NT)
    EnableWindow(GetDlgItem(hWnd,IDC_REMOTE_COMPUTER),FALSE);

  if (IsDlgButtonChecked(hWnd,IDC_LOCAL_COMPUTER)==BST_CHECKED)
  {
    EnableWindow(GetDlgItem(hWnd,IDC_COMPUTER),FALSE);
    SetDlgItemText(hWnd,IDC_COMPUTER,"");
    EnableWindow(GetDlgItem(hWnd,IDC_BROWSE),FALSE);
    EnableWindow(GetDlgItem(hWnd,IDC_LOGOFF),TRUE);
    EnableWindow(GetDlgItem(hWnd,IDC_POWEROFF),TRUE);
  }
  else
  {
    EnableWindow(GetDlgItem(hWnd,IDC_COMPUTER),TRUE);
    EnableWindow(GetDlgItem(hWnd,IDC_BROWSE),TRUE);
    EnableWindow(GetDlgItem(hWnd,IDC_LOGOFF),FALSE);
    EnableWindow(GetDlgItem(hWnd,IDC_POWEROFF),FALSE);
    if (IsDlgButtonChecked(hWnd,IDC_LOGOFF)==BST_CHECKED
     || IsDlgButtonChecked(hWnd,IDC_POWEROFF)==BST_CHECKED)
      CheckRadioButton(hWnd,IDC_LOGOFF,IDC_POWEROFF,IDC_SHUTDOWN);
  }
}

void EnableWarningFields(HWND hWnd)
{
  if (IsDlgButtonChecked(hWnd,IDC_WARNING)==BST_CHECKED)
  {
    EnableWindow(GetDlgItem(hWnd,IDC_WARNING_TIME),TRUE);
    EnableWindow(GetDlgItem(hWnd,IDC_WARNING_MESSAGE),TRUE);
    SetDlgItemText(hWnd,IDC_WARNING_TIME,"20");
  }
  else
  {
    EnableWindow(GetDlgItem(hWnd,IDC_WARNING_TIME),FALSE);
    EnableWindow(GetDlgItem(hWnd,IDC_WARNING_MESSAGE),FALSE);
    SetDlgItemText(hWnd,IDC_WARNING_TIME,"");
    SetDlgItemText(hWnd,IDC_WARNING_MESSAGE,"");
  }
}

void DoPoweroff(HWND hWnd)
{
  int force;
  char computer[MAX_PATH]="",tmpstr[100],message[250];
  DWORD timeout;

  force=(IsDlgButtonChecked(hWnd,IDC_FORCE)==BST_CHECKED)?1:0;
  GetDlgItemText(hWnd,IDC_COMPUTER,computer,MAX_PATH);
  if (IsDlgButtonChecked(hWnd,IDC_WARNING)==BST_CHECKED)
  {
    GetDlgItemText(hWnd,IDC_WARNING_TIME,tmpstr,100);
    timeout=atoi(tmpstr);
    GetDlgItemText(hWnd,IDC_WARNING_MESSAGE,message,250);
  }
  else
  {
    timeout=0;
  }

  if (IsDlgButtonChecked(hWnd,IDC_LOGOFF)==BST_CHECKED)
    PowerOff(hWnd,EWX_LOGOFF,force,computer,timeout,message);
  else if (IsDlgButtonChecked(hWnd,IDC_REBOOT)==BST_CHECKED)
    PowerOff(hWnd,EWX_REBOOT,force,computer,timeout,message);
  else if (IsDlgButtonChecked(hWnd,IDC_SHUTDOWN)==BST_CHECKED)
    PowerOff(hWnd,EWX_SHUTDOWN,force,computer,timeout,message);
  else if (IsDlgButtonChecked(hWnd,IDC_POWEROFF)==BST_CHECKED)
    PowerOff(hWnd,EWX_POWEROFF,force,computer,timeout,message);
}

BOOL FAR PASCAL DlgProc(HWND hWnd, unsigned message,DWORD wParam, LONG lParam)
{
  SYSTEMTIME systime,dlgtime,dlgdate;
  HWND ht,hd;
  char str[20];
  NOTIFYICONDATA nid;

  switch (message) 
  {
    case WM_INITDIALOG:
      SendMessage(hWnd,WM_SETICON,(WPARAM)ICON_BIG,(LPARAM)LoadIcon(hInst, MAKEINTRESOURCE(IDI_POWEROFF)));
      SendMessage(hWnd,WM_SETICON,(WPARAM)ICON_SMALL,(LPARAM)LoadIcon(hInst, MAKEINTRESOURCE(IDI_POWEROFF)));
      CheckRadioButton(hWnd,IDC_LOCAL_COMPUTER,IDC_REMOTE_COMPUTER,IDC_LOCAL_COMPUTER);
      EnableComputerFields(hWnd);
      EnableWarningFields(hWnd);
      CheckRadioButton(hWnd,IDC_LOGOFF,IDC_POWEROFF,IDC_POWEROFF);
      CheckDlgButton(hWnd,IDC_FORCE,BST_CHECKED);
      DateTime_SetFormat(GetDlgItem(hWnd,IDC_TIME),"HH:mm:ss");
      DateTime_SetFormat(GetDlgItem(hWnd,IDC_DATE),"dd/MM/yyy");
      g_time_tid=SetTimer(hWnd,TIME_TIMER,1000,NULL);
      return TRUE;
    case WM_CLOSE:
      PostQuitMessage(0);
      break;
    case WM_TIMER:
      ht=GetDlgItem(hWnd,IDC_TIME);
      if (ht==NULL)
        DisplayLastError(hWnd);
      hd=GetDlgItem(hWnd,IDC_DATE);
      if (hd==NULL)
        DisplayLastError(hWnd);
      DateTime_GetSystemtime(ht,&dlgtime);
      DateTime_GetSystemtime(hd,&dlgdate);
      GetLocalTime(&systime);
      sprintf(str,"%02d:%02d:%02d",systime.wHour,systime.wMinute,systime.wSecond);
      SetDlgItemText(hWnd,IDC_CTIME,str);
      sprintf(str,"%02d/%02d/%02d",systime.wDay,systime.wMonth,systime.wYear);
      SetDlgItemText(hWnd,IDC_CDATE,str);
      if ((UINT)wParam==SHUTDOWN_TIMER &&
        dlgtime.wHour==systime.wHour && 
		    dlgtime.wMinute==systime.wMinute && 
		    dlgtime.wSecond==systime.wSecond &&
		    dlgdate.wDay==systime.wDay &&
		    dlgdate.wMonth==systime.wMonth &&
		    dlgdate.wYear==systime.wYear)
      {
        DoPoweroff(hWnd);
      }
      break;
    case WM_COMMAND:
      switch (LOWORD(wParam))
      {
        case IDC_OK:
          if (g_tid==0)
          {
            g_tid=SetTimer(hWnd,SHUTDOWN_TIMER,1000,NULL);
            SetDlgItemText(hWnd,IDC_OK,"Stop");
          }
          else
          {
            KillTimer(hWnd,g_tid);
            g_tid=0;
            SetDlgItemText(hWnd,IDC_OK,"Start");
          }
          break;
        case IDC_CANCEL:
          PostQuitMessage(0);
          break;
        case IDC_HIDE:
          ShowWindow(hWnd,SW_HIDE);
          nid.cbSize=sizeof(NOTIFYICONDATA);
          nid.hWnd=hWnd;
          nid.uID=1;
          nid.uFlags=NIF_ICON|NIF_MESSAGE|NIF_TIP;
          nid.uCallbackMessage=NMESSAGE;
          nid.hIcon=LoadIcon(hInst, MAKEINTRESOURCE(IDI_POWEROFF));
          strcpy(nid.szTip,"Poweroff");
          Shell_NotifyIcon(NIM_ADD,&nid);
          break;
        case IDC_NOW:
          DoPoweroff(hWnd);
          break;
        case IDC_BROWSE:
          BrowseComputers(hWnd);
          break;
        case IDC_LOCAL_COMPUTER:
        case IDC_REMOTE_COMPUTER:
          EnableComputerFields(hWnd);
          break;
        case IDC_WARNING:
          EnableWarningFields(hWnd);
          break;
        default:
	        break;
      }
      break;
    case NMESSAGE:
      if (lParam==WM_LBUTTONDBLCLK)
      {
        nid.cbSize=sizeof(NOTIFYICONDATA);
        nid.hWnd=hWnd;
        nid.uID=1;
        nid.uFlags=0;
        Shell_NotifyIcon(NIM_DELETE,&nid);
        ShowWindow(hWnd,SW_SHOW);
      }
      break;               
  }
  return FALSE;
}

BOOL InitApplication(HINSTANCE hInstance)
{
  WNDCLASSEX  wc;

  // Fill in window class structure with parameters that describe the
  // main window.

  wc.cbSize=sizeof(WNDCLASSEX);
  wc.style = CS_HREDRAW | CS_VREDRAW;	// Class style(s).
  wc.lpfnWndProc = (WNDPROC)MainWndProc;	// Function to retrieve messages for
													  // windows of this class.
  wc.cbClsExtra = 0;	// No per-class extra data.
  wc.cbWndExtra = 0;	// No per-window extra data.
  wc.hInstance = hInstance;	// Application that owns the class.
  wc.hIcon = LoadIcon(hInstance,MAKEINTRESOURCE(IDI_POWEROFF));
  wc.hIconSm = LoadIcon(hInstance,MAKEINTRESOURCE(IDI_POWEROFF));
  wc.hCursor = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = GetStockObject(WHITE_BRUSH);
  wc.lpszMenuName = NULL;	// Name of menu resource in .RC file.
  wc.lpszClassName = "POWEROFF";	// Name used in call to CreateWindow.

  /* Register the window class and return success/failure code. */

  return (RegisterClassEx(&wc));
}

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
	HWND	hWnd;	// Main window handle.


	/* Create a main window for this application instance.  */

	hWnd = CreateWindow(
		"POWEROFF", 	               // See RegisterClass() call.
		"Poweroff",	  	      // Text for window title bar.
		WS_OVERLAPPEDWINDOW,	      // Window style.
		CW_USEDEFAULT, 			   // Default horizontal position.
		CW_USEDEFAULT,	   	   	// Default vertical position.
		CW_USEDEFAULT,	      		// Default width.
		CW_USEDEFAULT, 		   	// Default height.
		NULL,		   		         // Overlapped windows have no parent.
		NULL,	   			         // Use the window class menu.
		hInstance,  			      // This instance owns this window.
		NULL			   	         // Pointer not needed.
	);

	/* If window could not be created, return "failure" */

	if (!hWnd)
		return (FALSE);

	/* Make the window visible; update its client area; and return "success" */

 // ShowWindow(hWnd, nCmdShow);	// Show the window
//	UpdateWindow(hWnd);			// Sends WM_PAINT message
  g_hWnd=hWnd;
	return (TRUE);				// Returns the value from PostQuitMessage

}

BOOL InitDialog(HWND hWnd,HINSTANCE hInstance,int nCmdShow)
{
  FARPROC dlgproc;
  
  dlgproc=MakeProcInstance((FARPROC)DlgProc,hInstance);
	if (DialogBox(hInstance,MAKEINTRESOURCE(IDD_POWEROFF),hWnd,(DLGPROC)dlgproc)==-1)
    DisplayLastError(hWnd);
	FreeProcInstance(dlgproc);
  return TRUE;
}

void CreateNewArgv(char ***argv,int *argc,char *commandline,int start_pos,int end_pos)
{
  *argv=(char**)realloc(*argv,(*argc+1)*sizeof(char*));
  (*argv)[*argc]=malloc(end_pos-start_pos+1);
  memcpy((*argv)[*argc],&commandline[start_pos],end_pos-start_pos);
  (*argv)[*argc][end_pos-start_pos]=0;
  (*argc)++;
}

void ConvertCommandLine(char *commandline,int *argc,char ***argv)
{
  int lastpos=0,x;
  char quote=0;

  *argv=NULL;
  *argc=0;
  for (x=0;x<=(int)strlen(commandline);x++)
  {
    if (!quote && (commandline[x]==' ' || commandline[x]=='\t' || commandline[x]==0))
    {
      if (lastpos!=x)
        CreateNewArgv(argv,argc,commandline,lastpos,x);
      lastpos=x+1;
    }
    else if ((commandline[x]=='"' || commandline[x]=='\'') && (!quote || commandline[x]==quote))
    {
      if (!quote)
      {
        quote=commandline[x];
      }
      else
      {
        CreateNewArgv(argv,argc,commandline,lastpos,x);
        quote=0;
      }
      lastpos=x+1;
    }
  }
  *argv=(char**)realloc(*argv,(*argc+1)*sizeof(char*));
  (*argv)[*argc]=NULL;
}

void Usage(void)
{
  Error(NULL,
		"Usage: poweroff [poweroff | shutdown | reboot | logoff] [seconds_to_wait | HH:MM | dd/mm/yyyy:HH:MM] [options]\n\
Options:\n\
  -f\t\t: Force\n\
  -w n\t\t: Wait n seconds before checking the time\n\
  -r machine\t: Perform action on remote machine\n\
  -warn\t\t: Give a warning dialog box before perfoming the action\n\
  -m message\t: Specifies the message in the warning dialog box\n\
  -t n\t\t: Display the warning dialog box for n seconds");
  exit(0);
}


int PASCAL WinMain( HINSTANCE hInstance,HINSTANCE hPrevInstance,LPSTR lpCmdLine,int nCmdShow)
{
  MSG msg;
  INITCOMMONCONTROLSEX controls;
  int argc,force=0,shuttype,x,hour,minute,day=0,month=0,year=0,sleeptime;
  DWORD seconds=-1L;
  char **argv,*action=NULL,*shuttime=NULL,str[100];
  time_t t;
  int timeout=0;
  char *computer=NULL,*message=NULL;

  if (!hPrevInstance)
    if (!InitApplication(hInstance))
      return (FALSE);

	hInst=hInstance;
  ConvertCommandLine(lpCmdLine,&argc,&argv);
  for (x=0;x<argc;x++)
  {
    if (argv[x][0]=='-')
		{
			if (!strcmp(argv[x],"-f"))
			{
				force=1;
			}
      else if (!strcmp(argv[x],"-w"))
      {
        if (x<argc-1)
        {
          x++;
          sleeptime=atoi(argv[x]);
          Sleep(sleeptime*1000);
        }
        else
          Usage();
      }
      else if (!strcmp(argv[x],"-r"))
      {
        if (x<argc-1)
        {
          x++;
          computer=strdup(argv[x]);
        }
        else
          Usage();
      }
      else if (!strcmp(argv[x],"-m"))
      {
        if (x<argc-1)
        {
          x++;
          message=strdup(argv[x]);
        }
        else
          Usage();
      }
			else if (!strcmp(argv[x],"-warn"))
			{
				if (timeout==0)
          timeout=20;
			}
      else if (!strcmp(argv[x],"-t"))
      {
        if (x<argc-1)
        {
          x++;
          timeout=atoi(argv[x]);
        }
        else
          Usage();
      }
			else
				Usage();
		}
		else
		{
			if (!action)
				action=strdup(argv[x]);
			else if (seconds==-1)
      {
        if (!strchr(argv[x],':'))
  				seconds=atoi(argv[x]);
        else
        {
          shuttime=strdup(argv[x]);
          if (strlen(shuttime)!=5 && strlen(shuttime)!=16)
            Usage();
          if (sscanf(shuttime,"%d:%d",&hour,&minute)!=2)
					{
					  if (sscanf(shuttime,"%d/%d/%d:%d:%d",&day,&month,&year,&hour,&minute)!=5)
              Usage();
						if (day<=0 || day>23 || month<=0 || month>12 || year<=0 || year>9999)
							Usage();
					}
          if (hour<0 || hour>23 || minute<0 || minute>59)
            Usage();
        }
      }
			else
				Usage();
		}
	}
  if (action)
	{
    if (!_stricmp(action,"POWEROFF"))
      shuttype=EWX_POWEROFF;
    else if (!_stricmp(action,"SHUTDOWN"))
      shuttype=EWX_SHUTDOWN;
    else if (!_stricmp(action,"REBOOT"))
      shuttype=EWX_REBOOT;
    else if (!_stricmp(action,"LOGOFF"))
      shuttype=EWX_LOGOFF;
    else
      Usage();
    if (seconds!=-1)
      Sleep(seconds*1000L);
    else if (shuttime)
    {
      for(;;)
      {
        t=time(NULL);
				if (day==0)
          strftime(str,100,"%H:%M",localtime(&t));
			  else
					strftime(str,100,"%d/%m/%Y:%H:%M",localtime(&t));
        if (!strcmp(str,shuttime))
          break;
        Sleep(60000);
      }
    }
    PowerOff(NULL,shuttype,force,computer,timeout,message);
    return 0;
  }
  if (!InitInstance(hInstance, nCmdShow))
    return (FALSE);

  controls.dwSize=sizeof(INITCOMMONCONTROLSEX);
  controls.dwICC=ICC_DATE_CLASSES;
  InitCommonControlsEx(&controls);
  if (!InitDialog(NULL,hInstance,nCmdShow))
    return FALSE;

  while (GetMessage(&msg,NULL,0,0)==1)
  {
    TranslateMessage(&msg);
    DispatchMessage(&msg);
  }
  return (msg.wParam);
}

