;================================================================
; KEYSET By Eric Tauck
;
; This program sets the caps lock, num lock, and scroll lock
; state.  The usage is: 
; 
;   KEYSET [CAPS ON | OFF] [NUM ON | OFF] [SCROLL ON | OFF] 
; 
; Example: 
; 
;   KEYSET CAPS ON SCROLL OFF
; 
; This turns the caps lock on and the scroll lock off.
;
; If you have keyboard LED's, they will only be updated if you are 
; using an AT class computer.  The keyboard codes as returned by 
; the BIOS will reflect the state changes regardless of the type 
; of computer being used. 
;
; This program makes use of the WASM library files.

        INCLUDE '..\..\library\string.asm'
        INCLUDE '..\..\library\case1.asm'
        INCLUDE '..\..\library\case2.asm'
        INCLUDE '..\..\library\parms.asm'
        INCLUDE '..\..\library\message1.asm'

;--- state change flags

C_SET   EQU     0000000000000001B       ;set caps lock
N_SET   EQU     0000000000000010B       ;set num lock
S_SET   EQU     0000000000000100B       ;set scroll lock

;--- BIOS data area constants

C_BIT   EQU     01000000B       ;caps lock bit
N_BIT   EQU     00100000B       ;num lock bit
S_BIT   EQU     00010000B       ;scroll lock bit

BDATA_SEG       EQU     0040H   ;segment of target data area
BDATA_OFF       EQU     0017H   ;offset of target data area

;--- program entry

        mov     ax, OFFSET banner       ;banner address
        call    MesPutL                 ;display

        sub     ah, ah          ;zero bits
        sub     di, di          ;clear all flags
        jmps    main2           ;enter loop

main1   mov     al, C_BIT               ;hardware bit
        mov     bx, OFFSET clock        ;option string
        mov     cx, C_SET               ;flag bit
        call    checkopt                ;check if option
        jc      main2                   ;jump if so

        mov     al, N_BIT               ;hardware bit
        mov     bx, OFFSET nlock        ;option string
        mov     cx, N_SET               ;flag bit
        call    checkopt                ;check if option
        jc      main2                   ;jump if so

        mov     al, S_BIT               ;hardware bit
        mov     bx, OFFSET slock        ;option string
        mov     cx, S_SET               ;flag bit
        call    checkopt                ;check if option
        jc      main2                   ;jump if so
        jmp     error

main2   push    ax
        call    ParGet          ;get next parameter
        mov     si, ax
        pop     ax
        jnc     main1           ;loop back if availible

        or      di, di          ;check if any options
        jz      help            ;jump if none

;--- perform all settings, flags in DI and AH

        push    ds
        mov     dx, BDATA_SEG   ;
        mov     ds, dx          ;-- load address
        mov     bx, BDATA_OFF   ;
        mov     al, [bx]        ;load keyboard control byte

;--- set caps lock

        test    di, C_SET       ;check if set caps lock
        jz      main3
        and     al, NOT C_BIT   ;zero bit
        mov     dl, ah          ;load bit settings
        and     dl, C_BIT       ;mask bit
        or      al, dl          ;set bit on or off

;--- set num lock

main3   test    di, N_SET       ;check if set num lock
        jz      main4
        and     al, NOT N_BIT   ;zero bit
        mov     dl, ah          ;load bit settings
        and     dl, N_BIT       ;mask bit
        or      al, dl          ;set bit on or off

;--- set scroll lock

main4   test    di, S_SET       ;check if set scroll lock
        jz      main5
        and     al, NOT S_BIT   ;zero bit
        mov     dl, ah          ;load bit settings
        and     dl, S_BIT       ;mask bit
        or      al, dl          ;set bit on or off

;--- finished

main5   mov     [bx], al
        pop     ds

        mov     ax, 4C00H       ;exit function error code 0
        int     21H             ;execute

;--- no options, display help

help    mov     ax, OFFSET helpmes
        call    MesPutL
        mov     ax, 4CFFH
        int     21H

;--- invalid options, display error message

error   mov     ax, OFFSET errmes
        call    MesPutL
        mov     ax, 4CFFH
        int     21H

;========================================
; Try to match an option.
;
; In: SI= parameter string; BX= target
;     string; CX= flag bit; DI= current
;     flag bits; AH= current hardware
;     bits; AL= hardware bit.
;
; Out: DI & AH updated; CY= set if
;      option matched; SI= preserved.

checkopt PROC   NEAR
        push    ax
        push    cx
        push    di

        push    bx
        mov     ax, si
        call    StrUpr          ;convert to uppercase
        pop     bx
        mov     ax, si
        call    StrCmp          ;compare with target string
        jc      chkopt2         ;jump if no match

        call    ParGet          ;get ON/OFF
        jc      chkopt3         ;jump if error
        mov     di, ax
        call    StrUpr          ;convert to uppercase
        mov     ax, di
        mov     bx, OFFSET on_str
        call    StrCmp                  ;check if ON
        jnc     chkopt1                 ;jump if so
        mov     ax, di
        mov     bx, OFFSET off_str
        call    StrCmp                  ;check if OFF
        jc      chkopt3                 ;jump if not

;--- found OFF

        call    dispopt         ;display
        pop     di
        pop     cx
        pop     ax
        or      di, cx          ;set flag bit
        not     al              ;flip bits
        and     ah, al          ;zero hardware bit
        stc
        ret

;--- found ON

chkopt1 call    dispopt         ;display
        pop     di
        pop     cx
        pop     ax
        or      di, cx          ;set flag bit
        or      ah, al          ;set hardware bit
        stc
        ret

;--- string doesn't match

chkopt2 pop     di
        pop     cx
        pop     ax
        clc
        ret

;--- error, no ON or OFF

chkopt3 jmp     error
        ENDP

;========================================
; Display the new state.

dispopt PROC    NEAR
        mov     ax, si
        call    MesPut
        mov     ax, OFFSET ismes
        call    MesPut
        mov     ax, di
        call    MesPutL
        ret
        ENDP

;========================================
; Data.

clock   DB      'CAPS',0
nlock   DB      'NUM',0
slock   DB      'SCROLL',0

on_str  DB      'ON',0
off_str DB      'OFF',0

banner  DB      13,10,'KeySet By Eric Tauck',0
helpmes DB      'Usage: KEYSET [CAPS ON | OFF] [NUM ON | OFF]'
        DB      ' [SCROLL ON | OFF]',0
errmes  DB      'Error: run KEYSET without options for help',0
ismes   DB      ' is ',0
