/*
 *  PCTCP - the true worker of Waterloo TCP
 *   	  - contains all opens, closes, major read/write routines and
 *	    basic IP handler for incomming packets
 *	  - NOTE: much of the TCP/UDP/IP layering is done at the data structure
 *	    level, not in separate routines or tasks
 *
 */


#include "wattcp.h"
#include "dpmi.h"

#define MAXINT 32767

extern int _arp_handler( void *in);
void udp_handler(in_Header *ip);
udp_write(udp_Socket *s, ubyte *datap, int len, word offset);
int udp_read(udp_Socket *s, ubyte *datap, int maxlen);
void tcp_Retransmitter(void);


#define TCP_LOCAL 0x4000

/* statics */
static tcp_ProcessData(tcp_Socket *s, tcp_Header *tp, int len);

static initialized = 0;
static void (*system_yield)() = NULL;
extern int multihomes;
extern word _pktipofs;
void (*_dbugxmit)() = NULL;
void (*_dbugrecv)() = NULL;
void (*wattcpd)() = NULL;

char *_hostname = "012345678901234567890123456789012345678901234567890";

word _mss = ETH_MSS;

char *_wattcp = WATTCP_C;

static void udp_handler(in_Header *ip);
extern void icmp_handler(in_Header *ip );

static void tcp_unthread(tcp_Socket *ds);
static void tcp_abort(tcp_Socket *s);
tcp_rst( in_Header *his_ip, tcp_Header *oldtcpp);
static udp_close(udp_Socket *ds);


/*
 * ip user level timer stuff
 *   void ip_timer_init( void *s, int delayseconds )
 *   int  ip_timer_expired( void *s )
 *	- 0 if not expired
 */
static unsigned long *realclock = (unsigned long *)0x000046cL;
#define MAXTICKS 0x1800b0L

void ip_timer_init( udp_Socket *s , int delayseconds )
{
    if (delayseconds)
	s->usertimer = set_timeout( delayseconds );
    else
	s->usertimer = 0;
}

int ip_timer_expired( udp_Socket *s )
{
    if (! s->usertimer)	/* cannot expire */
	return( 0 );
    return( chk_timeout( s->usertimer));
}
longword MsecClock()
{
    return( (*realclock) * 055L);
}

/*
 * Local IP address
 */
longword my_ip_addr = 0L;	/* for external references */
longword sin_mask = 0xfffffe00L;
longword sin_gate = 0x0;


/*
 * IP identification numbers
 */

static int ip_id = 0;			/* packet number */
static int next_tcp_port = 1024;	/* auto incremented */
static int next_udp_port = 1024;
static tcp_Socket *tcp_allsocs = NULL;
static udp_Socket *udp_allsocs = NULL;

/* Timer definitions */
#define RETRAN_STRAT_TIME  1     /* in ticks - how often do we check retransmitter tables*/
#define tcp_RETRANSMITTIME 3     /* interval at which retransmitter is called */
#define tcp_LONGTIMEOUT 31       /* timeout for opens */
#define tcp_TIMEOUT 13           /* timeout during a connection */


/*
 * Shut down the card and all services
 */
void tcp_shutdown()
{
	while (tcp_allsocs)
		tcp_abort( tcp_allsocs );
	_eth_release();
	initialized = 0;
}

/*
 * tcp_Init - Initialize the tcp implementation
 *	    - may be called more than once without hurting
 */
int tcp_init()
{
	extern int _arp_last_gateway;

	if (!initialized) {
		/* initialize ethernet interface */
		initialized = 1;
		if (!_eth_init()) return 0;

		/* reset the various tables */
		_arp_last_gateway = 0;	/* reset the gateway table */
		*_hostname = 0;		/* reset the host's name */

		_eth_free( 0 );
		next_udp_port = next_tcp_port = 1024 + ((*realclock >> 7 )& 0x1ff);
	}
	return 1;
}



/* socket, localport, destaddress */
int udp_open(udp_Socket *s, word lport, longword ina, word port, procref datahandler)
{
	udp_close( s );
	memset( s, 0, sizeof( udp_Socket ));
	s->rdata = s->rddata;
	s->maxrdatalen = tcp_MaxBufSize;
	s->ip_type = UDP_PROTO;

	s->myport = lport;
	s->myaddr = my_ip_addr;

	/* check for broadcast */
	if ( (long)(ina) == -1 || !ina )
		memset( s->hisethaddr, 0xff, sizeof( eth_address ));
	else {
		//printf( "Resolving..." );
		if ( ! _arp_resolve(ina, (eth_address *) &s->hisethaddr[0], 0) )	{
			//printf( "Couldn't resolve it!\n" );
			return( 0 );
		}
		//printf( "\n" );
	}

	s->hisaddr = ina;
	s->hisport = port;
	s->dataHandler = datahandler;
	s->usr_yield = system_yield;
	s->safetysig = SAFETYUDP;
	s->next = udp_allsocs;
	udp_allsocs = s;
	return( 1 );
}

/*
 * Actively open a TCP connection to a particular destination.
 *	- 0 on error
 */



static udp_close(udp_Socket *ds)
{
    udp_Socket *s, **sp;

    sp = &udp_allsocs;
    for (;;) {
        s = *sp;
        if ( s == ds ) {
            *sp = s->next;
            break;
        }
	if ( !s ) break;
	if ( ! s->err_msg ) s->err_msg = "UDP Close called";
	sp = &s->next;
    }
    return( 0 );
}


/*
 * Abort a tcp connection
 */
static void tcp_abort(tcp_Socket *s)
{
    s->unhappy = false;
    s->datalen = 0;
    s->ip_type = 0;
    s->state = tcp_StateCLOSED;
    tcp_unthread(s);
}

void sock_abort(tcp_Socket *s )
{
	udp_close( (udp_Socket *)s );
}



/*
 * Unthread a socket from the tcp socket list, if it's there
 */
static void tcp_unthread(tcp_Socket *ds)
{
    tcp_Socket *s, **sp;

    if (!ds->rdatalen || (ds->state > tcp_StateESTCL))
        ds->ip_type = 0;                /* fail io */
    ds->state = tcp_StateCLOSED;   /* tcp_tick needs this */
    sp = &tcp_allsocs;
    for (;;) {
	s = *sp;
	if ( s == ds ) {
	    *sp = s->next;
            continue;           /* unthread multiple copies if necessary */
	}
	if ( !s ) break;
	sp = &s->next;
    }
}


/*
 * tcp_tick - called periodically by user application
 *	    - returns 1 when our socket closes
 *	    - called with socket parameter or NULL
 */
void tcp_tick()
{
	in_Header *ip;
	word packettype;

	while ( ip = (in_Header *)_eth_arrived( (word *) &packettype ) ) {

		switch ( packettype ) {
		case 0x008 :	/* do IP */
			if ( inchksum(ip, in_GetHdrlenubytes(ip)) == 0xffff ) {
				switch ( ip->proto ) {
				case TCP_PROTO:	break;
				case UDP_PROTO:	udp_handler(ip); break;
				case ICMP_PROTO:	icmp_handler(ip); break;
				}
			} else {
			}
			break;
		case /*0x806*/ 0x608 :
         /* do arp */
			_arp_handler(ip);
			break;
		default:
			break;
		}
		if (ip) _eth_free(ip);

		continue;
	}
}


/* returns 1 if connection is established */
int tcp_established(tcp_Socket *s)
{
	return( s->state >= tcp_StateESTAB );
}

/*
 * udp_write() handles fragmented UDP by assuming it'll be called
 *     once for all fragments with no intervening calls.  This is
 *     the case in sock_write().
 * Handles upto a hair-under 32K datagrams.  Could be made to handle
 *     upto a hair-under 64K easily...  wanna Erick?
 * Might be possible to test 'offset' for non/zero fewer times to be
 *     more efficient.  Might also be more efficient to use the old
 *     UDP checksum() call when more_frags is false in the first frag
 *     (i.e., not a fragmented dgram).
 * Uses _mss to decide splits which defaults to 1400.  Could pack
 *     more into an Ethernet packet.
 */
#define IP_MF 0x0020               // more fragments, net ubyte order

int udp_write(udp_Socket *s, ubyte *datap, int len, word offset)
{
	struct {                    // special pseudo header because need to
		tcp_PseudoHeader ph;    //    compute checksum in two parts (may not
		word checksum2;         //    have all of datagram built at once).
	} ph;
	struct _pkt {
		in_Header  in;
		udp_Header udp;
		int	   data;
	} *pkt;
	ubyte *dp;
	in_Header *inp;
	udp_Header *udpp;

	word maxlen;
	int more_frags;
	word origlen = len;

	pkt = (struct _pkt *)_eth_formatpacket(&s->hisethaddr[0], /*0x800*/ 8);

	if ( offset ) {              // this is not the first fragment
		dp = (ubyte *) &pkt->udp;    // data goes right after IP header
	} else {
		dp = (ubyte *) &pkt->data;
		udpp = &pkt->udp;

		/* udp header */
		udpp->srcPort = intel16( s->myport );
		udpp->dstPort = intel16( s->hisport );
		udpp->checksum = 0;
		udpp->length = intel16( UDP_LENGTH + len );
	}
	inp = &pkt->in;

	memset( inp, 0, sizeof( in_Header ));

	maxlen = _mss & 0xFFF8;             // make a multiple of 8
	if( !offset ) maxlen -= UDP_LENGTH; // note UDP_LENGTH is 8, so ok

	if( len > maxlen ) {
		len = maxlen;
		more_frags = 1;
	} else more_frags = 0;

	inp->length = intel16( sizeof(in_Header) + (offset ? 0 : UDP_LENGTH) + len );
	movmem(datap, dp, len );

	/* internet header */
	inp->hdrlen_ver = HDRLENVER;
	inp->tos = 0;

	/* if offset non-zero, then is part of a prev datagram so don't incr ID */
	inp->identification = intel16( offset ? ip_id : ++ip_id );   /* was post inc */
	inp->frags = (offset ? intel16((offset + UDP_LENGTH) >> 3) : 0);
	if(more_frags) inp->frags |= IP_MF;
	inp->ttl = 254;
	inp->proto = UDP_PROTO;	/* udp */
	inp->checksum = 0;
	inp->source = intel( s->myaddr );
	inp->destination = intel( s->hisaddr );
	inp->checksum = ~inchksum( inp, sizeof(in_Header));

	/* compute udp checksum if desired */
	if(!offset) {  // only first of frags has UDP header for entire UDP dgram
		if ( s->sock_mode & UDP_MODE_NOCHK )
			udpp->checksum = 0;
		else {
			ph.ph.src = inp->source;	/* already INTELled */
			ph.ph.dst = inp->destination;
			ph.ph.mbz = 0;
			ph.ph.protocol = UDP_PROTO;	/* udp */
			ph.ph.length = udpp->length;	/* already INTELled */

			/* can't use since may not have the whole dgram built at once */
			/* this way handles it */
			ph.ph.checksum = inchksum(&pkt->udp, UDP_LENGTH);
			ph.checksum2 = inchksum(datap, origlen);

			udpp->checksum =  ~inchksum(&ph, sizeof(ph));
		}
	}
    
	_eth_send( intel16( inp->length ));

	return ( len );
}


void _udp_cancel( in_Header *ip )
{
    int len;
    udp_Header *up;
    udp_Socket *s;

    /* match to a udp socket */
    len = in_GetHdrlenubytes(ip);
    up = (udp_Header *)((ubyte *)ip + len);	/* udp frame pointer */

    /* demux to active sockets */
    for ( s = udp_allsocs; s; s = s->next )
        if ( s->hisport != 0 &&
             intel16( up->dstPort ) == s->hisport &&
             intel16( up->srcPort ) == s->myport &&
             intel( ip->destination ) == s->hisaddr ) break;
    if ( !s ) {
	/* demux to passive sockets */
	for ( s = udp_allsocs; s; s = s->next )
	    if ( s->hisport == 0 && intel16( up->dstPort ) == s->myport ) break;
    }
    if (s) {
        s->rdatalen = 0;
	s->ip_type = 0;
    }
}



void _tcp_cancel(in_Header *ip, int code, char *msg, longword dummyip )
{
    static int in_icmp_redirect = 0;            // smart@actrix.gen.nz
    int len;
    tcp_Socket *s;
    tcp_Header *tp;

    len = in_GetHdrlenubytes(ip);	/* check work */

    tp = (tcp_Header *)((ubyte *)ip + len);	/* tcp frame pointer */

    /* demux to active sockets */
    for ( s = tcp_allsocs; s; s = s->next ) {
        if ( intel16( tp->srcPort) == s->myport &&
             intel16( tp->dstPort ) == s->hisport &&
             intel( ip->destination ) == s->hisaddr ) {
                switch (code) {
                    /* halt it */
                    case  1 : if (( s->stress ++ > s->rigid ) &&
                                  ( s->rigid < 100 )) {
                                  s->err_msg = (msg) ?
                                    msg : "ICMP closed connection";
                                  s->rdatalen = s->datalen = 0;
                                  s->unhappy = false;
                                  tcp_abort( s );
                /*      if (s->dataHandler) s->dataHandler(s, 0, -1); */
                                  break;
                              }
                              // follow through to next case

                    /* slow it down */
                    case  2 : s->cwindow = 1;
                              s->wwindow = 1;
                              s->rto <<= 2;
                              s->vj_sa <<= 2;
                              s->vj_sd <<= 2;
                              break;
                    /* icmp redirect for host */
                    case  5 : /* save his NEW network address */
                        /* Dummy is passed in NW form need to intel! */
                        /* This was a bug fixed QVS - smart@actrix.gen.nz */
                              if (!in_icmp_redirect)
                              {
                                  in_icmp_redirect = 1;
                                  _arp_resolve(intel(dummyip), (eth_address*)&s->hisethaddr[0], 0);
                                  in_icmp_redirect = 0;
                              }
                              break;
                }
        }
    }
}


/*
 * Handler for incoming packets.
 */
static void udp_handler(in_Header *ip)
{
    udp_Header *up;
    tcp_PseudoHeader ph;
    word len;
    ubyte *dp;
    longword temp;
    udp_Socket *s;

    temp = intel( ip->destination );

    // temp = ip number
    //     or 255.255.255.255
    //     or sin_mask.255.255

    if ( ((~temp & ~sin_mask) != 0) &&  /* not a broadcast packet*/
        ((( temp - my_ip_addr) > multihomes )   /* not my address */
        && my_ip_addr))                 /* and I know my address */
          return;


    len = in_GetHdrlenubytes(ip);
    up = (udp_Header *)((ubyte *)ip + len);	/* udp segment pointer */
    len = intel16( up->length );

    /* demux to active sockets */
    for ( s = udp_allsocs; s; s = s->next ) {
        if ( s->safetysig != SAFETYUDP ) {
            //if (debug_on) outs("chain error in udp\r\n");
        }
        if ( (s->hisport != 0) &&
             (intel16( up->dstPort ) == s->myport) &&
             (intel16( up->srcPort ) == s->hisport) &&
             ((intel( ip->destination ) & sin_mask)  == (s->myaddr & sin_mask)) &&
             (intel( ip->source ) == s->hisaddr )) break;
    }
    if (_dbugrecv) (*_dbugrecv)(s,ip,up,0);
    if ( !s ) {
        /* demux to passive sockets */
	for ( s = udp_allsocs; s; s = s->next )
            if ( ((s->hisaddr == 0) || (s->hisaddr == 0xffffffff))
              && intel16( up->dstPort ) == s->myport ) {

                // do we record this information ???
                if ( s->hisaddr == 0 ) {
                    s->hisaddr = intel( ip->source );
                    s->hisport = intel16( up->srcPort );
                    _arp_resolve(intel(ip->source), &s->hisethaddr[0], 0);
                    // take on value of expected destination unless it
                    // is broadcast
                    if ( (~ip->destination & ~sin_mask) != 0 )
                        s->myaddr = intel( ip->destination );
                }
		break;
	    }
    }
    if ( !s ) {
	/* demux to broadcast sockets */
	for ( s = udp_allsocs; s; s = s->next )
            if ( (s->hisaddr == 0xffffffff) &&
                 (intel16( up->dstPort ) == s->myport )) break;
    }

    if ( !s ) {
		//if (debug_on) outs("discarding...");
	return;
    }

    // these parameters are used for things other than just checksums
    ph.src = ip->source;    /* already INTELled */
    ph.dst = ip->destination;
    ph.mbz = 0;
    ph.protocol = UDP_PROTO;
    ph.length = up->length;
    if ( up->checksum ) {
	ph.checksum =  inchksum(up, len);
	if (inchksum(&ph, sizeof( tcp_PseudoHeader)) != 0xffff)
	    return;
    }

    /* process user data */
    if ( (len -= UDP_LENGTH ) > 0) {
	dp = (ubyte *)( up );
        if (s->dataHandler) s->dataHandler( s, &dp[ UDP_LENGTH ], len , &ph, up);
	else {
            if (len > s->maxrdatalen ) len = s->maxrdatalen;
	    movmem( &dp[ UDP_LENGTH ], s->rdata, len );
	    s->rdatalen = len;
	}
    }
}


void sock_close( sock_type *s )
{
	switch (s->udp.ip_type) {
	case UDP_PROTO :
		udp_close( s );
		break;
	}
}

/*
 * resolve()
 * 	convert domain name -> address resolution.
 * 	returns 0 if name is unresolvable right now
 */
longword resolve(char *name)
{
    if( !name ) return 0L;

    if ( isaddr( name ))
    	 return( aton( name ));

    return (0L);
}

char *inet_ntoa( char *s, longword x )
{

    itoa( x >> 24, s, 10 );
    strcat( s, ".");
    itoa( (x >> 16) & 0xff, strchr( s, 0), 10);
    strcat( s, ".");
    itoa( (x >> 8) & 0xff, strchr( s, 0), 10);
    strcat( s, ".");
    itoa( (x) & 0xff, strchr( s, 0), 10);
    return( s );
}

longword inet_addr( char *s )
{
    return( isaddr( s ) ? aton( s ) : 0 );
}

/*
 * sock_init - easy way to guarentee:
 *	- card is ready
 *	- shutdown is handled
 *	- cbreaks are handled
 *      - config file is read
 *
 * 0.1 : May 2, 1991  Erick - reorganized operations
 */

void sock_exit()
{
	tcp_shutdown();
}

int sock_init()
{
	dpmi_init(0);

	if (!tcp_init()) return 0;		/* must precede tcp_config because we need eth addr */
	atexit(sock_exit);	/* must not precede tcp_init() incase no PD */

	if (tcp_config( NULL )) {	/* if no config file use BOOTP w/broadcast */
		return 0;
	}
	return 1;
}

#if 0
int _send_ping( longword host, longword countnum, ubyte ttl, ubyte tos, longword *theid )
{
	eth_address dest;
	struct _pkt *p;
	in_Header *ip;
	struct icmp_echo *icmp;
	static word icmp_id = 0;

	if ((host & 0xff) == 0xff ) {
		//outs( "Cannot ping a network!\n\r");
		return( -1 );
	}
    
	if ( ! _arp_resolve( host, dest, 0 )) {
		//outs( "Cannot resolve host's hardware address\n\r");
		return( -1 );
	}


	if (debug_on) {
		//outs("\n\rDEBUG: destination hardware :");
		//outhexes( (ubyte *)dest, 6 );
		//outs("\n\r");
	}
	p = (struct _pkt*)_eth_formatpacket( dest, 8 );

	ip = &p->in;
	memset( ip, 0, sizeof( in_Header ));
	icmp = &p->icmp;

	icmp->type = 8;
	icmp->code = 0;
	icmp->index = countnum;
	*(longword *)(&icmp->identifier) = set_timeout( 1 );
	if( theid ) *theid = *(longword *)(&icmp->identifier);

/*
    icmp->identifier = ++icmp_id;
    icmp->sequence = icmp_id;
*/

	/* finish the icmp checksum portion */
	icmp->checksum = 0;
	icmp->checksum = ~inchksum( icmp, sizeof( struct icmp_echo));

	/* encapsulate into a nice ip packet */
	ip->hdrlen_ver = HDRLENVER;
	ip->length = intel16( sizeof( in_Header ) + sizeof( struct icmp_echo));
	ip->tos = tos;
	ip->identification = intel16( icmp_id ++);	/* not using ip id */
	ip->ttl = ttl;
	ip->proto = ICMP_PROTO;
	ip->checksum = 0;
	ip->source = intel( my_ip_addr );
	ip->destination = intel( host );
	ip->checksum = ~ inchksum( ip, sizeof( in_Header ));

	return( _eth_send( intel16( ip->length )));
}
#endif
