#include <assert.h>
#include <conio.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "amx.h"

#define MAXFILES        16
#define MAXBREAKS       16
#define SCREENLINES     20      /* number of lines that fit on a screen */
#define NAMEMAX         20

enum {
  BP_NONE,
  BP_CODE,
  BP_DATA,
  /* --- */
  BP_TYPES
};

typedef struct __symbol {
  char name[NAMEMAX];
  ucell addr;   /* line number for functions */
  int local;    /* true if it is a local variable */
  int type;
  int calllevel;/* nesting level (visibility) of the variable */
  struct __symbol *next;
} SYMBOL;

typedef struct {
  int type;             /* one of the BP_xxx types */
  cell addr;            /* line number, or previous value */
  int index;
  SYMBOL *sym;
} BREAKPOINT;

static char *filenames[MAXFILES];
static int curfileno=-1;
static char **cursource;
static int curline;
static BREAKPOINT breakpoints[MAXBREAKS];
static SYMBOL functab;
static SYMBOL vartab;

static void freesource(char **source)
{
  int i;

  assert(source!=NULL);
  for (i=0; source[i]!=NULL; i++)
    free(source[i]);
  free(source);
}

static char **loadsource(char *filename)
{
  char **source;
  FILE *fp;
  char line[256];
  int lines,i;

  /* open the file, number of characters */
  if ((fp=fopen(filename,"rt"))==NULL)
    return NULL;
  lines=0;
  while (fgets(line,255,fp)!=NULL)
    lines++;

  /* allocate memory, reload the file */
  if ((source=malloc((lines+1)*sizeof(char *)))==NULL) {
    fclose(fp);
    return NULL;
  } /* if */
  for (i=0; i<=lines; i++)      /* initialize, so that freesource() works */
    source[i]=NULL;
  rewind(fp);
  i=0;
  while (fgets(line,255,fp)!=NULL) {
    assert(i<lines);
    source[i]=strdup(line);
    if (source[i]==NULL) {
      fclose(fp);
      freesource(source);       /* free everything allocated so far */
      return NULL;
    } /* if */
    i++;
  } /* if */

  fclose(fp);
  return source;
}

static char *skipwhitespace(char *str)
{
  while (*str==' ' || *str=='\t')
    str++;
  return str;
}

static SYMBOL *add_symbol(SYMBOL *table,char *name,int type,ucell addr,
                          int local,int level)
{
  SYMBOL *sym;

  if ((sym=malloc(sizeof(SYMBOL)))==NULL)
    return NULL;
  memset(sym,0,sizeof(SYMBOL));
  assert(strlen(name)<NAMEMAX);
  strcpy(sym->name,name);
  sym->type=type;
  sym->addr=addr;
  sym->local=local;
  sym->calllevel=level;
  sym->next=table->next;
  table->next=sym;
  return sym;
}

static SYMBOL *find_symbol(SYMBOL *table,char *name,int level)
{
  SYMBOL *sym = table->next;

  while (sym!=NULL) {
    if (strcmp(name,sym->name)==0 && sym->calllevel==level)
      return sym;
    sym=sym->next;
  } /* while */
  return NULL;
}

static void delete_symbol(SYMBOL *table,ucell addr)
{
  SYMBOL *prev = table;
  SYMBOL *cur = prev->next;

  while (cur!=NULL) {
    if (cur->local && cur->addr<addr) {
      prev->next=cur->next;
      free(cur);
      cur=prev->next;
    } else {
      prev=cur;
      cur=cur->next;
    } /* if */
  } /* while */
}

static void delete_allsymbols(SYMBOL *table)
{
  SYMBOL *sym=table->next, *next;

  while (sym!=NULL) {
    next=sym->next;
    free(sym);
    sym=next;
  } /* while */
}

static cell get_symbolvalue(AMX *amx,SYMBOL *sym,int index)
{
  cell *value;
  cell base;

  if (sym->type==2 || sym->type==4) {   /* a reference */
    amx_GetAddr(amx,sym->addr,&value);
    base=*value;
  } else {
    base=sym->addr;
  } /* if */
  /* ??? for arrays, need to have range */
  amx_GetAddr(amx,base+index*sizeof(cell),&value);
  return *value;
}

static void initbreak(void)
{
  int i;

  for (i=0; i<MAXBREAKS; i++) {
    breakpoints[i].type=BP_NONE;
    breakpoints[i].addr=0;
    breakpoints[i].sym=NULL;
    breakpoints[i].index=-1;
  } /* for */
}

static void clearbreak(int index)
{
  if (index>=0 && index<=MAXBREAKS) {
    breakpoints[index].type=BP_NONE;
    breakpoints[index].addr=0;
    breakpoints[index].sym=NULL;
    breakpoints[index].index=-1;
  } /* if */
}

static int setbreak(AMX *amx,char *str,int calllevel)
{
  int index;
  SYMBOL *sym;

  /* find an empty spot */
  for (index=0; index<MAXBREAKS && breakpoints[index].type!=BP_NONE; index++)
    /* nothing */
  if (index>=MAXBREAKS)
    return -1;
  assert(breakpoints[index].sym==NULL);
  assert(breakpoints[index].addr==0);
  assert(breakpoints[index].index==-1);

  /* find type */
  str=skipwhitespace(str);
  if (isdigit(*str)) {
    breakpoints[index].type=BP_CODE;
    breakpoints[index].addr=atol(str);
  } else if ((sym=find_symbol(&functab,str,-1))!=NULL) {
    breakpoints[index].type=BP_CODE;
    breakpoints[index].addr=sym->addr;
    breakpoints[index].sym=sym;
  } else {
    char *idxptr=strchr(str,'[');
    int idx=-1;
    if (idxptr!=NULL) {
      idx=atoi(idxptr+1);
      *idxptr='\0';
    } /* if */
    if ((sym=find_symbol(&vartab,str,calllevel))!=NULL) {
      if ((sym->type==3 || sym->type==4) && idxptr==NULL)
        return -1;
      if (sym->type!=3 && sym->type!=4 && idxptr!=NULL)
        return -1;
      breakpoints[index].type=BP_DATA;
      breakpoints[index].addr=get_symbolvalue(amx,sym,idx>=0 ? idx : 0);
      breakpoints[index].sym=sym;
      breakpoints[index].index=idx;
    } else {
      return -1;
    } /* if */
  } /* if */
  return index;
}

static void listbreak(void)
{
  int index;

  for (index=0; index<MAXBREAKS; index++) {
    if (breakpoints[index].type==BP_NONE)
      continue;
    printf("%2d  ",index);
    if (breakpoints[index].type==BP_CODE) {
      printf("line: %d",breakpoints[index].addr);
      if (breakpoints[index].sym!=NULL)
        printf("  func: %s",breakpoints[index].sym->name);
      printf("\n");
    } else {
      /* must be variable */
      assert(breakpoints[index].sym!=NULL);
      printf("var: %s",breakpoints[index].sym->name);
      if (breakpoints[index].index>=0)
        printf("[%d]",breakpoints[index].index);
      printf("\n");
    } /* if */
  } /* for */
}

static int checkbreak(AMX *amx,int line)
{
  int index;

  for (index=0; index<MAXBREAKS; index++) {
    if (breakpoints[index].type==BP_CODE && breakpoints[index].addr==line) {
      return index;
    } else if (breakpoints[index].type==BP_DATA) {
      int idx=breakpoints[index].index;
      SYMBOL *sym=breakpoints[index].sym;
      cell value;
      assert(sym!=NULL);
      value=get_symbolvalue(amx,sym,idx>=0 ? idx : 0);
      if (breakpoints[index].addr!=value) {
        breakpoints[index].addr=value;
        return index;
      } /* if */
    } /* if */
  } /* for */
  return -1;
}

enum {
  GO,
  NEXT,
  STEP,
  QUIT,
  WATCH,
  BREAK_SET,
  BREAK_CLEAR,
  BREAK_LIST,
};

static void listcommands(char *command)
{
  if (command==NULL)
    command="";
  if (stricmp(command,"break")==0) {
    printf("\tBREAK\t\tlist all breakpoints\n"
           "\tBREAK n\t\tset a breakpoint at line \"n\"\n"
           "\tBREAK func\tset a breakpoint at function with name \"func\"\n"
           "\tBREAK var\tset a breakpoint at variable \"var\"\n"
           "\tBREAK var[i]\tset a breakpoint at array element \"var[i]\"\n");
  } else if (stricmp(command,"clear")==0) {
    printf("\tCLEAR n\t\tremove breakpoint number \"n\"\n"
           "\tCLEAR *\t\tremove all breakpoints\n");
  } else if (stricmp(command,"disp")==0 || stricmp(command,"d")==0) {
    printf("\tDISP may be abbreviated to D\n\n"
           "\tDISP\t\tdisplay all variables that are currently in scope\n"
           "\tDISP var\tdisplay the value of variable \"var\"\n"
           "\tDISP var[i]\tdisplay the value of array element \"var[i]\"\n");
  } else if (stricmp(command,"calls")==0
             || stricmp(command,"g")==0 || stricmp(command,"go")==0
             || stricmp(command,"n")==0 || stricmp(command,"next")==0
             || stricmp(command,"quit")==0
             || stricmp(command,"s")==0 || stricmp(command,"step")==0)
  {
    printf("\tno additional information\n");
  } else if (stricmp(command,"l")==0 || stricmp(command,"list")==0) {
    printf("\tLIST may be abbreviated to L\n\n"
           "\tLIST\t\tdisplay 10 lines around the current line\n"
           "\tLIST n\t\tdisplay 10 lines, starting from line \"n\"\n"
           "\tLIST n m\tdisplay \"m\" lines, starting from line \"n\"\n"
           "\tLIST f:n m\tdisplay \"m\" lines from file \"f\", starting from \"n\"\n"
           "\t\t\t(use LIST FILES to display the valid values for \"f\")\n"
           "\tLIST FILES\tdisplay all files\n"
           "\tLIST FUNCS\tdisplay all functions\n");
  } else {
    printf("\tBREAK\t\tset breakpoint at line number or variable name\n"
           "\tCLEAR\t\tremove breakpoint\n"
           /* "\tCALLS\t\tshow call stack\n" ??? not yet implemented */
           "\tD(isp)\t\tdisplay the value of a variable, list variables\n"
           "\tG(o)\t\trun program (until breakpoint)\n"
           "\tL(ist)\t\tdisplay source file and symbols\n"
           "\tN(ext)\t\tRun until next line, step over functions\n"
           "\tQUIT\t\texit debugger, terminate program\n"
           "\tS(tep)\t\tsingle step, step into functions\n"
           "\n\tUse \"? <command name>\" to view more information on a command\n");
  } /* if */
  /* ??? other commands that may be useful:
   *     GO until line number or until function exit
   *     modify a variable
   */
}

static int docommand(AMX *amx,int calllevel)
{
  char line[100], command[32];
  int result,i;
  SYMBOL *sym;
  char *ptr;

  for ( ;; ) {
    printf("- ");
    gets(line);
    result=sscanf(line,"%8s",command);
    if (result<=0) {
      listcommands(NULL);
      continue;
    } /* if */
    if (stricmp(command,"?")==0) {
      result=sscanf(line,"%*s %30s",command);
      listcommands(result ? command : NULL);
    } else if (stricmp(command,"quit")==0) {
      exit(0);
    } else if (stricmp(command,"g")==0 || stricmp(command,"go")==0) {
      return GO;
    } else if (stricmp(command,"s")==0 || stricmp(command,"step")==0) {
      return STEP;
    } else if (stricmp(command,"n")==0 || stricmp(command,"next")==0) {
      return NEXT;
    } else if (stricmp(command,"l")==0 || stricmp(command,"list")==0) {
      ptr=strchr(line,' ');
      ptr=(ptr!=NULL) ? skipwhitespace(ptr) : "";
      /* first check a few hard cases */
      if (stricmp(ptr,"files")==0) {
        for (i=0; i<MAXFILES && filenames[i]!=NULL; i++)
          printf("%5d\t%s\n",i,filenames[i]);
      } else if (stricmp(ptr,"funcs")==0) {
        for (sym=functab.next; sym!=NULL; sym=sym->next)
          printf("%5ld\t%s\n",(long)sym->addr,sym->name);
      } else {
        int lnum,lastline,numlines,file;
        lnum=curline-4;
        numlines=10;
        result=sscanf(line,"%*s %d:%d %d",&file,&lnum,&numlines);
        if (result>=2 && file<MAXFILES && filenames[file]!=NULL) {
          curfileno=file;
          if (cursource!=NULL)
            freesource(cursource);
          cursource=loadsource(filenames[curfileno]);
          if (cursource==NULL) {
            printf("\nSource file not found or insufficient memory\n");
            continue;
          } /* if */
        } else {
          file=curfileno;
          result=sscanf(line,"%*s %d %d",&lnum,&numlines);
        } /* if */
        lnum--;           /* if user filled in a line number, subtract 1 */
        if (lnum<0)
          lnum=0;
        lastline=lnum+numlines;
        /* seek to line number from the start (to avoid displaying something
         * beyond the file
         */
        for (result=0; cursource[result]!=NULL && result<lnum; result++)
          /* nothing */;
        if (cursource[result]!=NULL) {
          assert(result==lnum);
          while (cursource[lnum]!=NULL && lnum<lastline) {
            if (lnum==curline)
              printf("[%d]\t%s",lnum+1,cursource[lnum]);
            else
              printf(" %d \t%s",lnum+1,cursource[lnum]);
            lnum++;
          } /* while */
        } /* if */
      } /* if */
    } else if (stricmp(command,"break")==0) {
      ptr=strchr(line,' ');
      ptr=(ptr!=NULL) ? skipwhitespace(ptr) : "";
      if (*ptr=='\0') {
        listbreak();
      } else {
        result=setbreak(amx,ptr,calllevel);
        if (result<0)
          printf("Invalid breakpoint, or table full\n");
      } /* if */
    } else if (stricmp(command,"clear")==0) {
      ptr=strchr(line,' ');
      ptr=(ptr!=NULL) ? skipwhitespace(ptr) : "";
      if (*ptr=='*') {
        /* clear all breakpoints */
        for (i=0; i<MAXBREAKS; i++)
          clearbreak(i);
      } else if (isdigit(*ptr)) {
        clearbreak(atoi(ptr));
      } else {
        printf("\tInvalid command, use \"?\" to view all commands\n");
      } /* if */
    } else if (stricmp(command,"disp")==0 || stricmp(command,"d")==0) {
      ptr=strchr(line,' ');
      ptr=(ptr!=NULL) ? skipwhitespace(ptr) : "";
      if (*ptr=='\0') {
        /* display all */
        for (sym=vartab.next; sym!=NULL; sym=sym->next) {
          if (sym->calllevel==-1 || sym->calllevel==calllevel) {
            printf("%s\t%s\t%ld",sym->local ? "loc" : "glb",sym->name,
                   get_symbolvalue(amx,sym,0));
            if (sym->type==3 || sym->type==4)
              printf(" [...]");   /* it is an array */
            printf("\n");
          } /* if */
        } /* for */
      } else {
        char *indexptr=strchr(ptr,'[');
        if (indexptr!=NULL) {
          i=atoi(indexptr+1);
          *indexptr='\0';
        } else {
          i=0;
        } /* if */
        if ((sym=find_symbol(&vartab,ptr,calllevel))!=NULL) {
          if ((sym->type==3 || sym->type==4) && indexptr==NULL)
            printf("\tArray must be indexed\n");
          else if (sym->type!=3 && sym->type!=4 && indexptr!=NULL)
            printf("\tInvalid index, not an array\n");
          else
            printf("%s\t%s\t%ld\n",sym->local ? "loc" : "glb",sym->name,
                   get_symbolvalue(amx,sym,i));
        } else {
          printf("\tSymbol not found, or not a variable\n");
        } /* if */
      } /* if */
    } else {
      printf("\tInvalid command, use \"?\" to view all commands\n");
    } /* if */
  } /* for */
}

static int debugproc(AMX *amx)
{
static int tracelevel, go;
static int calllevel;
static SYMBOL *curfunc;
  int cmd,i,vclass,type,num;
  unsigned short flags;

  switch (amx->dbgcode) {
  case DBG_INIT:
    assert(amx->flags==AMX_FLAG_BROWSE);
    /* check whether we should run */
    amx_Flags(amx,&flags);
    if ((flags & AMX_FLAG_DEBUG)==0 || curfileno!=-1)
      return AMX_ERR_DEBUG;     /* the debugger cannot run */
    /* intialize the file table and other global variables */
    for (i=0; i<MAXFILES; i++)
      filenames[i]=NULL;
    cursource=NULL;
    curfileno=-1;
    initbreak();
    functab.next=NULL;
    vartab.next=NULL;
    /* initialize statics here */
    tracelevel=0;
    go=0;
    calllevel=0;
    curfunc=NULL;
    break;
  case DBG_FILE:        /* file number in curfile, filename in dbgname */
    assert(amx->flags==(AMX_FLAG_DEBUG | AMX_FLAG_BROWSE));
    if (amx->curfile>=MAXFILES) {
      printf("\nCritical error: too many source files\n");
      exit(1);
    } /* if */
    /* file should not already be set */
    num=(int)amx->curfile;
    assert(filenames[num]==NULL);
    filenames[num]=malloc(strlen(amx->dbgname)+1);
    if (filenames[num]!=NULL) {
      strcpy(filenames[num],amx->dbgname);
    } else {
      printf("\nCritical error: insufficient memory\n");
      exit(1);
    } /* if */
    break;
  case DBG_LINE:        /* line number in curline, file number in curfile */
    if ((amx->flags & AMX_FLAG_BROWSE)!=0) {
      /* check whether there is a function symbol that needs to be adjusted */
      if (curfunc!=NULL)
        curfunc->addr=amx->curline;
      curfunc=NULL;
      break;            /* ??? could build a list with valid breakpoints */
    } /* if */
    curline=(int)amx->curline-1;
    /* check breakpoints */
    if ((i=checkbreak(amx,(int)amx->curline))>=0) {
      printf("Stopped at breakpoint %d\n",i);
      go=0;
      tracelevel=calllevel;
    } /* if */
    if (go || tracelevel<calllevel)
      break;
    assert(amx->curfile>=0 && amx->curfile<MAXFILES);
    if (curfileno!=(int)amx->curfile) {
      curfileno=(int)amx->curfile;
      if (cursource!=NULL)
        freesource(cursource);
      cursource=loadsource(filenames[curfileno]);
      if (cursource==NULL) {
        printf("\nCritical error: source file not found or insufficient memory\n");
        exit(1);
      } /* if */
    } /* if */
    assert(cursource[curline]!=NULL);
    printf("[%d]\t%s",curline+1,cursource[curline]);
    cmd=docommand(amx,calllevel);
    go= (cmd==GO);
    tracelevel= calllevel + (cmd==STEP);
    /* otherwise, must be step */
    break;
  case DBG_SYMBOL:      /* address in dbgaddr, class/type in dbgparam,
                         * symbol name in dbgname */
    vclass=(int)(amx->dbgparam>>8);
    type=(int)amx->dbgparam & 0xff;
    if (type==9) {
      /* function */
      assert(vclass==0);
      assert(amx->flags==(AMX_FLAG_DEBUG | AMX_FLAG_BROWSE));
      curfunc=add_symbol(&functab,amx->dbgname,type,0,vclass,-1);
    } else {
      /* must modify address relative to frame */
      if (vclass==1)
        amx->dbgaddr += amx->frm;
      assert((amx->flags & AMX_FLAG_DEBUG)!=0);
      if ((amx->flags & AMX_FLAG_BROWSE)!=0)
        calllevel=-1;
      add_symbol(&vartab,amx->dbgname,type,amx->dbgaddr,vclass,calllevel);
    } /* if */
    break;
  case DBG_CLRSYM:      /* stk = stack address below which locals should be removed */
    assert((amx->flags & (AMX_FLAG_DEBUG | AMX_FLAG_BROWSE))==AMX_FLAG_DEBUG);
    delete_symbol(&vartab,amx->stk);
    break;
  case DBG_CALL:        /* function call */
    assert((amx->flags & (AMX_FLAG_DEBUG | AMX_FLAG_BROWSE))==AMX_FLAG_DEBUG);
    calllevel++;
    break;
  case DBG_RETURN:      /* function returns */
    assert((amx->flags & (AMX_FLAG_DEBUG | AMX_FLAG_BROWSE))==AMX_FLAG_DEBUG);
    calllevel--;
    if (tracelevel>calllevel)
      tracelevel=calllevel;
    break;
  case DBG_TERMINATE:   /* program ends */
    assert((amx->flags & (AMX_FLAG_DEBUG | AMX_FLAG_BROWSE))==AMX_FLAG_DEBUG);
    for (i=0; i<MAXFILES; i++)
      if (filenames[i]!=NULL)
        free(filenames[i]);
    if (cursource!=NULL)
      freesource(cursource);
    delete_allsymbols(&functab);
    delete_allsymbols(&vartab);
    curfileno=-1;
    /* ??? save breakpoints on exit */
    break;
  } /* switch */
  return AMX_ERR_NONE;
}

static void *loadprogram(AMX *amx,char *filename)
{
  FILE *fp;
  AMX_HEADER hdr;
  void *program = NULL;

  if ((fp = fopen(filename,"rb")) != NULL) {
    fread(&hdr, sizeof hdr, 1, fp);
    if ((program = malloc((int)hdr.stp)) != NULL) {
      rewind(fp);
      fread(program, 1, (int)hdr.size, fp);
      fclose(fp);
      if (amx_Init(amx,program,debugproc) == AMX_ERR_NONE)
        return program;
      free(program);
    } /* if */
  } /* if */
  return NULL;
}

int main(int argc,char *argv[])
{
extern AMX_NATIVE_INFO core_Natives[];
extern AMX_NATIVE_INFO console_Natives[];
extern void core_Init(void);
extern void core_Exit(void);

  AMX amx;
  cell ret;
  int err;
  void *program;
  unsigned short flags;

  printf("Small command line debugger\n\n");
  if (argc != 2 || (program = loadprogram(&amx,argv[1])) == NULL) {
    printf("Usage: SDBG <filename>\n\n"
           "The filename must include the extension\n");
    return 1;
  } /* if */
  amx_Flags(&amx,&flags);
  if ((flags & AMX_FLAG_DEBUG)==0) {
    printf("This program has no debug information\n");
    return 1;
  } /* if */

  core_Init();

  amx_Register(&amx, core_Natives, -1);
  err = amx_Register(&amx, console_Natives, -1);

  if (err == AMX_ERR_NONE)
    err = amx_Exec(&amx, &ret, AMX_EXEC_MAIN, 0);

  if (err != AMX_ERR_NONE)
    printf("Run time error %d on line %ld\n", err, amx.curline);
  else
    printf("Normal termination, return value %ld\n", (long)ret);

  free(program);
  core_Exit();

  return 0;
}

