/*
 * s1 --- print a binary file in Motorola ``S1'' format.  The input filename
 * and the starting address are command line arguments (unlike BINEX & S19 :-)
 *
 * Copyright (c) MCMXCIV by Luchezar Georgiev <lucho@midi.tu-varna.bg>
 *
 * Usage: s1 [-b hexaddr] binfile [ > sfile ]
 *
 * where hexaddr is the begin address, binfile is the input binary file,
 * sfile is the output file in S1-format.  If the -b option is not given,
 * the default begin address of 0 is used.  If sfile is not given, the
 * standard output is used.
 *
 * The E.J.Rupp's ``as*'' package (a universal assembler for Motorola CPUs)
 * was used as a template (see ``util.c'' from that package).
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <fcntl.h>
#include <io.h>

#define E_LIMIT 32	/* max # of bytes on a single line */
#define lowbyte(b)	((b) & 0377)

typedef unsigned char u_char;
typedef unsigned long u_long;

char	*hexstr = { "0123456789ABCDEF" };

void hexout(int byte)
{
	byte = lowbyte(byte);
	fprintf( stdout, "%c%c", hexstr[byte >> 4], hexstr[byte & 017] );
}

void f_record	    /* flush record out in ``S1'' format */
   (u_char *E_bytes,	/* Emitted held bytes	   */
    int	    E_total,	/* Total # of bytes emited */
    int	    E_pc)	/* Emitted Program Counter */
{
    int i, chksum;

    chksum = E_total + 3;	/* total bytes in this record */
    chksum += lowbyte(E_pc);
    chksum += E_pc >> 8;
    fprintf(stdout, "S1");	/* record header preamble */
    hexout(E_total + 3);	/* byte count + PC & chksum */
    hexout(E_pc >> 8);		/* high byte of PC */
    hexout(lowbyte(E_pc));	/* low byte of PC */
    for (i=0; i < E_total; i++) {
	chksum += lowbyte(E_bytes[i]);
	hexout(lowbyte(E_bytes[i])); /* data byte */
    }
    chksum =~ chksum;	     /* one's complement */
    hexout(lowbyte(chksum)); /* checksum */
    fprintf(stdout,"\n");
}

int get_hex(char *ascii_value)
{
    u_long hex;
    char *endptr;

    hex = strtoul (ascii_value, &endptr, 16);
    if (*endptr) {
	fprintf (stderr, "Invalid begin address substring: \"%s\"\n", endptr);
	exit(-3);
    }
    if (hex > USHRT_MAX) {
	fprintf (stderr, "Invalid begin address value: \"%s\"\n", ascii_value);
	exit(-4);
    }
    return (int)hex;
}

int open_bin(char *name)
{
    int handle;

    if ( (handle = open( name, O_BINARY | O_RDONLY )) == -1 ) {
	fprintf (stderr, "Can't open %s\n", name);
	exit(-2);
    }
    return handle;
}

void usage()
{
    fprintf (stderr, "Usage: s1 [-b hexaddr] binfile [ > sfile ]\n");
    exit(-1);
}

void main (int argc, char **argv)
{
    u_char bytes[E_LIMIT];
    int total, binfile = -1, pc = 0;

    if (argc == 4 && argv[1][0] == '-' && argv[1][1] == 'b') {
	pc = get_hex(argv[2]);
	binfile = open_bin(argv[3]);
    }
    else if (argc == 2) {
	pc = 0;
	binfile = open_bin(argv[1]);
    }
    else usage();

    /* no S0-record is really needed... */

    while( ( total = read(binfile, (void *)bytes, E_LIMIT) ) != 0 ) {
	if (total == -1) {
	    fprintf (stderr, "Can't read input file - why?\n");
	    exit(-5);
	}
	f_record (bytes, total, pc); /* flush record */
	pc += total;		   /* update the PC */
    }
    fprintf(stdout, "S9");	/* end-of-record */
    hexout(3);
    hexout(0);
    hexout(0);
    hexout(0xFC);
    printf("\n");
}
