{ PREP.PAS: Text preprocessor filter

  Title   : PREP
  Language: Borland Pascal v4.0 and higher
  Version : 2.2
  Date    : Feb 19, 2000
  Author  : J R Ferguson
  Usage   : refer procedure Help and file PREP.TXT
  Download: http://hello.to/ferguson
  E-mail  : j.r.ferguson@iname.com

This program and its source may be used and copied freely without charge,
but  only  for non-commercial purposes. The author is not responsible for
any damage or loss of data that may be caused by using it.

To compile this source file, you wil need  some  units  from  the  JRFPAS
Pascal  routine  library by the same author, which can be downloaded from
the Internet address mentioned above.
}

{$V-}
{$R+}

{$UNDEF OUTBUFHEAP}   { UNDEF to work around a BP 7.0 bug resulting in
                        erroneous file output }

program PREP;
uses DefLib, StpLib, StfLib, ChrLib, ObtLib, DOS;


const
  PROGIDN   = 'PREP';
  PROGVERS  = '2.2';
  FBUFSIZ   = 4096;
  MaxCmd    = 127; { Maximum number of characters in a command argument }
  MaxArg    = 10 ; { Maximum number of command arguments recognized }


  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRINP    = 2;
  ERROUT    = 3;
  ERRELSE   = 4;
  ERRENDC   = 5;

  ERRMSG    : array[ERRINP..ERRENDC] of StpTyp =
 ('File not found',
  'Can''t open output',
  'Unmatched ELSE',
  'Unmatched ENDIF'
 );

  CmndPos   = 1;
  CmdNone   = 0;
  CmdCmnd   = 1; CmndCmd   = 'COMMAND';
  CmdDefn   = 2; DefnCmd   = 'DEFINE';
  CmdUndf   = 3; UndfCmd   = 'UNDEF';
  CmdCond   = 4; CondCmd   = 'IF';
  CmdElse   = 5; ElseCmd   = 'ELSE';
  CmdEndc   = 6; EndcCmd   = 'ENDIF';
  CmdPref   = 7; PrefCmd   = 'PREFIX';
  CmdIncl   = 8; InclCmd   = 'INCLUDE';
  CmdVar1   = 9; Var1Cmd   = 'VARBEG';
  CmdVar2   =10; Var2Cmd   = 'VAREND';

type
  ArgInd    = 0..MaxArg;
  CmdInd    = 0..MaxCmd;
  CmdStp    = string[MaxCmd];
  CmdTyp    = CmdNone .. CmdVar2;
  FbufTyp   = array[1..FBUFSIZ] of char;
  FilePtr   = ^FileTyp;
  FileTyp   = record
                Fname : StpTyp;
                Fvar  : Text;
                Fbuf  : ^FbufTyp;
                Fopen : boolean;
              end;
  IdLstPtr  = ^IdLstRec;
  IdLstRec  = record ident, value: StpTyp; next: IdLstPtr end;
  IfStkPtr  = ^IfStkRec;
  IfStkRec  = record condition, invert: boolean; next: IfStkPtr end;

var
  ArgC      : ArgInd;                  { Actual number of command args }
  ArgV      : array[ArgInd] of CmdStp; { Actual command argument values }
  ArgI      : ArgInd;
  InitialRun: boolean;

  InpFname  : StpTyp;
  OutFile   : FilePtr;

  ErrCod    : integer;
  ErrFname  : StpTyp;
  ErrLine   : integer;

  CurLine   : StpTyp;
  CurCmd    : CmdTyp;
  CurArg    : StpTyp;
  CmndPref  : StpTyp;
  InclPref  : StpTyp;
  VarBeg    : StpTyp;
  VarEnd    : StpTyp;
  IdnList   : IdLstPtr;
  IfStack   : IfStkPtr;
{$IFNDEF OUTBUFHEAP}
  OutBuf    : FBufTyp;
{$ENDIF}


procedure Help;
begin
  writeln(PROGIDN+' '+PROGVERS+': File include processing filter');
  writeln('usage   : PREP [<in] [>out] [/option [...] ]');
  writeln('options : C<string>              Specify command-prefix string [*]');
  writeln('          D<identifier[=value]>  Define identifier [and assign a value]');
  writeln('          U<identifier>          Undefine identifier');
  writeln('          P<fileprefix>          Set file prefix for INCLUDEs');
  writeln('          [<identbegin>          Set identifier substitution begin mark');
  writeln('          ]<identend>            Set identifier substitution end mark');
  writeln('Read PREP.TXT for further information');
end;

{ --- Command argument handling --- }

procedure GetCmdArg(var cmd: CmdStp);
const
  GetPSP = $62;
  ArgOfs = $0080;
var
  CPU    : Registers;
  s      : CmdStp; sL: CmdInd absolute s;
begin with CPU do begin
  AH:= GetPSP; MsDos(CPU); Move(Mem[BX:ArgOfs],s,SizeOf(s));
  if sL > MaxCmd then sL:= MaxCmd;
  cmd:= s;
end end;

procedure GetArgs;
const
  Null   = #0;
var
  arg    : CmdStp; ArgL: CmdInd absolute arg;
  cmd    : CmdStp; CmdL: CmdInd absolute cmd;
  i      : CmdInd;
  c      : char;
  Quote  : char;

  function HasArg: boolean;

    function NxtChr(var c: char): char;
    begin
      if i >= CmdL then c:= Null
      else begin inc(i); c:= cmd[i] end;
      NxtChr:= c;
    end;

  begin { HasArg }
    if ArgC >= MaxArg then HasArg:= false
    else begin
      arg:= '';
      while IsSpace(NxtChr(c)) do ;
      if (c='"') or (c='''') then begin
        Quote:= c;
        if NxtChr(c) = Null then arg:= Quote
        else while (c <> Null) and (c <> Quote) do begin  { quoted arg }
          if ArgL < MaxCmd then arg:= arg + c;
          c:= NxtChr(c);
        end;
      end
      else while (c <> Null) and not IsSpace(c) do begin  { unquoted arg }
        if ArgL < MaxCmd then arg:= arg + c;
        c:= NxtChr(c);
      end;
      HasArg:= ArgL>0
    end;
  end;

begin { GetArgs }
  if initialRun then GetCmdArg(cmd) else cmd:= CurArg;
  for ArgC:= 1 to MaxArg do ArgV[ArgC]:= '';
  ArgC:= 0; i:= 0; ArgV[0]:= '';
  while HasArg do begin inc(ArgC); ArgV[ArgC]:= arg end;
  ArgI:= 0;
end;


function NextArg(var arg: CmdStp): boolean;
begin
  if ArgI < ArgC then begin
  Inc(ArgI);
    arg:= ArgV[ArgI]; NextArg:= true;
  end
  else begin
    arg:= ''; NextArg:= false;
  end;
end;


{
--- Condition Stack manipulation ---
    Single linked list used as a LIFO stack
}

procedure IfCreate;
begin IfStack:= nil end;

function IfEmpty: boolean;
begin IfEmpty:= IfStack = nil end;

procedure IfPush(cond: boolean);
var p: IfStkPtr;
begin
  new(p);
  with p^ do begin condition:= cond; invert:= false; next:= IfStack end;
  IfStack:= p;
end;

procedure IfPop;
var p: IfStkPtr;
begin if not IfEmpty then begin
  p:= IfStack; IfStack:= IfStack^.next; dispose(p);
end end;

function IfAll: boolean;
var ok: boolean; p: IfStkPtr;
begin
  ok:= true; p:= IfStack;
  while ok and (p<>nil) do begin ok:= p^.condition; p := p^.next end;
  IfAll:= ok;
end;

procedure IfInvert;
begin if not IfEmpty then with IfStack^ do begin
  condition:= not condition; invert:= true;
end end;

function IfInverted: boolean;
begin if IfEmpty then IfInverted:= true else IfInverted:= IfStack^.invert end;

procedure IfDiscard;
begin while not IfEmpty do IfPop end;


{
--- Identifier Table Manipulation ---
    Single linked list with unique but unordered identifiers.
}

procedure IdCreate;
begin IdnList:= nil end;

function IdEmpty: boolean;
begin IdEmpty:= IdnList = nil end;

function IdSearchVal(ident: StpTyp; var value: StpTyp): boolean;
var p: IdLstPtr; found: boolean;
begin
  StpUpp(ident);
  p:= IdnList; found:= false; value:= '';
  while (p<>nil) and not found do begin
    found:= StpCmp(p^.ident,ident) = 0;
    if found then value:= p^.value;
    p:= p^.next;
  end;
  IdSearchVal:= found;
end;

function IdSearch(ident: StpTyp): boolean;
var dummy: StpTyp;
begin IdSearch:= IdSearchVal(ident,dummy); end;

function IdHasVal(ident, value: StpTyp): boolean;
var IdnVal: StpTyp;
begin
  IdHasVal:= IdSearchVal(ident,IdnVal)
    and (IdnVal = value);
end;

procedure IdInsertVal(ident, value: StpTyp);
var p: IdLstPtr; found: boolean;
begin
  StpUpp(ident);
  p:= IdnList; found:= false;
  while (p<>nil) and not found do begin
    found:= StpCmp(p^.ident,ident) = 0;
    if found then p^.value:= value;
    p:= p^.next;
  end;
  if not found then begin
    new(p);
    p^.ident:= StfUpp(ident);
    p^.value:= value;
    p^.next := IdnList;
    IdnList := p;
  end;
end;

procedure IdRemoveIdn(ident: StpTyp);
var p0: IdLstPtr;
  procedure  IdRemoveRecursive(var p: IdLstPtr);
  begin if p<>nil then begin
    if p^.ident=ident then begin
      p0:= p;
      p := p^.next;
      dispose(p0);
    end
    else IdRemoveRecursive(p^.next);
  end; end;
begin
  StpUpp(ident);
  IdRemoveRecursive(IdnList);
end;

procedure IdDiscard;
var p: IdLstPtr;
begin while not IdEmpty do begin
  p:= IdnList;
  IdnList:= IdnList^.next;
  dispose(p);
end end;

{
--- Command line parsing routines ---
}

function ParseVars(S: StpTyp): StpTyp;
var S1, ident, value: StpTyp;
begin
  S1:= '';
  while S <> '' do begin
    S1:= S1 + StfBefore(S,VarBeg); S:= StfAfter(S,VarBeg);
    if S <> '' then begin
      ident:= StfBefore(S,VarEnd); S:= StfAfter(S,VarEnd);
      if IdSearchVal(ident,value) then S1:= S1 + value;
    end;
  end;
  ParseVars:= S1;
end;

procedure ReadOpt(arg: StpTyp);
begin
  StpDel(arg,1,1);
  case ToUpper(StpcGet(arg)) of
    'C' : CmndPref:= arg;
    'D' : IdInsertVal(StfBefore(arg,'='),StfAfter(arg,'='));
    'U' : IdRemoveIdn(arg);
    'P' : InclPref:= arg;
    '[' : VarBeg  := arg;
    ']' : VarEnd  := arg;
    else ErrCod:= ERRARG;
  end;
end;

procedure ReadArgs;
var i   : ArgInd;
    arg : StpTyp;
begin
  GetArgs;
  i:= 0;
  while (i < ArgC) and (ErrCod = ERROK) do begin
    Inc(i); StpCpy(arg,ArgV[i]);
    case StpcRet(arg,1) of
      '/' : ReadOpt(arg);
      '<' : if InitialRun then StpSub(InpFname,arg,2,MaxStp);
      '>' : if InitialRun then StpSub(OutFile^.Fname,arg,2,MaxStp);
      else  ErrCod:= ERRARG;
    end;
  end;
end;


{
--- I/O routines ---
}

procedure OpenInp(f: FilePtr);
begin with f^ do begin
  Fopen:= false;
  if ErrCod=ERROK then begin
    Assign(Fvar,Fname);
    new(Fbuf); SetTextBuf(Fvar,Fbuf^); {$I-} reset(Fvar); {$I+}
    if IOresult <> 0 then begin ErrCod:= ERRINP; ErrFname:= Fname end
    else Fopen:= true;
  end;
end end;

procedure OpenOut(f: FilePtr);
begin with f^ do begin
  Fopen:= false;
  if ErrCod=ERROK then begin
    Assign(Fvar,Fname);
{$IFDEF OUTBUFHEAP}
      new(Fbuf); SetTextBuf(Fvar,Fbuf^);
{$ELSE}
      SetTextBuf(Fvar,OutBuf); FBuf:= @OutBuf;
{$ENDIF}
    {$I-} rewrite(Fvar); {$I+}
    if IOresult <> 0 then begin ErrCod:= ERROUT; ErrFname:= Fname end
    else Fopen:= true;
  end;
end end;

procedure CloseInp(f: FilePtr);
begin with f^ do begin
  if Fopen then Close(Fvar);
  if ErrCod=ErrOK then dispose(Fbuf);
end end;

procedure CloseOut(f: FilePtr);
begin with f^ do begin
  if Fopen then Close(Fvar);
{$IFDEF OUTBUFHEAP}
  if ErrCod=ErrOK then dispose(Fbuf);
{$ENDIF}
end end;


{
--- Internal Command Parsing ---
}

function ReadCommand(line: StpTyp; var arg: StpTyp): CmdTyp;
var command: StpTyp;
begin
  StpCreate(arg);
  if StpPos(line,CmndPref) = CmndPos then begin
    StpDel(line,1,CmndPos+StpLen(CmndPref)-1);
    StpGtw(command,line); StpUpp(command); arg:= StfRLS(line);
    if StpCmp(command,CmndCmd)=0 then ReadCommand:= CmdCmnd else
    if StpCmp(command,DefnCmd)=0 then ReadCommand:= CmdDefn else
    if StpCmp(command,UndfCmd)=0 then ReadCommand:= CmdUndf else
    if StpCmp(command,CondCmd)=0 then ReadCommand:= CmdCond else
    if StpCmp(command,ElseCmd)=0 then ReadCommand:= CmdElse else
    if StpCmp(command,EndcCmd)=0 then ReadCommand:= CmdEndc else
    if StpCmp(command,PrefCmd)=0 then ReadCommand:= CmdPref else
    if StpCmp(command,InclCmd)=0 then ReadCommand:= CmdIncl else
    if StpCmp(command,Var1Cmd)=0 then ReadCommand:= CmdVar1 else
    if StpCmp(command,Var2Cmd)=0 then ReadCommand:= CmdVar2 else
    ReadCommand:= CmdNone;
  end
  else ReadCommand:= CmdNone;
end;


{
--- Main line ---
}

procedure ProcessFile(InpFname: StpTyp);
var InpFile: FilePtr; LineNo: integer; ident, value: StpTyp;
begin
  if not InitialRun then ReadArgs;
  new(InpFile);
  with InpFile^ do begin
    StpCpy(Fname,InpFname);
    OpenInp(InpFile); LineNo:= 0;
    while (ErrCod=ERROK) and not eof(Fvar) do begin
      ReadLn(Fvar,CurLine); LineNo:= LineNo + 1;
      CurLine:= ParseVars(CurLine);
      CurCmd:= ReadCommand(CurLine, CurArg);
      case CurCmd of
        CmdCond : begin
                    ident:= StfBefore(CurArg,'=');
                    value:= StfAfter (CurArg,'=');
                    if value = '' then
                      IfPush(IdSearch(ident))
                    else
                      IfPush(IdHasVal(ident,value));
                  end;
        CmdElse : if IfInverted then begin
                    ErrCod:= ERRELSE; ErrFname:= InpFname; ErrLine:= LineNo;
                  end
                  else IfInvert;
        CmdEndc : if IfEmpty then begin
                    ErrCod:= ERRENDC; ErrFname:= InpFname; ErrLine:= LineNo;
                  end
                  else IfPop;
        else if IfAll then case CurCmd of
          CmdNone : WriteLn(OutFile^.Fvar,ParseVars(CurLine));
          CmdCmnd : CmndPref:= CurArg;
          CmdDefn : IdInsertVal(StfBefore(CurArg,'='),StfAfter(CurArg,'='));
          CmdUndf : IdRemoveIdn(CurArg);
          CmdPref : InclPref:= CurArg;
          CmdVar1 : VarBeg  := CurArg;
          CmdVar2 : VarEnd  := CurArg;
          CmdIncl : begin
                      InitialRun:= false;
                      CurArg:= InclPref+CurArg;
                      StpGtw(ident,CurArg); CurArg:= StfRLS(CurArg);
                      ProcessFile(ident);
                    end;
        end;
      end;
    end;
    CloseInp(InpFile);
  end;
  dispose(InpFile);
end;

procedure MainInit;
begin
  InitialRun:= true;
  new(OutFile); OutFile^.Fname:= ''; InpFname:= '';
  CmndPref:= '*';
  VarBeg  := '[';
  VarEnd  := ']';
  InclPref:= '';
  IdCreate;
  IfCreate;
  ErrCod:= ERROK;
  ReadArgs;
  OpenOut(OutFile);
end;

procedure MainTerm;
begin
  CloseOut(OutFile);
  if ErrCod <> ERROK then begin
    if ErrCod=ERRARG then Help
    else begin
      write(ERRMSG[ErrCod]);
      if ErrCod = ERRINP  then write(': ',ErrFname)                  else
      if ErrCod = ERROUT  then write(': ',ErrFname)                  else
      if ErrCod = ERRELSE then write(': ',ErrFname,' line ',ErrLine) else
      if ErrCod = ERRENDC then write(': ',ErrFname,' line ',ErrLine);
      writeln;
    end;
  end;
  dispose(OutFile);
  IdDiscard;
  IfDiscard;
end;

begin { Main program }
  MainInit;
  if ErrCod = ERROK then ProcessFile(InpFname);
  MainTerm;
end.
