;*****
; windows.asm - 'Windows' system
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __WINDOWS__ equ 1
                include "pmlib.asd"
                include "windows.asi"
                include "debug.asi"
                include "error.asi"
                include "geometry.asi"
                include "heap.asi"
                include "list.asi"
                include "mouse.asi"
                include "spr.asi"
                include "stack.asi"
                include "svga.asi"
                include "timer.asi"
                include "win.asi"





;;
;                               BUFFER: DATA                               ;
;;

dBufferWin      dd      0       ;'Win' of the buffer
dChangesList    dd      0       ;list of rectangles to update





;;
;                             WALLPAPER: DATA                              ;
;;

dWallpaperPic   dd      0       ;wallpaper picture

DBG_RUS_MESSAGE cWallpaperErr "ୠ 梥⮢ 奬 ."
DBG_ENG_MESSAGE cWallpaperErr "Bad wallpaper color scheme."





;;
;                              WINDOW: DATA                                ;
;;

dWindow1        dd      0       ;first temporary 'Window'
dWindow2        dd      0       ;second temporary 'Window'

DBG_RUS_MESSAGE cErrNotWindow "ꥪ  ਭ  'Window'."
DBG_ENG_MESSAGE cErrNotWindow "Object is not a 'Window'."

macro           WINDOW_ASSERT   reg
IF              DBG
                cmp     [(window &reg).dSign], WINDOW_SIGN
                @ERROR_NE cErrNotWindow
ENDIF
endm





;;
;                                CURSOR: DATA                              ;
;;

CR_MAX_ROW      equ     50
CR_MAX_COL      equ     50

dCursPoint      point   <>              ;cursor position
dTimer          dd      0               ;timer value for autorepeat detection
bButtons        db      0               ;buttons state
bAutorepeat     db      0               ;autorepeat flag
dCursPic        dd      0               ;cursor picture
dCursSpr        dd      0               ;cursor sprite
dSavePic        dd      0               ;image that was hided by cursor





;;
;                              WINDOWS: DATA                               ;
;;

sParentWin      win     <>      ;
sParentWindow   window  <>      ;parent window for all opened 'Window's
dModalsStack    dd      0       ;modal 'Window's stack
dPointedWindow  dd      0       ;'Window' pointed by cursor
bWindowsUpdate  db      0       ;counter of 'WindowsUpdate' calls





;;
;                              BUFFER: CODE                                ;
;;

proc            BufferInit
                pushad

                ;;create 'Win' of buffer
                call    SvgaGetInfo
                add     eax, CR_MAX_ROW
                add     ebx, CR_MAX_COL
                call    WinNew
                mov     [dBufferWin], edi
                mov     esi, edi
                call    SvgaGetInfo
                call    WinSetArea
                xor     eax, eax
                call    WinFill

                ;;create changes list
                call    ListNew
                mov     [dChangesList], edi
                call    RectNew
                mov     [(rect edi).dRow], 0
                mov     [(rect edi).dCol], 0
                call    SvgaGetInfo
                mov     [(rect edi).dTotalRow], eax
                mov     [(rect edi).dTotalCol], ebx
                call    BufferInvalidateRect

                popad
                ret
endp



proc            BufferRestore
                mov     esi, [dChangesList]
                call    ListDelete
                mov     esi, [dBufferWin]
                call    WinDelete
                ret
endp



proc            BufferInvalidateRect
                ;in
                ;  edi - 'Rectangle'
                ;out
                ;  nothing
                ;description
                ;  Marks buffer region for later update.
                push    ebx esi

                ;;check if new region intersects with olders
                mov     esi, [dChangesList]
                call    ListFirst
@@loop1:        jc      @@no_cross
                push    eax esi
                mov     esi, eax
                call    RectIsIntersect
                pop     esi eax
                jnc     @@cross
                call    ListNext
                jmp     @@loop1

@@no_cross:     call    ListInsertLast
                jmp     @@exit

@@cross:        mov     esi, eax
                call    RectConcatenate
                mov     ebx, esi
                mov     esi, edi
                call    RectDelete
                lea     esi, [sParentWindow]
                call    BufferGrowRect

@@exit:         pop     esi ebx
                ret
endp



proc            BufferGrowRect
                ;in
                ;  ebx - joined 'Rectangle'
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Copies 'Window' in buffer if it is not there and it is
                ;  intersected by 'Rectangle'

                ;;intersection check
                push    edi
                call    WindowGetRect
                push    esi
                mov     esi, ebx
                call    RectIsIntersect
                pushf
                mov     esi, edi
                call    RectDelete
                popf
                pop     esi
                pop     edi
                jc      @@exit

                cmp     [(window esi).dNotInBuffer], 1
                @IF_E
                  call  WindowMarkWithChilds
                  jmp   @@exit
                @ENDIF

                push    esi
                mov     esi, [(window esi).dChildsList]
                call    ListGetPosition
                push    eax
                call    ListLast
@@loop1:        @IF_NC
                  push  esi
                  mov   esi, eax
                  call  BufferGrowRect
                  pop   esi
                  call  ListPrevious
                  jmp   @@loop1
                @ENDIF
                pop     eax
                call    ListSetPosition
                pop     esi

@@exit:         ret
endp



proc            BufferCommitChanges
                ;in
                ;  nothing
                ;out
                ;  nothing
                ;description
                ;  Copies changed regions of buffer to screen.
                push    esi

                mov     esi, [dChangesList]
                call    ListFirst
@@loop:         @IF_NC
                  push  esi
                  mov   esi, eax
                  call  BufferShowRect
                  call  RectDelete
                  pop   esi
                  call  ListNext
                  jmp   @@loop
                @ENDIF
                call    ListZap

                pop     esi
                ret
endp



proc            BufferShowRect
                ;in
                ;  esi - 'Rectangle'
                ;out
                ;  nothing
                ;description
                ;  Copies buffer's region to screen.
                pushad

                call    RectCheck
                mov     eax, [(rect esi).dRow]
                mov     ebx, [(rect esi).dCol]
                mov     ecx, [(rect esi).dTotalRow]
                mov     edx, [(rect esi).dTotalCol]
                mov     esi, [dBufferWin]

                ;;save parameters of buffer's visible area
                push    [(win esi).dVisibleRow]
                push    [(win esi).dVisibleCol]
                push    [(win esi).dShiftRow]
                push    [(win esi).dShiftCol]

                ;;set new parameters of visible area
                add     [(win esi).dShiftRow], eax
                add     [(win esi).dShiftCol], ebx
                mov     [(win esi).dVisibleRow], ecx
                mov     [(win esi).dVisibleCol], edx

                ;;copy visible area to screen
                call    SvgaWinToScr

                ;;restore parameters of buffer's visible area
                pop     [(win esi).dShiftCol]
                pop     [(win esi).dShiftRow]
                pop     [(win esi).dVisibleCol]
                pop     [(win esi).dVisibleRow]

                popad
                ret
endp





;;
;                            WALLPAPER: CODE                               ;
;;

proc            WallpaperInit
                ;in
                ;  esi - wallpaper picture
                ;out
                ;  nothing
                ;description
                ;  Initializes wallpaper object.
                pushad

                mov     [dWallpaperPic], esi
                mov     edi, esi
                mov     esi, [dBufferWin]
                call    WinPicFill

                popad
                ret
endp



struc           locals
  dDstTotalRow  dd      0       ;total rows to copy
  dSrcTotalRow  dd      0       ;total rows in wallpaper picture
  dSrcCurrRow   dd      0       ;current row in wallpaper picture
  ;;
  dDst          dd      0       ;current address in buffer
  dDstRowLen    dd      0       ;total length of buffer rect row in bytes
  dDstDelta     dd      0       ;value to add to skip to next line in buffer
  ;;
  dSrcStart     dd      0       ;first row address in wallpaper picture
  dSrcCurr      dd      0       ;current row address in wallpaper picture
  dSrcRowLen    dd      0       ;total length of row in bytes
  dSrcRowOfs    dd      0       ;offset in row in bytes
  dSrcColCount  dd      0       ;counter of copied columns in bytes
ends
proc            WallpaperFillRect
                ;in
                ;  edi - rectangle of buffer to fill
                ;out
                ;  nothing
                ;description
                ;  Fills buffer rectangle with wallpaper picture.
                pushad
                sub     esp, size locals
                mov     ebp, esp

                mov     esi, edi
                call    RectCheck

IF              DBG
                mov     esi, [dBufferWin]
                mov     eax, [(win esi).dBytesPerCell]
                mov     esi, [dWallpaperPic]
                cmp     eax, [(pic esi).dBytesPerCell]
                @ERROR_NE cWallpaperErr
ENDIF

                ;;total row in buffer rectangle
                mov     eax, [(rect edi).dTotalRow]
                mov     [(locals ebp).dDstTotalRow], eax
                ;;total row in wallpaper picture
                mov     esi, [dWallpaperPic]
                mov     eax, [(pic esi).dTotalRow]
                mov     [(locals ebp).dSrcTotalRow], eax
                ;;first row in wallpaper picture
                xor     edx, edx
                mov     eax, [(rect edi).dRow]
                div     [(pic esi).dTotalRow]
                mov     [(locals ebp).dSrcCurrRow], edx

                ;;destination address
                mov     eax, [(rect edi).dRow]
                mov     ebx, [dBufferWin]
                mul     [(win ebx).dTotalCol]
                add     eax, [(rect edi).dCol]
                mul     [(win ebx).dBytesPerCell]
                add     eax, [(win ebx).dImagePtr]
                mov     [(locals ebp).dDst], eax
                ;;total length of buffer rect row in bytes
                mov     eax, [(rect edi).dTotalCol]
                mul     [(win ebx).dBytesPerCell]
                mov     [(locals ebp).dDstRowLen], eax
                ;;delta
                mov     eax, [(win ebx).dTotalCol]
                mul     [(win ebx).dBytesPerCell]
                sub     eax, [(locals ebp).dDstRowLen]
                mov     [(locals ebp).dDstDelta], eax

                ;;first row address in wallpaper picture
                mov     eax, [(pic esi).dImagePtr]
                mov     [(locals ebp).dSrcStart], eax
                ;;current row address in wallpaper picture
                mov     eax, [(locals ebp).dSrcCurrRow]
                mul     [(pic esi).dTotalCol]
                mul     [(pic esi).dBytesPerCell]
                add     eax, [(pic esi).dImagePtr]
                mov     [(locals ebp).dSrcCurr], eax
                ;;total length of row in bytes
                mov     eax, [(pic esi).dTotalCol]
                mul     [(pic esi).dBytesPerCell]
                mov     [(locals ebp).dSrcRowLen], eax
                ;;offset in source row in bytes
                xor     edx, edx
                mov     eax, [(rect edi).dCol]
                div     [(pic esi).dTotalCol]
                mov     eax, edx
                mul     [(pic esi).dBytesPerCell]
                mov     [(locals ebp).dSrcRowOfs], eax

                ;;main loop
                mov     edi, [(locals ebp).dDst]

@@loop1:        ;;column init
                mov     [(locals ebp).dSrcColCount], 0

                ;;column left part (from offset to end of tile)
                mov     esi, [(locals ebp).dSrcCurr]
                mov     eax, [(locals ebp).dSrcRowOfs]
                add     esi, eax
                sub     eax, [(locals ebp).dSrcRowLen]
                neg     eax
                cmp     eax, [(locals ebp).dDstRowLen]
                @IF_A
                  mov   eax, [(locals ebp).dDstRowLen]
                @ENDIF
                add     [(locals ebp).dSrcColCount], eax
                mov     ecx, eax
                shr     ecx, 2
                rep     movsd
                mov     ecx, eax
                and     ecx, 3
                rep     movsb

                ;;column middle part (number of full tiles)
                mov     eax, [(locals ebp).dDstRowLen]
                sub     eax, [(locals ebp).dSrcColCount]
                xor     edx, edx
                div     [(locals ebp).dSrcRowLen]
                and     eax, eax
                @IF_NZ
                  @DO
                    mov ecx, [(locals ebp).dSrcRowLen]
                    add [(locals ebp).dSrcColCount], ecx
                    mov esi, [(locals ebp).dSrcCurr]
                    shr ecx, 2
                    rep movsd
                    mov ecx, [(locals ebp).dSrcRowLen]
                    and ecx, 3
                    rep movsb
                  @DJNZ eax
                @ENDIF

                ;;column right part (from tile end to rectangle end)
                mov     esi, [(locals ebp).dSrcCurr]
                mov     eax, [(locals ebp).dDstRowLen]
                sub     eax, [(locals ebp).dSrcColCount]
                mov     ecx, eax
                shr     ecx, 2
                rep     movsd
                mov     ecx, eax
                and     ecx, 3
                rep     movsb

                ;;next line in wallpaper picture
                inc     [(locals ebp).dSrcCurrRow]
                mov     eax, [(locals ebp).dSrcCurrRow]
                cmp     eax, [(locals ebp).dSrcTotalRow]
                @IF_E
                  mov   [(locals ebp).dSrcCurrRow], 0
                  mov   eax, [(locals ebp).dSrcStart]
                  mov   [(locals ebp).dSrcCurr], eax
                @ELSE
                  mov   eax, [(locals ebp).dSrcRowLen]
                  add   [(locals ebp).dSrcCurr], eax
                @ENDIF

                ;;next line in buffer rectangle
                add     edi, [(locals ebp).dDstDelta]

                dec     [(locals ebp).dDstTotalRow]
                jnz     @@loop1

                add     esp, size locals
                popad
                ret
endp





;;
;                             WINDOW: CODE                                 ;
;;

proc            WindowNew
                ;in
                ;  eax - row
                ;  ebx - column
                ;  edx - address of event handler
                ;  esi - 'Win'
                ;out
                ;  edi - 'Window'
                push    ecx
                lea     ecx, [sParentWindow]
                call    WindowChildNew
                pop     ecx
                ret
endp



proc            WindowChildNew
                ;in
                ;  eax - row
                ;  ebx - column
                ;  ecx - parent 'Window'
                ;  edx - address of event handler
                ;  esi - 'Win'
                ;out
                ;  edi - 'Window'

                push    eax
                mov     eax, size window
                call    HeapAlloc
                pop     eax
                call    WindowInit

                ret
endp



proc            WindowInit
                ;in
                ;  eax - row
                ;  ebx - column
                ;  ecx - parent 'Window'
                ;  edx - address of event handler
                ;  esi - 'Win'
                ;  edi - 'Window'
                ;out
                ;  nothing
                WINDOW_ASSERT   ecx

                push    esi

                mov     [(window edi).dSign], WINDOW_SIGN
                mov     [(window edi).dWin], esi
                mov     [(window edi).dRow], eax
                mov     [(window edi).dCol], ebx
                mov     [(window edi).dParent], ecx
                mov     [(window edi).dCursorIsOver], 0
                mov     [(window edi).dCursorIsIn], 0
                mov     [(window edi).dNotInBuffer], 1
                mov     [(window edi).fnEventHnd], edx

                mov     esi, edi
                call    ListNew
                mov     [(window esi).dChildsList], edi
                call    ListNew
                mov     [(window esi).dOverlapping], edi
                mov     edi, esi
                mov     esi, [(window ecx).dChildsList]
                call    ListInsertFirst

                push    edi
                mov     esi, ecx
                call    WindowUpdateChildsOverlapping
                call    WindowUpdateCursorPosition
                pop     edi

                mov     esi, edi
                call    WindowUpdate

                pop     esi
                ret
endp



proc            WindowDelete
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                pushad
                WINDOW_ASSERT   esi

                call    WindowsBeginUpdate

                call    WindowGetRect
                call    WallpaperFillRect
                call    WindowsMarkRect
                call    BufferInvalidateRect

                mov     edi, esi
                mov     ecx, [(window edi).dParent]
                mov     esi, [(window edi).dChildsList]
                call    ListDelete
                mov     esi, [(window edi).dOverlapping]
                call    ListDelete
                mov     esi, [(window ecx).dChildsList]
                call    ListRemove
                mov     eax, edi
                call    HeapFree
                mov     esi, ecx
                call    WindowUpdateChildsOverlapping
                call    WindowUpdateCursorPosition

                call    WindowsEndUpdate

                popad
                ret
endp



proc            WindowSetOwner
                ;in
                ;  eax - 'Window' owner object
                ;  esi - 'Window'
                ;out
                ;  nothing
                WINDOW_ASSERT   esi
                mov     [(window esi).dOwner], eax
                ret
endp



proc            WindowMakeModal
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                push    esi
                WINDOW_ASSERT   esi

                cmp     esi, [dPointedWindow]
                @IF_NE
                  push  esi
                  mov   al, EV_CURS_OUT
                  mov   esi, [dPointedWindow]
                  call  WindowPassEvent
                  pop   esi
                @ENDIF

                mov     eax, esi
                mov     esi, [dModalsStack]
                call    StackPush

                pop     esi
                ret
endp



proc            WindowMakeNotModal
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                push    esi
                WINDOW_ASSERT   esi

                push    esi
                mov     esi, [dModalsStack]
                call    StackPop
                pop     esi

                cmp     esi, [dPointedWindow]
                @IF_NE
                  mov   al, EV_CURS_IN
                  mov   esi, [dPointedWindow]
                  call  WindowPassEvent
                @ENDIF

                lea     esi, [sParentWindow]
                call    WindowUpdateCursorPosition

                pop     esi
                ret
endp



proc            WindowGetWin
                ;in
                ;  esi - 'Window'
                ;out
                ;  edi - 'Win'
                mov     edi, [(window esi).dWin]
                ret
endp



proc            WindowUpdate
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Copies 'Window' to buffer or to screen through buffer
                ;  or to screen directly.
                WINDOW_ASSERT   esi

                push    ebx esi edi

                ;;check if we are inside update process
                cmp     [bWindowsUpdate], 0
                @IF_NE
                  call  WindowMarkWithOverlapping
                  call  WindowGetRect
                  call  BufferInvalidateRect
                  jmp   @@exit
                @ENDIF

                ;;check if 'Window' is overlapped by cursor
                mov     ebx, [(window esi).dCursorIsOver]
                or      ebx, [(window esi).dCursorIsIn]
                @IF_NZ
                  call  WindowsBeginUpdate
                  call  WindowMarkWithOverlapping
                  call  WindowGetRect
                  call  BufferInvalidateRect
                  call  WindowsEndUpdate
                  jmp   @@exit
                @ENDIF

                ;;check if 'Window' is overlapped by another window or
                ;;have child windows
                push    esi
                xor     ebx, ebx
@@loop1:        mov     edi, esi
                mov     esi, [(window edi).dOverlapping]
                call    ListIsEmpty
                setc    bh
                or      bl, bh
                mov     esi, [(window edi).dParent]
                and     esi, esi
                jnz     @@loop1
                pop     esi
                ;;
                push    esi
                mov     esi, [(window esi).dChildsList]
                call    ListIsEmpty
                setc    bh
                or      bl, bh
                pop     esi
                ;;
                @IF_NZ
                  call  WindowsBeginUpdate
                  call  WindowMarkWithOverlapping
                  call  WindowGetRect
                  call  BufferInvalidateRect
                  call  WindowsEndUpdate
                  jmp   @@exit
                @ENDIF

                ;;'Window' is not overlapped
                mov     [(window esi).dNotInBuffer], 1
                call    WindowGetRowCol
                mov     esi, [(window esi).dWin]
                call    SvgaWinToScr

@@exit:         pop     edi esi ebx
                ret
endp



proc            WindowMove
                ;in
                ;  eax - row offset
                ;  ebx - column offset
                ;  esi - 'Window'
                ;out
                ;  nothing
                pushad
                WINDOW_ASSERT   esi

                pushad
                call    WindowGetRect
                call    WallpaperFillRect
                call    WindowsMarkRect
                call    BufferInvalidateRect
                popad

                add     [(window esi).dRow], eax
                add     [(window esi).dCol], ebx
                call    WindowGetRect
                call    BufferInvalidateRect
                push    esi
                mov     esi, [(window esi).dParent]
                call    WindowUpdateChildsOverlapping
                pop     esi
                call    WindowMarkWithOverlapping

                popad
                ret
endp



proc            WindowGetCursorState
                ;in
                ;  esi - 'Window'
                ;out
                ;  eax - cursor row in 'Window'
                ;  ebx - cursor column in 'Window'
                ;   cl - left mouse button state
                ;   ch - right mouse button state
                test    [bButtons], 1
                setnz   cl
                test    [bButtons], 2
                setnz   ch
                call    WindowGetRowCol
                sub     eax, [dCursPoint.dRow]
                neg     eax
                sub     ebx, [dCursPoint.dCol]
                neg     ebx
                ret
endp



proc            WindowUpdateChildsOverlapping
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Recalculates overlapped windows lists of child windows.
                pushad

                ;;go through child windows list from bottom to top
                mov     esi, [(window esi).dChildsList]
                call    ListGetPosition
                push    eax
                call    ListLast
@@loop1:        @IF_NC
                  push  esi

                  ;;clear overlapped windows list
                  mov   edi, esi
                  mov   [dWindow1], eax                 ;current 'Window'
                  mov   esi, [(window eax).dOverlapping]
                  call  ListZap
                  mov   esi, edi

                  ;;go through windows that are upper than current
                  call  ListGetPosition
                  push  eax
@@loop2:          call  ListPrevious
                  @IF_NC
                    push esi
                    mov [dWindow2], eax                 ;compared 'Window'

                    ;;check windows overlapping
                    mov  esi, [dWindow1]
                    call WindowGetRect
                    push edi
                    mov  esi, [dWindow2]
                    call WindowGetRect
                    pop  esi
                    call RectIsIntersect
                    pushf
                    call RectDelete
                    mov  esi, edi
                    call RectDelete
                    popf
                    @IF_NC                              ;overlapped
                      mov  esi, [dWindow1]
                      mov  esi, [(window esi).dOverlapping]
                      mov  edi, [dWindow2]
                      call ListInsertFirst
                    @ENDIF

                    pop esi
                    jmp @@loop2
                  @ENDIF
                  pop   eax
                  call  ListSetPosition

                  pop   esi
                  call  ListPrevious
                  jmp   @@loop1
                @ENDIF
                pop     eax
                call    ListSetPosition

                popad
                ret
endp



proc            WindowUpdateCursorPosition
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Updates cursor position for 'Window' and all it's
                ;  child windows.
                WINDOW_ASSERT   esi
                pushad
                mov     bl, 0
                call    _WindowUpdateCursorPosition
                popad
                ret
endp

proc            _WindowUpdateCursorPosition
                push    edi

                ;;check if cursor is over the 'Window'
                push    esi
                call    WindowGetRect
                push    edi
                call    CursorGetRect
                pop     esi
                call    RectIsIntersect
                pushf
                call    RectDelete
                mov     esi, edi
                call    RectDelete
                popf
                pop     esi
                @IF_NC                  ;cursor is over
                  cmp   [(window esi).dCursorIsOver], 1
                  mov   [(window esi).dCursorIsOver], 1
                  @IF_NE                ;previously it wasn't there
                    cmp [(window esi).dNotInBuffer], 1
                    @IF_E
                      call WindowMarkWithOverlapping
                    @ENDIF
                  @ENDIF
                @ELSE
                  mov   [(window esi).dCursorIsOver], 0
                @ENDIF

                ;;go through child windows list from top to bottom
                push    esi
                mov     esi, [(window esi).dChildsList]
                call    ListGetPosition
                push    eax
                call    ListFirst
@@loop:         @IF_NC
                  push  esi
                  mov   esi, eax
                  call  _WindowUpdateCursorPosition
                  pop   esi
                  call  ListNext
                  jmp   @@loop
                @ENDIF
                pop     eax
                call    ListSetPosition
                pop     esi

                ;;check if cursor points to 'Window'
                and     bl, bl
                jnz     @@cursor_out
                push    esi
                call    WindowGetRect
                push    edi
                call    CursorGetPoint
                pop     esi
                call    RectIsPointIn
                pushf
                call    RectDelete
                popf
                pop     esi
                @IF_NC                  ;cursor in 'Window'
                  mov   bl, 1
                  mov   [dPointedWindow], esi
                  cmp   [(window esi).dCursorIsIn], 1
                  @IF_NE                ;previously it wasn't there
                    mov  al, EV_CURS_IN
                    call WindowPassEvent
                    mov  [(window esi).dCursorIsIn], 1
                  @ELSE
                    mov  al, EV_CURS_MOVE
                    call WindowPassEvent
                  @ENDIF
                @ELSE                   ;cursor is not in 'Window'
@@cursor_out:     cmp [(window esi).dCursorIsIn], 1
                  @IF_E                 ;previously it was there
                    mov  al, EV_CURS_OUT
                    call WindowPassEvent
                    mov  [(window esi).dCursorIsIn], 0
                  @ENDIF
                @ENDIF

@@exit:         pop     edi
                ret
endp



proc            WindowUpdateClick
                ;in
                ;  al - mouse button event
                ;out
                ;  nothing
                ;description
                ;  Passes event to pointed 'Window'.
                push    esi
                mov     esi, [dPointedWindow]
                call    WindowPassEvent
                pop     esi
                ret
endp



proc            WindowGetRowCol
                ;in
                ;  esi - 'Window'
                ;out
                ;  eax - row
                ;  ebx - column
                ;description
                ;  Returns absolute 'Window' coordinates.
                WINDOW_ASSERT   esi
                push    esi
                xor     eax, eax
                xor     ebx, ebx
@@loop:         add     eax, [(window esi).dRow]
                add     ebx, [(window esi).dCol]
                mov     esi, [(window esi).dParent]
                and     esi, esi
                jnz     @@loop
                pop     esi
                ret
endp



proc            WindowGetRect
                ;in
                ;  esi - 'Window'
                ;out
                ;  edi - 'Rect'
                ;description
                ;  Creates 'Rect' with visible window area coordinates.
                push    ebx
                WINDOW_ASSERT   esi

                mov     ebx, [(window esi).dWin]

                ;;create 'Rect'
                call    RectNew
                mov     eax, [(win ebx).dVisibleRow]
                mov     [(rect edi).dTotalRow], eax
                mov     ebx, [(win ebx).dVisibleCol]
                mov     [(rect edi).dTotalCol], ebx
                call    WindowGetRowCol
                ;;top clip
                and     eax, eax
                @IF_S
                  add   [(rect edi).dTotalRow], eax
                  @IF_S
                    mov [(rect edi).dTotalRow], 0
                  @ENDIF
                  xor   eax, eax
                @ENDIF
                mov     [(rect edi).dRow], eax
                ;;left clip
                and     ebx, ebx
                @IF_S
                  add   [(rect edi).dTotalCol], ebx
                  @IF_S
                    mov [(rect edi).dTotalCol], 0
                  @ENDIF
                  xor   ebx, ebx
                @ENDIF
                mov     [(rect edi).dCol], ebx
                ;;bottom clip
                mov     ebx, [dBufferWin]
                mov     eax, [(rect edi).dRow]
                add     eax, [(rect edi).dTotalRow]
                sub     eax, [(win ebx).dVisibleRow]
                @IF_A
                  sub   [(rect edi).dTotalRow], eax
                @ENDIF
                ;;right clip
                mov     eax, [(rect edi).dCol]
                add     eax, [(rect edi).dTotalCol]
                sub     eax, [(win ebx).dVisibleCol]
                @IF_A
                  sub   [(rect edi).dTotalCol], eax
                @ENDIF

                pop     ebx
                ret
endp



proc            WindowMarkWithChilds
                ;in
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Marks 'Window' with all it's childs to be copied to buffer.
                push    esi
                mov     [(window esi).dNeedUpdate], 1

                ;;mark child windows
                mov     esi, [(window esi).dChildsList]
                call    ListGetPosition
                push    eax
                call    ListLast
@@loop:         @IF_NC
                  push  esi
                  mov   esi, eax
                  call  WindowMarkWithChilds
                  pop   esi
                  call  ListPrevious
                  jmp   @@loop
                @ENDIF
                pop     eax
                call    ListSetPosition

                pop     esi
                ret
endp



proc            WindowMarkWithOverlapping
                ;in
                ;  esi - 'Windwow'
                ;out
                ;  nothing
                ;description
                ;  Marks 'Window' with all it's childs and overlapping windows
                ;  to be copied to buffer.
                push    esi edi
                call    WindowMarkWithChilds

                ;;mark overlapping windows
                mov     edi, esi
@@loop1:        mov     esi, [(window edi).dOverlapping]
                call    ListGetPosition
                push    eax
                call    ListFirst
@@loop2:        @IF_NC
                  push  esi
                  mov   esi, eax
                  call  WindowMarkWithOverlapping
                  pop   esi
                  call  ListNext
                  jmp   @@loop2
                @ENDIF
                pop     eax
                call    ListSetPosition
                mov     edi, [(window edi).dParent]
                and     edi, edi
                jnz     @@loop1

@@exit:         pop     edi esi
                ret
endp



proc            WindowPassEvent
                ;in
                ;  al  - event code
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Passes event to 'Window' if there is no modal windows
                ;  or 'Window' is current modal window or his child. In
                ;  other case passes event to current modal window.
                pushad

                mov     ebx, eax
                mov     ecx, esi

                ;;eax - current modal window
                mov     esi, [dModalsStack]
                call    StackRead
                setc    dl

                ;;compare 'Window' and its parents with modal window
                @IF_NC
                  push  ecx
@@loop1:          cmp   ecx, eax
                  @IF_NZ
                    mov ecx, [(window ecx).dParent]
                    and ecx, ecx
                    jnz short @@loop1
                    dec ecx
                  @ENDIF
                  pop   ecx
                @ENDIF
                setz    dh

                or      dl, dh
                @IF_NZ
                  ;;no modal windows or 'Window' is current modal window
                  mov   esi, ecx
                @ELSE
                  ;;pass event to current modal window
                  mov   esi, eax
                @ENDIF
                mov     eax, ebx
                push    esi
                mov     esi, [(window esi).dParent]
                call    WindowPassEventToParents
                pop     esi

                mov     eax, ebx
                mov     edi, [(window esi).dOwner]
                call    [(window esi).fnEventHnd]

                popad
                ret
endp



proc            WindowPassEventToParents
                ;in
                ;  al  - event code
                ;  esi - 'Window'
                ;out
                ;  nothing
                ;description
                ;  Passes event to all parent windows of 'Window'.
                pushad

                or      eax, EV_CLASS_IN_THE_CHILD
                and     esi, esi
                @IF_NZ
                  push  esi
                  mov   esi, [(window esi).dParent]
                  call  WindowPassEventToParents
                  pop   esi
                  mov   edi, [(window esi).dOwner]
                  call  [(window esi).fnEventHnd]
                @ENDIF

                popad
                ret
endp



proc            WindowStdEventHnd
                ;in
                ;  al  - event code
                ;  esi - 'Window'
                ;  edi - 'Window' owner
                ;out
                ;  nothing
                ;description
                ;  Default event handler.
                ret
endp



dMove           dd      0       ;1-move mode
dRow            dd      0       ;cursor row while moving
dCol            dd      0       ;cursor column while moving
proc            WindowMoveEventHnd
                ;in
                ;   al - event code
                ;  esi - 'Window'
                ;out
                ;  nothing
                pushad

                cmp     al, EV_LB_DOWN
                @IF_E
                  mov   [dMove], 1
                  call  WindowMakeModal
                  call  WindowGetCursorState
                  mov   [dRow], eax
                  mov   [dCol], ebx
                  jmp   @@exit
                @ENDIF

                and     al, NOT EV_CLASS_IN_THE_CHILD
                cmp     al, EV_LB_UP
                @IF_E
                  mov   [dMove], 0
                  call  WindowMakeNotModal
                  jmp   @@exit
                @ENDIF

                cmp     [dMove], 1
                @IF_E
                  call  WindowGetCursorState
                  sub   eax, [dRow]
                  sub   ebx, [dCol]
                  call  WindowMove
                @ENDIF

@@exit:         popad
                ret
endp





;;
;                              CURSOR: CODE                                ;
;;

proc            CursorInit
                ;in
                ;  esi - cursor picture
                ;out
                ;  nothing

                mov     eax, [(pic esi).dTotalRow]
                mov     ebx, [(pic esi).dTotalCol]
                mov     ecx, [(pic esi).dColorScheme]
                call    PicNew
                mov     [dCursPic], esi
                mov     [dSavePic], edi

                ;;create cursor sprite
                xor     eax, eax
                call    SprNewFromPic
                mov     [dCursSpr], edi

                ;;initialize mouse driver
                call    MouseInit
                call    SvgaGetInfo
                call    CursorSetMouseLimits

                ;;setup initial 'Cursor' position
                call    SvgaGetInfo
                shr     eax, 1
                shr     ebx, 1
                mov     [dCursPoint.dRow], eax
                mov     [dCursPoint.dCol], ebx
                call    CursorSetMousePosition

                ;;show 'Cursor'
                call    CursorShow

                ret
endp



proc            CursorRestore
                ;in
                ;  esi - cursor picture
                ;out
                ;  nothing
                mov     esi, [dSavePic]
                call    PicDelete
                ret
endp



proc            CursorSetMouseLimits
                ;in
                ;  eax - total number of rows
                ;  ebx - total number of columns
                ;out
                ;  nothing
                pushad

                mov     edx, eax
                mov     eax, 0          ;eax - minimal column
                shl     ebx, 3          ;ebx - maximal column
                dec     ebx
                mov     ecx, 0          ;ecx - minimal row
                shl     edx, 3          ;edx - maximal row
                dec     edx
                call    MouseLimits

                ;;set mouse sensitivity
                mov     ax, 0Fh
                mov     cx, 1
                mov     dx, 1
                int     33h

                popad
                ret
endp



proc            CursorGetMouseState
                ;in
                ;  nothing
                ;out
                ;  eax - row
                ;  ebx - column
                ;  cl  - buttons state
                ;descroption
                ;  Returns mouse position and state.
                call    MouseState
                shr     eax, 3
                shr     ebx, 3
                ret
endp



proc            CursorSetMousePosition
                ;in
                ;  eax - row
                ;  ebx - column
                ;out
                ;  nothing
                shl     eax, 3
                shl     ebx, 3
                call    MouseSetPosition
                ret
endp



proc            CursorGetPoint
                ;in
                ;  nothing
                ;out
                ;  edi - 'Point'
                lea     edi, [dCursPoint]
                ret
endp



proc            CursorGetRect
                ;in
                ;  nothing
                ;out
                ;  edi - 'Rect'
                push    esi

                call    RectNew
                mov     eax, [dCursPoint.dRow]
                mov     [(rect edi).dRow], eax
                mov     eax, [dCursPoint.dCol]
                mov     [(rect edi).dCol], eax
                mov     esi, [dCursPic]
                mov     eax, [(pic esi).dTotalRow]
                mov     [(rect edi).dTotalRow], eax
                mov     eax, [(pic esi).dTotalCol]
                mov     [(rect edi).dTotalCol], eax

                pop     esi
                ret
endp



proc            CursorUpdate
                ;in
                ;  nothing
                ;out
                ;  nothing
                ;description
                ;  Called by user to update 'Cursor' and whole 'Windows' state
                pushad

                ;;check if 'Cursor' state changed
                call    CursorGetMouseState
                cmp     eax, [dCursPoint.dRow]
                setne   dl
                cmp     ebx, [dCursPoint.dCol]
                setne   dh
                or      dl, dh
                cmp     cl, [bButtons]
                setne   dh
                and     dx, dx
                jnz     @@state_chng

                ;;check for left button autorepeat
                test    [bButtons], 1
                @IF_NZ
                  ;;left button depressed
                  cmp     [bAutorepeat], 0
                  @IF_E
                    ;;no autorepeat yet
                    call  TimerGetSystemTicks
                    cmp   eax, [dTimer]
                    @IF_AE
                      sub eax, [dTimer]
                    @ENDIF
                    cmp   eax, 5
                    @IF_A
                      mov [bAutorepeat], 1
                    @ENDIF
                  @ELSE
                    ;;autorepeat in progress
                    call  TimerGetSystemTicks
                    cmp   eax, [dTimer]
                    ;@IF_NE
                      mov  [dTimer], eax
                      call WindowsBeginUpdate
                      mov  al, EV_LB_DOWN_REP
                      lea  esi, [sParentWindow]
                      call WindowUpdateClick
                      call WindowsEndUpdate
                    ;@ENDIF
                  @ENDIF
                @ENDIF
                jmp      @@exit

                ;;state of 'Cursor' changed - begin updates
@@state_chng:   mov     [bAutorepeat], 0

                push    eax
                call    WindowsBeginUpdate
                pop     eax

                ;;check if mouse position changed
                and     dl, dl
                @IF_NZ
                  mov   [dCursPoint.dRow], eax
                  mov   [dCursPoint.dCol], ebx
                  lea   esi, [sParentWindow]
                  call  WindowUpdateCursorPosition
                @ENDIF

                ;;check if left button state changed
                test    cl, 1
                setnz   al
                test    [bButtons], 1
                setnz   ah
                cmp     al, ah
                @IF_NE
                  and   al, al
                  mov   al, EV_LB_UP
                  @IF_NZ
                    call TimerGetSystemTicks
                    mov  [dTimer], eax
                    mov  al, EV_LB_DOWN
                  @ENDIF
                  lea   esi, [sParentWindow]
                  call  WindowUpdateClick
                @ENDIF

                ;;check if right button state changed
                test    cl, 2
                setnz   al
                test    [bButtons], 2
                setnz   ah
                cmp     al, ah
                @IF_NE
                  and   al, al
                  mov   al, EV_RB_UP
                  @IF_NZ
                    mov al, EV_RB_DOWN
                  @ENDIF
                  lea   esi, [sParentWindow]
                  call  WindowUpdateClick
                @ENDIF

                ;;store buttons state
                mov     [bButtons], cl

                ;;finish updates
                call    WindowsEndUpdate

@@exit:         popad
                ret
endp



proc            CursorShow
                pushad

                mov     eax, [dCursPoint.dRow]
                mov     ebx, [dCursPoint.dCol]
                mov     esi, [dSavePic]
                mov     edi, [dBufferWin]
                call    PicFromPic

                mov     eax, [dCursPoint.dRow]
                mov     ebx, [dCursPoint.dCol]
                mov     esi, [dCursSpr]
                mov     edi, [dBufferWin]
                call    SprToPic

                popad
                ret
endp



proc            CursorHide
                pushad
                mov     eax, [dCursPoint.dRow]
                mov     ebx, [dCursPoint.dCol]
                mov     esi, [dSavePic]
                mov     edi, [dBufferWin]
                call    PicToPic
                popad
                ret
endp





;;
;                            WINDOWS: CODE                                 ;
;;

proc            WindowsInit
                ;in
                ;  esi - wallpaper picture
                ;  edi - cursor picture
                ;out
                ;  nothing
                ;description
                ;  Initialises windows system.
                push    esi edi

                call    BufferInit
                call    WallpaperInit
                mov     esi, edi
                call    CursorInit
                call    BufferCommitChanges

                call    SvgaGetInfo
                lea     esi, [sParentWin]
                mov     [(win esi).dTotalRow], eax
                mov     [(win esi).dTotalCol], ebx
                mov     [(win esi).dVisibleRow], eax
                mov     [(win esi).dVisibleCol], ebx

                lea     esi, [sParentWindow]
                mov     [(window esi).dWin], offset sParentWin
                call    ListNew
                mov     [(window esi).dChildsList], edi
                call    ListNew
                mov     [(window esi).dOverlapping], edi

                lea     edi, [WindowStdEventHnd]
                mov     [(window esi).fnEventHnd], edi
                mov     [(window esi).dCursorIsIn], 1
                mov     [dPointedWindow], esi

                call    StackNew
                mov     [dModalsStack], edi

                pop     edi esi
                ret
endp



proc            WindowsRestore
                push    esi edi

                mov     esi, [dModalsStack]
                call    StackDelete

                lea     edi, [sParentWindow]
                mov     esi, [(window edi).dChildsList]
                call    ListDelete
                mov     esi, [(window edi).dOverlapping]
                call    ListDelete

                call    CursorRestore
                call    BufferRestore

                pop     edi esi
                ret
endp



proc            WindowsBeginUpdate
                ;in
                ;  nothing
                ;out
                ;  nothing
                ;description
                ;  Starts series of updates of 'Windows' system.
                cmp     [bWindowsUpdate], 0
                @IF_E
                  ;;hide 'Cursor'
                  call  CursorGetRect
                  call  BufferInvalidateRect
                  call  CursorHide
                @ENDIF

                inc     [bWindowsUpdate]
                ret
endp



proc            WindowsEndUpdate
                ;in
                ;  nothing
                ;out
                ;  nothing
                ;description
                ;  Ends series of updates of 'Windows' system.
                push    esi

                dec     [bWindowsUpdate]
                @IF_Z
                  call  CursorGetRect
                  call  BufferInvalidateRect
                  lea   esi, [sParentWindow]
                  call  @@Recurse
                  call  CursorShow
                  call  BufferCommitChanges
                @ENDIF
                pop     esi
                ret

                ;;copy all windows with 'dNeedUpdate' set to buffer
@@Recurse:      cmp     [(window esi).dNeedUpdate], 1
                @IF_E
                  push  esi
                  mov   [(window esi).dNeedUpdate], 0
                  mov   [(window esi).dNotInBuffer], 0
                  call  WindowGetRowCol
                  mov   esi, [(window esi).dWin]
                  mov   edi, [dBufferWin]
                  call  WinToWin
                  pop   esi
                @ENDIF
                mov     esi, [(window esi).dChildsList]
                call    ListLast
@@loop:         @IF_NC
                  push  esi
                  mov   esi, eax
                  call  @@Recurse
                  pop   esi
                  call  ListPrevious
                  jmp   short @@loop
                @ENDIF
                ret
endp



proc            WindowsMarkRect
                ;in
                ;  edi - 'Rect'
                ;out
                ;  nothing
                ;description
                ;  Marks all windows intersected by 'Rect' as to be updated.
                pushad
                mov     ebx, edi
                mov     esi, [(window sParentWindow).dChildsList]
                call    ListGetPosition
                push    eax
                call    ListLast
@@loop:         @IF_NC
                  mov   ecx, esi
                  mov   edx, eax
                  mov   esi, eax
                  call  WindowGetRect
                  mov   esi, ebx
                  call  RectIsIntersect
                  @IF_NC
                    mov  esi, edx
                    call WindowMarkWithOverlapping
                  @ENDIF
                  mov   esi, edi
                  call  RectDelete
                  mov   esi, ecx
                  call  ListPrevious
                  jmp   @@loop
                @ENDIF
                pop     eax
                call    ListSetPosition
                popad
                ret
endp

ends            code32
                end
