;*****
; timer.asm - system timer handling.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __TIMER__ equ 1
                include "pmlib.asd"
                include "timer.asi"



dUsrTimerHnd    dd      0       ;user handler address
fDosTimerHnd    df      0       ;DOS handler address
wUsrTimerCnt    dw      0       ;
wDosTimerCnt    dw      0       ;



proc            TimerSetHandler
                ;in
                ;  eax - user handler address
                ;out
                ;  nothing
                pushad

                mov     [dUsrTimerHnd], eax

                ;;save DOS handler
                mov     ax, 0204h
                mov     bl, 08h                         ;IRQ number
                int     31h
                mov     [wptr (fDosTimerHnd+4)], cx     ;selector
                mov     [dptr (fDosTimerHnd)], edx      ;offset

                ;;wait while DOS handler stops motor
                mov     edi, 440h
                cmp     [bptr fs:edi], 2
                @IF_A
                  mov   [bptr fs:edi], 2
                @ENDIF
@@loop1:        cmp     [bptr fs:edi], 0
                jnz     @@loop1

                call    TimerHandler

                popad
                ret
endp            TimerSetHandler



proc            TimerSetFrequency
                ;in
                ;  eax - new interrupt frequency in Hz.
                ;out
                ;  nothing
                pushad

                ;;find new counter value
                mov     ebx, eax
                xor     edx, edx
                mov     eax, 1193280
                div     ebx

                ;;store it
                mov     [wUsrTimerCnt], ax
                mov     ebx, eax

                ;;reprogramm timer
                mov     al, 00110110b           ;SC=00 RW=11 Mode=011 BCD=0
                out     43h, al
                jmp     short $+2
                mov     al, bl
                out     40h, al
                jmp     short $+2
                mov     al, bh
                out     40h, al

                call    TimerHandler

                popad
                ret
endp            TimerSetFrequency



proc            TimerRestore
                ;in
                ;  nothing
                ;out
                ;  nothing
                pushad

                ;;reprogramm timer as DOS does it
                mov     al, 00110110b    ;SC=00 RW=11 Mode=011 BCD=0
                out     43h, al
                jmp     short $+2
                xor     eax, eax
                out     40h, al
                jmp     short $+2
                out     40h, al

                ;;restore DOS handler
                mov     ax, 0205h                       ;set handler
                mov     bl, 08h                         ;IRQ number
                mov     cx, [wptr (fDosTimerHnd+4)]     ;selector
                mov     edx, [dptr (fDosTimerHnd)]      ;offset
                int     31h

                xor     eax, eax
                mov     [wUsrTimerCnt], ax
                mov     [wDosTimerCnt], ax

                popad
                ret
endp            TimerRestore



proc            TimerGetSystemTicks
                ;in
                ;  nothing
                ;out
                ;  eax - system timer value
                push    ecx edx

                xor     eax, eax
                int     1Ah
                push    cx
                push    dx
                pop     eax

                pop     edx ecx
                ret
endp



proc            TimerGetTimeStamp
                ;in
                ;  nothing
                ;out
                ;  ax - channel 0 counter value
                cli

                ;;latch the count register
                xor     al, al
                out     43h, al
                jmp     $+2
                jmp     $+2

                ;;read low byte
                in      al, 40h
                mov     ah, al
                jmp     $+2
                jmp     $+2

                ;;read hi byte
                in      al, 40h
                xchg    al, ah

                sti
                ret
endp



;****************************************************************************;
;                            LOCAL METHODS                                   ;
;****************************************************************************;



proc            TimerHandler
                pushad

                ;;select handler
                lea     edx, [TimerHandlerStd]
                cmp     [wUsrTimerCnt], 0
                @IF_NE
                  lea   edx, [TimerHandlerAlt]
                @ENDIF

                ;;set handler
                mov     ax, 0205h
                mov     bl, 08h                         ;IRQ number
                mov     cx, cs
                int     31h

                popad
                ret
endp            TimerHandler



proc            TimerHandlerStd
                ;description
                ;  Handler for standard frequency.
                push    eax edi
                call    [dUsrTimerHnd]
                mov     edi, 46Ch       ;current DOS time
                inc     [dptr fs:edi]
                cmp     [dptr fs:edi], 1800B0h
                @IF_AE
                  xor   eax, eax
                  mov   [fs:edi], eax
                  mov   edi, 470h       ;midnight cross flag
                  inc   [bptr fs:edi]
                @ENDIF
                mov     al, 20h
                out     20h, al
                pop     edi eax
                sti
                iretd
endp            TimerHandlerStd



proc            TimerHandlerAlt
                ;description
                ;  Handler for altered frequency.
                push    eax
                call    [dUsrTimerHnd]
                mov     ax, [wUsrTimerCnt]
                sub     [wDosTimerCnt], ax
                @IF_C
                  push  edi
                  mov   edi, 46Ch       ;current DOS time
                  inc   [dptr fs:edi]
                  cmp   [dptr fs:edi], 1800B0h
                  @IF_AE
                    xor eax, eax
                    mov [fs:edi], eax
                    mov edi, 470h       ;midnight cross flag
                    inc [bptr fs:edi]
                  @ENDIF
                  pop   edi
                @ENDIF
                mov     al, 20h
                out     20h, al
                pop     eax
                sti
                iretd
endp            TimerHandlerAlt

ends            code32
                end
