;*****
; svga.asm  -  SVGA handling.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __SVGA__ equ 1
                include "pmlib.asd"
                include "api.asi"
                include "color.asi"
                include "debug.asi"
                include "error.asi"
                include "svga.asi"
                include "win.asi"



dModeNum        dd      ?       ;mode number
dYSize          dd      ?       ;vertical screen size
dXSize          dd      ?       ;horizontal screen size
dColorScheme    dd      ?       ;color scheme
dBytesPerCell   dd      ?       ;
dXSizeInBytes   dd      ?       ;
dWindowGran     dd      ?       ;SVGA-window granularity (in bytes)
dWindowSize     dd      ?       ;SVGA-window size
dWindowAddr     dd      ?       ;SVGA-window linear address

RUS_MESSAGE     cErrNoSvga      "SVGA   ."
ENG_MESSAGE     cErrNoSvga      "SVGA video card not found."
RUS_MESSAGE     cErrNoSvgaMode  "  ন 㪠 ०."
ENG_MESSAGE     cErrNoSvgaMode  "Video card does not support requested mode."



proc            SvgaInit
                ;in
                ;  eax - number of rows
                ;  ebx - number of columns
                ;  ecx - color scheme
                ;out
                ;  nothing
                pushad

                ;;store video mode data
                mov     [dYSize], eax
                mov     [dXSize], ebx
                mov     [dColorScheme], ecx
                mov     eax, [adBytesPerCell+ecx*4]
                mov     [dBytesPerCell], eax
                mul     ebx
                mov     [dXSizeInBytes], eax

                ;;query SVGA support
                lea     edi, [sVregs]
                mov     [(vr edi).eax], 4f00h
                mov     ax, [wLowBufferSeg]
                mov     [(vr edi).es], ax
                mov     ax, [wLowBufferOff]
                mov     [(vr edi).di], ax
                mov     eax, 300h               ;real mode call
                mov     ebx, 10h                ;int 10h
                xor     ecx, ecx                ;no parameters on stack
                mov     [(vr edi).spss], ecx    ;PMODE will provide stack
                int     31h
                cmp     [(vr edi).ax], 004fh
                @IF_NE
                  lea   esi, [cErrNoSvga]
                  call  ErrorFatal
                @ENDIF

                ;;get pointer to mode list
                lea     edi, [uLowBuffer]
                movzx   eax, [(SvgaInfoBlock edi).wModesSeg]
                shl     eax, 4
                movzx   ebx, [(SvgaInfoBlock edi).wModesOff]
                add     eax, ebx
                sub     eax, [dBaseAddr]
                mov     esi, eax

                ;;find video mode
@@loop:         movzx   eax, [word esi]
                add     esi, 2
                mov     [dModeNum], eax
                cmp     ax, 0FFFFh
                @IF_E
                  lea   esi, [cErrNoSvgaMode]
                  call  ErrorFatal
                @ENDIF
                ;;query video mode data
                lea     edi, [sVregs]
                mov     [(vr edi).eax], 4f01h
                mov     [(vr edi).ecx], eax
                mov     ax, [wLowBufferSeg]
                mov     [(vr edi).es], ax
                mov     ax, [wLowBufferOff]
                mov     [(vr edi).di], ax
                mov     eax, 300h               ;real mode call
                mov     ebx, 10h                ;int 10h
                xor     ecx, ecx                ;no parameters on stack
                mov     [(vr edi).spss], ecx    ;PMODE will provide stack
                int     31h
                ;;compare to requested parameters
                lea     edi, [uLowBuffer]
                movzx   eax, [(ModeInfoBlock edi).wYResolution]
                cmp     eax, [dYSize]
                jne     @@loop
                movzx   eax, [(ModeInfoBlock edi).wXResolution]
                cmp     eax, [dXSize]
                jne     @@loop
                mov     al, [(ModeInfoBlock edi).bBitsPerPixel]
                cmp     al, 8
                jb      @@loop
                call    ColorScheme
                cmp     eax, [dColorScheme]
                jne     @@loop

                ;;mode found - set current color scheme
                call    ColorSchemeSet

                ;;mode found - store window data
                movzx   eax, [(ModeInfoBlock edi).wWinGranularity]
                shl     eax, 10
                mov     [dWindowGran], eax
                movzx   eax, [(ModeInfoBlock edi).wWinSize]
                shl     eax, 10
                mov     [dWindowSize], eax
                movzx   eax, [(ModeInfoBlock edi).wWinASegment]
                shl     eax, 4
                sub     eax, [dBaseAddr]
                mov     [dWindowAddr], eax

                ;;Set mode. Do not clear video memory
                mov     eax, 4f02h
                mov     ebx, [dModeNum]
                or      ebx, 8000h
                int     10h

                ;;clear screen
                call    SvgaCls

                popad
                ret
endp



proc            SvgaRestore
                ;in
                ;  nothing
                ;out
                ;  nothing
                mov     eax, 03h
                int     10h
                ret
endp



proc            SvgaGetInfo
                ;in
                ;  nothing
                ;out
                ;  eax - number of rows
                ;  ebx - number of columns
                ;  ecx - color scheme
                mov     eax, [dYSize]
                mov     ebx, [dXSize]
                mov     ecx, [dColorScheme]
                ret
endp



proc            SvgaCls
                ;in
                ;  nothing
                ;out
                ;  nothing
                pushad

                ;;initialize variables
                mov     eax, [dYSize]
                mul     [dXSizeInBytes]
                mov     ebp, eax
                xor     esi, esi

@@loop:         ;;window positioning
                mov     eax, esi
                cmp     eax, ebp
                jae     short @@exit
                xor     edx, edx
                div     [dWindowGran]
                mov     edx, eax
                xor     ebx, ebx
                mov     eax, 4F05h
                int     10h

                ;;window clear
                add     esi, [dWindowSize]
                mov     edi, [dWindowAddr]
                mov     ecx, [dWindowSize]
                shr     ecx, 2
                xor     eax, eax
                rep     stosd

                jmp     short @@loop

@@exit:         popad
                ret
endp



dCurrentRow     dd      0               ;current screen row
dLastRow        dd      0               ;last+1 screen row
dBreakRow       dd      0               ;row of SVGA-window end
dCurrentCol     dd      0               ;(current initial screen column)*BPP
dVisibleCol     dd      0               ;(number of points to copy)*BPP
dBreakCol       dd      0               ;(column of SVGA-window end)*BPP
dScrDelta       dd      0               ;value to add to skip to next line
dWinDelta       dd      0               ;value to add to skip to next line

macro           @FAST_MOVS
                mov     eax, ecx
                shr     ecx, 2
                rep     movsd
                mov     ecx, eax
                and     ecx, 3
                rep     movsb
endm

proc            SvgaWinToScr
                ;in
                ;  eax - row
                ;  ebx - column
                ;  esi - 'Win'
                ;out
                ;  nothing
                ;description
                ;  Copies visible area to screen.

                ;;save registers
                pushad
                push    [(win esi).dVisibleRow]
                push    [(win esi).dVisibleCol]

                ;;validate values
                cmp     eax, [dYSize]
                jae     @@exit
                cmp     ebx, [dXSize]
                jae     @@exit
                mov     ecx, [dYSize]
                sub     ecx, eax
                sub     ecx, [(win esi).dVisibleRow]
                @IF_B
                  add   [(win esi).dVisibleRow], ecx
                @ENDIF
                mov     ecx, [dXSize]
                sub     ecx, ebx
                sub     ecx, [(win esi).dVisibleCol]
                @IF_B
                  add   [(win esi).dVisibleCol], ecx
                @ENDIF

                ;;initialize variables
                mov     [dCurrentRow], eax
                ;;
                add     eax, [(win esi).dVisibleRow]
                mov     [dLastRow], eax
                ;;
                mov     eax, [(win esi).dVisibleCol]
                mul     [dBytesPerCell]
                mov     [dVisibleCol], eax
                ;;
                mov     eax, ebx
                mul     [dBytesPerCell]
                mov     [dCurrentCol], eax
                ;;
                mov     ebx, esi                        ;'Win'
                call    CalculateBreak
                mov     eax, [dXSize]
                sub     eax, [(win ebx).dVisibleCol]
                mul     [dBytesPerCell]
                mov     [dScrDelta], eax
                ;;
                mov     eax, [(win ebx).dTotalCol]
                sub     eax, [(win ebx).dVisibleCol]
                mul     [(win ebx).dBytesPerCell]
                mov     [dWinDelta], eax

                ;;main loop
                mov     eax, [(win ebx).dShiftRow]
                mul     [(win ebx).dTotalCol]
                add     eax, [(win ebx).dShiftCol]
                mul     [(win ebx).dBytesPerCell]
                mov     esi, [(win ebx).dImagePtr]
                add     esi, eax

@@loop1:        mov     eax, [dCurrentRow]
                cmp     eax, [dBreakRow]
                @IF_E
                  call  BreakLine
                @ELSE
                  mov   ecx, [dVisibleCol]
                  @FAST_MOVS
                  add   esi, [dWinDelta]
                  add   edi, [dScrDelta]
                  inc   [dCurrentRow]
                @ENDIF
                mov     eax, [dCurrentRow]
                cmp     eax, [dLastRow]
                jb      @@loop1

                IF      DBG
                  ;;pose window to top for screen grabers to work correctly
                  push  ebx
                  xor   edx, edx
                  xor   ebx, ebx
                  mov   eax, 4F05h
                  int   10h
                  pop   ebx
                ENDIF

                ;;restore registers
@@exit:         pop     [(win ebx).dVisibleCol]
                pop     [(win ebx).dVisibleRow]
                popad

                ret
endp

proc            BreakLine
                ;There are three cases:
                ;  1. Break point is laying before copied line
                ;  2. Break point is laying on copied line
                ;  3. Break point is laying after copied line

                ;;case 1 - switch SVGA-window than copy line
@@case1:        mov     eax, [dBreakCol]
                sub     eax, [dCurrentCol]
                ja      @@case2
                call    CalculateBreak
                mov     ecx, [dVisibleCol]
                @FAST_MOVS
                add     esi, [dWinDelta]
                add     edi, [dScrDelta]
                inc     [dCurrentRow]
                jmp     @@exit

                ;;case 2 - copy part of line, switch SVGA-window, copy rest
                ;;of line
@@case2:        cmp     eax, [dVisibleCol]
                jae     @@case3
                push    eax
                push    [dCurrentCol]
                add     [dCurrentCol], eax
                mov     ecx, eax
                @FAST_MOVS
                call    CalculateBreak
                pop     [dCurrentCol]
                pop     eax
                mov     ecx, [dVisibleCol]
                sub     ecx, eax
                @FAST_MOVS
                add     esi, [dWinDelta]
                add     edi, [dScrDelta]
                inc     [dCurrentRow]
                jmp     @@exit

                ;;case 3 - copy line than switch SVGA-window
@@case3:        mov     ecx, [dVisibleCol]
                @FAST_MOVS
                add     esi, [dWinDelta]
                inc     [dCurrentRow]
                call    CalculateBreak

@@exit:         ret
endp

proc            CalculateBreak
                ;in
                ;  [dCurrentRow]
                ;  [dCurrentCol]
                ;out
                ;  [dBreakRow]
                ;  [dBreakCol]
                ;  edi - ptr to current point in video buffer
                push    ebx edx

                ;;get in eax address of current point in video buffer
                mov     eax, [dCurrentRow]
                mul     [dXSizeInBytes]
                add     eax, [dCurrentCol]

                ;;get in eax window position (in 'dWindowGran')
                ;;get in edx offset of point in video buffer
                div     [dWindowGran]

                ;;get in edi ptr to current point in video buffer
                mov     edi, [dWindowAddr]
                add     edi, edx

                ;;pose a window
                mov     edx, eax
                xor     ebx, ebx
                mov     eax, 4F05h
                push    edx             ;some cards does not save edx.
                int     10h
                pop     edx

                ;;find break point
                mov     eax, [dWindowGran]
                mul     edx
                add     eax, [dWindowSize]

                xor     edx, edx
                div     [dXSizeInBytes]
                mov     [dBreakRow], eax
                mov     eax, edx
                xor     edx, edx
                mov     [dBreakCol], eax

                pop     edx ebx
                ret
endp

ends            code32
                end
