;*****
; sblaster.asm - Sound Blaster handling.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __SBLASTER__ equ 1
                include "pmlib.asd"
                include "dma.asi"
                include "irq.asi"
                include "sblaster.asi"



dIoAddr         dd      0               ;base address
dIrq            dd      0               ;IRQ number
dDma            dd      0               ;DMA channel
dType           dd      0               ;Sound Blaster card type



proc            SbGetSettings
                ;in
                ;  nothing
                ;out
                ;  eax - base address
                ;  ebx - IRQ number
                ;  ecx - DMA channel
                ;  edx - Sound Blaster card type
                mov     eax, [dIoAddr]
                mov     ebx, [dIrq]
                mov     ecx, [dDma]
                mov     edx, [dType]
                ret
endp



proc            SbDetect
                ;in
                ;  nothing
                ;out
                ;  CN: card present
                ;  CY: error (card not found)

                ;;setup initial values
                mov     eax, 210h
                mov     [dIoAddr], eax
                xor     eax, eax
                mov     [dIrq], eax

                ;;find base address
@@loop1:        call    SbReset
                @IF_C
                  add   [bptr dIoAddr], 10h
                  cmp   [bptr dIoAddr], 60h
                  jna   @@loop1
                  stc
                  jmp   @@exit
                @ENDIF

                ;;find IRQ
                xor     ecx, ecx                ;set IRQ traps
                irp     I, <2,3,5,7,10>
                mov     eax, I
                lea     ebx, [IrqTrap&I]
                call    IrqSetHandler
                endm
                ;
                mov     al, 0F2h                ;generate IRQ command
                call    SbOut
                ;
                mov     ecx, 10000h             ;delay
@@loop2:        cmp     [dIrq], 0
                loope   @@loop2
                ;
                irp     I, <2,3,5,7,10>
                mov     eax, I                  ;remove IRQ traps
                call    IrqRestoreHandler
                endm
                ;
                cmp     [dIrq], 0
                @IF_E
                  stc
                  jmp   @@exit
                @ENDIF

                call    SbDetectDmaWin
                jc      @@exit

                ;;find Sound Blaster type
                xor     eax, eax
                mov     al, 0E1h
                call    SbOut
                call    SbIn
                mov     ah, al
                call    SbIn
                mov     [dType], eax

@@exit:         ret
endp



proc            SbReset
                ;in
                ;  nothing
                ;out
                ;  CN: reset succeed
                ;  CY: reset failed
                ;description
                ;  Resets Sound Blaster.
                pushad

                ;;write 1 to reset register
                mov     edx, [dIoAddr]
                add     dl, 6
                mov     al, 1
                out     dx, al

                ;;delay before next write
                in      al, dx
                in      al, dx
                in      al, dx
                in      al, dx

                ;;write 0 to reset register
                xor     al, al
                out     dx, al

                call    SbIn
                cmp     al, 0AAh
                @IF_NE
                  stc
                @ENDIF

                popad
                ret
endp



proc            SbOut
                ;in
                ;  al - byte to output to register 0Ch
                ;out
                ;  CN: write succeed
                ;  CY: error
                push    ecx edx
                mov     ah, al
                mov     edx, [dIoAddr]
                add     dl, 0Ch
                mov     ecx, 1000               ;fail safe
@@loop1:        in      al, dx
                and     al, 80h
                @IF_NZ
                  loop  @@loop1
                  stc
                  jmp   @@exit
                @ENDIF
                mov     al, ah
                out     dx, al
@@exit:         pop     edx ecx
                ret
endp



proc            SbIn
                ;in
                ;  nothing
                ;out
                ;  CN: al - byte from register 0Ah
                ;  CY: error
                push    ecx edx
                mov     edx, [dIoAddr]
                add     dl, 0Eh
                mov     ecx, 1000               ;fail safe
@@loop1:        in      al, dx
                and     al, 80h
                @IF_Z
                  loop  @@loop1
                  stc
                  jmp   @@exit
                @ENDIF
                sub     dl, 4
                in      al, dx
@@exit:         pop     edx ecx
                ret
endp



proc            SbIrqAck
                ;in
                ;  nothing
                ;out
                ;  nothing
                ;description
                ;  Acknowledges IRQ.
                push    edx
                mov     edx, [dIoAddr]
                add     dl, 0Eh
                in      al, dx
                pop     edx
                ret
endp



;****************************************************************************;
;                             LOCAL METHODS                                  ;
;****************************************************************************;



dDmaBuffer      dd      0

proc            IrqHandler
                call    SbIrqAck
                ret
endp



proc            SbDetectDmaWin
                ;description
                ;  Determines 8-bit DMA channel number under Windows.

                ;;set IRQ handler
                mov     eax, [dIrq]
                lea     ebx, [IrqHandler]
                xor     ecx, ecx                ;second handler not needed
                call    IrqSetHandler

                ;;set frequency
                mov     al, 40h
                call    SbOut
                mov     al, 0FFh
                call    SbOut

                ;;programm DMA controllers
                irp     I, <0,1,3>
                mov     eax, I
                call    DmaMaskChannel
                mov     eax, I
                lea     ebx, [dDmaBuffer]
                call    DmaSetAddress
                mov     eax, I
                mov     ebx, 40h
                call    DmaSetSize
                mov     eax, I
                mov     bl, 48h
                call    DmaSetMode
                mov     eax, I
                call    DmaUnMaskChannel
                endm

                mov     al, 014h                ;begin DMA operation
                call    SbOut
                mov     al, 0FFh
                call    SbOut
                mov     al, 0FFh
                call    SbOut

                mov     ecx, 10000h
@@loop1:
                irp     I, <0,1,3>
                mov     eax, I
                mov     [dDma], eax
                call    DmaGetSize
                cmp     eax, 40h
                jb      @@found
                endm
                loop    @@loop1
                mov     [dDma], 0FFh

@@found:        call    SbReset

                irp     I, <0,1,3>
                mov     eax, I
                call    DmaMaskChannel
                endm

                ;;remove IRQ handler
                mov     eax, [dIrq]
                call    IrqRestoreHandler

                cmp     [dDma], 0FFh
                clc
                @IF_E
                  stc
                @ENDIF

                ret
endp



proc            SbDetectDmaDos
                ;description
                ;  Determines 8-bit DMA channel number under DOS.

                irp     I, <0,1,3>
                mov     eax, I
                call    DmaMaskChannel          ;mask channels
                mov     eax, I
                call    DmaClearRequest
                endm

                mov     al, 014h                ;begin DMA operation
                call    SbOut
                mov     al, 1
                call    SbOut
                mov     al, 0
                call    SbOut

                mov     ecx, 10000h
@@loop3:
                irp     I, <0,1,3>              ;
                mov     eax, I                  ;
                mov     [dDma], eax             ;wait for
                call    DmaGetState             ;unhandled
                and     al, 2                   ;request
                jnz     @@found                 ;
                endm                            ;
                loop    @@loop3                 ;
                stc
                jmp     @@exit

@@found:        call    SbReset

                irp     I, <0,1,3>
                mov     eax, I
                call    DmaClearRequest
                mov     eax, I
                call    DmaUnMaskChannel        ;unmask channels
                endm

@@exit:         ret
endp



                irp     I, <2,3,5,7,10>
proc            IrqTrap&I
                call    SbIrqAck
                mov     [dIrq], I
                ret
endp            IrqTrap&I
                endm

ends            code32
                end
