;*****
; pcx.asm - reading and storing 'Pic' objects as PCX files.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __PCX__ equ 1
                include "pmlib.asd"
                include "color.asi"
                include "debug.asi"
                include "error.asi"
                include "file.asi"
                include "heap.asi"
                include "pcx.asi"
                include "pic.asi"



struc           pcxhead         ;PCX header
  bManufacturer db      0Ah     ;0Ah = ZSoft PCX file
  bVersion      db      0       ;5 = version 3
  bEncoding     db      0       ;1 = RLL encoding
  bBitsPerPixel db      0       ;number of bits per pixel in color layer
  wXMin         dw      0       ;\
  wYMin         dw      0       ; |image window coordinates
  wXMax         dw      0       ; |
  wYMax         dw      0       ;/
  wHRes         dw      0       ;horisontal resolution
  wVRes         dw      0       ;vertical resolution
  abColorMap    db   48 dup (0) ;16-color palette
  bReserved     db      0       ;
  bNumPlanes    db      0       ;number of color layers
  wBytesPerLine dw      0       ;number of bytes per row in layer
  wPaletteInfo  dw      0       ;1=color/black&white, 2=gray
  abFiller      db   58 dup (0) ;filler to 128 bytes
ends



sPcxHead        pcxhead <>      ;PCX file header
dFileHandle     dd      0       ;PCX file handle
dRepCount       dd      0       ;repeat counter
dRepByte        dd      0       ;repeat byte
dRowLength      dd      0       ;row buffer length
dRowBuffer      dd      0       ;ptr to row buffer
dTotalRow       dd      0       ;total number of rows
dTotalCol       dd      0       ;total number of columns

RUS_MESSAGE     cErrBadFormat   " ଠ 䠩 PCX."
ENG_MESSAGE     cErrBadFormat   "Unknown PCX file format."
RUS_MESSAGE     cErrRead        "訡 ⥭  䠩 PCX."
ENG_MESSAGE     cErrRead        "PCX file reading error."



proc            PcxToPic
                ;in
                ;  esi - 'Str' object containing file name
                ;out
                ;  edi - 'Pic' handle
                ;description
                ;  Creates 'Pic' object from PCX file.
                push    ecx ebx edx esi

                ;;get file handle in ESI
                mov     ax, 'r'
                call    FileOpen

                ;;read header
                mov     eax, size pcxhead
                lea     edi, [sPcxHead]
                call    FileReadStr
                cmp     [(pcxhead edi).bManufacturer], 0Ah      ;ZSoft
                setne   bl
                cmp     [(pcxhead edi).bVersion], 5             ;Version 3
                setne   bh
                or      bl, bh
                cmp     [(pcxhead edi).bEncoding], 1            ;RLL
                setne   bh
                or      bl, bh
                cmp     [(pcxhead edi).bBitsPerPixel], 8        ;256 color
                setne   bh
                or      bl, bh
                @IF_NZ
                  lea   esi, [cErrBadFormat]
                  call  ErrorFatal
                @ENDIF

                ;;allocate string buffer
                pushad
                movzx   eax, [(pcxhead edi).wBytesPerLine]
                xor     ebx, ebx
                mov     bl, [(pcxhead edi).bNumPlanes]
                mul     ebx
                mov     [dRowLength], eax
                call    HeapAlloc
                mov     [dRowBuffer], edi
                mov     [dRepCount], 0
                popad

                ;;check for 256-color file
                cmp     [(pcxhead edi).bNumPlanes], 1
                je      @@256color

                ;;check for true-color file
                cmp     [(pcxhead edi).bNumPlanes], 3
                je      @@truecolor
                lea     esi, [cErrBadFormat]
                call    ErrorFatal

                ;;create 256color "Pic" object
@@256color:     push    edi
                call    PaletteNew
                xchg    edi, [esp]
                ;;
                movzx   eax, [(pcxhead edi).wYMax]
                movzx   edx, [(pcxhead edi).wYMin]
                sub     eax, edx
                inc     eax
                mov     [dTotalRow], eax                ;row number
                movzx   ebx, [(pcxhead edi).wXMax]
                movzx   edx, [(pcxhead edi).wXMin]
                sub     ebx, edx
                inc     ebx
                mov     [dTotalCol], ebx                ;column number
                mov     ecx, COLOR88
                call    PicNew
                ;;
                pop     [(pic edi).dPalette]

                ;;decode 256color image
                mov     ebx, [(pic edi).dImagePtr]
                mov     ecx, [dTotalRow]
                @DO
                  push  ecx
                  call  PcxGetLine
                  mov   edx, [dRowBuffer]
                  mov   ecx, [dTotalCol]
                  @DO
                    mov al, [edx]
                    inc edx
                    mov [ebx], al
                    inc ebx
                  @DJNZ ecx
                  pop   ecx
                @DJNZ   ecx

                ;;load 256color palette
@@loop3:        call    FileReadByte
                @IF_C
                  lea   esi, [cErrBadFormat]
                  call  ErrorFatal
                @ENDIF
                cmp     al, 0Ch
                jnz     @@loop3

                mov     ebx, [(pic edi).dPalette]
                lea     ebx, [(palette ebx).adRGB]
                mov     ecx, 256
                @DO
                  call  FileReadByte
                  mov   [ebx], al
                  call  FileReadByte
                  mov   [ebx+1], al
                  call  FileReadByte
                  mov   [ebx+2], al
                  mov   [bptr ebx+3], 0
                  add   ebx, 4
                @DJNZ   ecx
                jmp     @@exit

                ;;create truecolor "Pic" object
@@truecolor:    movzx   eax, [(pcxhead edi).wYMax]
                movzx   edx, [(pcxhead edi).wYMin]
                sub     eax, edx
                inc     eax
                mov     [dTotalRow], eax                ;row number
                movzx   ebx, [(pcxhead edi).wXMax]
                movzx   edx, [(pcxhead edi).wXMin]
                sub     ebx, edx
                inc     ebx
                mov     [dTotalCol], ebx                ;column number
                mov     ecx, COLOR24
                call    PicNew

                ;;decode truecolor image
                push    edi
                mov     ebx, [(pic edi).dImagePtr]
                mov     ecx, [dTotalRow]
                mov     edi, [dTotalCol]
                @DO
                  push  ecx
                  call  PcxGetLine
                  mov   edx, [dRowBuffer]
                  mov   ecx, [dTotalCol]
                  @DO
                    mov al, [edx+edi*2]
                    mov [ebx], al
                    inc ebx
                    mov al, [edx+edi]
                    mov [ebx], al
                    inc ebx
                    mov al, [edx]
                    mov [ebx], al
                    inc ebx
                    inc edx
                  @DJNZ ecx
                  pop   ecx
                @DJNZ   ecx
                pop     edi

                ;;free string buffer
@@exit:         mov     eax, [dRowBuffer]
                call    HeapFree

                ;;close file
                call    FileClose

                pop     esi edx ebx ecx
                ret
endp



proc            PcxGetLine
                push    ebx ecx edx

                mov     ecx, [dRowLength]
                mov     ebx, [dRowBuffer]
                mov     eax, [dRepCount]
                and     al, al
                jnz     short @@cont

@@loop1:        call    FileReadByte
                cmp     al, 192
                @IF_B
                  mov   [ebx], al
                  inc   ebx
                  dec   ecx
                  jnz   short @@loop1
                @ELSE
                  sub   al, 192
@@cont:           mov   dl, al
                  call  FileReadByte
@@loop2:          mov   [ebx], al
                  inc   ebx
                  dec   dl
                  dec   ecx
                  @IF_NZ
                    and dl, dl
                    jnz short @@loop2
                    jmp short @@loop1
                  @ENDIF
                @ENDIF

                mov     [dRepCount], edx

                pop     edx ecx ebx
                ret
endp



proc            PcxFromPic
                ;in
                ;  esi - 'Str' object containing file name
                ;  edi - 'Pic' handle
                ;out
                ;  nothing
                ;description
                ;  Creates PCX file from 'Pic' object.
                pushad

                mov     ebx, edi

                ;;get file handle in ESI
                mov     ax, 'w'
                call    FileOpen
                mov     [dFileHandle], esi

                ;;clear header
                xor     eax, eax
                mov     ecx, (size pcxhead)/4
                lea     edi, [sPcxHead]
                rep     stosd

                ;;fill header
                lea     edi, [sPcxHead]
                mov     [(pcxhead edi).bManufacturer], 0Ah      ;ZSoft
                mov     [(pcxhead edi).bVersion], 5             ;Version 3
                mov     [(pcxhead edi).bEncoding], 1            ;RLL
                mov     [(pcxhead edi).bBitsPerPixel], 8        ;256 color
                ;;;
                mov     eax, [(pic ebx).dTotalCol]
                dec     eax
                mov     [(pcxhead edi).wXMax], ax
                ;;;
                mov     eax, [(pic ebx).dTotalRow]
                dec     eax
                mov     [(pcxhead edi).wYMax], ax
                mov     [(pcxhead edi).bNumPlanes], 1           ;256 color
                ;;;
                mov     eax, [(pic ebx).dTotalCol]
                neg     eax
                and     eax, 1
                add     eax, [(pic ebx).dTotalCol]
                mov     [(pcxhead edi).wBytesPerLine], ax
                ;;;
                mov     [(pcxhead edi).wPaletteInfo], 1         ;color

                ;;write header
                mov     eax, size pcxhead
                call    FileWriteStr

                ;;allocate and clear string buffer
                pushad
                movzx   eax, [(pcxhead edi).wBytesPerLine]
                xor     ebx, ebx
                mov     bl, [(pcxhead edi).bNumPlanes]
                mul     ebx
                mov     [dRowLength], eax
                call    HeapAlloc
                mov     [dRowBuffer], edi
                xor     al, al
                mov     ecx, [dRowLength]
                rep     stosb
                popad

                ;;encode image
                mov     ecx, [(pic ebx).dTotalRow]
                mov     esi, [(pic ebx).dImagePtr]
@@encode_loop:  push    ecx
                ;;;copy current row to buffer
                mov     edi, [dRowBuffer]
                mov     ecx, [(pic ebx).dTotalCol]
                rep     movsb
                push    esi
                ;;;encode current row
                xor     ecx, ecx
                mov     edi, [dRowBuffer]
@@loop1:        cmp     ecx, [dRowLength]
                @IF_B
                  mov   al, [edi+ecx]
                  inc   ecx
                  cmp   ecx, [dRowLength]
                  @IF_B
                    cmp al, [edi+ecx]
                    sete dl
                  @ELSE
                    xor dl, dl
                  @ENDIF
                  and   dl, dl
                  @IF_Z
                    ;;can not be compressed
                    cmp al, 192
                    @IF_B
                      ;;just store
                      mov  esi, [dFileHandle]
                      call FileWriteByte
                    @ELSE
                      ;;store repeat counter=1 and this byte
                      push eax
                      mov  al, 193
                      mov  esi, [dFileHandle]
                      call FileWriteByte
                      pop  eax
                      call FileWriteByte
                    @ENDIF
                  @ELSE
                    ;;compress
                    mov edx, 194
                    mov [dRepCount], edx
@@loop2:            inc ecx
                    cmp ecx, [dRowLength]
                    @IF_B
                      cmp  [dRepCount], 0FFh
                      @IF_B
                        cmp  al, [edi+ecx]
                        @IF_E
                          inc  [dRepCount]
                          jmp  @@loop2
                        @ENDIF
                      @ENDIF
                    @ENDIF
                    push eax
                    mov  eax, [dRepCount]
                    mov  esi, [dFileHandle]
                    call FileWriteByte
                    pop  eax
                    call FileWriteByte
                  @ENDIF
                  jmp @@loop1
                @ENDIF
                pop     esi
                pop     ecx
                dec     ecx
                jnz     @@encode_loop

                ;;write palette
                mov     al, 0Ch
                mov     esi, [dFileHandle]
                call    FileWriteByte
                cmp     [(pic ebx).dColorScheme], COLOR88
                @IF_E
                  ;;take palette from 'Pic'
                  mov   edi, [(pic ebx).dPalette]
                  lea   edi, [(palette edi).adRGB]
                @ELSE
                  ;;take sysmen palette
                  lea   edi, [adSystemPalette]
                @ENDIF
                mov     ecx, 256
                @DO
                  mov   al, [edi]
                  inc   edi
                  call  FileWriteByte
                  mov   al, [edi]
                  inc   edi
                  call  FileWriteByte
                  mov   al, [edi]
                  inc   edi
                  inc   edi
                  call  FileWriteByte
                @DJNZ   ecx

                ;;free string buffer
                mov     eax, [dRowBuffer]
                call    HeapFree

                ;;close file
                call    FileClose

                popad
                ret
endp

ends            code32
                end
