;*****
; lm.asm  -  low memory management (like heap management)
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __LM__  equ 1
                include "pmlib.asd"
                include "debug.asi"
                include "error.asi"
                include "lm.asi"



SIGN            = 'LMem'        ;signature for run-time checks
THRESHOLD       = 16            ;minimal size of block in heap
                                ;(should be greater than header size)



struc           List            ;List
  dNext         dd      0       ;ptr to next 'List' object.
ends

struc           MemBlock        ;memory block (List descendant)
  dNext         dd      0       ;ptr to next 'MemBlock'
  dBlockSize    dd      0       ;size of memory block (including header)
  dSign         dd      SIGN    ;signature for run-time checks
ends



sHead           List    <>      ;ptr to first free memory block
RUS_MESSAGE     cErr            "⪠ ᢮ ᯮ祭   "
ENG_MESSAGE     cErr            "Attempt to free bad block of lower memory."
RUS_MESSAGE     cErrNoMem       "  "
ENG_MESSAGE     cErrNoMem       "Out of lower memory."



proc            LmInit
                ;in
                ;  eax - size
                ;  ecx - start
                ;out
                ;  nothing
                pushad

		lea	esi, [sHead]
                mov     edi, ecx

		mov	[(List esi).dNext], edi
                mov     [(List edi).dNext], 0
                mov     [(MemBlock edi).dBlockSize], eax
		mov	[(MemBlock edi).dSign], SIGN

                popad
                ret
endp



proc            LmAlloc
                ;in
                ;  eax - size of requested block
                ;out
                ;  edi - ptr to allocated block
		push	ebx esi

                ;;get in eax block size rounded to dword
                add     eax, size MemBlock
                test    al, 03h
                @IF_NZ
                  and   al, 0FCh
                  add   eax, 04h
                @ENDIF

		lea	edi, [sHead]
@@loop1:	mov	esi, edi
		mov	edi, [(List esi).dNext]
		and	edi, edi
                @IF_Z
                  lea   esi, [cErrNoMem]
                  call  ErrorFatal
                @ENDIF
		cmp	[(MemBlock edi).dBlockSize], eax
		jb	@@loop1

                ;;have found suitable block now check if we must split it
                ;;or allocate whole block
        	mov	ebx, [(MemBlock edi).dBlockSize]
        	sub	ebx, eax
		cmp	ebx, THRESHOLD
		@IF_A
                  ;;split block
        	  sub	[(MemBlock edi).dBlockSize], eax
		  add	edi, [(MemBlock edi).dBlockSize]
		  mov	[(MemBlock edi).dBlockSize], eax
		  mov	[(MemBlock edi).dSign], SIGN
		@ELSE
                  ;;allocate whole block
		  mov	ebx, [(List edi).dNext]
		  mov	[(List esi).dNext], ebx
		@ENDIF
 		add	edi, size MemBlock

                pop     esi ebx
		ret
endp



proc            LmFree
                ;in
                ;  eax - ptr to previously allocated block
                ;out
                ;  nothing
                push	ebx esi edi

		sub	eax, size MemBlock
                mov     ebx, eax                ;freeing block
		cmp	[(MemBlock ebx).dSign], SIGN
		@IF_NE
                  lea   esi, [cErr]
                  call  ErrorFatal
		@ENDIF

                lea     esi, [sHead]            ;previous block
@@loop1:        mov     edi, [(List esi).dNext] ;current block
		and	edi, edi
		jz	@@insert

                ;;try to join freeing block to current
		mov	eax, [(MemBlock edi).dBlockSize]
		add	eax, edi
		cmp	eax, ebx
		@IF_Z
                  ;;join two blocks in one not belonging to list
		  mov	eax, [(MemBlock ebx).dBlockSize]
		  add	[(MemBlock edi).dBlockSize], eax
		  mov	eax, [(List edi).dNext]
		  mov	[(List esi).dNext], eax
		  mov	ebx, edi
                  ;;try to join resulting block to any block from heap
		  mov	eax, [(MemBlock ebx).dBlockSize]
		  add	eax, ebx
 		  lea	edi, [sHead]
@@loop2:	  mov	esi, edi
		  mov	edi, [(List esi).dNext]
		  and	edi, edi
		  jz	@@insert
		  cmp	eax, edi
		  jnz	@@loop2
		  mov	eax, [(MemBlock edi).dBlockSize]
		  add	[(MemBlock ebx).dBlockSize], eax
		  mov	eax, [(List edi).dNext]
		  mov	[(List ebx).dNext], eax
		  mov	[(List esi).dNext], ebx
		  jmp	@@exit
		@ENDIF

                ;;try to join freeing block to the start of current block
		mov	eax, [(MemBlock ebx).dBlockSize]
		add	eax, ebx
		cmp	eax, edi
		@IF_Z
                  ;;join two blocks in one not belonging to list
		  mov	eax, [(MemBlock edi).dBlockSize]
		  add	[(MemBlock ebx).dBlockSize], eax
		  mov	eax, [(List edi).dNext]
		  mov	[(List esi).dNext], eax
                  ;;try to join resulting block to the end of any block
                  ;;from heap
 		  lea	edi, [sHead]
@@loop3:	  mov	esi, edi
		  mov	edi, [(List esi).dNext]
		  and	edi, edi
		  jz	@@insert
		  mov	eax, [(MemBlock edi).dBlockSize]
		  add	eax, edi
		  cmp	eax, ebx
		  jnz	@@loop3
		  mov	eax, [(MemBlock ebx).dBlockSize]
		  add	[(MemBlock edi).dBlockSize], eax
		  jmp	@@exit
		@ENDIF

		mov	esi, edi
		jmp	@@loop1

@@insert:       ;;insert to list at last position
		mov	[(List esi).dNext], ebx
		mov	[(List ebx).dNext], edi

@@exit:         pop	edi esi ebx
                ret
endp



IF              DBG
cMsg1           db      "Low memory heap dump:", 0
cMsg2		db	0
cMsg3		db	"End of dump.", 0

proc            LmDump
                ;in
                ;  nothing
                ;out
                ;  nothing
                pushad

		lea	esi, [cMsg2]
		call	DebugPrintStr
		lea	esi, [cMsg1]
		call	DebugPrintStr

		lea	edi, [sHead]
@@loop1:	mov	esi, edi
		mov	edi, [(List esi).dNext]
		and	edi, edi
		jz	@@exit

                ;;dump heap structure
		mov	eax, edi
		call	DebugPrintEaxHex
		mov	eax, [(List edi).dNext]
		call	DebugPrintEaxHex
        	mov	eax, [(MemBlock edi).dBlockSize]
                call    DebugPrintEaxDec

		jmp	@@loop1

@@exit:		lea	esi, [cMsg3]
		call	DebugPrintStr
		lea	esi, [cMsg2]
		call	DebugPrintStr
                popad
		ret
endp
ENDIF           ;DBG

ends            code32
                end
