;*****
; gfx.asm  -  graphics primitives.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __GFX__ equ 1
                include "pmlib.asd"
                include "gfx.asi"
                include "color.asi"
                include "debug.asi"
                include "error.asi"
                include "math.asi"
                include "win.asi"
                include "svga.asi"



proc            GfxSetPoint
                ;in
                ;  eax - point row
                ;  ebx - point column
                ;  ecx - point color
                ;  edi - 'Pic'
                ;out
                ;  nothing
                push    edx edi

                ;;validate coordinates
                cmp     eax, [(pic edi).dTotalRow]
                jae     @@exit
                cmp     ebx, [(pic edi).dTotalCol]
                jae     @@exit

                ;;get point address in the window
                mul     [(pic edi).dTotalCol]
                add     eax, ebx
                mul     [(pic edi).dBytesPerCell]
                mov     edx, [(pic edi).dBytesPerCell]
                mov     edi, [(pic edi).dImagePtr]
                add     edi, eax

                ;;plot point
                cmp     dl, 1
                @IF_E
                  mov   [edi], cl
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 2
                @IF_E
                  mov   [edi], cx
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 3
                @IF_E
                  mov   [edi], cx
                  shr   ecx, 16
                  mov   [edi+2], cl
                  jmp   short @@exit
                @ENDIF
                mov     [edi], ecx

@@exit:         pop     edi edx
                ret
endp



proc            GfxGetPoint
                ;in
                ;  eax - point row
                ;  ebx - point column
                ;  edi - 'Pic'
                ;out
                ;  ecx - point color, black if point is out of 'Pic'
                push    edx edi

                ;;initialize ecx with black color
                xor     ecx, ecx

                ;;validate coordinates
                cmp     eax, [(pic edi).dTotalRow]
                jae     @@exit
                cmp     ebx, [(pic edi).dTotalCol]
                jae     @@exit

                ;;get point address in the window
                mul     [(pic edi).dTotalCol]
                add     eax, ebx
                mul     [(pic edi).dBytesPerCell]
                mov     edx, [(pic edi).dBytesPerCell]
                mov     edi, [(pic edi).dImagePtr]
                add     edi, eax

                ;;read point
                cmp     dl, 1
                @IF_E
                  mov   cl, [edi]
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 2
                @IF_E
                  mov   cx, [edi]
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 3
                @IF_E
                  mov   cl, [edi+2]
                  shl   ecx, 16
                  mov   cx, [edi]
                  jmp   short @@exit
                @ENDIF
                mov     ecx, [edi]

@@exit:         pop     edi edx
                ret
endp



proc            GfxHLine
                ;in
                ;  eax - starting point row
                ;  ebx - starting point column
                ;  ecx - length of line
                ;  ebp - color
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                ;;validate coordinates
                cmp     eax, [(win edi).dTotalRow]
                jae     @@exit
                mov     edx, [(win edi).dTotalCol]
                sub     edx, ebx
                jbe     @@exit
                @MIN    ecx, edx

                ;;edi - ptr to starting point
                mul     [(win edi).dTotalCol]
                add     eax, ebx
                mul     [(win edi).dBytesPerCell]
                mov     edx, [(win edi).dBytesPerCell]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax

                ;;eax - color
                mov     eax, ebp

                ;;draw horizontal line
                cmp     dl, 1
                @IF_E
                  rep   stosb
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 2
                @IF_E
                  rep   stosw
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 3
                @IF_E
                  mov   edx, eax
                  shr   edx, 16
@@loop1:          mov   [edi], ax
                  mov   [edi+2], dl
                  add   edi, 3
                  loop  @@loop1
                  jmp   short @@exit
                @ENDIF
                ;;cmp     dl, 4
                ;;@IF_E
                  rep   stosd
                ;;  jmp   short @@exit
                ;;@ENDIF

@@exit:         popad
                ret
endp



proc            GfxVLine
                ;in
                ;  eax - starting point row
                ;  ebx - starting point column
                ;  ecx - length of line
                ;  ebp - color
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                ;;validate coordinates
                cmp     ebx, [(win edi).dTotalCol]
                jae     @@exit
                mov     edx, [(win edi).dTotalRow]
                sub     edx, eax
                jbe     @@exit
                @MIN    ecx, edx

                ;;edi - ptr to starting point
                mul     [(win edi).dTotalCol]
                add     eax, ebx
                mul     [(win edi).dBytesPerCell]
                mov     edx, [(win edi).dBytesPerCell]
                mov     ebx, [(win edi).dTotalColByte]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax

                ;;eax - color
                mov     eax, ebp

                ;;draw vertical line
                cmp     dl, 1
                @IF_E
@@loop1:          mov   [edi], al
                  add   edi, ebx
                  loop  @@loop1
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 2
                @IF_E
@@loop2:          mov   [edi], ax
                  add   edi, ebx
                  loop  @@loop2
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 3
                @IF_E
                  mov   edx, eax
                  shr   edx, 16
@@loop3:          mov   [edi], ax
                  mov   [edi+2], dl
                  add   edi, ebx
                  loop  @@loop3
                  jmp   short @@exit
                @ENDIF
                cmp     dl, 4
                @IF_E
@@loop4:          mov   [edi], eax
                  add   edi, ebx
                  loop  @@loop4
                  jmp   short @@exit
                @ENDIF

@@exit:         popad
                ret
endp





;;
;                                  LINE                                    ;
;;

proc            GfxLine
                ;in
                ;  eax - starting point row
                ;  ebx - starting point column
                ;  ecx - ending point row
                ;  edx - ending point column
                ;  ebp - color
                ;  edi - 'Win'
                ;out
                ;  nothing

                cmp     [(win edi).dBytesPerCell], 1
                @IF_E
                  jmp   GfxLine8
                @ENDIF

                cmp     [(win edi).dBytesPerCell], 4
                @IF_E
                  jmp   GfxLine32
                @ENDIF

                ret
endp



proc            GfxLine8
                ;in
                ;  eax - starting point row
                ;  ebx - starting point column
                ;  ecx - ending point row
                ;  edx - ending point column
                ;  ebp - color
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                ;;validate coordinates
                cmp     eax, [(win edi).dTotalRow]
                jae     @@exit
                cmp     ebx, [(win edi).dTotalCol]
                jae     @@exit
                cmp     ecx, [(win edi).dTotalRow]
                jae     @@exit
                cmp     edx, [(win edi).dTotalCol]
                jae     @@exit

                ;;starting point must be more left than ending
                cmp     ebx, edx
                @IF_A
                  xchg  eax, ecx
                  xchg  ebx, edx
                @ENDIF

                ;;esi - row increment
                mov     esi, [(win edi).dTotalColByte]

                ;;edi - ptr to starting point
                push    eax edx
                mul     [(win edi).dTotalCol]
                add     eax, ebx
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                pop     edx eax

                ;;esi adjustment
                cmp     eax, ecx
                @IF_A
                  xchg  eax, ecx
                  neg   esi
                @ENDIF

                ;;determine the main loop type: on rows or on columns
                sub     ecx, eax        ;number of rows
                sub     edx, ebx        ;number of columns
                cmp     edx, ecx
                mov     eax, ebp
                jb      @@cont_y

                ;;main loop on columns
                xchg    ecx, edx
                mov     ebp, ecx        ;number of columns
                mov     ebx, ecx        ;counter
                shr     ebx, 1
@@loop_x:       mov     [edi], al
                sub     ebx, edx
                inc     edi
                @IF_C
                  add   edi, esi
                  add   ebx, ebp
                @ENDIF
                sub     ecx, 1          ;for zero length correct handling
                jae     @@loop_x
                jmp     @@exit

                ;;main loop on rows
@@cont_y:       mov     ebp, ecx        ;number of rows
                mov     ebx, ecx        ;counter
                shr     ebx, 1
@@loop_y:       mov     [edi], al
                add     edi, esi
                sub     ebx, edx
                @IF_C
                  inc   edi
                  add   ebx, ebp
                @ENDIF
                sub     ecx, 1          ;for zero length correct handling
                jae     @@loop_y

@@exit:         popad
                ret
endp



proc            GfxLine32
                ;in
                ;  eax - starting point row
                ;  ebx - starting point column
                ;  ecx - ending point row
                ;  edx - ending point column
                ;  ebp - color
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                ;;validate coordinates
                cmp     eax, [(win edi).dTotalRow]
                jae     @@exit
                cmp     ebx, [(win edi).dTotalCol]
                jae     @@exit
                cmp     ecx, [(win edi).dTotalRow]
                jae     @@exit
                cmp     edx, [(win edi).dTotalCol]
                jae     @@exit

                ;;starting point must be more left than ending
                cmp     ebx, edx
                @IF_A
                  xchg  eax, ecx
                  xchg  ebx, edx
                @ENDIF

                ;;esi - row increment
                mov     esi, [(win edi).dTotalColByte]

                ;;edi - ptr to starting point
                push    eax edx
                mul     [(win edi).dTotalCol]
                add     eax, ebx
                shl     eax, 2
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                pop     edx eax

                ;;esi adjustment
                cmp     eax, ecx
                @IF_A
                  xchg  eax, ecx
                  neg   esi
                @ENDIF

                ;;determine the main loop type: on rows or on columns
                sub     ecx, eax        ;number of rows
                sub     edx, ebx        ;number of columns
                cmp     edx, ecx
                mov     eax, ebp
                jb      @@cont_y

                ;;main loop on columns
                xchg    ecx, edx
                mov     ebp, ecx        ;number of columns
                mov     ebx, ecx        ;counter
                shr     ebx, 1
@@loop_x:       mov     [edi], eax
                add     edi, 4
                sub     ebx, edx
                @IF_C
                  add   edi, esi
                  add   ebx, ebp
                @ENDIF
                sub     ecx, 1          ;for zero length correct handling
                jae     @@loop_x
                jmp     @@exit

                ;;main loop on rows
@@cont_y:       mov     ebp, ecx        ;number of rows
                mov     ebx, ecx        ;counter
                shr     ebx, 1
@@loop_y:       mov     [edi], eax
                add     edi, esi
                sub     ebx, edx
                @IF_C
                  add   edi, 4
                  add   ebx, ebp
                @ENDIF
                sub     ecx, 1          ;for zero length correct handling
                jae     @@loop_y

@@exit:         popad
                ret
endp





;;
;                            Convex Poligon                                ;
;                                                                          ;
; Terminology:                                                             ;
;   1.index - vertex number in the array of vertexes                       ;
;   2.line - length from one vertex to another                             ;
;   3.side - one or more lines going from top to bottom one after another  ;
;                                                                          ;
;;



;;side types
SIDE_UNDEF      equ     0       ;undefined; used for initialization
SIDE_LEFT       equ     1       ;left   used to determine initial offset
SIDE_RIGHT      equ     2       ;right  of horizontal line

;;direction of index move in the array of vertexes
DIR_FORWARD     equ     1
DIR_BACKWARD    equ     -1

macro           IdxFwd  reg     ;move index forward
                local   cont
                inc     &reg
                cmp     &reg, [(poligon esi).dVertexNum]
                jb      cont
                sub     &reg, [(poligon esi).dVertexNum]
cont:
endm

macro           IdxBack reg     ;move index backward
                local   cont
                sub     &reg, 1
                jnc     cont
                add     &reg, [(poligon esi).dVertexNum]
cont:
endm

struc           side            ;side
  dType         dd      0       ;side type (used for horizontal lines drawing)
  dTopIndex     dd      0       ;index of next line's top
  dIndexDir     dd      0       ;direction of index move
  dLineY        dd      0       ;current row
  dLineX        dd      0       ;current column
  dLineDeltaX   dd      0       ;column increment (+1 or -1)
  dLineCount1   dd      0       ;counter of total points number in line
  dLineCount2   dd      0       ;counter to determine step value on short axis
  dLineCountMin dd      0       ;number of points on the short axis
  dLineCountMax dd      0       ;number of points on the long axis
  dUpdateFunc   dd      0       ;function that updates state of this side
ends



dTopY           dd      0
dBotY           dd      0
dWinTotalCol    dd      0       ;total number of columns in 'Win'
sLSide          side    <>      ;left side
sRSide          side    <>      ;right side



proc            GfxConvexPoligon
                ;in
                ;  esi - poligon
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                ;;convert color from byte to dword
                mov     al, [bptr (poligon esi).dColor]
                mov     ah, al
                shl     eax, 8
                mov     al, ah
                shl     eax, 8
                mov     al, ah
                mov     [(poligon esi).dColor], eax

                ;;walk through the vertex list to find top and bottom points
                xor     ebx, ebx
                mov     eax, [(point esi+(size poligon)).dY]
                mov     [dTopY], eax    ;top point
                mov     [dBotY], eax    ;bottom point
                mov     ebp, ebx        ;top point index
@@loop1:        inc     ebx
                cmp     ebx, [(poligon esi).dVertexNum]
                @IF_B
                  mov   eax,[(point esi+(size poligon)+ebx*(size point)).dY]
                  cmp   eax, [dTopY]
                  @IF_B
                    mov ebp, ebx
                    mov [dTopY], eax
                  @ENDIF
                  cmp   eax, [dBotY]
                  @IF_A
                    mov [dBotY], eax
                  @ENDIF
                  jmp  @@loop1
                @ENDIF
                mov     eax, [dTopY]

                ;;suppose that vertexes placed clockwise
                mov     [sLSide.dIndexDir], DIR_BACKWARD
                mov     [sRSide.dIndexDir], DIR_FORWARD
                ;
                mov     ebx, ebp
                mov     ecx, [(poligon esi).dVertexNum]
@@loop2:        mov     [sLSide.dTopIndex], ebx ;index of left side top
                IdxBack ebx
                cmp     eax,[(point esi+(size poligon)+ebx*(size point)).dY]
                loope   @@loop2
                ;
                mov     ebx, ebp
                mov     ecx, [(poligon esi).dVertexNum]
@@loop3:        mov     [sRSide.dTopIndex], ebx ;index of right side top
                IdxFwd  ebx
                cmp     eax,[(point esi+(size poligon)+ebx*(size point)).dY]
                loope   @@loop3

                ;;initialize main loop
                mov     eax, [dTopY]
                mov     ebp, [dBotY]
                sub     ebp, eax                        ;loop counter
                mov     ebx, [(win edi).dTotalCol]
                mul     ebx
                mov     [dWinTotalCol], ebx
                mov     edx, [(win edi).dImagePtr]      ;current line start
                add     edx, eax                        ;

                ;;main loop imitation for determining how vertexes are placed
                ;;clockwise or anticlockwise
                mov     [sLSide.dType], SIDE_UNDEF
                mov     [sRSide.dType], SIDE_UNDEF
                push    [sLSide.dTopIndex]
                push    [sRSide.dTopIndex]
                push    [sLSide.dIndexDir]
                push    [sRSide.dIndexDir]
                lea     eax, [PoligonLineInit]
                mov     [sLSide.dUpdateFunc], eax
                mov     [sRSide.dUpdateFunc], eax
                mov     ecx, ebp
                inc     ecx                             ;loop counter
@@loop4:        push    ecx
                lea     edi, [sLSide]
                call    [(side edi).dUpdateFunc]
                lea     edi, [sRSide]
                call    [(side edi).dUpdateFunc]
                mov     eax, [sRSide.dLineX]
                sub     eax, [sLSide.dLineX]
                pop     ecx
                loope   @@loop4
                @IF_AE
                  ;;clockwise
                  pop   [sRSide.dIndexDir]
                  pop   [sLSide.dIndexDir]
                  pop   [sRSide.dTopIndex]
                  pop   [sLSide.dTopIndex]
                @ELSE
                  ;;anticlockwise
                  pop   [sLSide.dIndexDir]
                  pop   [sRSide.dIndexDir]
                  pop   [sLSide.dTopIndex]
                  pop   [sRSide.dTopIndex]
                @ENDIF
                mov     [sLSide.dType], SIDE_LEFT
                mov     [sRSide.dType], SIDE_RIGHT

                ;;main loop
                ;;go from top to bottom and draw horizontal lines from
                ;;left side to the right
                lea     eax, [PoligonLineInit]
                mov     [sLSide.dUpdateFunc], eax
                mov     [sRSide.dUpdateFunc], eax
@@loop5:        lea     edi, [sLSide]
                call    [(side edi).dUpdateFunc]
                lea     edi, [sRSide]
                call    [(side edi).dUpdateFunc]
                mov     ebx, [sLSide.dLineX]
                mov     ecx, [sRSide.dLineX]
                lea     edi, [edx+ebx]                  ;start
                sub     ecx, ebx                        ;length
                mov     eax, [(poligon esi).dColor]     ;color
                ;;
                mov     ebx, ecx
                shr     ecx, 2
                rep     stosd
                mov     ecx, ebx
                and     ecx, 3
                inc     ecx
                rep     stosb
                ;;
                add     edx, [dWinTotalCol]
                dec     ebp
                jns     @@loop5

@@exit:         popad
                ret
endp

proc            PoligonLineInit
                push    edx ebp
                mov     ebp, [(side edi).dTopIndex]

@@loop1:        mov     eax,[(point esi+(size poligon)+ebp*(size point)).dY]
                mov     ebx,[(point esi+(size poligon)+ebp*(size point)).dX]
                mov     [(side edi).dLineY], eax
                mov     [(side edi).dLineX], ebx

                ;;forward an index
                add     ebp, [(side edi).dIndexDir]
                @IF_S
                  add   ebp, [(poligon esi).dVertexNum]
                @ELSE
                  cmp   ebp, [(poligon esi).dVertexNum]
                  @IF_AE
                    sub ebp, [(poligon esi).dVertexNum]
                  @ENDIF
                @ENDIF

                mov     ecx,[(point esi+(size poligon)+ebp*(size point)).dY]
                mov     edx,[(point esi+(size poligon)+ebp*(size point)).dX]
                sub     ecx, eax
                sub     edx, ebx
                lea     eax, [PoligonLineV]
                mov     ebx, 1
                @IF_C
                  neg   edx
                  neg   ebx
                @ENDIF
                cmp     ecx, edx
                @IF_B
                  xchg  ecx, edx
                  lea   eax, [PoligonLineH]
                @ENDIF

                ;;skip vertex if it matches with previous
                and     ecx, ecx
                @IF_Z
                  ;;check if poligon turned to point
                  cmp   ebp, [(side edi).dTopIndex]
                  jne   @@loop1
                @ENDIF

                mov     [(side edi).dUpdateFunc], eax
                mov     [(side edi).dLineDeltaX], ebx
                mov     [(side edi).dLineCount1], ecx
                mov     [(side edi).dLineCountMax], ecx
                mov     [(side edi).dLineCountMin], edx
                shr     ecx, 1
                mov     [(side edi).dLineCount2], ecx

                ;;correct the column for horizontal line
                cmp     eax, off PoligonLineH
                sete    cl
                cmp     [(side edi).dType], SIDE_UNDEF
                setne   ch
                and     cl, ch
                cmp     [(side edi).dType], SIDE_LEFT
                @IF_E
                  and   ebx, ebx
                  sets  ch
                @ELSE
                  and   ebx, ebx
                  setns ch
                @ENDIF
                and     cl, ch
                @IF_NZ
@@loop2:          add   [(side edi).dLineX], ebx
                  dec   [(side edi).dLineCount1]
                  jz    @@loop1
                  sub   [(side edi).dLineCount2], edx
                  jae   @@loop2
                  sub   [(side edi).dLineX], ebx
                  inc   [(side edi).dLineCount1]
                  mov   eax, [(side edi).dLineCountMax]
                  add   [(side edi).dLineCount2], eax
                @ENDIF

                mov     [(side edi).dTopIndex], ebp
                pop     ebp edx
                ret
endp

proc            PoligonLineV
                dec     [(side edi).dLineCount1]
                @IF_Z
                  call  PoligonLineInit
                @ELSE
@@cont:           mov   eax, [(side edi).dLineCountMin]
                  sub   [(side edi).dLineCount2], eax
                  @IF_B
                    mov eax, [(side edi).dLineCountMax]
                    add [(side edi).dLineCount2], eax
                    mov eax, [(side edi).dLineDeltaX]
                    add [(side edi).dLineX], eax
                  @ENDIF
                @ENDIF
                ret
endp

proc            PoligonLineH
@@loop1:        dec     [(side edi).dLineCount1]
                @IF_Z
                  call  PoligonLineInit
                @ELSE
                  mov   eax, [(side edi).dLineCountMin]
                  mov   ebx, [(side edi).dLineDeltaX]
                  add   [(side edi).dLineX], ebx
                  sub   [(side edi).dLineCount2], eax
                  jae   @@loop1
                  mov   eax, [(side edi).dLineCountMax]
                  add   [(side edi).dLineCount2], eax
                @ENDIF
                ret
endp





;;
;                                   BOX                                    ;
;;

struc           box
  d11           dd      0       ;upper left point color
  d12           dd      0
  d21           dd      0       ;upper line color
  d22           dd      0
  d31           dd      0       ;upper right point color
  d32           dd      0
  d41           dd      0       ;right line color
  d42           dd      0
  d51           dd      0       ;lower right point color
  d52           dd      0
  d61           dd      0       ;lower line color
  d62           dd      0
  d71           dd      0       ;lower left point color
  d72           dd      0
  d81           dd      0       ;left line color
  d82           dd      0
  d9            dd      0       ;internal area color
ends

sBoxStandard8   box <29,30, 29,30, 26,26, 20,21, 20,21, 20,21, 26,26, 29,30, 26>
sBoxInverted8   box <21,20, 21,20, 26,26, 30,29, 30,29, 30,29, 26,26, 21,20, 26>
sBoxThinStd8    box <30,26, 30,26, 26,26, 00,26, 00,26, 00,26, 26,26, 30,26, 26>
sBoxThinInv8    box <00,26, 00,26, 26,26, 30,26, 30,26, 30,26, 26,26, 00,26, 26>
C1              =   (((0EFh SHR 3) SHL 10) + ((0EFh SHR 3) SHL 5) + (0EFh SHR 3))
C2              =   (((0CFh SHR 3) SHL 10) + ((0CFh SHR 3) SHL 5) + (0CFh SHR 3))
C3              =   (((0AFh SHR 3) SHL 10) + ((0AFh SHR 3) SHL 5) + (0AFh SHR 3))
C4              =   (((04Fh SHR 3) SHL 10) + ((04Fh SHR 3) SHL 5) + (04Fh SHR 3))
C5              =   (((02Fh SHR 3) SHL 10) + ((02Fh SHR 3) SHL 5) + (02Fh SHR 3))
C6              =   (((000h SHR 3) SHL 10) + ((000h SHR 3) SHL 5) + (000h SHR 3))
sBoxStandard15  box <C2,C1, C2,C1, C3,C3, C5,C4, C5,C4, C5,C4, C3,C3, C2,C1, C3>
sBoxInverted15  box <C4,C5, C4,C5, C3,C3, C1,C2, C1,C2, C1,C2, C3,C3, C4,C5, C3>
sBoxThinStd15   box <C1,C3, C1,C3, C3,C3, C6,C3, C6,C3, C6,C3, C3,C3, C1,C3, C3>
sBoxThinInv15   box <C6,C3, C6,C3, C3,C3, C1,C3, C1,C3, C1,C3, C3,C3, C6,C3, C3>
C1              =   (((0EFh SHR 3) SHL 11) + ((0EFh SHR 2) SHL 5) + (0EFh SHR 3))
C2              =   (((0CFh SHR 3) SHL 11) + ((0CFh SHR 2) SHL 5) + (0CFh SHR 3))
C3              =   (((0AFh SHR 3) SHL 11) + ((0AFh SHR 2) SHL 5) + (0AFh SHR 3))
C4              =   (((04Fh SHR 3) SHL 11) + ((04Fh SHR 2) SHL 5) + (04Fh SHR 3))
C5              =   (((02Fh SHR 3) SHL 11) + ((02Fh SHR 2) SHL 5) + (02Fh SHR 3))
C6              =   (((000h SHR 3) SHL 11) + ((000h SHR 2) SHL 5) + (000h SHR 3))
sBoxStandard16  box <C2,C1, C2,C1, C3,C3, C5,C4, C5,C4, C5,C4, C3,C3, C2,C1, C3>
sBoxInverted16  box <C4,C5, C4,C5, C3,C3, C1,C2, C1,C2, C1,C2, C3,C3, C4,C5, C3>
sBoxThinStd16   box <C1,C3, C1,C3, C3,C3, C6,C3, C6,C3, C6,C3, C3,C3, C1,C3, C3>
sBoxThinInv16   box <C6,C3, C6,C3, C3,C3, C1,C3, C1,C3, C1,C3, C3,C3, C6,C3, C3>
C1              =   0EFEFEFh
C2              =   0CFCFCFh
C3              =   0AFAFAFh
C4              =   04F4F4Fh
C5              =   02F2F2Fh
C6              =   0000000h
sBoxStandard24  box <C2,C1, C2,C1, C3,C3, C5,C4, C5,C4, C5,C4, C3,C3, C2,C1, C3>
sBoxInverted24  box <C4,C5, C4,C5, C3,C3, C1,C2, C1,C2, C1,C2, C3,C3, C4,C5, C3>
sBoxThinStd24   box <C1,C3, C1,C3, C3,C3, C6,C3, C6,C3, C6,C3, C3,C3, C1,C3, C3>
sBoxThinInv24   box <C6,C3, C6,C3, C3,C3, C1,C3, C1,C3, C1,C3, C3,C3, C6,C3, C3>



proc            GfxBoxStandard
                ;in
                ;  edi - 'Win'
                ;out
                ;  nothing
                push    esi

                cmp     [(win edi).dColorScheme], COLOR24
                @IF_AE
                  lea   esi, [sBoxStandard24]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR16
                @IF_E
                  lea   esi, [sBoxStandard16]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR15
                @IF_E
                  lea   esi, [sBoxStandard15]
                  jmp   @@draw
                @ENDIF
                lea     esi, [sBoxStandard8]
@@draw:         call    GfxBoxFill

                pop     esi
                ret
endp



proc            GfxBoxInverted
                ;in
                ;  edi - 'Win'
                ;out
                ;  nothing
                push    esi

                cmp     [(win edi).dColorScheme], COLOR24
                @IF_AE
                  lea   esi, [sBoxInverted24]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR16
                @IF_E
                  lea   esi, [sBoxInverted16]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR15
                @IF_E
                  lea   esi, [sBoxInverted15]
                  jmp   @@draw
                @ENDIF
                lea     esi, [sBoxInverted8]
@@draw:         call    GfxBoxFill

                pop     esi
                ret
endp



proc            GfxBoxThinStd
                ;in
                ;  edi - 'Win'
                ;out
                ;  nothing
                push    esi

                cmp     [(win edi).dColorScheme], COLOR24
                @IF_AE
                  lea   esi, [sBoxThinStd24]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR16
                @IF_E
                  lea   esi, [sBoxThinStd16]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR15
                @IF_E
                  lea   esi, [sBoxThinStd15]
                  jmp   @@draw
                @ENDIF
                lea     esi, [sBoxThinStd8]
@@draw:         call    GfxBoxFill

                pop     esi
                ret
endp



proc            GfxBoxThinInv
                ;in
                ;  edi - 'Win'
                ;out
                ;  nothing
                push    esi

                cmp     [(win edi).dColorScheme], COLOR24
                @IF_AE
                  lea   esi, [sBoxThinInv24]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR16
                @IF_E
                  lea   esi, [sBoxThinInv16]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR15
                @IF_E
                  lea   esi, [sBoxThinInv15]
                  jmp   @@draw
                @ENDIF
                lea     esi, [sBoxThinInv8]
@@draw:         call    GfxBoxFill

                pop     esi
                ret
endp



proc            GfxBoxDrawThinStd
                ;in
                ;  eax - row
                ;  ebx - col
                ;  ecx - total number of rows
                ;  edx - total number of columns
                ;  edi - 'Win'
                ;out
                ;  nothing
                push    esi

                cmp     [(win edi).dColorScheme], COLOR24
                @IF_AE
                  lea   esi, [sBoxThinStd24]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR16
                @IF_E
                  lea   esi, [sBoxThinStd16]
                  jmp   @@draw
                @ENDIF
                cmp     [(win edi).dColorScheme], COLOR15
                @IF_E
                  lea   esi, [sBoxThinStd15]
                  jmp   @@draw
                @ENDIF
                lea     esi, [sBoxThinStd8]
@@draw:         call    GfxBoxDraw

                pop     esi
                ret
endp



proc            GfxBoxFill
                ;in
                ;  esi - 'Box'
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                mov     eax, [(box esi).d9]
                xchg    esi, edi
                call    WinFill
                xchg    esi, edi

                mov     eax, [(win edi).dShiftRow]
                mov     ebx, [(win edi).dShiftCol]
                mov     ecx, [(win edi).dVisibleRow]
                mov     edx, [(win edi).dVisibleCol]
                call    GfxBoxDraw

                popad
                ret
endp



dRow            dd      0
dCol            dd      0
dTotalRow       dd      0
dTotalCol       dd      0
proc            GfxBoxDraw
                ;in
                ;  eax - row
                ;  ebx - col
                ;  ecx - total number of rows
                ;  edx - total number of columns
                ;  esi - ptr to 'box'
                ;  edi - 'Win'
                ;out
                ;  nothing
                pushad

                mov     [dRow], eax
                mov     [dCol], ebx
                mov     [dTotalRow], ecx
                mov     [dTotalCol], edx

                ;;plot upper left point 1
                mov     eax, [dRow]
                mov     ebx, [dCol]
                mov     ecx, [(box esi).d11]
                call    GfxSetPoint

                ;;draw upper horizontal line 1
                mov     eax, [dRow]
                mov     ebx, [dCol]
                inc     ebx
                mov     ecx, [dTotalCol]
                sub     ecx, 2
                mov     ebp, [(box esi).d21]
                call    GfxHLine

                ;;plot upper right point 1
                mov     eax, [dRow]
                mov     ebx, [dCol]
                add     ebx, [dTotalCol]
                dec     ebx
                mov     ecx, [(box esi).d31]
                call    GfxSetPoint

                ;;draw right vertical line 1
                mov     eax, [dRow]
                inc     eax
                mov     ebx, [dCol]
                add     ebx, [dTotalCol]
                dec     ebx
                mov     ecx, [dTotalRow]
                sub     ecx, 2
                mov     ebp, [(box esi).d41]
                call    GfxVLine

                ;;plot lower right point 1
                mov     eax, [dRow]
                add     eax, [dTotalRow]
                dec     eax
                mov     ebx, [dCol]
                add     ebx, [dTotalCol]
                dec     ebx
                mov     ecx, [(box esi).d51]
                call    GfxSetPoint

                ;;draw lower horizontal line 1
                mov     eax, [dRow]
                add     eax, [dTotalRow]
                dec     eax
                mov     ebx, [dCol]
                inc     ebx
                mov     ecx, [dTotalCol]
                sub     ecx, 2
                mov     ebp, [(box esi).d61]
                call    GfxHLine

                ;;plot lower left point 1
                mov     eax, [dRow]
                add     eax, [dTotalRow]
                dec     eax
                mov     ebx, [dCol]
                mov     ecx, [(box esi).d71]
                call    GfxSetPoint

                ;;draw left vertical line 1
                mov     eax, [dRow]
                inc     eax
                mov     ebx, [dCol]
                mov     ecx, [dTotalRow]
                sub     ecx, 2
                mov     ebp, [(box esi).d81]
                call    GfxVLine

                ;;plot upper left point 2
                mov     eax, [dRow]
                inc     eax
                mov     ebx, [dCol]
                inc     ebx
                mov     ecx, [(box esi).d12]
                call    GfxSetPoint

                ;;draw upper horizontal line 2
                mov     eax, [dRow]
                inc     eax
                mov     ebx, [dCol]
                add     ebx, 2
                mov     ecx, [dTotalCol]
                sub     ecx, 4
                mov     ebp, [(box esi).d22]
                call    GfxHLine

                ;;plot upper right point 2
                mov     eax, [dRow]
                inc     eax
                mov     ebx, [dCol]
                add     ebx, [dTotalCol]
                sub     ebx, 2
                mov     ecx, [(box esi).d32]
                call    GfxSetPoint

                ;;draw right vertical line 2
                mov     eax, [dRow]
                add     eax, 2
                mov     ebx, [dCol]
                add     ebx, [dTotalCol]
                sub     ebx, 2
                mov     ecx, [dTotalRow]
                sub     ecx, 4
                mov     ebp, [(box esi).d42]
                call    GfxVLine

                ;;plot lower right point 2
                mov     eax, [dRow]
                add     eax, [dTotalRow]
                sub     eax, 2
                mov     ebx, [dCol]
                add     ebx, [dTotalCol]
                sub     ebx, 2
                mov     ecx, [(box esi).d52]
                call    GfxSetPoint

                ;;draw lower horizontal line 2
                mov     eax, [dRow]
                add     eax, [dTotalRow]
                sub     eax, 2
                mov     ebx, [dCol]
                add     ebx, 2
                mov     ecx, [dTotalCol]
                sub     ecx, 4
                mov     ebp, [(box esi).d62]
                call    GfxHLine

                ;;plot lower left point 2
                mov     eax, [dRow]
                add     eax, [dTotalRow]
                sub     eax, 2
                mov     ebx, [dCol]
                inc     ebx
                mov     ecx, [(box esi).d72]
                call    GfxSetPoint

                ;;draw left vertical line 2
                mov     eax, [dRow]
                add     eax, 2
                mov     ebx, [dCol]
                inc     ebx
                mov     ecx, [dTotalRow]
                sub     ecx, 4
                mov     ebp, [(box esi).d82]
                call    GfxVLine

                popad
                ret
endp

ends            code32
                end
