;*****
; font.asm - font handling.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __FONT__ equ 1
                include "pmlib.asd"
                include "font.asi"
                include "fontmx.asi"
                include "color.asi"
                include "debug.asi"
                include "error.asi"
                include "heap.asi"
                include "str.asi"
                include "win.asi"



FONT_SIGN       equ     "Font"  ;signature for run-time checks

struc           font            ;Font
  dSign         dd  FONT_SIGN   ;signature
  dFontmx       dd      0       ;'Fontmx'
  dPalette      dd      0       ;font's palette
  dBytesPerCell dd      0       ;number of bytes per cell
  dColSpace     dd      0       ;space between characters
  dRowBuffer    dd      0       ;ptr to matrix row buffer
  fnPrintChar   dd      0       ;print function
ends



adPrintFuncs    dd      PrintFix1, PrintPro1, PrintFixSolid1, PrintProSolid1
                dd      PrintFix2, PrintPro2, PrintFixSolid2, PrintProSolid2
                dd      PrintFix3, PrintPro3, PrintFixSolid3, PrintProSolid3
                dd      PrintFix4, PrintPro4, PrintFixSolid4, PrintProSolid4



;;temporary variables
dChar           dd      0       ;character being printed
dRowBuffer      dd      0       ;ptr to matrix row buffer
dColSpace       dd      0       ;space between characters
dTotalCol       dd      0       ;number of columns in 'Win'
dDstDelta       dd      0       ;offset to next row
dOfs            dd      0       ;offset to character image
dLen            dd      0       ;character length



;;error messages
RUS_MESSAGE     cErrNotFont     "ꥪ  ਭ  'Font'."
ENG_MESSAGE     cErrNotFont     "Object is not a 'Font'."



proc            FontNew
                ;in
                ;  eax - space between characters (0 - fixed font)
                ;  ebx - font transparency (0 - transparent)
                ;  ecx - color scheme
                ;  esi - 'Fontmx'
                ;  edi - font's palette
                ;out
                ;  edi - 'Font'
                push    ebx ecx

                push    eax
                push    edi

                ;;allocate memory for matrix row buffer
                mov     eax, [(fontmx esi).dBytesPerRow]
                inc     eax
                call    HeapAlloc
                push    edi

                ;;allocate memory for 'Font'
                mov     eax, size font
                call    HeapAlloc

                ;;initialize variables
                mov     [(font edi).dSign], FONT_SIGN
                pop     [(font edi).dRowBuffer]
                pop     [(font edi).dPalette]
                mov     ecx, [adBytesPerCell+ecx*4]
                mov     [(font edi).dBytesPerCell], ecx
                pop     eax
                mov     [(font edi).dColSpace], eax
                mov     [(font edi).dFontmx], esi

                ;;select print function
                and     eax, eax
                setnz   al
                and     ebx, ebx
                setnz   bl
                dec     ecx
                shl     ecx, 2
                add     cl, bl
                add     cl, bl
                add     cl, al
                mov     eax, [adPrintFuncs+ecx*4]
                mov     [(font edi).fnPrintChar], eax

                ;;create palette according to current color scheme
                push    edi
                mov     ebx, [(font edi).dPalette]
                mov     eax, 4
                mov     ecx, [(fontmx esi).dBitPerPixel]
                shl     eax, cl
                call    HeapAlloc
                push    edi
                mov     eax, 1
                shl     eax, cl
                mov     ecx, eax
@@loop:         mov     eax, [ebx]
                add     ebx, 4
                call    ColorTrueToCurrent
                mov     [edi], eax
                add     edi, 4
                loop    @@loop
                pop     ebx
                pop     edi
                mov     [(font edi).dPalette], ebx

                pop     ecx ebx
                ret
endp



proc            FontDelete
                ;in
                ;  esi - 'Font'
                ;out
                ;  nothing
                mov     eax, [(font esi).dPalette]
                call    HeapFree
                mov     eax, [(font esi).dRowBuffer]
                call    HeapFree
                mov     eax, esi
                call    HeapFree
                ret
endp



proc            FontCheck
                ;in
                ;  esi - 'Font'
                ;out
                ;  nothing
                cmp     [(font esi).dSign], FONT_SIGN
                @IF_NE
                  lea   esi, [cErrNotFont]
                  call  ErrorFatal
                @ENDIF
                ret
endp



proc            FontGetRowPerChar
                ;in
                ;  esi - 'Font'
                ;out
                ;  eax - number of rows per char
                IF DBG
                  call  FontCheck
                ENDIF
                mov     eax, [(font esi).dFontmx]
                mov     eax, [(fontmx eax).dRowPerChar]
                ret
endp



proc            FontGetColPerChar
                ;in
                ;  esi - 'Font'
                ;out
                ;  eax - number of columns per char
                IF DBG
                  call  FontCheck
                ENDIF
                mov     eax, [(font esi).dFontmx]
                mov     eax, [(fontmx eax).dColPerChar]
                ret
endp



proc            FontGetColSpace
                ;in
                ;  esi - 'Font'
                ;out
                ;  eax - number of columns between chars
                IF DBG
                  call  FontCheck
                ENDIF
                mov     eax, [(font esi).dColSpace]
                ret
endp



proc            FontGetPaletteEntry
                ;in
                ;  eax - number of color in font palette
                ;  esi - 'Font'
                ;out
                ;  eax - color
                IF DBG
                  call  FontCheck
                ENDIF
                shl     eax, 2
                add     eax, [(font esi).dPalette]
                mov     eax, [eax]
                ret
endp



proc            FontGetStrLen
                ;in
                ;  esi - 'Font'
                ;  edi - 'Str'
                ;out
                ;  eax - string length in pixels
                push    ebx ecx edx edi ebp

                xor     eax, eax
                mov     ebx, [(font esi).dColSpace]
                xor     ecx, ecx
                mov     edx, [(font esi).dFontmx]

                and     ebx, ebx
                @IF_E
                  ;;fixed font
                  mov   edx, [(fontmx edx).dColPerChar]
                  mov   edi, [(str edi).dLength]
@@loop1:          add   ecx, edx
                  dec   edi
                  jnz   @@loop1
                @ELSE
                  ;;proportional font
                  mov   edx, [(fontmx edx).dLenPtr]
                  mov   ebp, [(str edi).dLength]
                  mov   edi, [(str edi).dDataPtr]
@@loop2:          mov   al, [edi]
                  inc   edi
                  mov   al, [edx+eax]
                  add   ecx, eax
                  add   ecx, ebx
                  dec   ebp
                  jnz   @@loop2
                @ENDIF
                sub     ecx, ebx
                mov     eax, ecx
                pop     ebp edi edx ecx ebx
                ret
endp



proc            FontGetStrzLen
                ;in
                ;  esi - 'Font'
                ;  edi - 'Strz'
                ;out
                ;  eax - string length in pixels
                push    ebx ecx edx edi

                xor     eax, eax
                mov     ebx, [(font esi).dColSpace]
                xor     ecx, ecx
                mov     edx, [(font esi).dFontmx]

                and     ebx, ebx
                @IF_E
                  ;;fixed font
                  mov   edx, [(fontmx edx).dColPerChar]
@@loop1:          mov   al, [edi]
                  inc   edi
                  and   al, al
                  @IF_NZ
                    add ecx, edx
                    jmp @@loop1
                  @ENDIF
                @ELSE
                  ;;proportional font
                  mov   edx, [(fontmx edx).dLenPtr]
@@loop2:          mov   al, [edi]
                  inc   edi
                  and   al, al
                  @IF_NZ
                    mov al, [edx+eax]
                    add ecx, eax
                    add ecx, ebx
                    jmp @@loop2
                  @ENDIF
                @ENDIF
                sub     ecx, ebx
                mov     eax, ecx
                pop     edi edx ecx ebx
                ret
endp



proc            FontPrintStr
                ;in
                ;  ebx - row (-1 to print in center)
                ;  ecx - column (-1 to print in center)
                ;  edx - 'Str'
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebp

                ;;select row
                and     ebx, ebx
                @IF_S
                  call  FontGetRowPerChar
                  mov   ebx, [(win edi).dTotalRow]
                  sub   ebx, eax
                  shr   ebx, 1
                @ENDIF

                ;;select column
                and     ecx, ecx
                @IF_S
                  push  edi
                  mov   edi, edx
                  call  FontGetStrLen
                  pop   edi
                  mov   ecx, [(win edi).dTotalCol]
                  sub   ecx, eax
                  shr   ecx, 1
                @ENDIF

                xor     ebp, ebp
@@loop:         cmp     ebp, [(str edx).dLength]
                @IF_NE
                  mov   eax, [(str edx).dDataPtr]
                  mov   al, [ds:ebp+eax]
                  call  [(font esi).fnPrintChar]
                  inc   ebp
                  jmp   @@loop
                @ENDIF

                pop     ebp
                ret
endp



proc            FontPrintStrz
                ;in
                ;  ebx - row (-1 to print in center)
                ;  ecx - column (-1 to print in center)
                ;  edx - 'Strz'
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                pushad

                call    FontCheck

                ;;select row
                and     ebx, ebx
                @IF_S
                  call  FontGetRowPerChar
                  mov   ebx, [(win edi).dTotalRow]
                  sub   ebx, eax
                  shr   ebx, 1
                @ENDIF

                ;;select column
                and     ecx, ecx
                @IF_S
                  push  edi
                  mov   edi, edx
                  call  FontGetStrzLen
                  pop   edi
                  mov   ecx, [(win edi).dTotalCol]
                  sub   ecx, eax
                  shr   ecx, 1
                @ENDIF

@@loop:         mov     al, [edx]
                inc     edx
                and     al, al
                @IF_NZ
                  call  [(font esi).fnPrintChar]
                  jmp   @@loop
                @ENDIF
                popad
                ret
endp



proc            FontPrintChar
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                call    [(font esi).fnPrintChar]
                ret
endp



proc            PrintFix1
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], al
                @ENDIF
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFix2
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                shl     eax, 1
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], ax
                @ENDIF
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFix3
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                lea     eax, [eax*2+eax]
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], ax
                  shr   eax, 16
                  mov   [edi+2], al
                @ENDIF
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFix4
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                shl     eax, 2
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], eax
                @ENDIF
                inc     edi
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintPro1
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns loop
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], al
                @ENDIF
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintPro2
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                shl     eax, 1
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns loop
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], ax
                @ENDIF
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintPro3
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                lea     eax, [eax*2+eax]
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns loop
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], ax
                  shr   eax, 16
                  mov   [edi+2], al
                @ENDIF
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintPro4
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                shl     eax, 2
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns loop
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                and     eax, eax
                @IF_NZ
                  mov   eax, [ds:ebp+eax*4]
                  mov   [edi], eax
                @ENDIF
                inc     edi
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFixSolid1
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], al
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFixSolid2
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                shl     eax, 1
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], ax
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFixSolid3
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                lea     eax, [eax*2+eax]
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], ax
                shr     eax, 16
                mov     [edi+2], al
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintFixSolid4
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                sub     eax, [(fontmx esi).dColPerChar]
                shl     eax, 2
                mov     [dDstDelta], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [(fontmx esi).dColPerChar]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx
                ;;initialize row buffer
                push    edi
                mov     edi, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
                @DO
                  mov   al, [ebx]
                  inc   ebx
                  mov   [edi], al
                  inc   edi
                @DJNZ   ecx
                pop     edi
                ;;columns loop
                mov     ecx, [(fontmx esi).dColPerChar]
@@loop3:        push    ecx
                ;;read one pixel
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
                push    edi
                @DO
                  mov   edi, [dRowBuffer]
                  mov   edx, [(fontmx esi).dBytesPerRow]
@@loop5:          rcl   [bptr edi+edx-1], 1
                  dec   edx
                  jnz   @@loop5
                  rcl   al, 1
                @DJNZ   ecx
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], eax
                inc     edi
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop3
                ;;
                add     edi, [dDstDelta]
                pop     ecx
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [(fontmx esi).dColPerChar]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintProSolid1
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     eax, [(font esi).dColSpace]
                mov     [dColSpace], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns row
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], al
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;fill space between pixels
                mov     al, [ds:ebp]
                mov     ecx, [dColSpace]
                rep     stosb
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintProSolid2
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     eax, [(font esi).dColSpace]
                mov     [dColSpace], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                shl     eax, 1
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns row
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], ax
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;fill space between pixels
                mov     ax, [ds:ebp]
                mov     ecx, [dColSpace]
                rep     stosw
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintProSolid3
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     eax, [(font esi).dColSpace]
                mov     [dColSpace], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                lea     eax, [eax*2+eax]
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns row
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], ax
                shr     eax, 16
                mov     [edi+2], al
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;fill space between pixels
                mov     eax, [ds:ebp]
                mov     ecx, [dColSpace]
                @DO
                  mov   [edi], ax
                  shr   eax, 16
                  mov   [edi+2], al
                  inc   edi
                  inc   edi
                  inc   edi
                @DJNZ   ecx
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                dec     ecx
                jnz     @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp



proc            PrintProSolid4
                ;in
                ;   al - character to print
                ;  ebx - row
                ;  ecx - column
                ;  esi - 'Font'
                ;  edi - 'Win'
                ;out
                ;  ecx - new column
                push    ebx edx edi ebp
                push    esi ecx

                ;;initialization
                mov     [bptr dChar], al
                mov     eax, [(font esi).dRowBuffer]
                mov     [dRowBuffer], eax
                mov     eax, [(font esi).dColSpace]
                mov     [dColSpace], eax
                mov     ebp, [(font esi).dPalette]
                mov     esi, [(font esi).dFontmx]
                mov     eax, [(win edi).dTotalCol]
                shl     eax, 2
                mov     [dTotalCol], eax
                ;
                xor     eax, eax
                mov     edx, [(fontmx esi).dOfsPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dOfs], eax
                mov     edx, [(fontmx esi).dLenPtr]
                add     edx, [dChar]
                mov     al, [edx]
                mov     [dLen], eax

                ;;validate coordinates
                mov     eax, ebx
                add     eax, [(fontmx esi).dRowPerChar]
                cmp     eax, [(win edi).dTotalRow]
                setae   dl
                mov     eax, ecx
                add     eax, [dLen]
                cmp     eax, [(win edi).dTotalCol]
                setae   dh
                or      dl, dh
                jnz     @@exit

                ;;get address of upper left point
                mov     eax, ebx
                mul     [(win edi).dTotalCol]
                mov     edi, [(win edi).dImagePtr]
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, eax
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx
                add     edi, ecx

                ;;;{main print loop}
                mov     eax, [dChar]
                mul     [(fontmx esi).dBytesPerChar]
                mov     ebx, [(fontmx esi).dImagePtr]
                add     ebx, eax
                ;;rows loop
                mov     ecx, [(fontmx esi).dRowPerChar]
@@loop1:        push    ecx edi
                ;;initialize row buffer
                mov     edx, [dRowBuffer]
                mov     ecx, [(fontmx esi).dBytesPerRow]
@@loop2:        mov     al, [ebx]
                inc     ebx
                mov     [edx], al
                inc     edx
                loop    @@loop2
                ;;
                mov     eax, [(fontmx esi).dBitPerPixel]
                mul     [dOfs]
                and     eax, eax
@@loop3:        @IF_NZ
                  mov   edx, [dRowBuffer]
                  mov   ecx, [(fontmx esi).dBytesPerRow]
@@loop4:          rcl   [bptr edx+ecx-1], 1
                  loop  @@loop4
                  dec   eax
                  jmp   @@loop3
                @ENDIF
                ;;columns row
                mov     ecx, [dLen]
@@loop5:        push    ecx
                ;;read one pixel
                push    edi
                xor     eax, eax
                mov     ecx, [(fontmx esi).dBitPerPixel]
@@loop6:        mov     edi, [dRowBuffer]
                mov     edx, [(fontmx esi).dBytesPerRow]
@@loop7:        rcl     [bptr edi+edx-1], 1
                dec     edx
                jnz     @@loop7
                rcl     al, 1
                loop    @@loop6
                pop     edi
                ;;put pixel to screen
                mov     eax, [ds:ebp+eax*4]
                mov     [edi], eax
                inc     edi
                inc     edi
                inc     edi
                inc     edi
                ;;
                pop     ecx
                loop    @@loop5
                ;;fill space between pixels
                mov     eax, [ds:ebp]
                mov     ecx, [dColSpace]
                rep     stosd
                ;;
                pop     edi ecx
                add     edi, [dTotalCol]
                loop    @@loop1
                ;;;{end of main print loop}

@@exit:         pop     ecx
                add     ecx, [dLen]
                pop     esi
                add     ecx, [(font esi).dColSpace]
                pop     ebp edi edx ebx
                ret
endp

ends            code32
                end
