;*****
; file.asm  -  file functions with input/output buffering.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __FILE__ equ 1
                include "pmlib.asd"
                include "api.asi"
                include "debug.asi"
                include "error.asi"
                include "file.asi"
                include "lm.asi"



;;constants
MODE_R          = 0000h         ;file opened for read only (DOS)
MODE_W          = 0001h         ;file opened for write only (DOS)
MODE_RW         = 0002h         ;file opened for both read and write (DOS)
DIRTY_FLAG      = 0100h         ;dirty buffer flag
SIGN            = 'FHnd'        ;signature for run-time checks

struc           fh              ;File Handler
dSign           dd      0       ;signature
wMode           dw      0       ;mode
wDosFileHandle  dw      0       ;DOS handle
dDosFileOffset  dd      0       ;current DOS ptr
dBufferOffset   dd      0       ;DOS ptr of buffer's start
dBufferPtr      dd      0       ;ptr to buffer's data
dBufferSize     dd      0       ;buffer size
dMaxBufferSize  dd      0       ;maximal buffer size
ends

struc           om              ;Open Mode
wChar           dw      0       ;character mode description
wMode           dw      0       ;numeric mode description
wAction         dw      0       ;action flags
ends

asModes         om <'r', MODE_R, 01h>   ;if exist-open, otherwise-error
                om <'w', MODE_W, 12h>   ;if exist-replace, otherwise-create
                om <'rw',MODE_RW,11h>   ;if exist-open, otherwise-create

dBufferSize     dd      1000h   ;buffer size

RUS_MESSAGE     cErrBadHandle   " ਯ 䠩."
ENG_MESSAGE     cErrBadHandle   "Bad file handle."
RUS_MESSAGE     cErrBadMode     " ०  䠩."
ENG_MESSAGE     cErrBadMode     "Bad file open mode."
RUS_MESSAGE     cErrOpen        "訡  䠩."
ENG_MESSAGE     cErrOpen        "File open error."
RUS_MESSAGE     cErrClose       "訡  䠩."
ENG_MESSAGE     cErrClose       "File close."
RUS_MESSAGE     cErrRead        "訡 ⥭ 䠩."
ENG_MESSAGE     cErrRead        "File read error."
RUS_MESSAGE     cErrWrite       "訡   䠩."
ENG_MESSAGE     cErrWrite       "File write error."
RUS_MESSAGE     cErrSeek        "訡 ६饭  䠩."
ENG_MESSAGE     cErrSeek        "File seek error."
RUS_MESSAGE     cErrOpenForRd   "  ⮫쪮  ⥭."
ENG_MESSAGE     cErrOpenForRd   "File opened for reading only."
RUS_MESSAGE     cErrOpenForWr   "  ⮫쪮  ."
ENG_MESSAGE     cErrOpenForWr   "File opened for writing only."



proc            FileBufferSize
                ;in
                ;  eax - buffer size
                ;out
                ;  eax - previous buffer size
                xchg    eax, [dBufferSize]
                ret
endp



proc            FileAttributes
                ;in
                ;  esi - 'Str' object containing file name
                ;out
                ;  CY: error
                ;    ax - error code
                ;  CN: success
                ;    cx - file attributes
                ;description
                ;  Searches requested file and returns its attributes.
                ;  You may use it to determine does the file exists.

                ;;copy file name to low buffer
                call    ApiStrToLowBuffer

                mov     ax, 04300h
                mov     bp, [wLowBufferSeg]     ;filename segment
                mov     dx, [wLowBufferOff]     ;filename offset
                call    ApiInt21h

                ret
endp



proc            FileOpen
                ;in
                ;  ax='r'  - open for read only
                ;  ax='w'  - open for write only
                ;  ax='rw' - open for both read and write
                ;  esi - 'Str' with file name
                ;out
                ;  esi - 'File'
                push    ebx ecx edx edi ebp

                ;;determine open mode
                mov     ecx, 3
                lea     edi, [asModes]
@@loop1:        cmp     ax, [(om edi).wChar]
                @IF_NE
                  add   edi, size om
                  loop  @@loop1
                  lea   esi, [cErrBadMode]
                  call  ErrorFatal
                @ENDIF
                push    edi

                ;;copy file name to low buffer
                call    ApiStrToLowBuffer

                ;;create 'File' object
                mov     ebx, [dBufferSize]
                mov     eax, ebx
                add     eax, size fh
                call    LmAlloc
                mov     [(fh edi).dSign], SIGN
                mov     [(fh edi).dMaxBufferSize], ebx

                ;;open file
                pop     esi
                mov     ah, 6Ch                 ;extended open
                mov     bx, [(om esi).wMode]    ;open mode
                mov     [(fh edi).wMode], bx
                xor     ecx, ecx                ;standard attributes
                mov     dx, [(om esi).wAction]  ;actions
                mov     bp, [wLowBufferSeg]     ;filename segment
                mov     si, [wLowBufferOff]     ;filename offset
                call    ApiInt21h
                @IF_C
                  mov   eax, edi
                  call  LmFree
                  lea   esi, [cErrOpen]
                  call  ErrorFatal
                @ENDIF
                mov     esi, edi
                mov     [(fh esi).wDosFileHandle], ax
                xor     eax, eax
                mov     [(fh esi).dDosFileOffset], eax
                call    LoadBuffer

                pop     ebp edi edx ecx ebx
                ret
endp



proc            FileClose
                ;in
                ;  esi - 'File'
                ;out
                ;  nothing
                push    ebx

                call    CheckHandle
                call    FlushBuffer

                ;;close file
                mov     ah, 3Eh
                mov     bx, [(fh esi).wDosFileHandle]
                int     21h
                @IF_C
                  lea   esi, [cErrClose]
                  call  ErrorFatal
                @ENDIF

                mov     [(fh esi).dSign], 0
                mov     eax, esi
                call    LmFree

                pop     ebx
                ret
endp



proc            FileSize
                ;in
                ;  esi - 'File'
                ;out
                ;  eax - size
                push    ebx ecx edx

                call    CheckHandle

                ;;skip to the end of file
                mov     ax, 4202h
                mov     bx, [(fh esi).wDosFileHandle]
                xor     ecx, ecx
                xor     edx, edx
                int     21h
                @IF_C
                  lea   esi, [cErrSeek]
                  call  ErrorFatal
                @ENDIF

                ;;push file size
                push    dx ax

                ;;skip to previous location
                mov     ax, 4200h
                mov     bx, [(fh esi).wDosFileHandle]
                mov     ecx, [(fh esi).dDosFileOffset]
                mov     edx, ecx
                shr     ecx, 16
                int     21h
                @IF_C
                  lea   esi, [cErrSeek]
                  call  ErrorFatal
                @ENDIF

                ;;pop file size
                pop     eax

                pop     edx ecx ebx
                ret
endp



proc            FileSeek
                ;in
                ;  esi - 'File'
                ;  eax - number of bytes from current position
                ;out
                ;  nothing
                pushad

                call    CheckHandle

                add     eax, [(fh esi).dBufferPtr]
                js      @@seek
                cmp     eax, [(fh esi).dBufferSize]
                jae     @@seek
                mov     [(fh esi).dBufferPtr], eax
                jmp     @@exit

@@seek:         call    FlushBuffer
                add     eax, [(fh esi).dBufferOffset]
                sub     eax, [(fh esi).dDosFileOffset]
                jz      @@load

                mov     edx, eax
                shld    ecx, eax, 16
                mov     bx, [(fh esi).wDosFileHandle]
                mov     ax, 4201h                       ;seek function
                int     21h
                @IF_C
                  lea   esi, [cErrSeek]
                  call  ErrorFatal
                @ENDIF
                push    dx ax
                pop     [(fh esi).dDosFileOffset]

@@load:         call    LoadBuffer

@@exit:         popad
                ret
endp



proc            FileReadStr
                ;in
                ;  eax - number of bytes to red
                ;  esi - 'File'
                ;  edi - ptr to buffer to receive string
                ;out
                ;  eax - number of bytes actually read
                ;  CN: - success
                ;  CY: - error
                push    ebx ecx edx edi

                mov     ebx, eax
                call    CheckHandle
                call    CanRead

                ;;main loop
                xor     edx, edx
@@loop:         and     ebx, ebx
                jz      @@exit

                ;;check for buffer or file end
                mov     eax, [(fh esi).dBufferSize]
                cmp     [(fh esi).dBufferPtr], eax
                @IF_AE
                  call  FlushBuffer
                  call  LoadBuffer
                  jc    @@exit
                @ENDIF

                ;;read from buffer
                mov     ecx, [(fh esi).dBufferSize]
                sub     ecx, [(fh esi).dBufferPtr]
                @MIN    ecx, ebx
                sub     ebx, ecx
                add     edx, ecx
                push    ecx esi
                add     esi, [(fh esi).dBufferPtr]
                add     esi, size fh
                rep     movsb
                pop     esi ecx
                add     [(fh esi).dBufferPtr], ecx

                jmp     @@loop

@@exit:         mov     eax, edx
                pop     edi edx ecx ebx
                ret
endp



proc            FileReadByte
                ;in
                ;  esi - 'File'
                ;out
                ;  CN - success
                ;     al - byte
                ;  CY - End Of File
                call    CheckHandle
                call    CanRead

                ;;check for buffer or file end
                mov     eax, [(fh esi).dBufferSize]
                cmp     [(fh esi).dBufferPtr], eax
                @IF_E
                  call  FlushBuffer
                  call  LoadBuffer
                  jc    @@exit
                @ENDIF

                mov     eax, [(fh esi).dBufferPtr]
                add     eax, size fh
                mov     al, [esi+eax]
                inc     [(fh esi).dBufferPtr]

@@exit:         ret
endp



proc            FileWriteStr
                ;in
                ;  eax - number of bytes
                ;  esi - 'File'
                ;  edi - ptr to string
                ;out
                ;  nothing
                push    ebx ecx edi

                mov     ebx, eax
                call    CheckHandle
                call    CanWrite

                ;;main loop
@@loop:         and     ebx, ebx
                jz      @@exit

                ;;check if we are within the buffer
                mov     eax, [(fh esi).dMaxBufferSize]
                cmp     [(fh esi).dBufferPtr], eax
                @IF_E
                  call  FlushBuffer
                  call  LoadBuffer
                @ENDIF

                ;;write to buffer
                or      [(fh esi).wMode], DIRTY_FLAG
                mov     ecx, [(fh esi).dMaxBufferSize]
                sub     ecx, [(fh esi).dBufferPtr]
                cmp     ecx, ebx
                @IF_A
                  mov   ecx, ebx
                @ENDIF
                sub     ebx, ecx
                push    ecx esi
                add     esi, [(fh esi).dBufferPtr]
                add     esi, size fh
                xchg    esi, edi
                rep     movsb
                xchg    esi, edi
                pop     esi ecx
                add     [(fh esi).dBufferPtr], ecx

                mov     eax, [(fh esi).dBufferPtr]
                cmp     eax, [(fh esi).dBufferSize]
                @IF_A
                  mov   [(fh esi).dBufferSize], eax
                @ENDIF

                jmp     @@loop

@@exit:         pop     edi ecx ebx
                ret
endp



proc            FileWriteByte
                ;in
                ;  al  - byte to write
                ;  esi - 'File'
                ;out
                ;  nothing
                push    ebx

                mov     ebx, eax
                call    CheckHandle
                call    CanWrite

                ;;check if we are within the buffer
                mov     eax, [(fh esi).dMaxBufferSize]
                cmp     [(fh esi).dBufferPtr], eax
                @IF_E
                  call  FlushBuffer
                  call  LoadBuffer
                @ENDIF

                or      [(fh esi).wMode], DIRTY_FLAG
                inc     [(fh esi).dBufferPtr]
                mov     eax, [(fh esi).dBufferPtr]
                cmp     eax, [(fh esi).dBufferSize]
                @IF_A
                  mov   [(fh esi).dBufferSize], eax
                @ENDIF
                add     eax, (size fh) - 1
                mov     [esi+eax], bl

                pop     ebx
                ret
endp



;****************************************************************************;
;                            LOCAL METHODS                                   ;
;****************************************************************************;



proc            CheckHandle
                ;in
                ;  esi - ptr to 'fh' structure
                ;out
                ;  nothing
                cmp     [(fh esi).dSign], SIGN
                @IF_NE
                  lea   esi, [cErrBadHandle]
                  call  ErrorFatal
                @ENDIF
                ret
endp



proc            CanRead
                ;in
                ;  esi - ptr to 'fh' structure
                ;out
                ;  nothing
                cmp     [bptr (fh esi).wMode], MODE_W
                @IF_E
                  lea   esi, [cErrOpenForWr]
                  call  ErrorFatal
                @ENDIF
                ret
endp



proc            CanWrite
                ;in
                ;  esi - ptr to 'fh' structure
                ;out
                ;  nothing
                cmp     [bptr (fh esi).wMode], MODE_R
                @IF_E
                  lea   esi, [cErrOpenForRd]
                  call  ErrorFatal
                @ENDIF
                ret
endp



proc            LoadBuffer
                ;in
                ;  esi - ptr to 'fh' structure
                ;out
                ;  CN  - success
                ;  CY  - EOF (0 bytes was read)
                pushad

                ;;check the open mode
                mov     ax, [(fh esi).wMode]
                cmp     al, MODE_W
                mov     ax, 0                   ;number of bytes
                @IF_NE
                  mov   ah, 3Fh                         ;read function
                  mov   bx, [(fh esi).wDosFileHandle]
                  mov   ecx, [(fh esi).dMaxBufferSize]  ;number fo bytes
                  lea   ebp, [esi+size fh]
                  add   ebp, [dBaseAddr]
                  mov   edx, ebp
                  shr   ebp, 4
                  and   edx, 0Fh
                  push  esi
                  call  ApiInt21h
                  pop   esi
                  @IF_C
                    lea   esi, [cErrRead]
                    call  ErrorFatal
                  @ENDIF
                @ENDIF

                push    [(fh esi).dDosFileOffset]
                pop     [(fh esi).dBufferOffset]
                movzx   eax, ax         ;number of bytes actually read
                mov     [(fh esi).dBufferSize], eax
                add     [(fh esi).dDosFileOffset], eax
                mov     [(fh esi).dBufferPtr], 0
                cmp     ax, 1

                popad
                ret
endp



proc            FlushBuffer
                ;in
                ;  esi - ptr to 'fh' structure
                ;out
                ;  nothing
                pushad

                ;;flush if buffer is dirty only
                test    [(fh esi).wMode], DIRTY_FLAG
                jz      @@exit

                ;;move DOS pointer to buffer start
                mov     eax, [(fh esi).dBufferOffset]
                sub     eax, [(fh esi).dDosFileOffset]
                @IF_NZ
                  mov   edx, eax
                  shld  ecx, eax, 16
                  mov   bx, [(fh esi).wDosFileHandle]   ;DOS file handle
                  mov   ax, 4201h                       ;seek function
                  int   21h
                  @IF_C
                    lea   esi, [cErrSeek]
                    call  ErrorFatal
                  @ENDIF
                @ENDIF

                ;;write the buffer
                mov     ah, 40h                 ;write function
                mov     bx, [(fh esi).wDosFileHandle]
                mov     ecx, [(fh esi).dBufferSize]
                lea     ebp, [esi+size fh]
                add     ebp, [dBaseAddr]
                mov     edx, ebp
                shr     ebp, 4
                and     edx, 0Fh
                call    ApiInt21h
                @IF_C
                  lea   esi, [cErrWrite]
                  call  ErrorFatal
                @ENDIF

                ;;change DOS ptr and buffer ptr
                mov     eax, [(fh esi).dBufferOffset]
                add     eax, [(fh esi).dBufferSize]
                mov     [(fh esi).dDosFileOffset], eax
                xor     eax, eax
                mov     [(fh esi).dBufferPtr], eax

@@exit:         popad
                ret
endp

ends            code32
                end
