;*****
; dma.asm - DMA handling.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __DMA__ equ 1
                include "pmlib.asd"
                include "api.asi"
                include "dma.asi"



struc           cr              ;channel registers
  bPage         db      0       ;page
  bBase         db      0       ;base address
  bCount        db      0       ;counter
  bCommand      db      0       ;command/state register
  bRequest      db      0       ;request register
  bPersMask     db      0       ;individual masks register
  bMode         db      0       ;mode register
  bFlipFlop     db      0       ;flip-flop flag
ends

asCr            cr      <87h, 00h, 01h, 08h, 09h, 0Ah, 0Bh, 0Ch>   ; 0
                cr      <83h, 02h, 03h, 08h, 09h, 0Ah, 0Bh, 0Ch>   ; 1
                cr      <81h, 04h, 05h, 08h, 09h, 0Ah, 0Bh, 0Ch>   ; 2
                cr      <82h, 06h, 07h, 08h, 09h, 0Ah, 0Bh, 0Ch>   ; 3
                cr      <8Fh,0C0h,0C2h,0D0h,0D2h,0D4h,0D6h,0D8h>   ; 4
                cr      <8Bh,0C4h,0C6h,0D0h,0D2h,0D4h,0D6h,0D8h>   ; 5
                cr      <89h,0C8h,0CAh,0D0h,0D2h,0D4h,0D6h,0D8h>   ; 6
                cr      <8Ah,0CCh,0CEh,0D0h,0D2h,0D4h,0D6h,0D8h>   ; 7



proc            DmaMaskChannel
                ;in
                ;  eax - channel number
                ;out
                ;  nothing
                ;description
                ;  Masks channel.
                pushad
                call    ChannelToPtr
                mov     eax, edi                ;channel number
                bts     eax, 2                  ;mask
                mov     dl, [(cr esi).bPersMask]
                out     dx, al
                popad
                ret
endp



proc            DmaUnMaskChannel
                ;in
                ;  eax - channel number
                ;out
                ;  nothing
                ;description
                ;  Unmasks channel.
                pushad
                call    ChannelToPtr
                mov     eax, edi                ;channel number
                mov     dl, [(cr esi).bPersMask]
                out     dx, al
                popad
                ret
endp



proc            DmaClearRequest
                ;in
                ;  eax - channel number
                ;out
                ;  nothing
                ;description
                ;  Resets channel request.
                pushad
                call    ChannelToPtr
                mov     eax, edi                ;channel number
                mov     dl, [(cr esi).bRequest]
                out     dx, al
                popad
                ret
endp



proc            DmaSetRequest
                ;in
                ;  eax - channel number
                ;out
                ;  nothing
                ;description
                ;  Sets channel request.
                pushad
                call    ChannelToPtr
                mov     eax, edi                ;channel number
                bts     eax, 2                  ;set request
                mov     dl, [(cr esi).bRequest]
                out     dx, al
                popad
                ret
endp



proc            DmaSetMode
                ;in
                ;  eax - channel number
                ;  bl  - code for state register without channel number
                ;out
                ;  nothing
                ;description
                ;  Sets channel work mode.
                pushad
                call    ChannelToPtr
                and     bl, 11111100b           ;remove channel number
                mov     eax, edi                ;channel number
                or      al, bl
                mov     dl, [(cr esi).bMode]
                out     dx, al
                popad
                ret
endp



proc            DmaSetAddress
                ;in
                ;  eax - channel number
                ;  ebx - buffer address below 1M.
                ;out
                ;  nothing
                ;description
                ;  Sets base address and and page.
                pushad
                call    ChannelToPtr

                ;;convert to physical address
                add     ebx, [dBaseAddr]

                ;;reset flip-flop register
                mov     dl, [(cr esi).bFlipFlop]
                out     dx, al                  ;any value

                ;;set base address
                mov     dl, [(cr esi).bBase]
                mov     al, bl
                out     dx, al
                jmp     short $+2
                mov     al, bh
                out     dx, al

                ;;set page
                mov     dl, [(cr esi).bPage]
                shld    eax, ebx, 16
                out     dx, al

                popad
                ret
endp



proc            DmaSetSize
                ;in
                ;  eax - channel number
                ;  ebx - buffer size
                ;out
                ;  nothing
                ;description
                ;  Sets buffer size.
                pushad
                call    ChannelToPtr

                ;;reset flip-flop register
                mov     dl, [(cr esi).bFlipFlop]
                out     dx, al                  ;any value

                ;;set the counter
                dec     ebx
                mov     dl, [(cr esi).bCount]
                mov     al, bl
                out     dx, al
                jmp     short $+2
                mov     al, bh
                out     dx, al

                popad
                ret
endp



proc            DmaGetSize
                ;in
                ;  eax - channel number
                ;out
                ;  eax - current buffer pointer
                ;description
                ;  Returns current buffer pointer
                push    edx edi esi
                call    ChannelToPtr

                ;;reset flip-flop register
                mov     dl, [(cr esi).bFlipFlop]
                out     dx, al                  ;any value

                ;;read the counter
                xor     eax, eax
                mov     dl, [(cr esi).bCount]
                in      al, dx
                mov     ah, al
                jmp     short $+2
                in      al, dx
                xchg    al, ah
                inc     eax

                pop     esi edi edx
                ret
endp



proc            DmaGetState
                ;in
                ;  eax - channel number
                ;out
                ;  eax - channel state
                ;        bit 0 set - channel work finished
                ;        bit 1 set - unhandled request present
                ;description
                ;  Returns channel state.
                push    ecx edx edi esi
                call    ChannelToPtr

                ;;read the channel state
                mov     dl, [(cr esi).bCommand]
                xor     eax, eax
                in      al, dx
                mov     ecx, edi
                shr     al, cl
                and     al, 11h
                mov     cl, al
                shr     cl, 3
                or      al, cl

                pop     esi edi edx ecx
                ret
endp



;****************************************************************************;
;                            LOCAL METHODS                                   ;
;****************************************************************************;



proc            ChannelToPtr
                ;in
                ;  eax - channel number
                ;out
                ;  edx - 0
                ;  edi - lower two bits of channel number
                ;  esi - ptr to 'cr' structure
                ;description
                ;  Converts channel number to 'cr' structure ptr.
                push    ebx
                mov     edi, eax
                and     edi, 3
                mov     ebx, size cr
                mul     ebx
                lea     esi, [asCr]
                add     esi, eax
                pop     ebx
                ret
endp

ends            code32
                end
