;*****
; cdf.asm - Common Data File. Information is stored in blocks. Blocks may be
;           nested.
;



segment         code32 public use32 'CODE'
                assume cs:code32, ds:code32

                __CDF__ equ 1
                include "pmlib.asd"
                include "cdf.asi"
                include "debug.asi"
                include "error.asi"
                include "file.asi"
                include "heap.asi"
                include "stack.asi"



CDF_SIGN        equ     'CDF '



struc           cdf             ;Common Data File
  dSign         dd   CDF_SIGN   ;sugnature for run-time checks
  dFile         dd      0       ;file descriptor
  dSizStack     dd      0       ;sizes stack
  dLenStack     dd      0       ;lengthes stack (readed bytes counter)
  dPreReadFlag  dd      0       ;block pre-readed
  dType         dd      0       ;type of pre-readed block
  dSize         dd      0       ;size of pre-readed block
ends

struc           block           ;block
  dType         dd      0       ;block type
  dSize         dd      0       ;block size
ends



abIdentifier    db      0FFh, "CDF", 00Dh, 00Ah, 01Ah, 000h
sTmpBlock       block   <>



DBG_RUS_MESSAGE cErrBadMode     " ०  CDF."
DBG_ENG_MESSAGE cErrBadMode     "CDF open mode invalid."
DBG_RUS_MESSAGE cErrNotCdf      "ꥪ  'CDF'."
DBG_ENG_MESSAGE cErrNotCdf      "Object is not 'CDF'."



proc            CdfOpen
                ;in
                ;  ax='r' - open for read only
                ;  ax='w' - open for write only
                ;  esi - ptr to file name
                ;out
                ;  NC:edi - descriptor of opened CDF
                ;   C:    - file is not a CDF
                push    edx esi

                ;;open for read only
                cmp     ax, 'r'
                @IF_E
                  call  FileOpen
                  ;;check the identifier
                  lea   edi, [abIdentifier]
@@loop1:          call  FileReadByte
                  setc  dl
                  cmp   al, [edi]
                  setne dh
                  or    dl, dh
                  @IF_NZ
                    call FileClose
                    stc
                    jmp  short @@exit
                  @ENDIF
                  mov   al, [edi]
                  inc   edi
                  and   al, al
                  jnz   short @@loop1
                  jmp   short @@cont
                @ENDIF

                ;;open for write only
                cmp     ax, 'w'
                @IF_E
                  call  FileOpen
                  ;;write the identifier
                  lea   edi, [abIdentifier]
@@loop2:          mov   al, [edi]
                  call  FileWriteByte
                  mov   al, [edi]
                  inc   edi
                  and   al, al
                  jnz   short @@loop2
                  jmp   short @@cont
                @ENDIF

                ;;bad open mode
                IF DBG
                  lea   esi, [cErrBadMode]
                  call  ErrorFatal
                ENDIF

                ;;create and initialise 'CDF' object
@@cont:         mov     eax, size cdf
                call    HeapAlloc
                mov     [(cdf edi).dSign], CDF_SIGN
                mov     [(cdf edi).dFile], esi
                mov     esi, edi
                call    StackNew
                mov     [(cdf esi).dSizStack], edi
                call    StackNew
                mov     [(cdf esi).dLenStack], edi
                mov     [(cdf esi).dPreReadFlag], 0
                mov     edi, esi

@@exit:         pop     esi edx
                ret
endp



IF DBG
proc            CdfCheck
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                cmp     [(cdf esi).dSign], CDF_SIGN
                @IF_NE
                  lea   esi, [cErrNotCdf]
                  call  ErrorFatal
                @ENDIF
		ret
endp
ENDIF



proc            CdfClose
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                push    edi
                IF DBG
                  call  CdfCheck
                ENDIF
                mov     edi, esi
                mov     esi, [(cdf edi).dLenStack]
                call    StackDelete
                mov     esi, [(cdf edi).dSizStack]
                call    StackDelete
                mov     esi, [(cdf edi).dFile]
                call    FileClose
                mov     eax, edi
                call    HeapFree
                pop     edi
                ret
endp



proc            CdfPreReadBlock
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  eax - code of block
                ;  ebx - size of block
                IF DBG
                  call  CdfCheck
                ENDIF

                push    edx esi edi
                mov     edx, esi
                lea     edi, [sTmpBlock]

                ;;read the block header
                mov     eax, size block
                mov     esi, [(cdf edx).dFile]
                call    FileReadStr
                mov     eax, -(size block)
                call    FileSeek

                ;;return type and size
                mov     eax, [(block edi).dType]
                mov     ebx, [(block edi).dSize]
                pop     edi esi edx
		ret
endp



proc            CdfBeginReadBlock
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  eax - code of block
                ;  ebx - size of block
                IF DBG
                  call  CdfCheck
                ENDIF

                push    edx esi edi
                mov     edx, esi
                lea     edi, [sTmpBlock]

                ;;read the block header
                mov     eax, size block
                mov     esi, [(cdf edx).dFile]
                call    FileReadStr

                ;;initialise stacks
                mov     eax, [(block edi).dSize]
                mov     esi, [(cdf edx).dSizStack]
                call    StackPush
                xor     eax, eax
                mov     esi, [(cdf edx).dLenStack]
                call    StackPush

                ;;return type and size
                mov     eax, [(block edi).dType]
                mov     ebx, [(block edi).dSize]
                pop     edi esi edx
		ret
endp



proc            CdfReadByte
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  CN: al - byte from current block
                ;  CF: error, no bytes left in current block
                IF DBG
                  call  CdfCheck
                ENDIF

                push    ecx edx esi
                mov     edx, esi

                ;;get size of current block
                mov     esi, [(cdf edx).dSizStack]
                call    StackRead
                jc      @@exit
                mov     ecx, eax

                ;;get length of current block
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                jc      @@exit

                inc     eax
                cmp     eax, ecx
                @IF_BE
                  call  StackWrite
                  ;;read byte
                  mov   esi, [(cdf edx).dFile]
                  call  FileReadByte
                @ELSE
                  stc
                @ENDIF

@@exit:         pop     esi edx ecx
		ret
endp



proc            CdfReadDword
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  CN: eax - dword from current block
                ;  CF: error, no dwords left in current block
                IF DBG
                  call  CdfCheck
                ENDIF

                push    ebx ecx edx esi edi
                mov     edx, esi
                mov     ecx, 4
                lea     edi, [sTmpBlock]

                ;;get size of current block
                mov     esi, [(cdf edx).dSizStack]
                call    StackRead
                jc      @@exit
                mov     ebx, eax

                ;;get length of current block
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                jc      @@exit

                add     eax, ecx
                cmp     eax, ebx
                @IF_BE
                  call  StackWrite
                  ;;read string
                  mov   eax, ecx
                  mov   esi, [(cdf edx).dFile]
                  call  FileReadStr
                @ELSE
                  stc
                @ENDIF

                ;;return dword
                mov     eax, [edi]

@@exit:         pop     edi esi edx ecx ebx
		ret
endp



proc            CdfReadStr
                ;in
                ;  eax - string length
                ;  esi - descriptor of opened CDF
                ;  edi - ptr to string buffer
                ;out
                ;  CN: OK
                ;  CF: error, not enougth bytes left in block
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi
                mov     ecx, eax

                ;;get size of current block
                mov     esi, [(cdf edx).dSizStack]
                call    StackRead
                jc      @@exit
                mov     ebx, eax

                ;;get length of current block
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                jc      @@exit

                add     eax, ecx
                cmp     eax, ebx
                @IF_BE
                  call  StackWrite
                  ;;read string
                  mov   eax, ecx
                  mov   esi, [(cdf edx).dFile]
                  call  FileReadStr
                @ELSE
                  stc
                @ENDIF

@@exit:         popad
		ret
endp



proc            CdfEndReadBlock
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi

                ;;skip to end of block
                mov     esi, [(cdf edx).dSizStack]
                call    StackRead
                mov     ecx, eax
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                sub     eax, ecx
                @IF_NZ
                  neg   eax
                  mov   esi, [(cdf edx).dFile]
                  call  FileSeek
                @ENDIF

                ;;increment counter of upper level block
                mov     esi, [(cdf edx).dSizStack]
                call    StackPop
                mov     esi, [(cdf edx).dLenStack]
                call    StackPop
                call    StackRead
                add     eax, ecx
                add     eax, size block
                call    StackWrite

                popad
		ret
endp



proc            CdfBeginWriteBlock
                ;in
                ;  eax - code of block
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi

                ;;write block header
                mov     esi, [(cdf edx).dFile]
                lea     edi, [sTmpBlock]
                mov     [(block edi).dType], eax
                mov     eax, size block
                call    FileWriteStr

                ;;initialise counter
                xor     eax, eax
                mov     esi, [(cdf edx).dLenStack]
                call    StackPush

                popad
                ret
endp



proc            CdfWriteByte
                ;in
                ;   al - byte to write
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi

                ;;write byte
                mov     esi, [(cdf edx).dFile]
                call    FileWriteByte

                ;;increment counter
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                inc     eax
                call    StackWrite

                popad
		ret
endp



proc            CdfWriteDword
                ;in
                ;   al - dword to write
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi

                ;;write string
                mov     esi, [(cdf edx).dFile]
                lea     edi, [sTmpBlock]
                mov     [edi], eax
                mov     eax, 4
                call    FileWriteStr

                ;;increment counter
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                add     eax, 4
                call    StackWrite

                popad
		ret
endp



proc            CdfWriteStr
                ;in
                ;  eax - string length
                ;  esi - descriptor of opened CDF
                ;  edi - ptr to string
                ;out
                ;  nothing
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi
                mov     ecx, eax

                ;;write string
                mov     esi, [(cdf edx).dFile]
                call    FileWriteStr

                ;;increment counter
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                add     eax, ecx
                call    StackWrite

                popad
		ret
endp



proc            CdfEndWriteBlock
                ;in
                ;  esi - descriptor of opened CDF
                ;out
                ;  nothing
                IF DBG
                  call  CdfCheck
                ENDIF

                pushad
                mov     edx, esi

                ;;read counter
                mov     esi, [(cdf edx).dLenStack]
                call    StackRead
                mov     ecx, eax

                ;;write counter value into file
                add     eax, 4
                neg     eax
                mov     esi, [(cdf edx).dFile]
                call    FileSeek
                lea     edi, [(block sTmpBlock).dSize]
                mov     [edi], ecx
                mov     eax, 4
                call    FileWriteStr
                mov     eax, ecx
                call    FileSeek

                ;;increment counter of upper level block
                mov     esi, [(cdf edx).dLenStack]
                call    StackPop
                mov     ecx, eax
                call    StackRead
                add     eax, ecx
                add     eax, size block
                call    StackWrite

                popad
		ret
endp

ends            code32
                end
