unit Demo16U2;
//
// PixelGraphicLibrary - Demo 16
// Version: 1.0 beta 5 for Delphi 2 and Delphi 3
// Copyright  1996-1998 Peter Beyersdorf, Lbeck, Germany
// http://www.beyersdorf.com/
//
//
// NOTE: Please notify me if you want to write file format readers and
//       writers for not currently supported formats that you want to make
//       available to other programmers.
//       I'm interested that good and solid file format readers and writers
//       will become available for as much file formats as possible.
//       - So, I would like to answer questions about how to use the reader
//         writer mechanism of the PixelGraphicLibary.
//       - I would like to test the readers and writers.
//       - I would like to add links to file format readers and writers to my
//         web site.
//
interface

uses PGraphic, Classes;

type
   // add support for the new "myjpeg" extension
   TMyJPGFileFormatReader = class(TJPGFileFormatReader)
   public
      constructor Create;
   end;

   TMyJPGFileFormatWriter = class (TJPGFileFormatWriter)
   public
      constructor Create;
   end;

   // add support for the new 8 bit format "xyz"
   TXYZFileFormatReader = class(TFileFormatReader)
   public
      constructor Create;
      function IsFormat(Bytes0_16: TBytes0_16): Boolean; override;
      procedure ReadFromStream(Stream: TStream; PixelGraphic: TPixelGraphic; Index: Integer); override;
      procedure GetInfoFromStream(Stream: TStream; Index: Integer; var iSubType, iOptions: string; var iWidth, iHeight: Integer; var iColors: string; var iBitsPerPixel, iGraphicCount: Integer); override;
   end;

   TXYZFileFormatWriter = class (TFileFormatWriter)
   public
      constructor Create;
      procedure WriteToStream(Stream: TStream; PixelGraphic: TPixelGraphic); override;
   end;

implementation

constructor TMyJPGFileFormatReader.Create;
   begin
   inherited Create;
   Extension:='myjpg';
   end;

constructor TMyJPGFileFormatWriter.Create;
   begin
   inherited Create;
   Extension:='myjpg';
   end;

constructor TXYZFileFormatReader.Create;
// Some fields have to be set to give the PixelGraphicLibarary some
// information about the file format.

   begin
   inherited Create;
   // The format should appear in the TPGOpenDialog
   AddToPGOpenSaveDialogFilter:=true;
   // The file extension; always lower case
   Extension:='xyz';
   // The format name
   FormatName:='XYZ-Format';
   // First bytes of the files contain always XYZ. That means the format can
   // be identified. So CanIdentifyFormat is set to true and the IsFormat
   // method must be overriden.
   CanIdentifyFormat:=true;
   end;

function TXYZFileFormatReader.IsFormat(Bytes0_16: TBytes0_16): Boolean;
// The method is called by the PixelGraphicLibrary if it wants to know if this
// file format reader can read a file. The PixelFGraphicLibrary passes the first
// 17 bytes of that file.
// The first 3 bytes of XYZ files contain always XYZ so we return true if the
// first 3 of the passed bytes contain XYZ.
// Note that this method must be overriden if the constructor sets
// CanIdentifyFormat to true.
// Some formats, e.g. tga, do not contain typical first bytes. In this case the
// constructor should set CanIdentifyFormat to false and not override the
// IsFormat method.

   begin
   result := (Bytes0_16[0] = Ord('X')) and (Bytes0_16[1] = Ord('Y')) and (Bytes0_16[2] = Ord('Z'));
   end;

type
   PBytes = ^TBytes;
   TBytes = array [0..MaxInt-1] of Byte;

   TThreeBytes = array [0..2] of Byte;

procedure TXYZFileFormatReader.ReadFromStream(Stream: TStream; PixelGraphic: TPixelGraphic; Index: Integer);
   var
      aWidth: Integer;
      aHeight: Integer;
      aInteger: Integer;
      aExtended: Extended;
      i, x, y: Integer;
      buffer: PBytes;
      ThreeBytes: TThreeBytes;
      Abort: Boolean;
   begin
   if Index>1 then
      raise EPixelGraphic.Create('Index > 1');
   Abort:=false;
   with PixelGraphic do
      begin
      Stream.ReadBuffer(ThreeBytes, 3);
      if not ((ThreeBytes[0] = Ord('X')) and (ThreeBytes[1] = Ord('Y')) and (ThreeBytes[2] = Ord('Z'))) then
         raise EPixelGraphic.Create('Not a XYZ file');
      Stream.ReadBuffer(aWidth, SizeOf(aWidth));
      Stream.ReadBuffer(aHeight, SizeOf(aHeight));
      SetDimension(aWidth, aHeight, bc8);
      // Most formats don't know about Gamma. But XYZ does for demonstration purpose.
      // Don't toutch Gamma if the file doesn't give you a gamma value.
      Stream.ReadBuffer(aExtended, SizeOf(aExtended));
      Gamma:=aExtended;
      // If the format allows information for background or transparency, you should read them, too.
      Stream.ReadBuffer(aInteger, SizeOf(aInteger));
      NumPaletteEntries:=aInteger;
      if NumPaletteEntries>0 then
         for i:= 0 to NumPaletteEntries-1 do
            begin
            Stream.ReadBuffer(aInteger, SizeOf(aInteger));
            PaletteEntries[i]:=aInteger;
            end;
      if (Width>0) and (Height>0) then
         begin
         // Writing single bytes to a stream is slow. So we read a scanline
         // form the stream into a buffer and write the entire scanline
         // to the PxielGraphic.
         GetMem(buffer, aWidth);
         try
            if Assigned(OnProgressStart) then
               PixelGraphic.OnProgressStart(Self);
            try
               for y:=0 to aHeight-1 do
                  begin
                  Stream.ReadBuffer(Buffer^, aWidth);
                  for x:=0 to aWidth-1 do
                     Bits[x, y]:=buffer^[x];
                  if Assigned(OnProgress) then
                     PixelGraphic.OnProgress(Self, y, aHeight-1, Abort);
                  if Abort then
                     break;
                  end;
            finally
               if Assigned(OnProgressEnd) then
                  PixelGraphic.OnProgressEnd(Self);
            end; // try/finally
         finally
            FreeMem(Buffer);
         end; // try/finally
         end; // of then
      end; // of with
   end;

procedure TXYZFileFormatReader.GetInfoFromStream(Stream: TStream; Index: Integer; var iSubType, iOptions: string; var iWidth, iHeight: Integer; var iColors: string; var iBitsPerPixel, iGraphicCount: Integer);
   var
      ThreeBytes: TThreeBytes;
   begin
   inherited;
   if Index>1 then
      raise EPixelGraphic.Create('Index > 1');
   Stream.ReadBuffer(ThreeBytes, 3);
   if not ((ThreeBytes[0] = Ord('X')) and (ThreeBytes[1] = Ord('Y')) and (ThreeBytes[2] = Ord('Z'))) then
      raise EPixelGraphic.Create('Not a XYZ file');
   Stream.ReadBuffer(iWidth, SizeOf(iWidth));
   Stream.ReadBuffer(iHeight, SizeOf(iHeight));
   iColors:='256';
   iBitsPerPixel:=8;
   iGraphicCount:=1;
   end;

constructor TXYZFileFormatWriter.Create;
// Some fields have to be set to give the PixelGraphicLibarary some
// information about the file format.

   begin
   inherited Create;
   // Tell the PixelGraphicLibrary that this writer can write 8 bits per pixel
   // files containing a palette. If necessary the PixelGraphicLibrary will
   // generate a temporary graphic fitting to a color resolution included in
   // CanWriteResolutions. In this case the PixelGraphicLibarary will generate
   // and pass a temprary PixelGraphic with 8 bit per pixel and an optimzed
   // palette graphic with optimized palette to the WriteToStream method if
   // the PixelGraphic which has to be saved is a true color graphic.
   CanWriteResolutions := [cwrPalette8Bit];
   // The format should appear in the TPGSaveDialog
   AddToPGOpenSaveDialogFilter:=true;
   // The file extension; always lower case
   Extension:='xyz';
   // The format name
   FormatName:='XYZ-Format';
   end;

procedure TXYZFileFormatWriter.WriteToStream(Stream: TStream; PixelGraphic: TPixelGraphic);
   var
      aInteger: Integer;
      aExtended: Extended;
      i, x, y: Integer;
      buffer: PBytes;
      Abort: Boolean;
   begin
   Abort:=false;
   with PixelGraphic do
      begin
      Stream.WriteBuffer('XYZ', 3);
      aInteger:=Width;
      Stream.WriteBuffer(aInteger, SizeOf(aInteger));
      aInteger:=Height;
      Stream.WriteBuffer(aInteger, SizeOf(aInteger));
      // Most formats don't know about Gamma. But XYZ does to demonstrate how to handle Gamma in the reader.
      aExtended:=Gamma;
      Stream.WriteBuffer(aExtended, SizeOf(aExtended));
      // If the format allows information for background or transparency, you should write them, too.
      aInteger:=NumPaletteEntries;
      Stream.WriteBuffer(aInteger, SizeOf(aInteger));
      if NumPaletteEntries>0 then
         for i:= 0 to NumPaletteEntries-1 do
            begin
            aInteger:=Integer(PaletteEntries[i]);
            Stream.WriteBuffer(aInteger, SizeOf(aInteger));
            end;
      if (Width>0) and (Height>0) then
         begin
         // Writing single bytes to a stream is slow. So we read a scanline
         // from the PixelGraphic into a buffer and write the entire scanline
         // to the stream.
         GetMem(buffer, Width);
         try
            if Assigned(OnProgressStart) then
               PixelGraphic.OnProgressStart(Self);
            try
               for y:=0 to Height-1 do
                  begin
                  for x:=0 to Width-1 do
                     buffer^[x]:=Byte(Bits[x, y]);
                  Stream.WriteBuffer(Buffer^, Width);
                  if Assigned(OnProgress) then
                     PixelGraphic.OnProgress(Self, y, Height-1, Abort);
                  // ignore Abort to avoid incomplete files
                  end;
            finally
               if Assigned(OnProgressEnd) then
                  PixelGraphic.OnProgressEnd(Self);
            end; // try/finally
         finally
            FreeMem(Buffer);
         end; // try/finally
         end; // of then
      end; // of with
   end;

initialization
FileFormatReaders.Add(TMyJPGFileFormatReader.Create);
FileFormatWriters.Add(TMyJPGFileFormatWriter.Create);
FileFormatReaders.Add(TXYZFileFormatReader.Create);
FileFormatWriters.Add(TXYZFileFormatWriter.Create);
end.
