{ PASTOC.PAS : Convert Pascal to C (raw conversion)

  Title   : PASTOC
  Version : 1.3
  Date    : Feb 19, 2000
  Author  : J.R. Ferguson, Amsterdam, The Netherlands
  E-mail  : j.r.ferguson@iname.com
  Download: http://hello.to/ferguson
            To compile this sourcefile, you will need some units from the
            Pascal library JRFPAS that can be found at the same Internet
            site.
  Usage   : Refer procedure Help

  This software may be used and copied without charge, but only for
  non-commercial purposes. The author can not be held responsible for any
  damage or loss of data that may be caused by the use of this software.
}

{$V-}
{$R+}

{$UNDEF OUTBUFHEAP}   { UNDEF to work around a BP 7.0 bug resulting in
                        erroneous file output }

program PASTOC;


uses DefLib, ArgLib, StpLib, StfLib, ChrLib, Dos;


const
  MAXFNM    = 79;     { Max filespec length (integrated environment) }
  DFLMSG    = 'CON';  { Default message output destination }
  INPBUFSIZ = 4096;   { Input buffer size in bytes }
  OUTBUFSIZ = 4096;   { Output buffer size in bytes }
  DFLINPEXT = '.PAS'; { Default input file extent }
  DFLOUTEXT = '.C';   { Default output file extent }
  ELN       = #000;   { End of line mark character }
  EFC       = #026;   { End of file mark character }

  { Error codes and messages: }
  ERROK     = 0;
  ERRARG    = 1;
  ERRFNF    = 2;
  ERRCRE    = 3;
  ERRREA    = 4;
  ERRWRI    = 5;

  ERRMSG    : array[ERRFNF..ERRWRI] of StpTyp =
 ('File not found',
  'File creation error',
  'Read error',
  'Write error'
 );

type
  InpBufTyp = array[1..INPBUFSIZ] of char; InpBufPtr = ^InpBufTyp;
  OutBufTyp = array[1..OUTBUFSIZ] of char; OutBufPtr = ^OutBufTyp;

var
  ErrCod    : integer;
  InpFnm,
  OutFnm    : StpTyp;
  Msg,
  Inp,
  Out       : Text;
  InpBuf    : InpBufPtr;
{$IFDEF OUTBUFHEAP}
  OutBuf    : OutBufPtr;
{$ELSE}
  OutBuf    : OutBufTyp;
{$ENDIF}
  InpOpn,
  OutOpn    : boolean;
  OptHlp    : boolean;

  InpLine   : StpTyp;
  OutLine   : StpTyp;
  CurTok  : StpTyp;
  UppTok  : StpTyp;
  SavTok  : StpTyp;
  CurChr    : char;
  ChrStack  : StpTyp;


{--- General routines ---}


procedure Help;
  procedure wr(s: StpTyp);  begin write  (Msg,s) end;
  procedure wl(s: StpTyp);  begin writeln(Msg,s) end;
begin
wl('PASTOC v1.3 - Convert Pascal to C (raw conversion)');
wl('Usage  : PASTOC inpfile [outfile] [/option[...] [...]]');
wl('Where  : infile  = [path]progname[.ext]');
wl('                   default path     = current directory');
wl('                   default .ext     = .PAS');
wl('         outfile = [path]progname[.ext]');
wl('                   default path     = current directory');
wl('                   default progname = same as in inpfile');
wl('                   default .ext     = .C');
wl('');
wl('Options: H   Send this help text to (redirected) output.');
wl('');
wl('Remarks: No complete translation. Only meant as a project start.');
end;


{--- Command line parsing routines ---}


procedure AdjustFileNames;
var Dir: DirStr; InpName, OutName: NameStr; Ext: ExtStr;
begin
  FSplit(FExpand(InpFnm),Dir,InpName,Ext);
  if StpEmpty(Ext) then StpCpy(Ext,DFLINPEXT);
  InpFnm:= Dir+InpName+Ext;

  FSplit(FExpand(OutFnm),Dir,OutName,Ext);
  if StpEmpty(OutName) then StpCpy(OutName,InpName);
  if StpEmpty(Ext)     then StpCpy(Ext,DFLOUTEXT);
  outFnm:= Dir+OutName+Ext;
end;


procedure ReadOpt(var arg: StpTyp);
var nextopt: boolean;
begin
  StpDel(arg,1,1);
  repeat
    if StpEmpty(arg) or (StpcRet(arg,1) = '/') then ErrCod:= ERRARG
    else begin
      nextopt:= false;
      while (ErrCod=ERROK) and not nextopt and not StpEmpty(arg) do
      case StpcGet(arg) of
        'H': OptHlp:= true;
        '/': nextopt:= true;
        else ErrCod:= ERRARG;
      end;
    end;
  until (ErrCod <> ERROK) or not nextopt;
end;


procedure ReadArgs;
var i,n : ArgInd;
    arg : StpTyp;
begin
  GetArgs; i:= 0; n:= 0;
  while (ErrCod = ERROK) and (i < ArgC) do begin
    Inc(i); StpCpy(arg,ArgV[i]); StpUpp(arg);
    case StpcRet(arg,1) of
      '/' : ReadOpt(arg);
      else  case n of
              0 :  begin Inc(n); StpNCpy(InpFnm,arg,MAXFNM); end;
              1 :  begin Inc(n); StpNCpy(OutFnm,arg,MAXFNM); end;
              else ErrCod:= ERRARG;
            end;
    end;
  end;
  if OptHlp then if ErrCod = ERROK then ErrCod:= ERRARG else OptHlp:= false;
  if n<1 then ErrCod:= ERRARG else AdjustFileNames;
end;



{--- Low-level I/O routines ---}


procedure OpenMsg;
begin
  if OptHlp then Assign(Msg,OutFnm) else Assign(Msg,DFLMSG);
  rewrite(Msg);
end;


procedure CloseMsg;
begin Close(Msg) end;


procedure OpenInp;
begin
  Assign(Inp,InpFnm); new(InpBuf); SetTextBuf(Inp,InpBuf^);
  {$I-} reset(Inp); {$I+}
  if IOresult <> 0 then ErrCod:= ERRFNF else InpOpn:= true;
end;


procedure CloseInp;
begin
  dispose(InpBuf);
  {$I-} Close(Inp); {$I+}
  if IOresult = 0 then InpOpn:= false;
end;


procedure ReadInp(var line: StpTyp);
begin
  {$I-} readln(Inp,line); {$I+}
  if IOresult <> 0 then ErrCod:= ERRREA;
end;


function GetLine: boolean;
begin
  if (ErrCod <> ERROK) or eof(Inp) then GetLine:= false
  else begin
    ReadInp(InpLine);
    GetLine:= ErrCod = ERROK;
  end;
end;


procedure OpenOut;
begin
  Assign(Out,OutFnm);
{$IFDEF OUTBUFHEAP}
  new(OutBuf); SetTextBuf(Out,OutBuf^);
{$ELSE}
  SetTextBuf(Out,OutBuf);
{$ENDIF}
  {$I-} rewrite(Out); {$I+}
  if IOresult <> 0 then ErrCod:= ERRCRE else OutOpn:= true;
end;


procedure CloseOut;
begin
{$IFDEF OUTBUFHEAP}
  dispose(OutBuf);
{$ENDIF}
  {$I-} Close(Out); {$I+}
  if IOresult = 0 then OutOpn:= false else begin
    if ErrCod = ERROK then ErrCod:= ERRWRI;
  end
end;


procedure WriteOut(line: StpTyp);
begin
  {$I-} writeln(Out,line); {$I+}
  if IOresult <> 0 then ErrCod:= ERRWRI;
end;


{--- Parsing routines ---}


procedure PutChr(c: char);
begin
  if c=ELN then begin WriteOut(OutLine); OutLine:= ''; end
  else StpcCat(OutLine,c);
end;


procedure PutStp(s: StpTyp);
var i: StpInd;
begin for i:= 1 to StpLen(s) do PutChr(s[i]); end;


procedure PushChr;
begin StpcIns(ChrStack,CurChr,1); end;


function PopChr: boolean;
begin
  if StpLen(ChrStack) = 0 then PopChr:= false
  else begin
    CurChr:= StpcGet(ChrStack);
    PopChr:= true;
  end;
end;


function NxtChr: char;
begin
  if not PopChr then begin
    CurChr := StpcGet(InpLine);
    if CurChr = ELN then begin
      if not GetLine then begin CurChr:= EFC; PushChr; end;
      CurChr:= ELN;
    end;
  end;
  NxtChr:= CurChr;
end;


function GetChr: char;
var Skipping: boolean;
begin
  case NxtChr of
    '''':begin
           CurChr:= '"'; Skipping:= true;
           while Skipping do begin
             PutChr(CurChr);
             if NxtChr = '''' then begin
               if NxtChr = '''' then PutChr('"')
               else begin PushChr; Skipping:= false; end;
               CurChr:= '"';
             end;
           end;
         end;
    '{' :begin
           PutChr('/'); CurChr:= '*';
           repeat
             PutChr(CurChr);
             CurChr:= NxtChr;
           until CurChr = '}';
           PutChr('*'); CurChr:= '/';
         end;
    '(' :if NxtChr = '*' then begin
           PutChr('/'); CurChr:= '*';
           Skipping:= true;
           while Skipping do begin
             PutChr(CurChr);
             if NxtChr = '*' then begin
               if NxtChr = ')' then begin
                 PutChr('*'); CurChr:= '/';
                 Skipping:= false;
               end
               else begin PushChr; CurChr:= '*'; end;
             end;
           end;
         end
         else begin PushChr; CurChr:= '('; end;
  end;
  GetChr:= CurChr;
end;


function GetTok: boolean;
begin
  while not (GetChr in
     ['A'..'Z','a'..'z','_',':','=','<','>',';','^',ELN,EFC]) do
    if CurChr <> EFC then PutChr(CurChr);
  StpcCpy(CurTok,CurChr);
  case CurChr of
    ';' : ;
    ELN : ;
    '^' : if GetChr = '.' then StpCpy(CurTok,'->')    else PushChr;
    ':' : if GetChr = '=' then StpcCat(CurTok,CurChr) else PushChr;
    '=' : if GetChr = '>' then StpcCat(CurTok,CurChr) else
          if CurChr = '<' then StpcCat(CurTok,CurChr) else PushChr;
    '<' : if GetChr = '=' then StpcCat(CurTok,CurChr) else
          if CurChr = '>' then StpcCat(CurTok,CurChr) else PushChr;
    '>' : if GetChr = '=' then StpcCat(CurTok,CurChr) else
          if CurChr = '<' then StpcCat(CurTok,CurChr) else PushChr;
    EFC : StpCreate(CurTok);
    else  begin
            while GetChr in ['A'..'Z','a'..'z','_','0'..'9'] do
              StpcCat(CurTok,CurChr);
            PushChr;
          end;
  end;
  UppTok:= StfUpp(CurTok);
  GetTok:= CurTok <> '';
end;


procedure PopTok;
begin PutStp(SavTok); SavTok:= ''; end;


{--- Main line ---}


procedure ProcessTok;
begin
  if      UppTok = ';'         then begin PopTok; PutStp(';'); end
  else if UppTok = 'BEGIN'     then PutStp('{')
  else if UppTok = 'END'       then begin PopTok; PutStp('}'); end
  else if UppTok = 'IF'        then PutStp('if (')
  else if UppTok = 'THEN'      then PutStp(')')
  else if UppTok = 'ELSE'      then begin PopTok; PutStp('; else'); end
  else if UppTok = 'WHILE'     then PutStp('while (')
  else if UppTok = 'DO'        then PutStp(')')
  else if UppTok = 'REPEAT'    then PutStp('do {')
  else if UppTok = 'UNTIL'     then begin
                                      PutStp('} while !(');
                                      SavTok:= ' ) ';
                                    end
  else if UppTok = 'PROCEDURE' then PutStp('void')
  else if UppTok = 'FUNCTION'  then PutStp('')
  else if UppTok = 'BOOLEAN'   then PutStp('int')
  else if UppTok = 'INTEGER'   then PutStp('int')
  else if UppTok = 'LONGINT'   then PutStp('long int')
  else if UppTok = 'REAL'      then PutStp('float')
  else if UppTok = 'WORD'      then PutStp('unsigned int')
  else if UppTok = 'BYTE'      then PutStp('unsigned char')
  else if UppTok = 'FALSE'     then PutStp('0')
  else if UppTok = 'TRUE'      then PutStp('1')
  else if UppTok = 'AND'       then PutStp('&&')
  else if UppTok = 'OR'        then PutStp('||')
  else if UppTok = 'NOT'       then PutStp('!')
  else if UppTok = ':='        then PutStp('=')
  else if UppTok = '='         then PutStp('==')
  else if UppTok = '<>'        then PutStp('!=')
  else if UppTok = '><'        then PutStp('!=')
  else if UppTok = 'NIL'       then PutStp('NULL')
  else if UppTok = 'CASE'      then PutStp('switch (')
  else if UppTok = 'OF'        then PutStp(') {')
  else if UppTok = 'RECORD'    then PutStp('struct {')
  else if UppTok = 'POINTER'   then PutStp('void *')
  else if UppTok = 'PRIVATE'   then PutStp('private:')
  else PutStp(CurTok);
end;


procedure MainProcess;
var line: StpTyp;
begin
  writeln(Msg,InpFnm,' ==> ',OutFnm);
  CurTok  := '';
  SavTok  := '';
  CurChr    := ' ';
  ChrStack  := '';
  InpLine   := '';
  OutLine   := '';
  if GetLine then
    while GetTok do
      ProcessTok;
end;


procedure MainInit;
begin
  ErrCod:= ERROK;
  StpCreate(InpFnm); InpOpn:= false;
  StpCreate(OutFnm); OutOpn:= false;
  OptHlp:= false;
  ReadArgs;
  OpenMsg;
  if ErrCod = ERROK then OpenInp;
  if ErrCod = ERROK then OpenOut;
end;


procedure MainTerm;
begin
  if InpOpn then CloseInp;
  if OutOpn then CloseOut;
  if ErrCod <> ERROK then begin
    if (ErrCod=ERRARG) then Help
    else writeln(Msg,'PASTOC: ',ERRMSG[ErrCod]);
  end;
  CloseMsg;
end;


begin { Main program }
  MainInit;
  if ErrCod = ERROK then MainProcess;
  MainTerm;
end.
