{ SRTLIB.PAS : In-memory array sort routine library

  Title    : SRTLIB
  Version  : 4.0
  Date     : Dec 22,1997
  Author   : J R Ferguson
  Language : Turbo Pascal v5.5 through 7.0 (all targets)
             Delphi 1.0 through 3.0
  Usage    : Unit
}

UNIT SRTLIB;

INTERFACE

type
  SrtElmPtr   = pointer;    { Element pointer    }
  SrtSizTyp   = word;       { Element size type  }
  SrtIndTyp   = word;       { Element index type }

  SrtCmpFnc   = function(e1,e2: SrtElmPtr): integer;
                            { Compare function
                              return value <0 if e1^ < e2^
                                           =0 if e1^ = e2^
                                           >0 if e1^ > e2^
                              Must be defined as a far function (using $F+)
                            }


procedure SrtSelect(base  : SrtElmPtr;   { pointer to first array element }
                    count : SrtIndTyp;   { number of elements to sort     }
                    size  : SrtSizTyp;   { size of array elements to sort }
                    cmp   : SrtCmpFnc);  { compare function               }
{
  Selection sort.

  Sorts count elements of given size at adjacent memory addresses within
  a (64k - 16) byte range starting at base, using the ordering as defined
  by the cmp function.

  Stable, natural. Minimizes the number of moves.
}


procedure SrtBinIns(base  : SrtElmPtr;   { pointer to first array element }
                    count : SrtIndTyp;   { number of elements to sort     }
                    size  : SrtSizTyp;   { size of array elements to sort }
                    cmp   : SrtCmpFnc);  { compare function               }
{
  Binary search insertion sort.

  Sorts count elements of given size at adjacent memory addresses within
  a (64k - 16) byte range starting at base, using the ordering as defined
  by the cmp function.

  Stable, natural (quasi uniform). Minimizes number of compares.
  Preferable in most cases.
}


procedure SrtQuick (base  : SrtElmPtr;   { pointer to first array element }
                    count : SrtIndTyp;   { number of elements to sort     }
                    size  : SrtSizTyp;   { size of array elements to sort }
                    cmp   : SrtCmpFnc);  { compare function               }
{
  Quicksort.

  Sorts count elements of given size at adjacent memory addresses within
  a (64k - 16) byte range starting at base, using the ordering as defined
  by the cmp function.

  Non-stable, uniform. Minimizes the number of both compares and moves,
  but has a significant housekeeping overhead and needs lots of stack.
  May be preferable when size is big and compares are slow, in particular
  when initial ordering may be (almost) reverse.
}



IMPLEMENTATION


{--- Selection sort ---}

procedure SrtSelect(base  : SrtElmPtr;   { pointer to first array element }
                    count : SrtIndTyp;   { number of elements to sort     }
                    size  : SrtSizTyp;   { size of array elements to sort }
                    cmp   : SrtCmpFnc);  { compare function               }
{$ifdef WIN32}
var
  first      : PChar;       { start of first element }
  last       : PChar;       { start of last element }
  p          : PChar;       { work pointer }
  current    : PChar;       { work pointer }
  smallest   : PChar;       { work pointer }
  tmp        : SrtElmPtr;   { temp element pointer }
begin if count > 1 then begin
  GetMem(tmp,size);
  first:= base; last:= first + (count-1)*size;
  p:= First;
  while p <= last do begin
    smallest:= p;
    current:= smallest;
    while current < last do begin
      Inc(current,size);
      if cmp(current,smallest) < 0 then smallest:= current;
    end;
    if smallest <> p then begin
      Move(p^       , tmp^     , size);
      Move(smallest^, p^       , size);
      Move(tmp^     , smallest^, size);
    end;
    Inc(p,size);
  end;
  FreeMem(tmp,size);
end end;
{$else}
var
  BaseSeg    : word;        { segment }
  BaseOfs    : word;        { offset  }
  LastSeg    : word;        { segment }
  LastOfs    : word;        { offset  }
  p          : SrtElmPtr;   { pointer }
  i0,j0,k0   : word;        { offset  }

begin if count > 1 then begin
  GetMem(p,size);
  BaseSeg:= Seg(base^); BaseOfs:= Ofs(base^); LastOfs:= BaseOfs + (count-1)*size;
  i0:= BaseOfs;
  while i0 <= LastOfs do begin
    k0:= i0;
    j0:= k0;
    while j0 < LastOfs do begin
      Inc(j0,size);
      if cmp(Ptr(BaseSeg,j0), Ptr(BaseSeg,k0)) < 0 then k0:= j0;
    end;
    if k0 <> i0 then begin
      Move(Ptr(BaseSeg,i0)^, p^              , size);
      Move(Ptr(BaseSeg,k0)^, Ptr(BaseSeg,i0)^, size);
      Move(p^              , Ptr(BaseSeg,k0)^, size);
    end;
    Inc(i0,size);
  end;
  FreeMem(p,size);
end end;
{$endif}


{--- Binary insertion sort ---}

procedure SrtBinIns(base  : SrtElmPtr;   { pointer to first array element }
                    count : SrtIndTyp;   { number of elements to sort     }
                    size  : SrtSizTyp;   { size of array elements to sort }
                    cmp   : SrtCmpFnc);  { compare function               }
{$ifdef WIN32}
var
  first      : PChar;       { start of first element }
  i,l,r,m    : SrtIndTyp;   { work index }
  tmp        : SrtElmPtr;   { temp element pointer }

begin if count > 1 then begin
  GetMem(tmp,size);
  first:= base;
  i:= 0;
  while i < count-1 do begin
    Inc(i); Move((first + i*size)^, tmp^, size);
    l:= 0; r:= i;
    repeat
      m:= l + (r - l) div 2;
      if cmp(first + m*size, tmp) > 0
      then r:= m
      else l:= succ(m);
    until l >= r;
    if l < i then begin
      Move((first + l*size)^, (first + succ(l)*size)^, (i-l) * size);
      Move(tmp^, (first + l*size)^, size);
    end;
  end;
  FreeMem(tmp,size);
end end;
{$else}
var
  BaseSeg    : word;        { segment }
  BaseOfs    : word;        { offset  }
  i,l,r,m    : SrtIndTyp;   { index   }
  p          : SrtElmPtr;   { pointer }

begin if count > 1 then begin
  GetMem(p,size);
  BaseSeg:= Seg(base^); BaseOfs:= Ofs(base^);
  i:= 0;
  while i < count-1 do begin
    Inc(i); Move(Ptr(BaseSeg, BaseOfs + i*size)^, p^, size);
    l:= 0; r:= i;
    repeat
      m:= l + (r - l) div 2;
      if cmp(Ptr(BaseSeg,BaseOfs + m*size), p) > 0
      then r:= m
      else l:= succ(m);
    until l >= r;
    if l < i then begin
      Move(Ptr(BaseSeg, BaseOfs + l*size)^,
           Ptr(BaseSeg, BaseOfs + succ(l)*size)^,
           (i-l) * size);
      Move(p^, Ptr(BaseSeg, BaseOfs + l*size)^, size);
    end;
  end;
  FreeMem(p,size);
end end;
{$endif}


{--- Quicksort ---}

procedure SrtQuick (base  : SrtElmPtr;   { pointer to first array element }
                    count : SrtIndTyp;   { number of elements to sort     }
                    size  : SrtSizTyp;   { size of array elements to sort }
                    cmp   : SrtCmpFnc);  { compare function               }
{$ifdef WIN32}
var
  first   : PChar;      { start of first element }
  pt,pw   : SrtElmPtr;  { work pointers }

  procedure Sort(lft,rgt: SrtIndTyp);
  var
    i ,j  : longint;    { index   }
    i0,j0 : PChar;      { pointer }
  begin
    i:= lft; i0:= first + i*size;
    j:= rgt; j0:= first + j*size;
    Move((first + (lft + (rgt - lft) div 2) * size)^, pw^, size);
    while i <= j do begin
      while cmp(i0, pw) < 0 do Inc(i0,size);
      while cmp(j0, pw) > 0 do Dec(j0,size);
        i:= (i0 - first) div size;
        j:= (j0 - first) div size;
        if i <= j then begin
          if i < j then begin
             Move(i0^, pt^, size);
             Move(j0^, i0^, size);
             Move(pt^, j0^, size);
          end;
        Inc(i); Inc(i0,size);
        Dec(j); Dec(j0,size);
      end;
    end;
    if lft < j then Sort(lft,j);
    if rgt > i then Sort(i,rgt);
  end;

begin if count > 1 then begin { SrtQuick }
  GetMem(pt,size); GetMem(pw,size);
  first:= base;
  Sort(0,count-1);
  FreeMem(pt,size); FreeMem(pw,size);
end end;
{$else}
var
  BaseSeg : word;       { segment }
  BaseOfs : word;       { offset  }
  pt,pw   : SrtElmPtr;  { pointer }

  procedure Sort(lft,rgt: SrtIndTyp);
  var
    i ,j : LongInt;   { index  }
    i0,j0: word;      { offset }
  begin
    i:= lft; i0:= BaseOfs + i*size;
    j:= rgt; j0:= BaseOfs + j*size;
    Move(Ptr(BaseSeg, BaseOfs + (lft + (rgt - lft) div 2) * size)^, pw^, size);
    while i <= j do begin
      while cmp(Ptr(BaseSeg,i0), pw) < 0 do Inc(i0,size);
      while cmp(Ptr(BaseSeg,j0), pw) > 0 do Dec(j0,size);
        i:= (i0 - BaseOfs) div size;
        j:= (j0 - BaseOfs) div size;
        if i <= j then begin
          if i < j then begin
             Move(Ptr(BaseSeg,i0)^, pt^             , size);
             Move(Ptr(BaseSeg,j0)^, Ptr(BaseSeg,i0)^, size);
             Move(pt^             , Ptr(BaseSeg,j0)^, size);
          end;
        Inc(i); Inc(i0,size);
        Dec(j); Dec(j0,size);
      end;
    end;
    if lft < j then Sort(lft,j);
    if rgt > i then Sort(i,rgt);
  end;

begin if count > 1 then begin { SrtQuick }
  GetMem(pt,size); GetMem(pw,size);
  BaseSeg:= Seg(base^); BaseOfs:= Ofs(base^);
  Sort(0,count-1);
  FreeMem(pt,size); FreeMem(pw,size);
end end;
{$endif}


END.
