{ NUMLIB.PAS : Numerical routine library

  Title   : NUMLIB
  Version : 5.0
  Date    : Dec 21,1997
  Language: Borland Turbo Pascal 4.0 through 7.0 (all targets)
  Author  : J R Ferguson
  Usage   : Unit
  Remarks : Programs and units that use this unit must be compiled
            with the $N+ compiler option (use 80x87 numerical copocessor).
            Use the $E+ option to add 80x87 emulation code if needed.
}

{$N+} {use or emulate numerical coprocessor}

UNIT NumLib;

INTERFACE



const
                   {   ....v....1....v....2  }
{Pi     =            3.14159265358979323846 ;}{Pi }
 DblPi  =   2.0*Pi;{ 6.28318530717958647692 ;}{2 * Pi}
 HalfPi =   0.5*Pi;{ 1.57079632679489661923 ;}{Pi / 2}
 Radian = 180.0/Pi;{57.295779513082320876   ;}{degrees per radian (180/Pi)}
 Ln2    ={ln( 2.0);} 0.6931471805599453094  ; {natural logarithm of 2}
 Ln10   ={ln(10.0);} 2.30258509299404568402 ; {natural logarithm of 10}

function PostInc(var n: integer): integer;
{ Increment integer n; return its original value. }

function PreInc(var n: integer): integer;
{ Increment integer n; return its new value. }

function PostDec(var n: integer): integer;
{ Decrement integer n; return its original value. }

function PreDec(var n: integer): integer;
{ Decrement integer n; return its new value. }

function even(n: integer): boolean;
{ Test if integer n is even. }

{ function odd(n: integer): boolean;     (standard Pascal function)}
{ Test if integer n is odd. }

function Imax(i,j: integer): integer;
{ Return the greatest of two integers.
  See also: <Imin>, <Rmax>, <Rmin>, <Emax>, <Emin>. }

function Imin(i,j: integer): integer;
{ Return the smallest of two integers.
  See also: <Imax>, <Rmax>, <Rmin>, <Emax>, <Emin>. }

function Rmax(x,y: real): real;
{ Return the greatest of two reals.
  See also: <Imax>, <Imin>, <Rmin>, <Emax>, <Emin>. }

function Emax(x,y: extended): extended;
{ Return the greatest of two extended values.
  See also: <Imax>, <Imin>, <Rmax>, <Rmin>, <Emin>. }

function Rmin(x,y: real): real;
{ Return the smallest of two reals.
  See also: <Imax>, <Imin>, <Rmax>, <Emax>, <Emin>. }

function Emin(x,y: extended): extended;
{ Return the smallest of two extended values.
  See also: <Imax>, <Imin>, <Rmax>, <Rmin>, <Emax>. }


function Isign(n: integer): integer;
{ Return +1 if n>0, 0 if n=0, -1 if n<0.
  See also: <Rsign>, <Esign>. }

function Rsign(x: real): real;
{ Return +1.0 if x>0, 0.0 if x=0, -1.0 if x<0.
  See also: <Isign>, <Esign>. }

function Esign(x: extended): extended;
{ Return +1.0 if x>0, 0.0 if x=0, -1.0 if x<0.
  See also: <Isign>, <Rsign>. }


function log2(x: real): real;
{ Return the logarithm of x with base 2.
  See also: <Elog2>. }

function Elog2(x: extended): extended;
{ Return the logarithm of x with base 2.
  See also: <log2>. }

function log10(x: real): real;
{ Return the logarithm of x with base 10.
  See also: <Elog10>. }

function Elog10(x: extended): extended;
{ Return the logarithm of x with base 10.
  See also: <log10>. }

function logg(g,x: real): real;
{ Return the logarithm of x with base g.
  See also: <Elogg>. }

function Elogg(g,x: extended): extended;
{ Return the logarithm of x with base g.
  See also: <logg>. }

{function ln(x: real): real;         (standard Pascal function)}
{ Return the natural logarithm of x. }
function Eln(x: extended): extended;
{ Return the natural logarithm of x. }



function RpowerR(x,e: real): real; { x ** e }
{ Return x to the power of e.
  x and e are both of type real. The result is also of type real.
  See also: <EpowerE>, <EpowerI>, <RpowerI>, <IpowerI>. }

function EpowerE(x,e: extended): extended; { x ** e }
{ Return x to the power of e.
  x and e are both of type extended. The result is also of type extended.
  See also: <RpowerR>, <EpowerI>, <RpowerI>, <IpowerI>. }

function RpowerI(x: real; n: integer): real; { x ** n }
{ Return x to the power of n.
  x is of type real, n is of type integer. The result is of type real.
  See also: <EpowerE>, <RpowerR>, <EpowerI>, <IpowerI>. }

function EpowerI(x: extended; n: integer): extended; { x ** n }
{ Return x to the power of n.
  x is of type extended, n is of type integer. The result is of type extended.
  See also: <EpowerE>, <RpowerR>, <RpowerI>, <IpowerI>. }

function IpowerI(i,n: integer): integer; { i ** n }
{ Return i to the power of n.
  i and n are both of type integer. The result is also of type integer.
  n must be non-negative.
  See also: <EpowerE>, <RpowerR>, <EpowerI>, <RpowerI>. }


function rad(x: real): real;
{ Returns the value of angle x in radians, when x is in degrees.
  Defined as rad(x) = x * Pi/180.0.
  See also: <deg>, <Erad>, <Edeg>. }

function Erad(x: extended): extended;
{ Returns the value of angle x in radians, when x is in degrees.
  Defined as rad(x) = x * Pi/180.0.
  See also: <Edeg>, <rad>, <deg>. }

function deg(x: real): real;
{ Returns the value of angle x in degrees, when x is in radians.
  Defined as deg(x) = x * 180.0/Pi.
  See also: <rad>, <Edeg>, <Erad>. }

function Edeg(x: extended): extended;
{ Returns the value of angle x in degrees, when x is in radians.
  Defined as deg(x) = x * 180.0/Pi.
  See also: <Erad>, <deg>, <rad>. }

function tan(x: real): real;
{ Tangent of x radians.
  Definition : tan(x) = sin(x)/cos(x)
  Condition  : x <> Pi/2 + k*Pi   where k is an integer
  See also   : <Etan> }

function Etan(x: extended): extended;
{ Tangent of x radians.
  Definition : tan(x) = sin(x)/cos(x)
  Condition  : x <> Pi/2 + k*Pi   where k is an integer
  See also   : <tan> }

function cot(x: real): real;
{ Cotangent of x radians.
  Definition : cot(x) = cos(x)/sin(x)
  Condition  : x <> k*Pi          where k is an integer
  See also   : <Ecot> }

function Ecot(x: extended): extended;
{ Cotangent of x radians.
  Definition : cot(x) = cos(x)/sin(x)
  Condition  : x <> k*Pi          where k is an integer
  See also   : <cot> }

function sec(x: real): real;
{ Secant of x radians.
  Definition : sec(x) = 1/cos(x)
  Condition  : x <> Pi/2 + k*Pi   where k is an integer
  See also   : <Esec> }

function Esec(x: extended): extended;
{ Secant of x radians.
  Definition : sec(x) = 1/cos(x)
  Condition  : x <> Pi/2 + k*Pi   where k is an integer
  See also   : <sec> }

function cosec(x: real): real;
{ Cosecant of x radians.
  Definition : cosec(x) = 1/sin(x)
  Condition  : x <> k*Pi          where k is an integer
  See also   : <Ecosec> }

function Ecosec(x: extended): extended;
{ Cosecant of x radians.
  Definition : cosec(x) = 1/sin(x)
  Condition  : x <> k*Pi          where k is an integer
  See also   : <cosec> }

function ArcSin(x: real): real;
{ Arc sine of x in radians.
  Definition : sin(ArcSin(x)) = 1
  Condition  : -1.0 <= x <=  1.0
  Range      : [-Pi/2 , +Pi/2]
  See Also   : <EArcSin> }

function EArcSin(x: extended): extended;
{ Arc sine of x in radians.
  Definition : sin(ArcSin(x)) = 1
  Condition  : -1.0 <= x <=  1.0
  Range      : [-Pi/2 , +Pi/2]
  See Also   : <ArcSin> }

function ArcCos(x: real): real;
{ Arc cosine of x in radians.
  Definition : cos(ArcCos(x)) = 1
  Condition  : -1.0 <= x <=  1.0
  Range      : [0.0 , Pi]
  See Also   : <EArcCos> }

function EArcCos(x: extended): extended;
{ Arc cosine of x in radians.
  Definition : cos(ArcCos(x)) = 1
  Condition  : -1.0 <= x <=  1.0
  Range      : [0.0 , Pi]
  See Also   : <ArcCos> }

function ArcCot(x: real): real;
{ Arc cotangent of x in radians.
  Definition : cot(ArcCot(x)) = 1
  Condition  : none
  Range      : <0 , Pi>
  See Also   : <EArcCot> }

function EArcCot(x: extended): extended;
{ Arc cotangent of x in radians.
  Definition : cot(ArcCot(x)) = 1
  Condition  : none
  Range      : <0 , Pi>
  See Also   : <ArcCot> }

function EArcTan(x: extended): extended;
{ Arc tangent of x in radians.
  Definition : tan(ArcTan(x)) = 1
  Condition  : none
  Range      : <-Pi/2 , Pi/2> }


{function ArcTan(x: real): real; (standard Pascal)}
{ Arc tangent of x in radians.
  Definition : tan(ArcTan(x)) = 1
  Condition  : none
  Range      : <-Pi/2 , Pi/2> }

function sinH(x: real): real;
{ Hyperbolic sine of x radians.
  Definition : sinH(x) = ( exp(x) - exp(-x) )/2
  See Also   : <EsinH> }

function EsinH(x: extended): extended;
{ Hyperbolic sine of x radians.
  Definition : sinH(x) = ( exp(x) - exp(-x) )/2
  See Also   : <sinH> }

function cosH(x: real): real;
{ Hyperbolic cosine of x radians.
  Definition : cosH(x) = ( exp(x) + exp(-x) )/2
  See Also   : <EcosH> }

function EcosH(x: extended): extended;
{ Hyperbolic cosine of x radians.
  Definition : cosH(x) = ( exp(x) + exp(-x) )/2
  See Also   : <cosH> }

function tanH(x: real): real;
{ Hyperbolic tangent of x radians.
  Definition : tanH(x) = sinH(x) / cotH(x)
  See Also   : <EtanH> }

function EtanH(x: extended): extended;
{ Hyperbolic tangent of x radians.
  Definition : tanH(x) = sinH(x) / cotH(x)
  See Also   : <tanH> }

function cotH(x: real): real;
{ Hyperbolic cotangent of x radians.
  Definition : cotH(x) = cosH(x) / sinH(x)
  Condition  : x <> 0.0
  See Also   : <EcotH> }

function EcotH(x: extended): extended;
{ Hyperbolic cotangent of x radians.
  Definition : cotH(x) = cosH(x) / sinH(x)
  Condition  : x <> 0.0
  See Also   : <cotH> }



IMPLEMENTATION


function PostInc(var n: integer): integer;
begin PostInc:= n; Inc(n) end;

function PreInc(var n: integer): integer;
begin Inc(n); PreInc:= n end;

function PostDec(var n: integer): integer;
begin PostDec:= n; Dec(n) end;

function PreDec(var n: integer): integer;
begin Dec(n); PreDec:= n end;

function even(n: integer): boolean;
begin even:= not odd(n) end;

function Imax(i,j: integer): integer;
begin if i>j then Imax:= i else Imax:= j end;

function Imin(i,j: integer): integer;
begin if i<j then Imin:= i else Imin:= j end;

function Rmax(x,y: real): real;
begin if x>y then Rmax:= x else Rmax:= y end;

function Rmin(x,y: real): real;
begin if x<y then Rmin:= x else Rmin:= y end;

function Emax(x,y: extended): extended;
begin if x>y then Emax:= x else Emax:= y end;

function Emin(x,y: extended): extended;
begin if x<y then Emin:= x else Emin:= y end;

function Isign(n: integer): integer;
begin if n=0 then Isign:= 0 else if n>0 then Isign:= 1 else Isign:= -1 end;

function Rsign(x: real): real;
begin
  if x=0.0 then Rsign:= 0.0 else if x>0.0 then Rsign:= 1.0 else Rsign:= -1.0
end;

function Esign(x: extended): extended;
begin
  if x=0.0 then Esign:= 0.0 else if x>0.0 then Esign:= 1.0 else Esign:= -1.0
end;

function log2(x: real): real;
begin log2:= ln(x)/Ln2 end;

function Elog2(x: extended): extended;
begin Elog2:= ln(x)/Ln2 end;

function log10(x: real): real;
begin log10:= ln(x)/Ln10 end;

function Elog10(x: extended): extended;
begin Elog10:= ln(x)/Ln10 end;

function logg(g,x: real): real;
begin logg:= ln(x)/ln(g) end;

function Elogg(g,x: extended): extended;
begin Elogg:= ln(x)/ln(g) end;

function Eln(x: extended): extended;
begin Eln:= ln(x) end;

function RpowerR(x,e: real): real;
begin if x=0.0 then RpowerR:= 0.0 else RpowerR:= exp(e*ln(x)) end;

function EpowerE(x,e: extended): extended;
begin if x=0.0 then EpowerE:= 0.0 else EpowerE:= exp(e*ln(x)) end;

function EpowerI(x: extended; n: integer): extended;
{ By Dennis E. Hamilton v1.08 Nov 11,1985
  Dr Dobbs Journal of Software Tools #112, Feb 1986 }
var i: integer; r: extended;
begin
  if n=0
  then EpowerI:= x/x
  else begin
    i:= abs(n);
    while not odd(i) do begin x:= sqr(x); i:= i shr 1 end;
    r:= x;
    while i<>1 do begin
      repeat x:= sqr(x); i:= i shr 1 until odd(i);
      r:= r*x
    end;
    if n<0 then EpowerI:= 1.0/r else EpowerI:= r
  end
end;

function RpowerI(x: real; n: integer): real;
begin RpowerI:= EpowerI(x,n) end;

function IpowerI(i,n: integer): integer;
var y: integer;
begin
  y:= 1;
  while n>0 do begin
    while not odd(n) do begin n:= n shr 1; i:= sqr(i) end;
    n:= pred(n); y:= y*i
  end;
  IpowerI:= y
end;

function rad(x: real): real;
begin rad:=x/Radian end;

function Erad(x: extended): extended;
begin Erad:=x/Radian end;

function deg(x: real): real;
begin deg:= x*Radian end;

function Edeg(x: extended): extended;
begin Edeg:= x*Radian end;

function tan(x: real): real;
begin tan:= sin(x)/cos(x) end;

function Etan(x: extended): extended;
begin Etan:= sin(x)/cos(x) end;

function cot(x: real): real;
begin cot:= cos(x)/sin(x) end;

function Ecot(x: extended): extended;
begin Ecot:= cos(x)/sin(x) end;

function sec(x: real): real;
begin sec:= 1.0/cos(x) end;

function Esec(x: extended): extended;
begin Esec:= 1.0/cos(x) end;

function cosec(x: real): real;
begin cosec:= 1.0/sin(x) end;

function Ecosec(x: extended): extended;
begin Ecosec:= 1.0/sin(x) end;

function ArcSin(x: real): real;
begin
  if x=1.0 then ArcSin:= HalfPi
  else if x=-1.0 then ArcSin:= -HalfPi
  else ArcSin:= ArcTan(x/sqrt(1.0-sqr(x)))
end;

function EArcSin(x: extended): extended;
begin
  if x=1.0 then EArcSin:= HalfPi
  else if x=-1.0 then EArcSin:= -HalfPi
  else EArcSin:= ArcTan(x/sqrt(1.0-sqr(x)))
end;

function ArcCos(x: real): real;
begin
  if x=1.0 then ArcCos:= 0.0
  else if x=-1.0 then ArcCos:= Pi
  else ArcCos:= HalfPi-ArcTan(x/sqrt(1.0-sqr(x)))
end;

function EArcCos(x: extended): extended;
begin
  if x=1.0 then EArcCos:= 0.0
  else if x=-1.0 then EArcCos:= Pi
  else EArcCos:= HalfPi-ArcTan(x/sqrt(1.0-sqr(x)))
end;

function ArcCot(x: real): real;
begin ArcCot:= HalfPi-ArcTan(x) end;

function EArcCot(x: extended): extended;
begin EArcCot:= HalfPi-ArcTan(x) end;

function EArcTan(x: extended): extended;
begin EArcTan:= ArcTan(x) end;

function sinH(x: real): real;
begin sinH:= (exp(x)-exp(-x))/2.0 end;

function EsinH(x: extended): extended;
begin EsinH:= (exp(x)-exp(-x))/2.0 end;

function cosH(x: real): real;
begin cosH:= (exp(x)+exp(-x))/2.0 end;

function EcosH(x: extended): extended;
begin EcosH:= (exp(x)+exp(-x))/2.0 end;

function tanH(x: real): real;
begin tanh:= 1.0-2.0/(exp(2.0*x)+1.0) end;

function EtanH(x: extended): extended;
begin Etanh:= 1.0-2.0/(exp(2.0*x)+1.0) end;

function cotH(x: real): real;
begin cotH:= 1.0+2.0/(exp(2.0*x)-1.0) end;

function EcotH(x: extended): extended;
begin EcotH:= 1.0+2.0/(exp(2.0*x)-1.0) end;


END.
