/* STZLIB.H -- String handling (zero-ended strings)

  Title   : STZLIB
  Version : 2.0
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Date    : Nov 24,1996
  Author  : J.R. Ferguson
  Usage   : Header file


  The routines in this library use the standard C string type of char[n]
  or char *. A zero-byte is used as end-of-string marker. The strings
  themselves can therefore never include a character with the ordinal
  value 0. The maximum string length is 65535 characters.

  Character positions are numbered starting with 0.

  String pointers having the value NULL are not supported.
  The routines that copy, insert or append characters into strings do not
  perform any length checking. The receiving string variables must have
  enough room for the (extra) characters they will receive.
*/

/*.hlptx skip 1*/
#ifndef Stz_Lib
#define Stz_Lib

#include <string.h>

typedef size_t Stz_Ind;
typedef char * Stz_Ptr;

#define Stz_NOTFOUND  0xFFFF

/*.hlptx skip 4*/
#ifdef __cplusplus
extern "C" {
#endif

Stz_Ptr Stz_After(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns a pointer to the first character in string src after pattern
   pat, or to the end-of-string marker if pat is not a part of src or if
   pat is an empty string.

   See also: <Stz_Before>, <Stz_Right>, <Stz_Sub>
*/

#define Stz_Alloc(s)  ((Stz_Ptr)strdup((const char *)(s)))
/* Stz_Ptr Stz_Alloc(const Stz_Ptr s);
   Allocates memory storage for Stz_Len(s)+1 characters and copies the
   contents of string s to this new allocated block of memory. 
   Returns a pointer to this allocated string, or NULL if there is not
   enough memory left to allocate the string.

   See also: <Stz_Free>, <Stz_ReAlloc>.
*/

Stz_Ptr Stz_Before(Stz_Ptr dst, const Stz_Ptr src, Stz_Ptr pat);
/* Extracts into dst the part from string src that comes before pattern
   pat, and adds an end-of-string marker.
   If pat is not a part of src, string src is copied to dst without
   change. Returns dst.

   See also: <Stz_After>, <Stz_Right>, <Stz_Sub>
*/

#define Stz_Cat(dst,src) (Stz_Ptr)strcat((char *)(dst),(const char *)(src))
/* Stz_Ptr Stz_Cat(Stz_Ptr dst, const Stz_Ptr src);
   Appends string src to string dst.
   Returns dst.
   src and dst must be separate, non-overlapping strings.

   See also:
   <Stz_ECat>, <Stz_NCat>, <Stz_cCat>, <Stz_cECat>
*/

Stz_Ptr Stz_Center(Stz_Ptr s, Stz_Ind n);
/* Centers string s to be printed on a line of n characters wide by
   inserting spaces in front of it.
   Returns s.
*/

#define Stz_Cmp(s1,s2)  strcmp((const char *)(s1),(const char *)(s2))
/* int Stz_Cmp(const Stz_Ptr s1, const Stz_Ptr s2)
   Compares strings s1 and s2, based on the ordinal value of their
   characters.
   The result is negative if s1<s2, zero if s1==s2 and positive if s1>s2.

   See also:
   <Stz_LexCmp>, <Stz_UppCmp>, <Stz_NCmp>, <Stz_LexNCmp>, <Stz_UppNCmp>
*/

#define Stz_Cpy(dst,src) (Stz_Ptr)strcpy((char *)(dst),(const char *)(src))
/* Stz_Ptr Stz_Cpy(Stz_Ptr dst, const Stz_Ptr src);
   Copies string src to dst.
   Returns dst.

   See also:
   <Stz_ECpy>, <Stz_NCpy>, <Stz_cCpy>, <Stz_cECpy>, <Stz_StpCpy>
*/

Stz_Ptr Stz_Create(Stz_Ptr s);
/* Creates an empty string s, containing an end-of-string marker only.
   Returns s.

   See also: <Stz_Fill>, <Stz_Make>
*/

Stz_Ptr Stz_cCat(Stz_Ptr dst, char c);
/* Appends character c to string dst.
   If c == '\0', string dst is not altered.
   Returns dst.

   See also: 
   <Stz_Cat>, <Stz_ECat>, <Stz_NCat>, <Stz_cECat>
*/

Stz_Ptr Stz_cCpy(Stz_Ptr dst, char c);
/* Copies character c to the first position of string dst.
   If c != '\0', an end-of-string marker is added.
   Returns dst.

   See also:
   <Stz_Cpy>, <Stz_ECpy>, <Stz_NCpy>, <Stz_cECpy>, <Stz_StpCpy>
*/

Stz_Ptr Stz_cECat(Stz_Ptr dst, char c);
/* Appends characters c to the end of string dst.
   If c == '\0', string dst is not altered.
   Returns a pointer to the end-of-string marker of dst.

   See also:
   <Stz_Cat>, <Stz_ECat>, <Stz_NCat>, <Stz_cCat>
*/

Stz_Ptr Stz_cECpy(Stz_Ptr dst, char c);
/* Copies character c to the first position of string dst.
   If c != '\0', an end-of-string marker is added.
   Returns a pointer to the end-of-string marker of dst.

   See also:
   <Stz_Cpy>, <Stz_ECpy>, <Stz_NCpy>, <Stz_cCpy>, <Stz_StpCpy>
*/

char Stz_cGet(Stz_Ptr s);
/* Returns the first character from string s and removes that character
   from string s. 
   If s is an empty string, a character with the ordinal value 0 is
   returned.

   See also: <Stz_Gtw>, <Stz_cRet>
*/

Stz_Ptr Stz_cIns(Stz_Ptr dst, char c, Stz_Ind i);
/* Inserts character c into string dst at position i.
   If i >= Stz_Len(s), the character is appended to string dst.
   If c == '\0', string dst is not altered.
   Returns dst.

   See also: <Stz_Ins>, <Stz_NIns>
*/

Stz_Ind Stz_cIPos(const Stz_Ptr s, char c);
/* Returns the first position of character c in string s
   If c == '\0', the value of Stz_Len(s) will be returned.
   If c cannot be found in s, the value Stz_NOTFOUND is returned.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>
*/

#define Stz_cPos(s,c)  (Stz_Ptr)strchr((const char *)(s),(int)(c))
/* Stz_Ptr Stz_cPos(const Stz_Ptr s, char c);
   Returns a pointer to the first position of character c in string s,
   or NULL if c cannot be found in s.
   If c == '\0', an pointer to the end-of-string marker of s is returned.


   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>
*/

char Stz_cRet(const Stz_Ptr s, Stz_Ind i);
/* Returns the character at position i in string s. 
   String s will not be altered.
   If i >= Stz_Len(s), the character '\0' will be returned.

   See also: <Stz_cGet>, <Stz_Gtw>
*/

Stz_Ind Stz_cRIPos(const Stz_Ptr s, char c);
/* Returns the last position of character c in string s, or the value
   Stz_NOTFOUND if c cannot be found in s. 
   If c == '\0', the value of Stz_Len(s) will be returned.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>
*/

#define Stz_cRPos(s,c) (Stz_Ptr)strrchr((const char *)(s),(int)(c))
/* Stz_Ptr Stz_cRPos(const Stz_Ptr s, char c);
   Returns a pointer to the last position of character c in string s, or
   NULL if c cannot be found in s. 
   If c == '\0', a pointer to the end-of-string marker of s is returned.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>
*/

Stz_Ptr Stz_cUpd(Stz_Ptr s, char c, Stz_Ind i);
/* Replaces the character at position i in string s with character c.
   If i >= Stz_Len(s), string s is left unchanged.
   Returns s.

   See also: <Stz_Repl>, <Stz_NRepl>
*/

Stz_Ind Stz_cUppIPos(const Stz_Ptr s, char c);
/* Returns the first position of character c in string s, or the value
   Stz_NOTFOUND if c cannot be found in s.
   If c == '\0', the value of Stz_Len(s) will be returned.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>
*/

Stz_Ptr Stz_cUppPos(const Stz_Ptr s, char c);
/* Returns a pointer to the first position of character c in string s, or
   NULL if c cannot be found in s.
   If c == '\0', an pointer to the end-of-string marker of s is returned.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>
*/

Stz_Ind Stz_cUppRIPos(const Stz_Ptr s, char c);
/* Returns the last position of character c in string s, or the value
   Stz_NOTFOUND if c cannot be found in s.
   If c == '\0', the value of Stz_Len(s) will be returned.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>
*/

Stz_Ptr Stz_cUppRPos(const Stz_Ptr s, char c);
/* Returns a pointer to the last position of character c in string s, or
   NULL if c cannot be found in s.
   If c == '\0', a pointer to the end-of-string marker of s is returned.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRIPos>
*/

Stz_Ptr Stz_Del(Stz_Ptr s, Stz_Ind i, Stz_Ind n);
/* Deletes n characters from string s, starting at position i. 
   Returns s.
*/

Stz_Ptr Stz_Detab(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
/* Expands tabs in string src into space groups, using n as the tab field
   width. The result is in dst, and the function also returns dst.
   This function recognizes the following control characters:
     HT : Horizontal tab : expand to spaces.
     BS : Back space     : decrement column position by 1.
     CR : Carriage return: reset column position to 1.
   Other characters with a value in the range 0..31 are considered as
   non-printable. They are copied without change, but don't alter the
   current column position.
   Characters with a value in the range 128..255 are considered to be
   printable, so they are copied and increment the column position by 1.
   Remarks:
   - The column positioning may be at fault when string src containts BS
     characters immediately following HT or other control characters.
   - If n==0, string src is copied to dst without change.
   - src and dst must be separate, non-overlapping strings.

   See also: <Stz_Entab>
*/

Stz_Ptr Stz_ECat(Stz_Ptr dst, const Stz_Ptr src);
/* Appends string src to string dst.
   Returns a pointer to the end-of-string marker of dst.
   src and dst must be separate, non-overlapping strings.

   See also:
   <Stz_Cat>, <Stz_NCat>, <Stz_cCat>, <Stz_cECat>
*/

#define Stz_ECpy(dst,src) \
 (Stz_Ptr)stpcpy((char *)(dst),(const char *)(src))
/* Stz_Ptr Stz_ECpy(Stz_Ptr dst, const Stz_Ptr src);
   Copies string src to dst.
   Returns a pointer to the end-of-string marker of dst.

   See also:
   <Stz_Cpy>, <Stz_NCpy>, <Stz_cCpy>, <Stz_cECpy>, <Stz_StpCpy>
*/

int Stz_Empty(const Stz_Ptr s);
/* Returns 1 if string s is empty, and 0 if it contains one ore more
   characters apart from the end-of-string marker.
*/

Stz_Ptr Stz_End(const Stz_Ptr s);
/* Returns a pointer to the end-of-string marker of s. */

Stz_Ptr Stz_Entab(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
/* Replaces space groups in src by horizontal tab characters, using
   multiples of n as tab columns. Single spaces are never replaced by
   tabs. The result is in dst, and the function also returns dst.
   This function recognizes the following control characters:
     HT : Horizontal tab : expand to spaces.
     BS : Back space     : decrement column position by 1.
     CR : Carriage return: reset column position to 1.
   Other characters with a value in the range 0..31 are considered as
   non-printable. They are copied without change, but don't alter the
   current column position. Characters with an ordinal value in the range
   128..255 are considered to be printable, so they are copied and
   increment the column position by 1.
   Remarks:
   - The column positioning may be at fault when string src containts BS
     characters immediately following a space group, a HT or another 
     control character.
   - If n == 0, string src is copied to dst without change.
   - src and dst must be separate, non-overlapping strings.

   See also: <Stz_Detab>
*/

Stz_Ptr Stz_Fill(Stz_Ptr s, char c, Stz_Ind n);
/* Fills (lengthens) string s to a length of n, by appending characters c
   (also if c = '\0'), and then appending an end-of-string marker to s.
   If n <= Stz_Len(s), string s is left unchanged.
   Returns s.

   See also: <Stz_Create>, <Stz_Make>
*/

void Stz_Free(Stz_Ptr *p);
/* Frees the memory block used for string *p, previously allocated with
   StzAlloc or StzRealloc (not regarding the current length of string *p),
   and sets p to NULL.

   See also: <Stz_Alloc>, <Stz_ReAlloc>
*/

Stz_Ptr Stz_Gtw(Stz_Ptr dst, Stz_Ptr src);
/* Returns the first word from string src into dst and removes this word
   from src.
   If string src does not contain a word, dst and src are both made empty.

   The folowing ASCII characters are seen as word separators: carriage
   return, line feed, form feed, horizontal tab, vertical tab and space.

   See also: <Stz_cGet>, <Stz_cRet>
*/

Stz_Ptr Stz_Ins(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i);
/* Inserts string src at position i into string dst.
   If i >= Stz_Len(src), string src will be appended to dst.
   Returns dst.

   src and dst must be separate, non-overlapping strings.

   See also: <Stz_NIns>, <Stz_cIns>
*/

Stz_Ind Stz_IPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns the first position of string pattern pat in string src, 
   or the value Stz_NOTFOUND if pat cannot be found in src.

   See also:
   <Stz_Pos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

#define Stz_Len(s)  (Stz_Ind)strlen((const char *)(s))
/* Stz_Ind Stz_Len(const Stz_Ptr s);
   Returns the number of characters i string s, not counting the end-of-
   string marker. */

int Stz_LexCmp(const Stz_Ptr s1, const Stz_Ptr s2);
/* Compares strings s1 and s2, based on the Chr_Lexorder character order.
   The result is negative if s1<s2, zero if s1==s2 and positive if s1>s2.

   See also: 
   <Stz_Cmp>, <Stz_UppCmp>, <Stz_NCmp>, <Stz_LexNCmp>, <Stz_UppNCmp>,
   <Stz_SoundAlike>, <Stz_SoundEx>
*/

int Stz_LexNCmp(const Stz_Ptr s1, const Stz_Ptr s2, Stz_Ind n);
/* Compares a maximum of n characters from strings s1 and s2, based on the
   Chr_Lexorder character order.
   The result is negative if s1<s2, zero if s1==s2 and positive if s1>s2.

   See also: 
   <Stz_Cmp>, <Stz_LexCmp>, <Stz_UppCmp>, <Stz_NCmp>, <Stz_UppNCmp>,
   <Stz_SoundAlike>, <Stz_SoundEx>
*/

Stz_Ptr Stz_LexSrt(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
/* Like StzSrt, using the Chr_LexOrder character order.

   See also: <Stz_Srt>, <Stz_UppSrt>
*/

#define Stz_Low(s) (Stz_Ptr)strlwr((Stz_Ptr)(s))
/* Stz_Ptr Stz_Low(Stz_Ptr s);
   Converts all upper case letters in string s to lower case.
   Returns s.

   See also: <Stz_Upp>
*/

Stz_Ptr Stz_Make(Stz_Ptr s, char c, Stz_Ind n);
/* Fills string s with n chacarters c and appends an end-of-string-marker
   to it. Returns s.

   See also: <Stz_Create>, <Stz_Fill>
*/

#define Stz_NCat(dst,src,n) \
 (Stz_Ptr)strncat((char *)(dst),(const char *)(src),(size_t)(n))
/* Stz_Ptr Stz_NCat(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
   Appends a maximum of n characters from string src to string dst.
   Returns dst. 
   src and dst must be separate, non-overlapping strings.

   See also:
   <Stz_Cat>, <Stz_ECat>, <Stz_cCat>, <Stz_cECat>
*/

#define Stz_NCmp(s1,s2,n) \
 strncmp((const char *)(s1),(const char *)(s2),(size_t)(n))
/* int Stz_NCmp(const Stz_Ptr s1, const Stz_Ptr s2, Stz_Ind n);
   Compares a maximum of n characters from strings s1 and s2, based on the
   ordinal value of their characters.
   The result is negative if s1<s2, zero if s1==s2 and positive if s1>s2.

   See also:
   <Stz_Cmp>, <Stz_LexCmp>, <Stz_UppCmp>, <Stz_LexNCmp>, <Stz_UppNCmp>,
   <Stz_SoundAlike>, <Stz_SoundEx>
*/

Stz_Ptr Stz_NCpy(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
/* Copies a maximum of n characters from string src to dst, and appends an
   end-of-string marker to dst. Returns dst.

   Remark: Stz_NCpy() is different from the ANSI C function strncpy(), in
   that it will not add extra 0-bytes to string dst if n>strlen(src).

   See also:
   <Stz_Cpy>, <Stz_ECpy>, <Stz_cCpy>, <Stz_cECpy>, <Stz_StpCpy>
*/

Stz_Ptr Stz_NIns(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i, Stz_Ind n);
/* Inserts a maximum of n characters from string src at position i into
   string dst.
   If i >= Stz_Len(src), the characters from string src will be appended
   to dst. 
   Returns dst

   src and dst must be separate, non-overlapping strings.

   See also: <Stz_Ins>, <Stz_cIns>
*/

Stz_Ptr Stz_NRepl(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i, Stz_Ind n);
/* Like StzRepl, replacing a maximum of n characters in string dst.

   See also: <Stz_cUpd>, <Stz_Repl>
*/

Stz_Ptr Stz_Pos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns a pointer to the first position of string pattern pat in string
   src, or NULL if pat is empty or cannot be found in src.

   See also:
   <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

Stz_Ptr Stz_RAS(Stz_Ptr s);
/* Remove All Spaces: Removes all carriage return, line feed, form feed,
   horizontal tab, vertical tab and space characters from string s.
   Returns s.

   See also: <Stz_RLS>, <Stz_RTS>
*/

Stz_Ptr Stz_ReAlloc(Stz_Ptr *p, const Stz_Ptr s);
/* Frees the memory block used for string *p, previously allocated with
   StzAlloc or StzRealloc, disregarding the current length of *p;
   allocates new memory storage for Stz_Len(s)+1 characters and copies the
   contents of string s to this new allocated block of memory. 
   Returns a pointer to this newly allocated string and also sets p to
   this pointer value.

   See also: <Stz_Alloc>, <Stz_Free>
*/

Stz_Ptr Stz_Repl(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i);
/* Replaces characters from dst by those of string src, starting at
   position i in dst. The resulting string dst may be longer than its
   original value. If i>Stz_Len(dst), spaces are added to dst until its
   length is i, and then string src is appended to it. 
   Returns dst.

   src and dst must be separate non-overlapping strings.
  
   See also: <Stz_cUpd>, <Stz_NRepl>
*/

#define Stz_Rev(s)  (Stz_Ptr)strrev((char *)(s))
/* Stz_Ptr Stz_Rev(Stz_Ptr s);
   Reverses the character order of string s.
   Returns dst.
*/

Stz_Ptr Stz_Right(const Stz_Ptr s, Stz_Ind n);
/* Returns a pointer to the the n rightmost characters of string s.
   If n==0, a pointer to the end-of-string marker of s is returned.

  See also: <Stz_Sub>, <Stz_Before>, <Stz_After>
*/

Stz_Ind Stz_RIPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns the last position of string pat in string src. 
   If pat is empty or cannot be found in src, the value Stz_NOTFOUND is
   returned.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

Stz_Ptr Stz_RLS(Stz_Ptr s);
/* Removes Leading Spaces: Remove all leading carriage return, line feed,
   form feed, horizontal tab, vertical tab and space characters from
   string s.
   Returns s.

   See also: <Stz_RAS>, <Stz_RTS>
*/

Stz_Ptr Stz_RPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns a pointer to the last position of string pattern pat in string
   src, or NULL if pat is empty or cannot be found in src.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

Stz_Ptr Stz_RTS(Stz_Ptr s);
/* Removes Trailing Spaces: Remove all trailing carriage return, line
   feed, form feed, horizontal tab, vertical tab and space characters from
   string s.
   Returns s.

   See also: <Stz_RAS>, <Stz_RLS>
*/

int Stz_SoundAlike(const Stz_Ptr s1, const Stz_Ptr s2, Stz_Ind i);
/* Tests if Stz_SoundEx(d1,s1,i) and Stz_SoundEx(d2,s2,i) yield the same
   results d1 and d2.
   Return TRUE if the Stz_SoundEx results are equal, FALSE if different.

   See also:
   <Stz_SoundEx>, <Stz_LexCmp>, <Stz_UppCmp>, <Stz_NCmp>, <Stz_LexNCmp>,
   <Stz_UppNCmp>
*/

Stz_Ptr Stz_SoundEx(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i);
/* Sound Expression: Creates into dst a string that is derived from string
   src by copying the first i characters, translating lower case letters
   to upper case, and then adding a 'sound code' for the remaining
   characters.
   Two strings that yield the same Stz_SoundEx result will probably sound
   alike in English. This function can therefore be used when searching a
   name or another string value in a database where the correct spelling
   is not certain.
   Returns dst.

   The exact algorithm for deriving the function result is as follows:
   1. Translate the first i characters to upper case.
   2. Translate the remaining characters to upper case, then code them
      as follows:
        'B','F','P','V'                 become '1'
        'C','G','J','K','Q','S','X','Z' become '2'
        'D','T'                         become '3'
        'L'                             become '4'
        'M','N'                         become '5'
        'R'                             become '6'
        alle other characters           are skipped.
      Moreover, never append the same code digit twice in this coding
      process.

  See also:
  <Stz_SoundAlike>, <Stz_LexCmp>, <Stz_UppCmp>, <Stz_NCmp>, <Stz_LexNCmp>,
  <Stz_UppNCmp>
*/

Stz_Ptr Stz_Srt(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
/* Sorts substring fields of string src, having a fixed field length n,
   comparing the ord-value of the characters, and puts the result in dst.
   If n==0, string src is copied to dst without change. 
   Returns dst.

   Example:  src = 'IF    THEN  BEGIN END   WHILE  REPEATDO    ', n=6
             dst = 'BEGIN DO    END   IF    REPEAT THEN  WHILE '

   src and dst must be separate, non-overlapping strings.

   See also: <Stz_LexSrt>, <Stz_UppSrt>
*/

Stz_Ptr Stz_StpCpy(Stz_Ptr dst, const char src[]);
/* Converts the (Turbo) Pascal type string src into type StzPtr string
   dst, appending an end-of-string marker.
   Returns dst.

   String src should start with a 1 byte length field, followed by its
   contents.

   See also:
   <Stz_Cpy>, <Stz_ECpy>, <Stz_NCpy>, <Stz_cCpy>, <Stz_cECpy>
*/

Stz_Ptr Stz_Sub(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i, Stz_Ind n);
/* Returns into dst a substring from s, consisting of a maximum of n
   characters starting at position i.
   Returns dst.

   src and dst must be separate, non-overlapping strings.

   See also: <Stz_Right>, <Stz_Before>, <Stz_After>
*/

Stz_Ptr Stz_Trunc(Stz_Ptr s, Stz_Ind i);
/* Replaces the character at position i in string s with an end-of-string
   marker, effectively truncating string s to a length of i.
   If i >= Stz_Len(s), string s will not be altered. 
   Returns s. 
*/

#define Stz_Upp(s)  (Stz_Ptr)strupr((char *)(s))
/* Stz_Ptr Stz_Upp(Stz_Ptr s);
   Converts all lower case letters in string s to upper case.
   Returns s.

   See also: <Stz_Low>
*/

#define Stz_UppCmp(s1,s2) stricmp((const char *)(s1),(const char *)(s2))
/* int Stz_UppCmp(const Stz_Ptr s1, const Stz_Ptr s2);
   Compares strings s1 and s2, seeing upper and lower case characters as
   equal.
   The result is negative if s1<s2, zero if s1==s2 and positive if s1>s2.

   See also:
   <Stz_Cmp>, <Stz_LexCmp>, <Stz_NCmp>, <Stz_LexNCmp>, <Stz_UppNCmp>,
   <Stz_SoundAlike>, <Stz_SoundEx>
*/

Stz_Ind Stz_UppIPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns the first position of string pattern pat in string src,
   or the value Stz_NOTFOUND if pat cannot be found in src.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppRPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

#define Stz_UppNCmp(s1,s2,n) \
 strnicmp((const char *)(s1),(const char *)(s2),(size_t)(n))
/* int Stz_UppNCmp(const Stz_Ptr s1, const Stz_Ptr s2, Stz_Ind n);
   Compares a maximum of n characters of strings s1 and s2, seeing upper
   and lower case characters as equal.
   The result is negative if s1<s2, zero if s1==s2 and positive if s1>s2.

   See also:
   <Stz_Cmp>, <Stz_LexCmp>, <Stz_UppCmp>, <Stz_NCmp>, <Stz_LexNCmp>,
   <Stz_SoundAlike>, <Stz_SoundEx>
*/

Stz_Ptr Stz_UppPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns a pointer to the first position of string pattern pat in string
   src, or NULL if pat is empty or cannot be found in src.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppIPos>, <Stz_UppRPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

Stz_Ind Stz_UppRIPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns the last position of string pattern pat in string src,
   or the value Stz_NOTFOUND if pat cannot be found in src.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRPos>,
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

Stz_Ptr Stz_UppRPos(const Stz_Ptr src, const Stz_Ptr pat);
/* Returns a pointer to the last position of string pattern pat in string
   src, or NULL if pat is empty or cannot be found in src.
   Upper and lower case characters are seen as equal.

   See also:
   <Stz_Pos>, <Stz_IPos>, <Stz_RPos>, <Stz_RIPos>,
   <Stz_UppPos>, <Stz_UppIPos>, <Stz_UppRIPos>
   <Stz_cPos>, <Stz_cIPos>, <Stz_cRPos>, <Stz_cRIPos>,
   <Stz_cUppPos>, <Stz_cUppIPos>, <Stz_cUppRPos>, <Stz_cUppRIPos>,
*/

Stz_Ptr Stz_UppSrt(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n);
/* Like Stz_Srt, seeing upper and lower case characters as equal.

   See also: <Stz_Srt>, <Stz_LexSrt>
*/

/*.hlptx skip 4*/
#ifdef __cplusplus
}
#endif
#endif /* ifndef Stz_Lib */
