/* STZLIB.C -- String handling (zero-ended strings)

  Title   : STZLIB
  Version : 2.0
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  date    : Nov 24,1996
  Author  : J.R. Ferguson
  Usage   : Function library
*/

#include <ctype.h>
#include <string.h>
#include <alloc.h>
#include <stdlib.h>
#include "deflib.h"
#include "chrlib.h"
#include "stzlib.h"

#define BS ASCII_BS
#define HT ASCII_HT
#define CR ASCII_CR

const static char _SoundChr[256] =
/*      0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF0123456789ABCDEF*/
/*0-3*/"                                                                "
/*4-7*/"  123 12  22455 12623 1 2 2       123 12  22455 12623 1 2 2     "
/*8-B*/"                                                                "
/*C-F*/"                                                                ";

Stz_Ptr Stz_After(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ptr p;

  p= Stz_Pos(src,pat);
  if (p==NULL) return Stz_End(src);
  else return p + Stz_Len(pat);
}

Stz_Ptr Stz_Before(Stz_Ptr dst, const Stz_Ptr src, Stz_Ptr pat)
{ Stz_Ptr p;

  p= Stz_Pos(src,pat);
  if (p==NULL) return Stz_Cpy(dst,src);
  else return Stz_NCpy(dst,src,p-src);
}

Stz_Ptr Stz_Center(Stz_Ptr s, Stz_Ind n)
{ Stz_Ind l; Stz_Ptr p;

  Stz_RLS(s);
  Stz_RTS(s);
  l= Stz_Len(s);
  if (l && (n > l)) {
    p= s + ((n - l) >> 1);
    memmove(p,s,l+1);
    memset(s,' ',p-s);
  }
  return s;
}

Stz_Ptr Stz_Create(Stz_Ptr s)
{
  *s= '\0';
  return s;
}

Stz_Ptr Stz_cCat(Stz_Ptr dst, char c)
{ Stz_Ptr p;

  if (c) { p= Stz_End(dst); *p++= c; *p= '\0'; }
  return dst;
}

Stz_Ptr Stz_cCpy(Stz_Ptr dst, char c)
{ Stz_Ptr p= dst;

  *p = c;
  if (c) *(++p)= '\0';
  return dst;
}

Stz_Ptr Stz_cECat(Stz_Ptr dst, char c)
{ Stz_Ptr p= Stz_End(dst);

  if (c) { *p= c; *(++p)= '\0'; }
  return p;
}

Stz_Ptr Stz_cECpy(Stz_Ptr dst, char c)
{ Stz_Ptr p= dst;

  *p= c;
  if (c) *(++p)= '\0';
  return p;
}

char Stz_cGet(Stz_Ptr s)
{ char c= *s;

  if (c) Stz_Cpy(s,s+1);
  return c;
}

Stz_Ptr Stz_cIns(Stz_Ptr dst, char c, Stz_Ind i)
{ char tmp[1];

  if (c) {
    tmp[0]= c; tmp[1]= '\0';
    return Stz_Ins(dst,tmp,i);
  }
  else return dst;
}

Stz_Ind Stz_cIPos(const Stz_Ptr s, char c)
{ Stz_Ptr p= Stz_cPos(s,c);

  return (p==NULL) ? Stz_NOTFOUND : p-s;
}

char Stz_cRet(const Stz_Ptr s, Stz_Ind i)
{ return (i > Stz_Len(s)) ? '\0' : s[i]; }

Stz_Ind Stz_cRIPos(const Stz_Ptr s, char c)
{ Stz_Ptr p= Stz_cRPos(s,c);

  return (p==NULL) ? Stz_NOTFOUND : p-s;
}

Stz_Ptr Stz_cUpd(Stz_Ptr s, char c, Stz_Ind i)
{
  if (i < Stz_Len(s)) s[i]= c;
  return s;
}

Stz_Ind Stz_cUppIPos(const Stz_Ptr s, char c)
{ Stz_Ptr p; char c1, c2;

  c1=Chr_ToUpper(c);
  for (p=s; ((c2=Chr_ToUpper(*p++)) != c1) && c2; );
  return (c1==c2) ? (--p)-s : Stz_NOTFOUND;
}

Stz_Ptr Stz_cUppPos(const Stz_Ptr s, char c)
{ Stz_Ptr p; char c1, c2;

  c1=Chr_ToUpper(c);
  for (p=s; ((c2=Chr_ToUpper(*p++)) != c1) && c2; );
  return (c1==c2) ? --p : NULL;
}

Stz_Ind Stz_cUppRIPos(const Stz_Ptr s, char c)
{ Stz_Ptr p; char c1, c2;

  c1=Chr_ToUpper(c);
  for (p=Stz_End(s); ((c2=Chr_ToUpper(*p)) !=  c1) && (p>s); --p);
  return (c1==c2) ? p-s : Stz_NOTFOUND;
}

Stz_Ptr Stz_cUppRPos(const Stz_Ptr s, char c)
{ Stz_Ptr p; char c1, c2;

  c1=Chr_ToUpper(c);
  for (p=Stz_End(s); ((c2=Chr_ToUpper(*p)) != c1) && (p>s); --p);
  return (c1==c2) ? p : NULL;
}

Stz_Ptr Stz_Del(Stz_Ptr s, Stz_Ind i, Stz_Ind n)
{ Stz_Ind l;

  if (n>0) {
    l=Stz_Len(s);
    if (i < l) {
      if ((i+n) >= l) s[i]= '\0'; else Stz_Cpy(s+i,s+i+n);
    }
  }
  return s;
}

Stz_Ptr Stz_Detab(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n)
{ Stz_Ind k;              /* current display column */
  Stz_Ptr p1;             /* src pointer            */
  Stz_Ptr p2;             /* dst pointer            */
if (n == 0) return Stz_Cpy(dst,src);
  else {
    p1= src; p2= dst; k= 0;
    while (*p1) {
      switch (*p1) {
        case HT : do {*(p2++)= ' '; ++k; } while (k % n); break;
        case CR : *(p2++)= CR; k= 0; break;
        case BS : *(p2++)= BS; if (k) --k; break;
        default : *p2++ = *p1; if (!iscntrl(*p1)) ++k; break;
      }
      ++p1;
    }
    *p2= '\0'; return dst;
  }
}

Stz_Ptr Stz_ECat(Stz_Ptr dst, const Stz_Ptr src)
{
  return stpcpy(dst+strlen(dst),src);
}

int Stz_Empty(const Stz_Ptr s)
{
  return *s == '\0';
}

Stz_Ptr Stz_End(const Stz_Ptr s)
{
  return s+strlen(s);
}

Stz_Ptr Stz_Entab(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n)
{
  Stz_Ptr tmp;
  int     k0;              /* starting column of last spacegroup */
  int     k1;              /* current display column             */
  char    c;               /* current character                  */
  Stz_Ptr p1;              /* tmp pointer                        */
  Stz_Ptr p2;              /* dst pointer                        */

  if ((n==0) || (src[0]=='\0')) return Stz_Cpy(dst,src);
  else {
    /* pass 1, src -> tmp : replace tabs by space groups */
    Stz_Detab(dst,src,n);
    if ((tmp= strdup(dst)) == NULL) return Stz_Cpy(dst,src);
    else {
    /* pass 2, tmp -> dst : replace space groups by tabs */
      k0= k1= 0; p1= tmp; p2= dst;
      c= *p1;
      while (c) {
        switch (c) {
          case ' ': ++k1;
                    if ((k1 % n) == 0) {
                      if (k1 - k0 > 1) c= HT;
                      *(p2++)= c; k0= k1;
                    }
                    break;
          case CR: *(p2++)= CR;
                   k0= k1= 0;
                   break;
          case BS: *(p2++)= BS;
                   if (k1 > 0) --k1;
                   if (k0 > k1) k0= k1;
                   break;
          default: while (k0 < k1) { *(p2++)= ' '; ++k0; }
                   *(p2++)= c;
                   if (!iscntrl(c)) ++k0;
                   k1= k0;
                   break;
        }
        c= *(++p1);
      }
      while (k0 < k1) { *(p2++)= ' '; ++k0; }
      *p2= '\0';
      free(tmp); return dst;
    }
  }
}

Stz_Ptr Stz_Fill(Stz_Ptr s, char c, Stz_Ind n)
{ Stz_Ind i= Stz_Len(s);

  if (n>i) { memset(s+i,c,n-i); s[n]= '\0'; }
  return s;
}

void Stz_Free(Stz_Ptr *p)
{ free(*p); *p= NULL; }

Stz_Ptr Stz_Gtw(Stz_Ptr dst, Stz_Ptr src)
{ Stz_Ptr p1= src, p2= dst;

  while (isspace(*p1)) ++p1;
  while ((*p1) && (!isspace(*p1))) *p2++= *p1++;
  *p2= '\0';
  Stz_Cpy(src,p1);
  return dst;
}

Stz_Ptr Stz_Ins(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i)
{ Stz_Ind l; Stz_Ptr p1, p2;

  if (i >= Stz_Len(dst)) return Stz_Cat(dst,src);
  else {
    l= Stz_Len(src); p1= dst+i; p2= p1+l;
    memmove(p2,p1,Stz_Len(p1)+1);
    memmove(p1,src,l);
    return dst;
  }
}

Stz_Ind Stz_IPos(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ptr p= Stz_Pos(src,pat);

  return (p==NULL) ? Stz_NOTFOUND : p-src;
}

int Stz_LexCmp(const Stz_Ptr s1, const Stz_Ptr s2)
{ Stz_Ptr p1, p2;

  p1= s1; p2= s2;
  while ((*p1) && (toupper(*p1) == toupper(*p2))) {
    ++p1; ++p2;
  }
  return Chr_LexOrder(*p1,*p2);
}

Stz_Ptr Stz_LexSrt(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n)
{ Stz_Ind i, k, l=Stz_Len(src); Stz_Ptr tmp;

  if ((n == 0) || (n >= l)) return Stz_Cpy(dst,src); else {
    tmp = strdup(src); *dst= '\0';
    while (*tmp) {
      k=0; i=n;
      while (i <= l) {
        if (Stz_LexNCmp(tmp+i, tmp+k, n) < 0) k= i; i += n;
      }
      Stz_NCat(dst, tmp+k, n); Stz_Del(tmp, k, n); l -= n;
    }
    free(tmp); return dst;
  }
}

int Stz_LexNCmp(const Stz_Ptr s1, const Stz_Ptr s2, Stz_Ind n)
{ Stz_Ind i; Stz_Ptr p1, p2;

  if (n==0) return 0;
  else {
    p1= s1; p2= s2; i= 1;
    while ((i < n) && (*p1) && (toupper(*p1) == toupper(*p2))) {
      ++i; ++p1; ++p2;
    }
    return Chr_LexOrder(*p1,*p2);
  }
}

Stz_Ptr Stz_NCpy(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n)
{
  *dst= '\0';
  return strncat(dst,src,n);
}

Stz_Ptr Stz_Make(Stz_Ptr s, char c, Stz_Ind n)
{ *s= '\0'; return Stz_Fill(s,c,n); }

Stz_Ptr Stz_NIns(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i, Stz_Ind n)
{ Stz_Ind l; Stz_Ptr p1, p2;

  l= Stz_Len(dst);
  if (i >= l) return Stz_NCat(dst,src,n);
  else {
    l= Stz_Len(src); if (l>n) l= n;
    p1= dst+i; p2= p1+l;
    memmove(p2,p1,Stz_Len(p1)+1);
    memmove(p1,src,l);
    return dst;
  }
}

Stz_Ptr Stz_NRepl(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i, Stz_Ind n)
{ Stz_Ind l= Stz_Len(src);

  if (n>l) n= l;
  if (i+n > 1)
    Stz_Fill(dst,' ',i+n-1);
  return Stz_NIns(Stz_Del(dst,i,n),src,i,n);
}

Stz_Ptr Stz_Pos(const Stz_Ptr src, const Stz_Ptr pat)
{ return (*pat == '\0') ? NULL : strstr(src,pat); }

Stz_Ptr Stz_RAS(Stz_Ptr s)
{ Stz_Ptr p1=s, p2=s;

  while (*p2) {
    if (!isspace(*p2)) *p1++= *p2;
    ++p2;
  }
  *p1= '\0';
  return s;
}

Stz_Ptr Stz_ReAlloc(Stz_Ptr *p, const Stz_Ptr s)
{ free(*p); return *p= Stz_Alloc(s); }

Stz_Ptr Stz_Repl(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i)
{ return Stz_NRepl(dst,src,i,Stz_Len(src)); }

Stz_Ptr Stz_Right(const Stz_Ptr s, Stz_Ind n)
{ Stz_Ind l;

  if (n==0) return Stz_End(s);
  else {
   l= Stz_Len(s);
   if (n >= l) return s ;else return s + (l-n);
  }
}

Stz_Ind Stz_RIPos(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ptr p= Stz_RPos(src,pat);

  return (p==NULL) ? Stz_NOTFOUND : p-src;
}

Stz_Ptr Stz_RLS(Stz_Ptr s)
{ Stz_Ptr p= s;

  while (isspace(*p)) ++p;
  return Stz_Cpy(s,p);
}

Stz_Ptr Stz_RPos(const Stz_Ptr src, const Stz_Ptr pat)
{ unsigned int n1,n2,cmp; Stz_Ptr p;

  if (((n1=Stz_Len(pat))==0) || ((n2=Stz_Len(src))==0) || (n2<n1))
    return NULL;
  else {
    n2 -= n1;
    for (p= src+n2; ((cmp=memcmp(p,pat,n1)) != 0) && (p > src); --p);
    return cmp ? NULL : p;
  }
}

Stz_Ptr Stz_RTS(Stz_Ptr s)
{ Stz_Ind i= Stz_Len(s);

  while (i && isspace(s[i-1])) --i;
  s[i]= '\0';
  return s;
}

Stz_Ptr Stz_Srt(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n)
{ Stz_Ind i, k, l=Stz_Len(src); Stz_Ptr tmp;

  if ((n == 0) || (n >= l)) return Stz_Cpy(dst,src);
  else {
    tmp = strdup(src); *dst= '\0';
    while (*tmp) {
      k=0; i=n;
      while (i <= l) {
        if (Stz_NCmp(tmp+i, tmp+k, n) < 0) k= i; i += n;
      }
      Stz_NCat(dst, tmp+k, n); Stz_Del(tmp, k, n); l -= n;
    }
    free(tmp); return dst;
  }
}

int Stz_SoundAlike(const Stz_Ptr s1, const Stz_Ptr s2, Stz_Ind i)
{ Stz_Ptr d1, d2; int result;

  d1= Stz_Alloc(Stz_Len(s1));
  d2= Stz_Alloc(Stz_Len(s2));
  result= Stz_Cmp(Stz_SoundEx(d1,s1,i),Stz_SoundEx(d2,s2,i));
  Stz_Free(&d1); Stz_Free(&d2);
  return result==0;
}

Stz_Ptr Stz_SoundEx(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i)
{ Stz_Ind i0, i1; char c, c0;

  i= min(i,Stz_Len(src));
  for (i0=0;i0<i;i0++) dst[i0]= Chr_ToUpper(src[i0]);
  i0= i; i1= i; c0=' ';
  while (src[i0] != '\0') {
    c= _SoundChr[src[i0++]];
    if ((c!=' ') && (c!=c0)) dst[i1++]= c0= c;
  }
  dst[i1]= '\0';
  return dst;
}

Stz_Ptr Stz_StpCpy(Stz_Ptr dst, const char src[])
{ char *p1= (char *)src; Stz_Ptr p2= dst; unsigned char n= src[0];

  while (n--) *p2++ = *(++p1);
  *p2= '\0';
  return dst;
}

Stz_Ptr Stz_Sub(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind i, Stz_Ind n)
{
  if (i >= Stz_Len(src)) { *dst= '\0'; return dst; }
  else return Stz_NCpy(dst,src+i,n);
}

Stz_Ptr Stz_Trunc(Stz_Ptr s, Stz_Ind i)
{
  if (i < Stz_Len(s)) s[i]= '\0';
  return s;
}

Stz_Ind Stz_UppIPos(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ptr s, p; Stz_Ind i= Stz_NOTFOUND;

  if (((s= Stz_Alloc(src)) != NULL) && ((p= Stz_Alloc(pat)) != NULL))
    i= Stz_IPos(Stz_Upp(s),Stz_Upp(p));
  if (s) Stz_Free(&s); if (p) Stz_Free(&p);
  return i;
}

Stz_Ptr Stz_UppPos(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ind i;

  return (i=Stz_UppIPos(src,pat)) == Stz_NOTFOUND ? NULL : src+i;
}

Stz_Ind Stz_UppRIPos(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ptr s, p; Stz_Ind i = Stz_NOTFOUND;

  if (((s= Stz_Alloc(src)) != NULL) && ((p= Stz_Alloc(pat)) != NULL))
    i= Stz_RIPos(Stz_Upp(s),Stz_Upp(p));
  if (s) Stz_Free(&s); if (p) Stz_Free(&p);
  return i;
}

Stz_Ptr Stz_UppRPos(const Stz_Ptr src, const Stz_Ptr pat)
{ Stz_Ind i;

  return (i=Stz_UppRIPos(src,pat)) == Stz_NOTFOUND ? NULL : src+i;
}

Stz_Ptr Stz_UppSrt(Stz_Ptr dst, const Stz_Ptr src, Stz_Ind n)
{ Stz_Ind i, k, l=Stz_Len(src); Stz_Ptr tmp;

  if ((n == 0) || (n >= l)) return Stz_Cpy(dst,src); else {
    tmp = strdup(src); *dst= '\0';
    while (*tmp) {
      k=0; i=n;
      while (i <= l) {
        if (Stz_UppNCmp(tmp+i, tmp+k, n) < 0) k= i; i += n;
      }
      Stz_NCat(dst, tmp+k, n); Stz_Del(tmp, k, n); l -= n;
    }
    free(tmp); return dst;
  }
}
