/* SRTTST.C : Test SRTLIB

   Title   : SRTTST
   Version : 2.1
   Date    : Nov 27,1996
   Author  : J R Ferguson
   Language: Turbo C 2.0
   Usage   : Refer function Help

[ Project file:
srttst   (srtlib.c)
srtlib   (srtlib.h)
tcfd.lib
]
*/

#define  DEBUG

#include <stdio.h>
#include "deflib.h"
#include "srtlib.h"
#include "stzlib.h"
#include "chrlib.h"

#define FALSE      0
#define TRUE       1
#define MAXSTZIND  255    /* Max line length for Inp/Out processing */
#define MAXLINE    1000   /* Max nr of text lines in input          */
#define MAXFNM     79     /* Max filespec length (IDE)              */
#define INPBUFSIZ  4096   /* Input buffer size in bytes             */
#define OUTBUFSIZ  4096   /* Output buffer size in bytes            */

/* Error codes and messages: */
#define ERROK      0
#define ERRARG     1
#define ERRFNF     2
#define ERRCRE     3
#define ERRREA     4
#define ERRWRI     5
#define ERRMAX     6
#define ERRMEM     7

char *ErrMsg[ERRMEM+1] =
 {"",
  "Command syntax error",
  "File not found",
  "File creation error",
  "Read error",
  "Write error",
  "Too many lines",
  "Out of memory"
 };

typedef Stz_Ptr       PtrBufTyp[MAXLINE]; /* we will sort pointers */
typedef unsigned int  PtrBufInd;

/* Global variables */

int        ErrCod;
char       InpFnm[MAXFNM], OutFnm[MAXFNM];
FILE       *Msg, *Inp, *Out;
int        InpOpn, OutOpn;
int        OptHlp;
Stz_Ind    OptKeyBeg;
Stz_Ind    OptKeyLen;
PtrBufTyp  PtrBuf;
PtrBufInd  PtrCnt;
enum       {NoSort,BinInsSort,SelectSort,QuickSort} OptSrt;


/* --- General routines --- */


#define wr(s) fprintf(Msg,s)
#define wi(i) fprintf(Msg,"%i",i)
#define wl(s) fprintf(Msg,"%s\n",s)

void Help(void)
{
wl("SRTTST v2.0 (Test op Srt_Lib) : text sort filter");
wl("Usage  : SRTTST [<in] [>out] [/option[...] [...]]");
wl("Options: I           Binary insertion sort");
wl("         S           Selection sort");
wl("         Q           Quicksort");
wr("         K[c1][,c2]  Define key first col c1 [1], last col c2 [");
  wi(MAXSTZIND); wl("].");
wr("         K[c1][,+c2] Define key first col c1 [1], nr of cols c2 [");
  wi(MAXSTZIND); wl("].");
wl("         H           Send this help text to (redirected) output.");
wl("");
wr("Remarks: - If no option is provided, input is copied to output ");
  wl("without sorting.");
wr("         - Max number of input lines is "); wi(MAXLINE); wl(".");
wr("         - Max number of characters per line is "); wi(MAXSTZIND);
  wl(".");
}


/* --- Command line parsing routines --- */


int ReadUns(Stz_Ptr arg)
{ char c; int n= 0;

  while ((ErrCod == ERROK) && Chr_IsDigit(c= Stz_cRet(arg,0))) {
    Stz_Del(arg,0,1);
    n= 10 * n + (c - '0');
  }
  return n;
}


void ReadKeyDesc(Stz_Ptr arg)
{ int n;

  if (Chr_IsDigit(Stz_cRet(arg,0))) OptKeyBeg= ReadUns(arg) % MAXSTZIND;
  if (Stz_cRet(arg,0) == ',') {
    Stz_Del(arg,0,1);
    if (Stz_cRet(arg,0) == '+') {
      Stz_Del(arg,0,1);
      if (Chr_IsDigit(Stz_cRet(arg,0))) OptKeyLen= ReadUns(arg) % MAXSTZIND;
      else ErrCod= ERRARG;
    }
    else {
      if (Chr_IsDigit(Stz_cRet(arg,0))) {
        n= (ReadUns(arg) % MAXSTZIND) - OptKeyBeg + 1;
        if (n < 0) ErrCod= ERRARG; else OptKeyLen= n;
      }
      else ErrCod= ERRARG;
    }
  }
}


void ReadOpt(Stz_Ptr arg)
{ int nextopt;

  Stz_Del(arg,0,1);
  do {
    if (Stz_Empty(arg) || (Stz_cRet(arg,0) == '/')) ErrCod= ERRARG;
    else {
      nextopt= FALSE;
      while ((ErrCod==ERROK) && !nextopt && !Stz_Empty(arg)) {
        switch (Stz_cGet(arg)) {
          case 'I': if ((OptSrt == NoSort) || (OptSrt == BinInsSort))
                      OptSrt= BinInsSort; else ErrCod= ERRARG; break;
          case 'S': if ((OptSrt == NoSort) || (OptSrt == SelectSort))
                      OptSrt= SelectSort; else ErrCod= ERRARG; break;
          case 'Q': if ((OptSrt == NoSort) || (OptSrt == QuickSort))
                      OptSrt= QuickSort ; else ErrCod= ERRARG; break;
          case 'K': ReadKeyDesc(arg); break;
          case 'H': OptHlp= TRUE;  break;
          case '/': nextopt= TRUE; break;
          default : ErrCod= ERRARG;
    } } }
  } while ((ErrCod == ERROK) && nextopt);
}


void ReadArgs(int argc, char *argv[])
{ int i= 0;
  char arg[MAXCMD];

  while ((ErrCod == ERROK) && (++i < argc)) {
    Stz_Cpy(arg,argv[i]); Stz_Upp(arg);
    switch (Stz_cRet(arg,0)) {
      case '/': ReadOpt(arg); break;
      case '<': Stz_Sub(InpFnm,arg,1,MAXFNM); break; /* simulate for IDE */
      case '>': Stz_Sub(OutFnm,arg,1,MAXFNM); break; /* simulate for IDE */
      default : ErrCod= ERRARG;
    }
  }
  if (OptHlp) if (ErrCod == ERROK) ErrCod= ERRARG; else OptHlp= FALSE;
}


/* --- Low-level I/O routines --- */


void OpenMsg(void)
{
  if (OptHlp) {
    if (Stz_Empty(OutFnm)) Msg= stdout;
    else if ((Msg= fopen(OutFnm,"wt")) == NULL) Msg= stderr;
  }
  else Msg= stderr;
}


void CloseMsg(void)
{ if ((Msg != stderr) && (Msg != stdout)) fclose(Msg); }


void OpenInp(void)
{
  if (Stz_Empty(InpFnm)) { Inp= stdin; InpOpn= TRUE; }
  else {
    if ((Inp= fopen(InpFnm,"rt")) == NULL) ErrCod= ERRFNF;
    else { setvbuf(Inp,NULL,_IOFBF,INPBUFSIZ); InpOpn= TRUE; }
} }


void CloseInp(void)
{
  if (Inp == stdin) InpOpn= FALSE;
  else { if (fclose(Inp) == 0) InpOpn= FALSE; }
}


int ReadInp(char line[])
{
  if (fgets(line,MAXSTZIND,Inp) == NULL) {
    if (!feof(Inp)) ErrCod= ERRREA;
    return 0;
  }
  else return 1;
}


void OpenOut(void)
{
  if (Stz_Empty(OutFnm)) { Out= stdout; OutOpn= TRUE; }
  else {
    if ((Out= fopen(OutFnm,"wt")) == NULL) ErrCod= ERRCRE;
    else { setvbuf(Out,NULL,_IOFBF,OUTBUFSIZ); OutOpn= TRUE; }
} }


void CloseOut(void)
{
  if (Out == stdout) OutOpn= FALSE;
  else { if (fclose(Out) == 0) OutOpn= FALSE; else ErrCod= ERRWRI; }
}


void WriteOut(const Stz_Ptr line)
{
  if (fputs(line,Out) == EOF) ErrCod= ERRWRI;
}



/* --- High level I/O and sorting --- */



void ReadBuffer(void)
{ char line[MAXSTZIND];

  PtrCnt= 0;
  while ((ErrCod==ERROK) && ReadInp(line)) {
    if (PtrCnt >= MAXLINE-1) ErrCod= ERRMAX;
    else {
      if ((PtrBuf[PtrCnt++]= Stz_Alloc(line)) == NULL) ErrCod= ERRMEM;
    }
  }
}


void WriteBuffer(void)
{ PtrBufInd i= 0;

  while ((ErrCod==ERROK) && (i < PtrCnt)) {
    WriteOut(PtrBuf[i]);
    Stz_Free(&PtrBuf[i]);
    ++i;
  }
}


int SortOrder(const Srt_ElmPtr e1, const Srt_ElmPtr e2)
#ifdef DEBUG
{ char s1[MAXSTZIND], s2[MAXSTZIND]; int result;

  Stz_Sub(s1, *(Stz_Ptr *)e1, OptKeyBeg-1, OptKeyLen);
  Stz_Sub(s2, *(Stz_Ptr *)e2, OptKeyBeg-1, OptKeyLen);
  result= Stz_Cmp(s1,s2);
  return result;
}
#else
{
  return Stz_Cmp(Stz_Sub(*(Stz_Ptr *)e1,OptKeyBeg-1,OptKeyLen),
                 Stz_Sub(*(Stz_Ptr *)e2,OptKeyBeg-1,OptKeyLen));
}
#endif


void SortBuffer(void)
{
  switch (OptSrt) {
    case NoSort    : /*do nothing*/
                     break;
    case BinInsSort: Srt_BinIns(PtrBuf,PtrCnt,sizeof(Stz_Ptr),SortOrder);
                     break;
    case SelectSort: Srt_Select(PtrBuf,PtrCnt,sizeof(Stz_Ptr),SortOrder);
                     break;
    case QuickSort : Srt_Quick (PtrBuf,PtrCnt,sizeof(Stz_Ptr),SortOrder);
                     break;
  }
}



/* --- Main line --- */


void MainProcess(void)
{
  ReadBuffer();
  if (ErrCod==ERROK) {
    SortBuffer();
    if (ErrCod==ERROK) WriteBuffer();
  }
}


void MainInit(int argc, char *argv[])
{
  ErrCod= ERROK;
  Stz_Create(InpFnm); InpOpn= FALSE;
  Stz_Create(OutFnm); OutOpn= FALSE;
  OptHlp= FALSE;
  OptSrt= NoSort;
  OptKeyBeg= 1;
  OptKeyLen= MAXSTZIND;
  ReadArgs(argc,argv);
  if (ErrCod == ERROK) OpenInp();
  if (ErrCod == ERROK) OpenOut();
}


void MainTerm(void)
{
  if (InpOpn) CloseInp();
  if (OutOpn) CloseOut();
  if (ErrCod != ERROK) {
    OpenMsg();
    if ((ErrCod==ERRARG)) Help();
    else fprintf(Msg,"SRTTST: %s\n",ErrMsg[ErrCod]);
    CloseMsg();
  }
}


int main(int argc, char *argv[])
{
  MainInit(argc,argv);
  if (ErrCod == ERROK) MainProcess();
  MainTerm();
  return ErrCod;
}
