/* SETLIB.H - Character set manipulation functions

  Title    : SETLIB
  Version  : 1.1
  Date     : Nov 27,1996
  Author   : J R Ferguson
  Language : Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage    : Header file

Supports Pascal-like set type of ennumerated data types, such as
unsigned int and char, having values in the range of 0 through 255.

   Internal representation of Set_Typ:          Formula's:

     +------+------+- // -+------+              word = elm    / 16
     |   0  |   1  |      |  15  |              bit  = elm    % 16
     +------+------+- // -+------+              mask = 0x8000 >> bit
      0   15 16  31 32        255
*/


/*.hlptx skip 1*/
#ifndef Set_Lib
#define Set_Lib

#define  Set_MIN   0
#define  Set_MAX   255
#define  Set_SIZE  16

typedef unsigned char Set_Elm;
typedef unsigned int  Set_Typ [Set_SIZE];
typedef unsigned int *Set_Ptr;

/*.hlptx skip 4*/
#ifdef __cplusplus
extern "C" {
#endif

unsigned Set_Card(const Set_Ptr set);
/* 
  Returns the cardinality of the given set (the number of elements in it).
*/

Set_Ptr Set_Clear(Set_Ptr set);
/* 
  Initializes set, making it empty 
*/

int Set_Contains(const Set_Ptr superset, const Set_Ptr subset);
/* 
  Returns 1 if superset contains all elements in subset, 0 if it does not.
*/

Set_Ptr Set_Cpy(Set_Ptr dst, const Set_Ptr src);
/* 
  Copies the contents of set src to set dst, returns dst. 
  src and dst may be the same sets.  
*/

Set_Ptr Set_DeleteVal(Set_Ptr set, const Set_Elm elm);
/* 
  Deletes value val from an existing set, returning the resulting set. 
*/

Set_Ptr Set_DeleteRng(Set_Ptr set, const Set_Elm low, const Set_Elm high);
/* 
  Deletes the range of values from low to high inclusive from an existing
  set, returning the resulting set.
*/

Set_Ptr Set_Difference(      Set_Ptr result,
                       const Set_Ptr set1,
                       const Set_Ptr set2);
/* 
  Sets result to the difference of sets set1 and set2, consisting of the
  elements in set1 not contained in set2. Returns a pointer to the result.
  The result set may be the same as set1 or set2.
*/

int Set_Empty(const Set_Ptr set);
/* 
  Returns 1 if the given set is empty, 0 if it contains one or more
  elements. 
*/

int Set_Eq(const Set_Ptr set1, const Set_Ptr set2);
/* 
  Returns 1 if set1 and set2 are equal (contain exactly the same 
  elements), else 0. 
*/

int Set_In(const Set_Ptr set, const Set_Elm elm);
/* 
 Returns 1 if elm is an element of the given set, 0 if it is not.
*/

Set_Ptr Set_InsertVal(Set_Ptr set, const Set_Elm elm);
/* 
  Inserts value val into an existing set, returning the resulting set.
*/

Set_Ptr Set_InsertRng(Set_Ptr set, const Set_Elm low, const Set_Elm high);
/* 
  Inserts the range of values from low to high inclusive into an existing
  set, returning the resulting set. 
*/

Set_Ptr Set_Intersection(      Set_Ptr result,
                         const Set_Ptr set1,
                         const Set_Ptr set2);
/* 
  Sets result to the intersection of sets set1 and set2, containing the
  elements that set1 and in set2 have in common.
  Returns a pointer to result.
  The result set may be the same as set1 or set2. 
*/

Set_Ptr Set_Inverse(Set_Ptr dst, const Set_Ptr src);
/* 
  The dst set will be made to be the inverse of the src set, having all
  elements in the range 0..Set_Max that src did not have, and not having
  any of the elements of src. 
  Returns dst. 
  The src and dst sets may be the same. 
*/

Set_Ptr Set_Union(      Set_Ptr result,
                  const Set_Ptr set1,
                  const Set_Ptr set2);
/* 
  Sets result to the union of sets set1 and set2, containing all elements
  that are part of either set1 or set2.
  Returns a pointer to result.
  The result set may be the same as set1 or set2. 
*/

/*.hlptx skip 4*/
#ifdef __cplusplus
}
#endif
#endif /* ifndef Set_Lib */
