/* SETLIB.C - Character set manipulation functions

  Title    : SETLIB
  Version  : 1.1
  Date     : Nov 27,1996
  Author   : J R Ferguson
  Language : Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage    : Function library
*/

#include <stdlib.h>
#include <mem.h>
#include "setlib.h"


/* === local definitions === */


static Set_Ptr _Set_Or(Set_Ptr set1, const Set_Ptr set2)
/* return set1 = set1 or set2 */
{ int i;

  for (i=0; i<Set_SIZE; ++i) set1[i] |= set2[i];
  return set1;
}


static Set_Ptr _Set_And(Set_Ptr set1, const Set_Ptr set2)
/* return set1 = set1 and set2 */
{ int i;

  for (i=0; i<Set_SIZE; ++i) set1[i] &= set2[i];
  return set1;
}


static Set_Ptr _Set_Not(Set_Ptr set)
/* return set = not set */
{ int i;

  for (i=0; i<Set_SIZE; ++i) set[i] = ~set[i];
  return set;
}


static int _Set_Tst(const Set_Ptr set1, const Set_Ptr set2)
/* return 1 if set1 contains all elements of set2, else 0 */
{ int i, r;

  for (i=0, r=0; (r==0) && (i<Set_SIZE); ++i) r= ~set1[i] & set2[i];
  return r==0;
}


static Set_Ptr _Set_Rng(Set_Ptr result,const Set_Elm low,const Set_Elm high)
/* creates a result set with elements ranging low..high and returns a 
   pointer to it. */
{ int elm;

  Set_Clear(result);
  for (elm= max(low,Set_MIN); elm <= min(high,Set_MAX); ++elm)
    result[elm / 16] |= (0x8000 >> elm % 16);
  return result;
}


/* === global definitions === */


unsigned Set_Card(const Set_Ptr set)
{ int i; unsigned n, mask;

  for (n=0,i=0; i<Set_SIZE; ++i) {
    for (mask=0x8000; mask; mask >>= 1)
      if (set[i] &  mask) ++n;
  }
  return n;
}

Set_Ptr Set_Clear(Set_Ptr set)
{ return (Set_Ptr) memset(set,0,sizeof(Set_Typ)); }

int Set_Contains(const Set_Ptr superset, const Set_Ptr subset)
{ return _Set_Tst(superset,subset); }

Set_Ptr Set_Cpy(Set_Ptr dst, const Set_Ptr src)
{ return memmove(dst,src,sizeof(Set_Typ)); }

Set_Ptr Set_DeleteVal(Set_Ptr set, const Set_Elm elm)
{ return Set_DeleteRng(set,elm,elm); }

Set_Ptr Set_DeleteRng(Set_Ptr set, const Set_Elm low, const Set_Elm high)
{ Set_Typ tmp;

  return _Set_And(set, _Set_Not(_Set_Rng(tmp,low,high)));
}

Set_Ptr Set_Difference(Set_Ptr result,const Set_Ptr set1,const Set_Ptr set2)
{ Set_Typ tmp;

  return Set_Cpy(result,_Set_And(_Set_Not(Set_Cpy(tmp,set2)),set1));
}

int Set_Empty(const Set_Ptr set)
{ int i, ok;

  for (ok=1,i=0; ok && (i<Set_SIZE); ++i) {
    ok= (set[i] == 0);  
  }
  return ok; 
}

int Set_Eq(const Set_Ptr set1, const Set_Ptr set2)
{ return memcmp(set1,set2,sizeof(Set_Typ)) == 0; }

int Set_In(const Set_Ptr set, const Set_Elm elm)
{ Set_Typ tmp;

  return _Set_Tst(set, _Set_Rng(tmp,elm,elm));
}

Set_Ptr Set_InsertVal(Set_Ptr set, const Set_Elm elm)
{ return Set_InsertRng(set,elm,elm); }

Set_Ptr Set_InsertRng(Set_Ptr set, const Set_Elm low, const Set_Elm high)
{ Set_Typ tmp;

  return _Set_Or(set, _Set_Rng(tmp,low,high));
}

Set_Ptr Set_Intersection(      Set_Ptr result,
                         const Set_Ptr set1,
                         const Set_Ptr set2)
{ Set_Typ tmp;

  return Set_Cpy(result,_Set_And(Set_Cpy(tmp,set1), set2));
}

Set_Ptr Set_Inverse(Set_Ptr dst, const Set_Ptr src)
{ return _Set_Not(Set_Cpy(dst,src)); }

Set_Ptr Set_Union(      Set_Ptr result,
                  const Set_Ptr set1,
                  const Set_Ptr set2)
{ Set_Typ tmp;

  return Set_Cpy(result,_Set_Or(Set_Cpy(tmp,set1), set2));
}
