/* RRGLIB.C : Real range handling routines

  Title   : RRGLIB
  Version : 4.0
  Date    : Nov 26,1996
  Author  : J R Ferguson
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage   : Function library
*/

#include <alloc.h>
#include <values.h>
#include "rrglib.h"


/* --- local definitions --- */

static void order(float *lb, float *hb, int *li, int *hi)
{
  if (*lb > *hb) {
    float tb; int ti;
    tb= *lb; *lb= *hb; *hb= tb;
    ti= *li; *li= *hi; *hi= ti;
  }
}


static int below(const Rrg_Typ r, float i)
{
  return (r==NULL) ? 0 : (r->li) ? (i < r->lb) : (i <= r->lb);
}


static int bndbelow(const Rrg_Typ r, float hb, int hi)
{
  return (r==NULL) ? 0 : (hi && r->li) ? (hb < r->lb) : (hb <= r->lb);
}


static int above(const Rrg_Typ r, float i)
{
  return (r==NULL) ? 0 : r->hi ? (i > r->hb) : (i >= r->hb);
}


static int bndabove(const Rrg_Typ r, float lb, int li)
{
  return (r==NULL) ? 0 : (li && r->hi) ? (lb > r->hb) : (lb >= r->hb);
}


static int lbndinside(const Rrg_Typ r, float lb, int li)
{
  return (li && !r->li) ? (lb > r->lb) : (lb >= r->lb);
}

static int hbndinside(const Rrg_Typ r, float hb, int hi)
{
  return (hi && !r->hi) ? (hb < r->hb) : (hb <= r->hb);
}


static int inside(const Rrg_Typ r, float i)
{
  return (r==NULL) ? 0 : !( below(r,i) || above(r,i) );
}


static int bndinside(const Rrg_Typ r, float lb, float hb, int li, int hi)
{
  return (r==NULL) ? 0
		   : (lbndinside(r,lb,li) && hbndinside(r,hb,hi));
}


static int connected(const Rrg_Typ r1,const Rrg_Typ r2)
{
  return (r2==NULL)         ? 0 :
         (r1->hi || r2->li) ? (r1->hb >= r2->lb) : (r1->hb > r2->lb);
}


static int empty(const Rrg_Typ r)
{
  return (r->li && r->hi) ? (r->lb > r->hb) : (r->lb >= r->hb);
}


static Rrg_Typ *cleanup(Rrg_Typ *r)
/* clear empty subranges, merge connected subranges,
   return pointer to result */
{
  if (*r != NULL) {
    Rrg_Typ r0;

    if (empty(*r)) {
      r0= *r; *r= (*r)->nxt; free(r0);
      cleanup(r);
    }
    else if (connected(*r, (*r)->nxt)) {
      r0= (*r)->nxt;
      (*r)->hb= r0->hb; (*r)->hi= r0->hi; (*r)->nxt= r0->nxt;
      free(r0);
      cleanup(r);
    }
  else cleanup(&((*r)->nxt));
  }
  return r;
}


static void getminimum(float b1, int i1,
                       float b2, int i2,
                       float *b, int *i)
{
  if (      b1  < b2 ) { *b= b1; *i= i1;       }
  else if ( b1  > b2 ) { *b= b2; *i= i2;       }
  else    /*b1 == b2*/ { *b= b1; *i= i1 || i2; }
}


static void getmaximum(float b1, int i1,
                       float b2, int i2,
                       float *b, int *i)
{
  if (      b1  > b2 ) { *b= b1; *i= i1;       }
  else if ( b1  < b2 ) { *b= b2; *i= i2;       }
  else    /*b1 == b2*/ { *b= b1; *i= i1 || i2; }
}


/* --- global routines --- */


Rrg_Typ *Rrg_Create(Rrg_Typ *r)
{ *r=NULL; return r; }


void Rrg_Dispose(Rrg_Typ *r)
{ if (*r != NULL) { Rrg_Dispose(&((*r)->nxt)); free(*r); } }


Rrg_Typ *Rrg_Clear(Rrg_Typ *r)
{ Rrg_Dispose(r); return Rrg_Create(r); }


int Rrg_Empty(const Rrg_Typ *r)
{ return *r==NULL; }


int Rrg_Inside(const Rrg_Typ *r, float i)
{
  return (*r==NULL)    ? 0 :
         (above(*r,i)) ? Rrg_Inside(&((*r)->nxt),i) : inside(*r,i);
}

Rrg_Typ *Rrg_Insert(Rrg_Typ *r,     /* empty or existing range */
                    float    lb,    /* low boundary value */
                    int      li,    /* low boundary included */
                    float    hb,    /* high boundary value */
                    int      hi)    /* high boundary included */
{ Rrg_Typ r0;

  order(&lb, &hb, &li, &hi);
  if (above(*r,lb)) Rrg_Insert(&((*r)->nxt),lb,li,hb,hi);
  else if ((*r==NULL) || below(*r,hb)) {
    if ((r0=(Rrg_Typ)malloc(sizeof(_Rrg_Rec))) != NULL) {
      r0->lb= lb; r0->li= li;
      r0->hb= hb; r0->hi= hi;
      r0->nxt= *r;
      *r= r0;
    }
  }
  else if (!bndinside(*r,lb,hb,li,hi)) {
    getminimum(lb, li, (*r)->lb, (*r)->li, &((*r)->lb), &((*r)->li));
    getmaximum(hb, hi, (*r)->hb, (*r)->hi, &((*r)->hb), &((*r)->hi));
  }
  return cleanup(r);
}


Rrg_Typ *Rrg_Delete(Rrg_Typ *r,     /* empty or existing range */
                    float    lb,    /* low boundary value */
                    int      li,    /* low boundary included */
                    float    hb,    /* high boundary value */
                    int      hi)    /* high boundary included */
{ Rrg_Typ r0;

  if (*r != NULL) {
    order(&lb, &hb, &li, &hi);
    if (bndabove(*r,lb,li)) Rrg_Delete(&((*r)->nxt),lb,li,hb,hi);
    else if (!bndbelow(*r,hb,hi)) {
      if (lbndinside(*r,lb,li)) {
        if (hbndinside(*r,hb,hi)) {
  	if ((r0=(Rrg_Typ)malloc(sizeof(_Rrg_Rec))) != NULL) {
            r0->lb = hb       ; r0->li= !hi;
            r0->hb = (*r)->hb ; r0->hi= (*r)->hi;
            r0->nxt= (*r)->nxt;
            (*r)->hb = lb     ; (*r)->hi = !li;
            (*r)->nxt= r0;
          }
        }
        else {
          (*r)->hb= lb; (*r)->hi= !li;
          Rrg_Delete(&((*r)->nxt),lb,li,hb,hi);
        }
      }
      else {
        if (hbndinside(*r,hb,hi)) {
          (*r)->lb= hb; (*r)->li= !hi;
        }
        else {
          r0= *r; *r= (*r)->nxt; free(r0);
          Rrg_Delete(r,lb,li,hb,hi);
        }
      }
    }
  }
  return cleanup(r);
}


Rrg_Typ *Rrg_Merge(Rrg_Typ *r1, const Rrg_Typ *r2)
{
  if (*r2 != NULL) {
    Rrg_Insert(r1, (*r2)->lb, (*r2)->li, (*r2)->hb, (*r2)->hi);
    Rrg_Merge(r1, &((*r2)->nxt));
  }
  return r1;
}


Rrg_Typ *Rrg_Remove(Rrg_Typ *r1, const Rrg_Typ *r2)
{
  if (*r2 != NULL) {
    Rrg_Delete(r1, (*r2)->lb, (*r2)->li, (*r2)->hb, (*r2)->hi);
    Rrg_Remove(r1, &((*r2)->nxt));
  }
  return r1;
}


Rrg_Typ *Rrg_Copy(Rrg_Typ *dst, const Rrg_Typ *src)
{
  return Rrg_Merge(Rrg_Create(dst),src);
}


Rrg_Typ *Rrg_Invert(Rrg_Typ *r)
{ Rrg_Typ r0, t;

  Rrg_Insert(Rrg_Create(&r0),MINFLOAT,1,MAXFLOAT,1);
  Rrg_Remove(&r0,r);
  t= *r; *r= r0; r0= t;
  Rrg_Dispose(&r0);
  return r;
}


Rrg_Typ *Rrg_Unite(Rrg_Typ *result, const Rrg_Typ *r1, const Rrg_Typ *r2)
{
  return Rrg_Merge(Rrg_Copy(result,r1),r2);
}


Rrg_Typ *Rrg_Intersect(Rrg_Typ *result, const Rrg_Typ *r1, const Rrg_Typ *r2)
{ Rrg_Typ t1,t2;

  Rrg_Remove(Rrg_Copy(&t1,r1),r2);
  Rrg_Remove(Rrg_Copy(&t2,r2),r1);
  Rrg_Unite(result,r1,r2);
  Rrg_Remove(result,&t1); Rrg_Dispose(&t1);
  Rrg_Remove(result,&t2); Rrg_Dispose(&t2);
  return result;
}
