/* PRJGEN.C : Extract project file from C source

  Title   : PRJGEN
  Version : 1.2
  Date    : Nov 27,1996
  Author  : J R Ferguson
  Language: Turbo C 2.0
  Usage   : Refer function Help

[ Project file:
PRJGEN
TCFD.LIB
]
*/

#include <stdio.h>
#include "deflib.h"
#include "stzlib.h"
#include "chrlib.h"
#include "conlib.h"


#define PROGIDN    "PRJGEN"
#define PROGVER    "1.2"
#define FALSE      0
#define TRUE       1
#define MAXLINE    255    /* Max line length for Inp/Out processing */
#define MAXFNM     79     /* Max filespec length (IDE) */
#define INPBUFSIZ  4096   /* Input buffer size in bytes */
#define DFLINPEXT  ".C"
#define DFLOUTEXT  ".PRJ"
#define TRIGGER1   "[ PROJECT FILE:"
#define TRIGGER2   "]"


/* --- Error codes and messages: --- */


#define ERROK      0
#define ERRARG     1
#define ERRFNF     2
#define ERRCRE     3
#define ERRREA     4
#define ERRWRI     5

char *ErrMsg[ERRWRI+1] =
 {"",
  "Command syntax error",
  "File not found",
  "File creation error",
  "Read error",
  "Write error"
 };


/* --- Global variables --- */


int        ErrCod;
char       InpFnm[MAXFNM], OutFnm[MAXFNM];
FILE       *Msg, *Inp, *Out;
int        InpOpn, OutOpn;
int        OptHlp;
int        OptYes;
char       CurLine[MAXLINE];


/* --- General routines --- */


#define wr(s) fprintf(Msg,s)
#define wl(s) fprintf(Msg,"%s\n",s)

void Help(void)
{
wl(PROGIDN" v"PROGVER" - Extract project file from C source");
wl("Usage  : "PROGIDN" infile[.ext] [/option... [...]]");
wl("Default: .ext = .C");
wl("Options: /H Send this help text to (redirected) output.");
wl("         /Y Overwrite any existing project file without confirmation.");
wl("");
wl("The input file is scanned for project file information in the format:");
wl("  [ PROJECT FILE: filename.ext");
wl("  ...");
wl("  ]");
wl("The '[' and ']' brackets should be in column 1. The first line is the");
wl("trigger line. The opening '[' and capital letter part must be exactly");
wl("as above, although letters may be in lower case. 'filename.ext' may be");
wl("omitted, in which case it defaults to 'infile"DFLOUTEXT"'.");
wl("");
wl("If this trigger line is found, all lines between the '[' and ']' lines");
wl("are copied to a file created with the (defaulted) name 'filename.ext',");
wl("omitting any leading white space.");
wl("");
wl("When 'filename.ext' already exists and the /Y option is not specified,");
wl("the program asks for confirmation to overwrite the existing file.");
}

int exist(const char *filename)
{ FILE *f;

  if ((f= fopen(filename,"rb")) == NULL) return FALSE;
  else { fclose(f); return TRUE; }
}


/* --- Command line parsing routines --- */


void ReadOpt(Stz_Ptr arg)
{ int nextopt;

  Stz_Del(arg,0,1);
  do {
    if (Stz_Empty(arg) || (Stz_cRet(arg,0) == '/')) ErrCod= ERRARG;
    else {
      nextopt= FALSE;
      while ((ErrCod==ERROK) && !nextopt && !Stz_Empty(arg)) {
        switch (Stz_cGet(arg)) {
          case 'H': OptHlp= TRUE;  break;
          case 'Y': OptYes= TRUE;  break;
          case '/': nextopt= TRUE; break;
          default : ErrCod= ERRARG;
    } } }
  } while ((ErrCod == ERROK) && nextopt);
}


void ReadArgs(int argc, char *argv[])
{ int i= 0, n=0;
  char arg[MAXCMD];

  while ((ErrCod == ERROK) && (++i < argc)) {
    Stz_Cpy(arg,argv[i]); Stz_Upp(arg);
    switch (Stz_cRet(arg,0)) {
      case '/': ReadOpt(arg); break;
      default : if (n) ErrCod= ERRARG;
                else { ++n; Stz_NCpy(InpFnm,arg,MAXFNM); }
    }
  }
  if (ErrCod == ERROK) {
    if ((n<1) || OptHlp) ErrCod= ERRARG;
    else if (Stz_cPos(InpFnm,'.') == NULL) Stz_Cat(InpFnm,DFLINPEXT);
  } else OptHlp= FALSE;
}


/* --- Low-level I/O routines --- */


void OpenMsg(void)
{ if (OptHlp) Msg= stdout; else Msg= stderr; }


void CloseMsg(void)
{ /*empty*/ }


void OpenInp(void)
{
  if ((Inp= fopen(InpFnm,"rt")) == NULL) ErrCod= ERRFNF;
  else { setvbuf(Inp,NULL,_IOFBF,INPBUFSIZ); InpOpn= TRUE; }
}


void CloseInp(void)
{ if (fclose(Inp) == 0) InpOpn= FALSE; }


int ReadInp(char line[])
{
  if (fgets(line,MAXLINE,Inp) == NULL) {
    if (!feof(Inp)) ErrCod= ERRREA;
    return 0;
  }
  else return 1;
}


void OpenOut(void)
{
  if ((Out= fopen(OutFnm,"wt")) == NULL) ErrCod= ERRCRE;
  else OutOpn= TRUE;
}


void CloseOut(void)
{ if (fclose(Out) == 0) OutOpn= FALSE; else ErrCod= ERRWRI; }


void WriteOut(const char line[])
{ if (fputs(line,Out) == EOF) ErrCod= ERRWRI; }


/* --- Program functions --- */


int FindTrigger(void)
{ int ok= 0; int n= Stz_Len(TRIGGER1);

  while ((ErrCod == ERROK) && !ok && ReadInp(CurLine)) {
    ok= Stz_UppNCmp(CurLine,TRIGGER1,n) == 0;
  }
  if (ok) {
    if (Stz_Empty(Stz_Gtw(OutFnm,
                  Stz_Trunc(Stz_RLS(Stz_Del(CurLine,0,n)),MAXFNM))))
      Stz_Cat(Stz_Before(OutFnm,InpFnm,"."),DFLOUTEXT);
    fprintf(stderr,PROGIDN" v"PROGVER": %s ==> %s\n",InpFnm,OutFnm);
  }
  return ok;
}

int WriteAllowed(void)
{ char c;

  if (OptYes || !exist(OutFnm)) return 1;
  else {
    fprintf(stderr,"Destination file exists. Overwrite? [Y/N]: ");
    while ( ((c=Con_UppKey()) != 'Y') && (c != 'N') );
    fprintf(stderr,"%c\n",c);
    return c == 'Y';
  }
}

void WriteProjectFile(void)
{ int stop= 0; int n= Stz_Len(TRIGGER2);

  OpenOut();
  while ((ErrCod == ERROK) && !stop && ReadInp(CurLine)) {
    stop= Stz_UppNCmp(CurLine,TRIGGER2,n) == 0;
    if (!stop) WriteOut(Stz_RLS(CurLine));
  }
}


/* --- Main line --- */


void MainInit(int argc, char *argv[])
{
  ErrCod= ERROK;
  Stz_Create(InpFnm); InpOpn= FALSE;
  Stz_Create(OutFnm); OutOpn= FALSE;
  OptHlp= FALSE;
  OptYes= FALSE;
  ReadArgs(argc,argv);
  if (ErrCod == ERROK) OpenInp();
}


void MainExit(void)
{
  if (InpOpn) CloseInp();
  if (OutOpn) CloseOut();
  if (ErrCod != ERROK) {
    OpenMsg();
    if (ErrCod==ERRARG) Help();
    else {
      fprintf(Msg,PROGIDN" - %s",ErrMsg[ErrCod]);
      switch (ErrCod) {
        case ERRFNF:
        case ERRREA: fprintf(Msg,": %s\n",InpFnm); break;
        case ERRCRE:
        case ERRWRI: fprintf(Msg,": %s\n",OutFnm); break;
        default    : fprintf(Msg,"\n");
      }
    }
    CloseMsg();
  }
}


int main(int argc, char *argv[])
{
  MainInit(argc,argv);
  if ( (ErrCod == ERROK) && FindTrigger() && WriteAllowed() )
    WriteProjectFile();
  MainExit();
  return ErrCod;
}
