/* OPTLIB.H : Argument option handling

  Title    : OPTLIB
  Version  : 2.0
  Date     : Nov 23,1996
  Author   : J R Ferguson
  Language : Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage    : Header file

  This library contains code to process command-line options of the form
    /<option>[[/]<option>...]
  where <option> has the form
    <idn>[<value>]
  <idn>   is a unique one-character option identifier. Upper and lower
          case letters are treated as equal.
  <value> describes the value that will be assigned to the specified
          option. Its syntax depends on the option type as described
          below.

  Five option types are recognized:
  1. Switch options can have the values 1 or 0.
     For these options <value> can be + (for 1) or - (for 0).
     The default for <value> is +.
  2. Number options describe lists of one or more signed or unsigned
     integer values.
     For these options <value> has the format
        [<integer>][,[<integer>]...]
     where <integer> is empty or consists of an optional + or - sign
     followed by a sequence of 1 to 4 digits. For each item in the
     list a switch specifying whether the <integer> value may be omitted
     is provided when the option item is defined. Also a default as well
     as a minimum and a maximum value for each integer item is specified
     at definition time.
  5. Range option have integer range set values as defined in unit rnglib,
     and can be used to specify ranges of line numbers or text columns.
     For these options <value> specifies a left and right boundary pair,
     using the format
       [<integer>][,[<integer>]]
     where <integer> is empty or consists of an optional + or - sign
     followed by a sequence of 1 to 4 digits. For both integers a switch
     specifying whether the value may be omitted is provided when the
     option is defined. Also a default as well as a minimum and a
     maximum value for each integer is specified at definition time.
     When the same range option is parsed more than once the ranges will
     be combined (merged).
  4. Character options have character values. Their <value> consist of
     the character immediately following the option identifier and can
     only be omitted with the last option in a command argument. The
     default value for a character option is specified when the option is
     defined.
  5. String options have string values. Their <value> consist of all
     remaining characters in the command-argument. If <value> is omitted,
     an empty string is assigned to the option.

  To use this unit, a program should declare a variable of type optLstPtr
  that is referenced to as the option list variable in all functions in the
  unit.
  The program must contain initializing code executed once, containing:
  - A call to function Opt_Init.
  - A call to functions Opt_DefSw, Opt_DefNm, Opt_DefRg, Opt_DefCh or
    Opt_DefSt to define each option, specifying its identifying character.
    For number options function Opt_DefNm is called once for each item in
    the number value list.
  - A call to function Opt_Parse for each command-line argument that might
    contain an option.

  During program execution, the functions Opt_Scanned, Opt_ValSw,
  Opt_ValNm, Opt_ValRg, Opt_ValCh, Opt_ValSt and Opt_RngInside can be used
  as often as desired to find out if an option has been scanned by
  Opt_Parse, to get the current value of an option, or to check if an
  integer is inside a range option value.

  At the end of the program or after all option processing is completed an
  optional call to function Opt_Done can be executed to free all heap space
  allocated for the option list.

  Remarks:
  - It is possible to maintain separate groups of options in one program,
    for instance to support command-argument syntax of the form
      COMMAND <option-group-1> <file-argument> [...] <option-group-2>
    In this case each option-group has its own option list variable.
  - As option identifiers have to be unique within an option list, any
    redefinition of an option identifier that has been defined before
    is ignored, except for number options where redefinition results in
    adding an entry to the list of numbers that go with that option.
*/

/*.hlptx skip 1*/
#ifndef Opt_Lib
#define Opt_Lib

#include "rnglib.h"

#define Opt_PREFIX '/'   /* prefix char for option arguments             */
#define Opt_NUMSEP ','   /* separator char for number option value lists */
#define Opt_MAXDIG 4     /* max number of digits in a number option item */


/* The following type definitions can be referenced externally */

typedef int                    Opt_SwTyp;
typedef int                    Opt_NmTyp;
typedef Rng_Typ *              Opt_RgTyp;
typedef char                   Opt_ChTyp;
typedef char *                 Opt_StTyp;
typedef struct opt_LstStruc *  Opt_LstPtr;


/* The remaining type definitions are for internal use only */

typedef enum   {
          _Opt_SWITCH,
          _Opt_NUMBER,
          _Opt_RANGE,
          _Opt_CHAR,
          _Opt_STRING
        } _Opt_Typ;

typedef struct {
           int        nmreq;
           Opt_NmTyp  nmdfl, nmmin, nmmax, nmval;
        } _Opt_NmRec;

typedef struct _Opt_NmLstStruc *  _Opt_NmLstPtr;
typedef struct _Opt_NmLstStruc {
          _Opt_NmRec    val;
          _Opt_NmLstPtr nxt;
        } _Opt_NmLstRec;

typedef struct {
           int        rgreq;
           Opt_NmTyp  rgdfl, rgmin, rgmax;
        } _Opt_RgBndRec;

typedef struct _Opt_RgStruc *  _Opt_RgPtr;
typedef struct _Opt_RgStruc {
          _Opt_RgBndRec  rglft, rgrgt;
	   Opt_RgTyp     rgval;
	} _Opt_RgRec;

typedef struct {
          char       letter;
          int        scanned;
         _Opt_Typ    otype;
          union {
            Opt_SwTyp     swval;
           _Opt_NmLstPtr  nmlst;
           _Opt_RgPtr     rgptr;
            struct {
              Opt_ChTyp   chdfl, chval;
	    } c;
            Opt_StTyp     stptr;
	  } u;
        } _Opt_Rec;

typedef struct opt_LstStruc {
          _Opt_Rec     val;
           Opt_LstPtr  nxt;
        } _Opt_LstRec;

/*.hlptx skip 4*/
#ifdef __cplusplus
extern "C" {
#endif

void Opt_Init(Opt_LstPtr *optlst);
/* Initialize the option list pointed to by optlst. Execute this routine
   once for each option list at the start of the program.
*/

void Opt_Done(Opt_LstPtr *optlst);
/* Clear heap space allocated for the option list pointed to by optlst.
   This routine can be executed once at the end of the program or as soon
   as all option processing is done (including references to any options in
   the list). Execution of this function is not neccessary.
*/

void Opt_DefSw(Opt_LstPtr *optlst, char idn);
/* Add definition of switch type option idn into the option list pointed to
   by optlst, giving it the initial value 0.
*/

void Opt_DefNm(Opt_LstPtr *optlst,
               char        idn,          /* identifying option character */
               int         required,     /* value required */
               Opt_NmTyp   min,          /* minimum value */
               Opt_NmTyp   max,          /* maximum value */
               Opt_NmTyp   dfl           /* default value */
              );
/* Add definition of number type option idn into option list pointed to by
   optlst, specifying if it is required and also specifying the minimum,
   maximum and, default values min, max and dfl. The value is initially set
   to the default value. 
   When a number option with the same identifier has been defined before
   using this function, a number item is added to the list of items for
   this number option.
*/

void Opt_DefRg(Opt_LstPtr *optlst,
               char       idn,          /* identifying option character */
               int        lrequired,    /* left boundary required       */
               Opt_NmTyp  lmin,         /* left boundary minimum value  */
               Opt_NmTyp  lmax,         /* left boundary maximum value  */
               Opt_NmTyp  ldfl,         /* left boundary default value  */
               int        rrequired,    /* right boundary required      */
               Opt_NmTyp  rmin,         /* right boundary minimum value */
               Opt_NmTyp  rmax,         /* right boundary maximum value */
               Opt_NmTyp  rdfl          /* right boundary default value */
              );
/* Add definition of range type option idn into option list pointed to by
   optlst, specifying for both left and right boundary if it is required,
   and what values are its minimum, maximum and default. The initial value
   is the empty range.
*/

void Opt_DefCh(Opt_LstPtr *optlst, char idn, Opt_ChTyp dfl);
/* Add definition of character type option idn into option list pointed to
   by optlst, specifying its default value, which also is its initial
   value.
*/

void Opt_DefSt(Opt_LstPtr *optlst, char idn);
/* Add definition of string type option idn into option list pointed to by
   optlst, giving it the empty string as an initial value.
*/

int Opt_Parse(Opt_LstPtr *optlst, char *arg, int *ok);
/* Parse command-line argument arg for valid options, updating option list
   pointed to by optlst.
   The routine Opt_Parse is typically executed once for each command-line
   argument in the program initialization part.
   The function result reflects if the command-line argument contains one
   or more options.
   - If the function result is 1, variable *ok reflects successfull
     scanning of the argument, so a value *ok==0 signals a command-syntax
     error.
   - If the function result is 0, the argument does not represent one
     or more options. It could be a file-name or some other argument and
     should be interpreted separately. In this case the value of variable
     *ok is not defined. If no other argument type is expected this
     condition signals a command-syntax error as well.

   Each option recognized by this routine will be marked as being scanned,
   which can be tested by function Opt_Scanned. When the same range option
   is parsed more than once, the range values are combined (merged) into
   one possibly broken range of integer values. When other options are
   parsed more then once, the last value interpreted overwrites that of any
   previous instances. This property can be used to force certain
   dependencies between options by calling Opt_Parse with a
   program-supplied "arg" variable after scanning the real command
   arguments.
*/

int Opt_Scanned(const Opt_LstPtr *optlst, char idn);
/* Test if option idn was scanned by function Opt_Parse. This function can
   be used to see if the option was specified at all.
   If the option was not defined, the value 0 will be returned.
*/

Opt_SwTyp Opt_ValSw(const Opt_LstPtr *optlst, char idn);
/* Returns the current value of switch type option idn in the option list
   pointed to by optlst. If no option idn was defined, or if it is not
   defined as a switch option, the value 0 will be returned.
   This function can be called as often as required during the execution of
   a program after the option has been defined using function Opt_DefSw and
   the command-line arguments have been scanned for options using function
   Opt_Parse.
*/

Opt_NmTyp Opt_ValNm(const Opt_LstPtr *optlst, char idn, int i);
/* Returns the current value of item i in the list of values for the number
   type option idn in option list pointed to by optlst. If no option idn
   was defined, or if it is not defined as a number option, or i is less
   than one or grater than the number of items defined for this number
   option, the value 0 will be returned.
   This function can be called as often as required during the execution of
   a program after the option has been defined using function Opt_DefNm and
   the command-line arguments have been scanned for options using function
   Opt_Parse.
*/

Opt_RgTyp Opt_ValRg(const Opt_LstPtr *optlst, char idn);
/* Returns the current value of range option idn in the option list pointed
   to by optlst. If no option idn was defined, or if it is not defined as a
   range option, the NULL pointer value will be returned.
   This function can be called as often as required during the execution of
   a program after the option has been defined using function Opt_DefRg and
   the command-line arguments have been scanned for options using function
   Opt_Parse.
*/

int Opt_RngInside(const Opt_LstPtr *optlst, char idn, Opt_NmTyp n);
/* Specifies if the value n is within the current range for range option
   idn in the option list pointed to by optlst. 
   If no option idn was defined, or if it is not defined as a range option,
   the value 0 will be returned. 
   This function can be called as often as required during the execution of
   a program after the option has been defined using function Opt_DefRg and
   the command-line arguments have been scanned for options using function
   Opt_Parse.
*/

Opt_ChTyp Opt_ValCh(const Opt_LstPtr *optlst, char idn);
/* Returns the current value of character option idn in the option list
   pointed to by optlst.
   If no option idn was defined, or if it is not defined as a character
   option, the character value 0 will be returned.
   This function can be called as often as required during the execution of
   a program after the option has been defined using function Opt_DefCh and
   the command-line arguments have been scanned for options using function
   Opt_Parse.
*/

Opt_StTyp Opt_ValSt(const Opt_LstPtr *optlst, char idn);
/* Returns a pointer to the the current value of string option idn in the
   option list pointed to by optlst. If no option idn was defined, or if it
   is not defined as a string option, a pointer to an empty string will be
   returned.
   This function can be called as often as required during the execution of
   a program after the option has been defined using function Opt_DefSt and
   the command-line arguments have been scanned for options using function
   Opt_Parse.
*/

/*.hlptx skip 4*/
#ifdef __cplusplus
}
#endif
#endif /* #ifndef Opt_Lib */
