/* OPTLIB.C : Argument option handling

  Title    : OPTLIB
  Version  : 2.0
  Date     : Nov 23,1996
  Author   : J R Ferguson
  Language : Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage    : Function library
*/

#include <alloc.h>
#include <ctype.h>
#include <string.h>
#include "optlib.h"
#include "chrlib.h"
#include "stzlib.h"
#include "deflib.h"
#include "rnglib.h"

typedef char * ChrPtr;

/* -- local definitions -- */

static char empty[1] = "";

/* === Number List routines === */
/* Single linked List */

static void nmlstcreate(_Opt_NmLstPtr *nmlst)
{ *nmlst= NULL; }

static void nmlstdone(_Opt_NmLstPtr *nmlst)
{ if (*nmlst != NULL) { nmlstdone(&((*nmlst)->nxt)); free(*nmlst); } }

static void nmlstinsert(_Opt_NmLstPtr *nmlst, _Opt_NmRec *elm) 
/* Insert at tail */
{ _Opt_NmLstPtr p;

  if (*nmlst != NULL) nmlstinsert(&((*nmlst)->nxt), elm);
  else {
    if ((p= (_Opt_NmLstPtr)malloc(sizeof(_Opt_NmLstRec))) != NULL) {
      p->val= *elm; p->nxt= NULL; *nmlst= p;
} } }

static int nmlstfind(const _Opt_NmLstPtr *nmlst,int i, _Opt_NmLstPtr *p)
{ int ok;

  if ((*nmlst==NULL) || (i<1)) ok= 0;
  else {
    *p= *nmlst;
    while ((i>1) && ((*p)->nxt != NULL))  { *p= (*p)->nxt; --i; }
    ok= i==1;
  }
  return ok;
}

static int nmlstupdate(_Opt_NmLstPtr *p, Opt_NmTyp n)
{
  if ((n < (*p)->val.nmmin) || (n > (*p)->val.nmmax)) return 0;
  else { (*p)->val.nmval= n; return 1; }
}


/* === option List routines === */
/* Single linked ordered circular list with dummy header item */

static int optlstorder(_Opt_Rec e1, _Opt_Rec e2)
{ return Chr_UppOrder(e1.letter, e2.letter); }

static void oplstcreate(Opt_LstPtr *optlst)
{
  if ((*optlst= (Opt_LstPtr) malloc(sizeof(_Opt_LstRec))) != NULL)
    (*optlst)->nxt= *optlst;
}

static void oplstclear(Opt_LstPtr *optlst)
{ free(*optlst); }

static int oplstempty(Opt_LstPtr *optlst)
{ return (*optlst)->nxt == *optlst; }

static void oplstinsert(_Opt_Rec *elm, Opt_LstPtr *optlst)
{ Opt_LstPtr pt0, pt1;

  (*optlst)->val= *elm; /*sentinel*/
  pt0= *optlst;
  while (optlstorder(pt0->nxt->val, *elm) < 0)  pt0= pt0->nxt;
  if ((pt1= (Opt_LstPtr)malloc(sizeof(_Opt_LstRec))) != NULL) {
    pt1->val= *elm; pt1->nxt= pt0->nxt; pt0->nxt= pt1;
} }

static void oplstretrieve(_Opt_Rec *elm, Opt_LstPtr *optlst)
/* Retrieve smallest element *elm and discard from *optlst.
   If *optlst is empty, the value of *elm is not defined. */
{ Opt_LstPtr ptr;

  ptr= (*optlst)->nxt; *elm= ptr->val; (*optlst)->nxt= ptr->nxt;
  if (ptr->nxt != ptr) free(ptr);
}

static int oplstfind(char idn, const Opt_LstPtr *optlst, Opt_LstPtr *p)
/* Find pointer *p to option list record with option idn. The returnvalue
   indicates success of find action. The value of *p is not defined if idn
   could not be found in optlst. */
{
  (*optlst)->val.letter= idn; /*sentinel*/
  *p= *optlst;
  while (optlstorder((*p)->nxt->val,(*optlst)->val) < 0)  *p= (*p)->nxt;
  *p= (*p)->nxt;
  return (*p != *optlst) && (optlstorder((*p)->val, (*optlst)->val) == 0);
}


static void parseswitch(ChrPtr *sp, Opt_SwTyp *swval)
{
  switch (**sp) {
    case '+' : ++(*sp); *swval= 1; break;
    case '-' : ++(*sp); *swval= 0; break;
    default  : *swval= 1; break;
} }


static int parseint(ChrPtr    *sp,       /* command argument pointer    */
                    Opt_NmTyp *n,        /* result value if ok          */
                    int        required, /* value required (1=yes,0=no) */
                    Opt_NmTyp  dflval,   /* default value               */
                    Opt_NmTyp  minval,   /* minimum value accepted      */
                    Opt_NmTyp  maxval)   /* maximum value accepted      */
/* Parse integer. Report success in function result.
   In case of error the value of n is left unchanged.
   If no value is given and required is 0, n will get the value dflval.
   If a value is found and all is correct it is placed into n. */
{ char c; int plus; int digits; Opt_NmTyp value; int ok;

  ok= 0;
  if ( !( isdigit(c= **sp) || (c=='-') || (c=='+') ) ) {
    if (required == 0) { *n= dflval; ok= 1; }
  }
  else {
    switch (c) {
      case '+' : c=*(++(*sp)); plus= 1; break;
      case '-' : c=*(++(*sp)); plus= 0; break;
      default  : plus= 1; break;
    }
    value= 0;
    if (isdigit(c)) {
      digits= 0;
      do {
        value= 10 * (value) - (c-'0'); ++digits; c= *(++(*sp));
      } while ( isdigit(c) && (digits < Opt_MAXDIG) );
      if (plus) value= -value;
      ok= (!isdigit(c)) && (value >= minval) && (value <= maxval);
    }
    if (ok) *n= value;
  }
  return ok;
}


static int parsenumber(ChrPtr *sp, _Opt_NmLstPtr *nmlst, int i)
{ _Opt_NmLstPtr p;

  if (nmlstfind(nmlst,i,&p) == 0) return 0;
  else return parseint(sp, &(p->val.nmval),
                            (p->val.nmreq),
                            (p->val.nmdfl),
                            (p->val.nmmin),
                            (p->val.nmmax));
}


static int parsenmlst(ChrPtr *sp, _Opt_NmLstPtr *nmlst)
{ int ok; int i; _Opt_NmLstPtr p;

  i= 1; ok= parsenumber(sp,nmlst,i++);
  while (ok && (**sp == Opt_NUMSEP)) {
    ++(*sp); ok= parsenumber(sp,nmlst,i++);
  }
  while (ok && nmlstfind(nmlst,i++,&p)) {
    if (p->val.nmreq) ok= 0; else p->val.nmval= p->val.nmdfl;
  }
  return ok;
}


static int parsergnum(ChrPtr *sp, _Opt_RgBndRec *bnd, Opt_NmTyp *n)
{return parseint(sp, n, bnd->rgreq, bnd->rgdfl, bnd->rgmin, bnd->rgmax);}


static int parserange(ChrPtr *sp, _Opt_RgPtr rgptr)
{ int ok; Opt_NmTyp low,high;
  static char *dummy = empty;

   ok= parsergnum(sp, &(rgptr->rglft), &low);
   if (ok) {
    if (**sp == Opt_NUMSEP) {
      ++(*sp); ok= parsergnum(sp, &(rgptr->rgrgt), &high);
    }
    else ok= parsergnum(&dummy, &(rgptr->rgrgt), &high);
  }
  if (ok) Rng_Insert(rgptr->rgval, low, high);
  return ok;
}


static int parsechar(ChrPtr *sp, Opt_ChTyp *chval)
{
  if (**sp == '\0') return 0;
  else { *chval= *(*sp)++; return 1; }
}


static void parsestring(ChrPtr *sp, Opt_StTyp *p)
{ Stz_NCpy(*p,*sp,MAXCMD); (*sp)+= Stz_Len(*sp); }


/* --- global routines --- */

void Opt_Init(Opt_LstPtr *optlst)
{ oplstcreate(optlst); }


void Opt_Done(Opt_LstPtr *optlst)
{ _Opt_Rec r;

  while (!oplstempty(optlst))  {
    oplstretrieve(&r,optlst);
    switch (r.otype) {
      case _Opt_SWITCH : /* nothing */; break;
      case _Opt_NUMBER : nmlstdone(&(r.u.nmlst)); break;
      case _Opt_RANGE  : Rng_Dispose(r.u.rgptr->rgval);
                         free(r.u.rgptr);
                         break;
      case _Opt_CHAR   : /* nothing */; break;
      case _Opt_STRING : free(r.u.stptr); break;
    }
  }
  oplstclear(optlst);
}


void Opt_DefSw(Opt_LstPtr *optlst, char idn)
{ _Opt_Rec r; Opt_LstPtr p;

  if (!oplstfind(idn,optlst,&p)) {
    r.letter  = toupper(idn);
    r.scanned = 0;
    r.otype   = _Opt_SWITCH;
    r.u.swval = 0;
    oplstinsert(&r,optlst);
  }
}


void Opt_DefNm(Opt_LstPtr *optlst,
               char        idn,
               int         required,
               Opt_NmTyp   min,
               Opt_NmTyp   max,
               Opt_NmTyp   dfl
              )
{ _Opt_Rec r; Opt_LstPtr p; _Opt_NmRec nmrec;

  nmrec.nmreq= required;
  nmrec.nmdfl= dfl;
  nmrec.nmmin= min;
  nmrec.nmmax= max;
  nmrec.nmval= dfl;
  if (oplstfind(idn, optlst, &p)) nmlstinsert(&(p->val.u.nmlst), &nmrec);
  else {
    r.letter = toupper(idn);
    r.scanned= 0;
    r.otype  = _Opt_NUMBER;
    nmlstcreate(&(r.u.nmlst));
    nmlstinsert(&(r.u.nmlst),&nmrec);
    oplstinsert(&r,optlst);
  }
}


void Opt_DefRg(Opt_LstPtr *optlst,
               char        idn,
               int         lrequired,
               Opt_NmTyp   lmin,
               Opt_NmTyp   lmax,
               Opt_NmTyp   ldfl,
               int         rrequired,
               Opt_NmTyp   rmin,
               Opt_NmTyp   rmax,
               Opt_NmTyp   rdfl
              )
{ _Opt_Rec r; Opt_LstPtr p;

  if (!oplstfind(idn,optlst,&p)) {
    r.letter  = toupper(idn);
    r.scanned = 0;
    r.otype   = _Opt_RANGE;
    if ((r.u.rgptr= (_Opt_RgPtr) malloc(sizeof(_Opt_RgRec))) != NULL) {
      r.u.rgptr->rglft.rgreq= lrequired;
      r.u.rgptr->rgrgt.rgreq= rrequired;
      r.u.rgptr->rglft.rgdfl= ldfl;
      r.u.rgptr->rgrgt.rgdfl= rdfl;
      r.u.rgptr->rglft.rgmin= lmin;
      r.u.rgptr->rgrgt.rgmin= rmin;
      r.u.rgptr->rglft.rgmax= lmax;
      r.u.rgptr->rgrgt.rgmax= rmax;
      Rng_Create(r.u.rgptr->rgval);
    }
    oplstinsert(&r,optlst);
  }
}


void Opt_DefCh(Opt_LstPtr *optlst, char idn, Opt_ChTyp dfl)
{ _Opt_Rec r; Opt_LstPtr p;

  if (!oplstfind(idn,optlst,&p)) {
    r.letter    = toupper(idn);
    r.scanned   = 0;
    r.otype     = _Opt_CHAR;
    r.u.c.chdfl = dfl;
    r.u.c.chval = dfl;
    oplstinsert(&r,optlst);
  }
}


void Opt_DefSt(Opt_LstPtr *optlst, char idn)
{ _Opt_Rec r; Opt_LstPtr p;

  if (!oplstfind(idn,optlst,&p)) {
    r.letter  = toupper(idn);
    r.scanned = 0;
    r.otype   = _Opt_STRING;
    if ( (r.u.stptr= (Opt_StTyp) malloc(MAXCMD+1)) != NULL)
      Stz_NCpy(r.u.stptr, empty, MAXCMD);
    oplstinsert(&r,optlst);
  }
}


int Opt_Parse(Opt_LstPtr *optlst, char *arg, int *ok)
{ Opt_LstPtr p;

  *ok= 1;
  if (*arg++ != Opt_PREFIX) return 0;
  else {
    do {
      if (!oplstfind(*arg++,optlst,&p)) *ok= 0;
      else {
        p->val.scanned= 1;
        switch (p->val.otype) {
         case _Opt_SWITCH: parseswitch (&arg,&(p->val.u.swval));   break;
         case _Opt_NUMBER: *ok=parsenmlst (&arg,&(p->val.u.nmlst));break;
         case _Opt_RANGE : *ok=parserange (&arg,p->val.u.rgptr);   break;
         case _Opt_CHAR  : if (!parsechar  (&arg,&(p->val.u.c.chval)))
                             p->val.u.c.chval= p->val.u.c.chdfl;
                           break;
         case _Opt_STRING: parsestring(&arg,&(p->val.u.stptr));    break;
        }
      }
      if ((*ok) && (*arg == Opt_PREFIX)) { *ok= !Stz_Empty(++arg); }
    } while (!Stz_Empty(arg) && *ok);
    if (*ok) *ok= Stz_Empty(arg);
    return 1;
  }
}

int Opt_Scanned(const Opt_LstPtr *optlst, char idn)
{ Opt_LstPtr p;

  return (oplstfind(idn,optlst,&p)) ? p->val.scanned : 0;
}


Opt_SwTyp Opt_ValSw(const Opt_LstPtr *optlst, char idn)
{ Opt_LstPtr p;

  return ((oplstfind(idn,optlst,&p)) && (p->val.otype == _Opt_SWITCH))
  ? p->val.u.swval : 0;
}


Opt_NmTyp Opt_ValNm(const Opt_LstPtr *optlst, char idn, int i)
{ Opt_LstPtr pl; _Opt_NmLstPtr pn;

  return ( (oplstfind(idn,optlst,&pl))          &&
           (pl->val.otype == _Opt_NUMBER)       &&
           (nmlstfind(&(pl->val.u.nmlst),i,&pn))
         )
  ? pn->val.nmval : 0;
}


Opt_RgTyp Opt_ValRg(const Opt_LstPtr *optlst, char idn)
{ Opt_LstPtr p;

  return ( oplstfind(idn,optlst,&p) && (p->val.otype == _Opt_RANGE) )
  ? p->val.u.rgptr->rgval : NULL;
}


int Opt_RngInside(const Opt_LstPtr *optlst, char idn, Opt_NmTyp n)
{ return Rng_Inside(Opt_ValRg(optlst,idn),n); }


Opt_ChTyp Opt_ValCh(const Opt_LstPtr *optlst, char idn)
{ Opt_LstPtr p;

  return (oplstfind(idn,optlst,&p) && (p->val.otype == _Opt_CHAR))
  ? p->val.u.c.chval : '\0';
}


Opt_StTyp Opt_ValSt(const Opt_LstPtr *optlst, char idn)
{ Opt_LstPtr p;

  return (oplstfind(idn,optlst,&p) && (p->val.otype == _Opt_STRING))
  ? p->val.u.stptr : empty;
}
