/* OLLLIB.C : Ordered linked list routines

  Title   : OLLLIB
  Version : 1.1
  Date    : Nov 23,1996
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Author  : J R Ferguson
  Usage   : Function library

The list is implemented as a circular single linked list with a dummy entry
pointed to by the head of the list. This dummy entry is used as a sentinel
during the search action performed by function Oll_Insert.
*/

#include <stdlib.h>
#include "olllib.h"


/* ========================= local definitions ======================= */


#define  _Oll_Valid(list)  ((list)->head != NULL)


static int _Oll_Copy(const Oll_ListTyp *list, void *elm)
/* Copies the contents of the first element of the list to a memory area
   pointed to by elm, without removing it from the list.
   Returns 1 on success, 0 if the list is empty or not properly initialized.
*/
{
  if (Oll_Empty(list))
    return 0;
  else {
    memmove(elm, list->head->nxt->elm, list->size);
    return 1;
  }
}


static int _Oll_Remove(Oll_ListTyp *list)
/* Removes the first element of the list, discarding its contents.
   Returns 1 on seccess, 0 if the list was empty or not properly initialized.
*/
{ Oll_Ptr p;

  if (Oll_Empty(list))
    return 0;
  else {
    p= list->head->nxt;
    list->head->nxt= p->nxt;
    list->curr= list->head;
    free(p);
    return 1;
  }
}


/* ========================= global definitions ====================== */


int Oll_Create( Oll_ListTyp *list,
                 size_t       elmsize,
                 int        (*cmp) (const void *, const void *)
               )
{
  if ((list->head= (Oll_Ptr)malloc(sizeof(Oll_Ptr) + elmsize)) != NULL) {
    list->head->nxt= list->head;
    list->curr= list->head;
    list->size= elmsize;
    list->cmp = cmp;
    return 1;
  }
  else return 0;
}


void Oll_Clear(Oll_ListTyp *list)
{
  if (_Oll_Valid(list)) {
    while (_Oll_Remove(list));
    free(list->head);
    list->head= NULL;
    list->curr= NULL;
  }
}


int Oll_Empty(const Oll_ListTyp *list)
{
  return !_Oll_Valid(list) || (list->head->nxt == list->head);
}


int Oll_Insert (Oll_ListTyp *list, const void *elm)
{ Oll_Ptr p0, p1;

  if (_Oll_Valid(list)) {
    memmove(list->head->elm, elm, list->size);  /*plaats sentinel*/
    p0= list->head;
    while ((*(list->cmp))(p0->nxt->elm, elm) < 0)
      p0= p0->nxt;
    if ((p1= (Oll_Ptr)malloc(sizeof(Oll_Ptr) + list->size)) != NULL) {
      memmove(p1->elm, elm, list->size);
      p1->nxt= p0->nxt;
      p0->nxt= p1;
      return 1;
    }
    else return 0;
  }
  else return 0;
}


int Oll_Retrieve(Oll_ListTyp *list, void *elm)
{
  return _Oll_Copy(list,elm) && _Oll_Remove(list);
}


int Oll_GetFirst(Oll_ListTyp *list, void *elm)
{
   list->curr= list->head->nxt;
   return _Oll_Copy(list,elm);
}


int Oll_GetNext(Oll_ListTyp *list, void *elm)
{
  if (!_Oll_Valid(list) || (list->curr->nxt == list->head))
    return 0;
  else {
    list->curr= list->curr->nxt;
    memmove(elm, list->curr->elm, list->size);
    return 1;
  }
}
