// OBTLIB.H : Ordered Binary Tree objects
//
// Title    : OBTLIB
// Version  : 2.2
// Date     : Nov 23,1996
// Author   : J R Ferguson
// Language : Turbo C++ 3.1 for Windows
// Usage    : Header file

//.hlptx skip 1
#ifndef Obt_Lib
#define Obt_Lib

#include <_defs.h>
#include "objlib.h"


// --- Private definitions ---

typedef struct _Obt_Node {
  void              *Item;
  struct _Obt_Node  *Prev, *Next;
} T_Obt_Node;

typedef T_Obt_Node  *PT_Obt_Node;


//.hlptx skip 1
// --- Public definitions ---

_CLASSDEF(T_Obt_Tree);
class T_Obt_Tree : public T_Object 
// Unbalanced Binary Tree. The nodes need not have unique keys.
// There are no member functions to find or process individual nodes, once
// they are inserted.
{
public:
  PT_Obt_Node      Root;
  PT_CompareFunc   CompareFunc;

              T_Obt_Tree(PT_CompareFunc V_CompareFunc);
  // Create new (empty) tree, setting the compare function on which to
  // base its ordering.
  // Override: sometimes.
  virtual     ~T_Obt_Tree(void);
    // Discard tree after deleting and disposing of all items in it by
    // calling DeleteAll.
    // Override: sometimes.
  virtual void FreeItem(void *V_Item);
  // Disposes of the specified item, freeing any memory allocated by it.
  // The default FreeItem does nothing. In most cases you will override
  // this function. When V_Item is an object class pointer of type PT_Item,
  // the code could be:
  //   delete (P_Item)V_Item;
  // FreeItem is called by DeleteAll for each node item that is not NULL,
  // before that node itself is deallocated. It should never be called
  // directly, because that would leave a node with an invalid Item pointer
  // in the tree.
  // Override: often.
  virtual void DeleteAll(void);
  // Delete and dispose of all nodes in the tree. Before a node is deleted,
  // FreeItem is called with its Item pointer as a parameter, unless the
  // Item pointer is NULL.
  // The result is an empty tree.
  // Override: seldom.
  virtual int  Empty(void);
  // Test if the tree is empty
  // Override: never.
  virtual int  Insert(void *V_Item);
  // Insert an item into the tree.
  // The function result is 1 on success, and 0 if there is not enough
  // memory to insert the item.
  // Override: never.
  virtual void InOrder(PT_ProcessProc V_ProcessProc);
  // InOrder tree traversal, processing all node items with the specified
  // procedure in the order defined by CompareFunc.
  // Override: never;
  virtual void PreOrder(PT_ProcessProc V_ProcessProc);
  // PreOrder tree traversal, processing all node items with the specified
  // procedure.
  // The nodes are processed after their child subtrees have been processed
  // (in the order of the predecessor subtree followed by the successor
  // subtree, as defined by CompareFunc).
  // Override: never;
  virtual void PostOrder(PT_ProcessProc V_ProcessProc);
  // PostOrder tree traversal, processing all node items with the specified
  // procedure.
  // The nodes are processed before their child subtrees are processed
  // (in the order of the predecessor subtree followed by the successor
  // subtree, as defined by CompareFunc).
  // Override: never;
private:
  virtual void _Discard(PT_Obt_Node V_Node);
  virtual int  _SubTreeInsert(PT_Obt_Node *V_Node, void *V_Item);
  virtual void _DoInOrder(PT_Obt_Node V_Node, PT_ProcessProc V_Process);
  virtual void _DoPreOrder(PT_Obt_Node V_Node, PT_ProcessProc V_Process);
  virtual void _DoPostOrder(PT_Obt_Node V_Node, PT_ProcessProc V_Process);
};

_CLASSDEF(T_Obt_SearchTree)
class T_Obt_SearchTree : public T_Obt_Tree
// Unbalanced Binary Search Tree. This is a descendant of class T_Obt_Tree
// which maintains the following constraints:
//  - All elements in the Prev subtree of each node precede that node.
//  - All elements in the Next subtree of each node follow that node.
// As a consequence, all tree elements must have unique keys.
// In addition to the member functions provided by T_Obt_Tree, this class
// also has a Search member function to find an individual node item.
{
public:
               T_Obt_SearchTree(PT_CompareFunc V_CompareFunc)
               : T_Obt_Tree(V_CompareFunc) {};
  virtual int  Insert(void *V_Item);
  // Insert an item into the tree, in such a way that the binary search
  // tree constraint is maintained.
  // The result is 1 if the instertion succeeded, and 0 if the key of
  // V_Item was already found in the tree. The result is also 0 if there
  // is not enough memory to insert the item.
  // Override: never.
  virtual void *Search(void *V_Item);
  // Search for a tree node with an item that compares equal to V_Item.
  // If found, return a pointer to its item. Otherwise, return NULL.
  // Override: never.
  virtual int Delete(void *V_Item);
  // Search for a tree node with an item that compares equal to V_Item.
  // If found, delete that node from the tree and dispose of it, after
  // calling FreeItem(V_Item) if V_Item != NULL.
  // Returns 1 on success, 0 if the item was not found in the tree.
  // Overide: never.
private:
  virtual int  _Locate(void *V_Item, PT_Obt_Node *V_Node);
  virtual void _SubDel(PT_Obt_Node *elm, PT_Obt_Node *DelElm);
  virtual int  _Del(PT_Obt_Node *V_Node, PT_Obt_Node *DelElm,
                    void *V_Item);
};

//.hlptx skip 1
#endif // #ifndef Obt_Lib
