/* NUMLIB.H : Numerical routines

  Title   : NUMLIB
  Version : 4.0
  Date    : Nov 23,1996
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Author  : J R Ferguson
  Usage   : Header file

  The following C datat types are supported:

  Number type       Arguments   Supported data types   
  ---------------   ---------   ------------------------------
  whole numbers     i,j,n       char, short int, int, long int
                                (both signed and unsigned)

  real numbers      e,g,x,y     float, double
*/

/*.hlptx skip 1*/
#ifndef  Num_Lib
#define  Num_Lib

#include <stdlib.h>
#include <math.h>

#define  Num_PI          M_PI

#define  Num_DBLPI      (M_PI * 2.0)
#define  Num_HALFPI      M_PI_2

#define  Num_RADIAN     (180.0 / M_PI)
/* number of degrees in een radian (180 / Pi) */

#define  Num_LN2         M_LOG2E
/* natural logarithm of 2 */

/*.hlptx skip 4*/
#ifdef __cplusplus
extern "C" {
#endif

#define  Num_Odd(n)      (((n) & 1) != 0)
#define  Num_Even(n)     (((n) & 1) == 0)

#define  Num_IMax(i,j)   max((i),(j))
#define  Num_IMin(i,j)   min((i),(j))
#define  Num_RMax(x,y)   max((x),(y))
#define  Num_RMin(x,y)   min((x),(y))

#define  Num_Ln(x)       log(x)
#define  Num_Log2(e)     (log(x)*Num_LN2)
#define  Num_Log10(x)    log10(x)
#define  Num_LogG(g,x)   (log(x)/log(g))

double     Num_RSign(double x);
/* Returns +1.0 if x>0, 0.0 if x==0, -1.0 if x<0 */

long int   Num_ISign(long int n);
/* Returns +1 if n>0, 0 if n==0, -1 if n<0. */

#define    Num_RPowerR(x,e)   pow((x),(e))
/* Returns x to the power of e. */

double     Num_RPowerI(double x, long int n);
/* Returns x to the power of n. */

long int   Num_IPowerI(long int i, long int n);
/* Returns i to the power of n. 
   n must be non-negative.
*/

#define Num_Rad(x)     ((x)/Num_RADIAN)
/* double Num_Rad(double x);   
   Returns the value of angle x in radians, when x is in degrees.
   Definition : Num_Rad(x) = x * Num_PI/180.
*/

#define  Num_Deg(x)     ((x)*Num_RADIAN)
/* double Num_Deg(double x); 
   Returns the value of angle x in degrees, when x is in radians.
   Defined as deg(x) = x * 180/Pi.
*/

#define   Num_Tan(x)    tan(x)
/* double Num_Tan(double x); 
   Tangent of x radians.
   Definition : tan(x) = sin(x)/cos(x).
   Condition  : x != Pi/2 + k*Pi   where k is an integer.
*/

#define  Num_Cot(x)     (1.0/tan(x))
/* double Num_Cot(double x);
   Cotangent of x radians.
   Definition : cot(x) = cos(x)/sin(x).
   Condition  : x != k*Pi          where k is an integer.
*/

#define  Num_Sec(x)     (1.0/cos(x))
/* double Num_Sec(double x); 
   Secant of x radians.
   Definition : sec(x) = 1/cos(x).
   Condition  : x != Pi/2 + k*Pi   where k is an integer.
*/

#define  Num_Cosec(x)   (1.0/sin(x))
/* double Num_Cosec(double x); 
   Cosecant of x radians.
   Definition : cosec(x) = 1/sin(x).
   Condition  : x != k*Pi          where k is an integer. 
*/

#define  Num_ArcSin(x)  asin(x)
/* double Num_ArcSin(double x); 
   Arc sine of x in radians.
   Definition : sin(ArcSin(x)) = 1
   Condition  : -1.0 <= x <= 1.0
   Range      : [-Pi/2 , +Pi/2]   
*/

#define  Num_ArcCos(x)   acos(x)
/* double Num_ArcCos(double x); 
   Arc cosine of x in radians.
   Definition : cos(ArcCos(x)) = 1
   Condition  : -1.0 <= x <=  1.0
   Range      : [0.0 , Pi] 
*/

#define  Num_ArcTan(x)   atan(x)
/* double Num_ArcTan(double x); 
   Arc tangent of x in radians.
   Definition : tan(ArcTan(x)) = 1
   Condition  : none
   Range      : <-Pi/2 , Pi/2> }
*/

#define  Num_ArcCot(x)   (Num_HALFPI-atan(x))
/* double Num_ArcCot(double x); 
{ Arc cotangent of x in radians.
  Definition : cot(ArcCot(x)) = 1
  Condition  : none
  Range      : <0 , Pi>  }
*/

#define  Num_SinH(x)     sinh(x)
/* double Num_SinH(double x);   
   Hyperbolic sine of x radians.
   Definition : sinH(x) = ( exp(x) - exp(-x) )/2
*/

#define  Num_CosH(x)     cosh(x)
/* double Num_CosH(double x);         
   Hyperbolic cosine of x radians.
   Definition : cosH(x) = ( exp(x) + exp(-x) )/2
*/

#define  Num_TanH(x)     tanh(x)
/* double Num_TanH(double x);         
   Hyperbolic tangent of x radians.
   Definition : tanH(x) = sinH(x) / cotH(x)
*/

#define  Num_CotH(x)   (1.0+2.0/(exp(2.0*(x))-1.0))
/* double Num_CotH(double x); 
   Hyperbolic cotangent of x radians.
   Definition : cotH(x) = cosH(x) / sinH(x)
   Condition  : x <> 0.0
*/

/*.hlptx skip 4*/
#ifdef __cplusplus
}
#endif
#endif /* ifndef  Num_Lib */
