/* MATLIB.H : Matrix handling fuctions

  Title   : MATLIB
  Version : 4.0
  Date    : Nov 23,1996
  Author  : J.R. Ferguson
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage   : Header file
*/
/*.hlptx skip 1*/
#ifndef Mat_Lib
#define Mat_Lib

#include <float.h>

#define    Mat_MAX    8         /* max number of rows/columns */
#define    Mat_EPS    1.0E-10   /* computational precision */

typedef    unsigned int  Mat_Ind;
typedef    double        Mat_ElmTyp;

typedef    struct Mat_Struct *Mat_Ptr;
typedef    struct Mat_Struct {
             Mat_Ind      nrow,ncol;
             Mat_ElmTyp   elm [Mat_MAX] [Mat_MAX];
           } Mat_Typ;

/* error codes */
#define    Mat_ERROK  0        /* OK */
#define    Mat_ERRDIM 1        /* dimension error */
#define    Mat_ERREXP 2        /* negative exponent */
#define    Mat_ERRMEM 3        /* out of memory */

extern unsigned Mat_ErrCod;    /* used internally to store the current 
                                  error code */
extern char    *Mat_ErrMsg[4]; /* array of error messages (max 20 chars), 
                                  to be indexed by the returnvalue of 
                                  function Mat_Error() below */

/*.hlptx skip 4*/
#ifdef __cplusplus
extern "C" {
#endif

unsigned Mat_Error(void);
/* Retrievs the current value of Mat_ErrCod,
   then sets Mat_ErrCod to Mat_ERROK.
*/

void Mat_Dim(Mat_Ptr A, Mat_Ind n, Mat_Ind m);
/* Sets dimensions for the matrix pointed to by A to n rows, m columns.
*/

void Mat_Insert(Mat_Ptr A, Mat_Ind row, Mat_Ind col, Mat_ElmTyp x);
/* Sets A->elm[row,col] to x.

   The row and col values are range checked against the current dimensions
   of matrix A. If the check fails, Mat_ErrCod will be set to Mat_ERRDIM,
   and matrix A is left unchanged.
*/

void Mat_Retrieve(Mat_Ptr A, Mat_Ind row, Mat_Ind col, Mat_ElmTyp *x);
/* Retrieves A->elm[row,col] into x.

   The row and col values are range checked against the current dimensions
   of matrix A. If the check fails, Mat_ErrCod will be set to Mat_ERRDIM,
   and variable x is left unchanged.
*/

void Mat_Copy(const Mat_Ptr source, Mat_Ptr dest);
/* Copies the source matrix to dest (both dimensions and elements).
*/

void Mat_Zero(Mat_Ptr A);
/* Sets all elements for the current dimensions of A to zero.
*/

void Mat_Unify(Mat_Ptr A);
/* Unifies the contents of the matrix pointed to by A for its current 
   dimensions.
   A must be a square matrix (A->Nrow == A->Ncol).
   The main diagonal elements (having equal indices for row and column)
   are set to 1. All other elements are set to zero.

   If A is not a square matrix, Mat_ErrCod will be set to Mat_ERRDIM, and
   matrix A is left unchanged.
*/

void Mat_Add(const Mat_Ptr A, const Mat_Ptr B, Mat_Ptr result);
/* Matrix addition: sets result = A + B

   If A and B do not have the same dimensions, Mat_ErrCod will be set to
   Mat_ERRDIM and the result matrix is left unchanged.
*/

void Mat_Subtract(const Mat_Ptr A, const Mat_Ptr B, Mat_Ptr result);
/* Matrix subtraction: sets result = A - B
   If A and B do not have the same dimensions, Mat_ErrCod will be set to
   Mat_ERRDIM and the result matrix is left unchanged.
*/

void Mat_ScalarProd(Mat_ElmTyp factor, const Mat_Ptr A, Mat_Ptr result);
/* Scalar product: result = factor . A
*/

void Mat_MatrixProd(const Mat_Ptr A, const Mat_Ptr B, Mat_Ptr result);
/* Matrix product: result = A * B
   A->Ncol must be equal to B->Nrow. The result dimensions will be
   result->Nrow = A->Nrow, result->Ncol = B->NCol.

   If A->Ncol != B->Nrow, Mat_ErrCod will be set to Mat_ERRDIM and 
   the result matrix is left unchanged.
*/

void Mat_Transpose(const Mat_Ptr A, Mat_Ptr result);
/* Constructs a result matrix that is the transpose of matrix A
   (rows and columns exchanged).
*/

void Mat_Power(const Mat_Ptr A, int exponent, Mat_Ptr result);
/* result = A * A * ... A  (exponent times)
   Constructs a result matrix with the same dimensions of square matrix A
   which holds the result of exponent matrix multiplications of A with
   itself. The exponent may not be negative.
   If the exponent is 0, the result matrix will be a unified matrix.
   If the exponent is 1, the result matrix will be the same of A.

   If A is not a square matrix, Mat_ErrCod will be set to Mat_ERRDIM, and
   the result matrix is left unchanged.
   If the exponent is negative, Mat_ErrCod will be set to Mat_ERREXP, and
   the result matrix is left unchanged.
*/

void Mat_Ssq(const Mat_Ptr A, Mat_ElmTyp *ssq);
/* Computes ssq to be the sum of squares of all elements of matrix A,
   which must be a one-row or a one-column matrix.

   If A is not a one-row or a one-column matrix, Mat_ErrCod is set
   to Mat_ERRDIM, and ssq is left unchanged.
*/

void Mat_GaussJordan(Mat_Ptr A, Mat_Ptr A1, Mat_ElmTyp *det);
/* Performs the Gauss Jordan elimination process on square matrix A,
   which sets A to its normalized state. Matrix A1 will be set to the
   inverse of A, and det to the determinant of A.

   If A is not a square matrix, Mat_ErrCod will be set to Mat_ERRDIM, and
   A1 and det are left unchanged.
*/

void Mat_Invert(Mat_Ptr A, Mat_Ptr result, int *singular);
/* Constructs the result matrix the inverse of square matrix A, if
   possible (if which case singular is set to 0). If A does not have
   an inverse, singular wil be set to 1 and the result matrix is left
   unchanged.
   Matrix A is not changed by this procedure.

   If A is not a square matrix, Mat_ErrCod will be set to Mat_ERRDIM, and
   result and singular are left unchanged.
*/

void Mat_Det(Mat_Ptr A, Mat_ElmTyp *det);
/* Sets det to be the determinant of square matrix pointed to by A.
   Matrix A is not changed by this procedure.

   If A is not a square matrix, Mat_ErrCod will be set to Mat_ERRDIM, and
   det is left unchanged.
*/

void Mat_Solve(Mat_Ptr A, const Mat_Ptr b, Mat_Ptr x, int *singular);
/* Solves a set of linear equations represented by the matrix
   formula A * b = x.
   If there is a solution, it will be represented by a one-column
   result matrix x and singular will be set to false.
   If there is no solution, singular is set to true, and matrix
   x is left unchanged.

   A must be a square matrix (A->Nrow == A->Ncol) and b must be a 
   one-column matrix (b->Ncol == 1) with b->Nrow == A->nrow.
   If one or more of these conditions is not met, Mat_ErrCod is set
   to Mat_ERRDIM, and the result matrix is left unchanged.
*/

/*.hlptx skip 4*/
#ifdef __cplusplus
}
#endif
#endif /* ifndef Mat_Lib */
