/* MATLIB.C : Matrix manipulation fuctions

  Title   : MATLIB
  Version : 4.0
  Date    : Nov 23,1996
  Author  : J.R. Ferguson
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Usage   : Function library
*/


#include <stdlib.h>
#include <mem.h>
#include <math.h>
#include "matlib.h"


unsigned Mat_ErrCod = Mat_ERROK;
     
char    *Mat_ErrMsg[4] =
         { "",
           "Dimension error",
           "Negative exponent",
           "Out of memory" };


unsigned Mat_Error(void)
{ unsigned errcod;

  errcod= Mat_ErrCod;
  Mat_ErrCod= Mat_ERROK;
  return errcod;
}


Mat_Ptr _Mat_Alloc(void)
{ Mat_Ptr p;

  if ( (p= (Mat_Ptr )malloc(sizeof(Mat_Typ)) ) == NULL) {
    Mat_ErrCod= Mat_ERRMEM;
    return NULL;
  }
  else return p;
}


void Mat_Dim(Mat_Ptr A, Mat_Ind n, Mat_Ind m)
{
  A->nrow= n;
  A->ncol= m;
}


void Mat_Insert(Mat_Ptr A, Mat_Ind row, Mat_Ind col, Mat_ElmTyp x)
{
  if ((row > A->nrow) || (col > A->ncol))
    Mat_ErrCod= Mat_ERRDIM;
  else
    A->elm[row-1][col-1]= x;
}


void Mat_Retrieve(Mat_Ptr A, Mat_Ind row, Mat_Ind col, Mat_ElmTyp *x)
{
  if ((row > A->nrow) || (col > A->ncol))
    Mat_ErrCod= Mat_ERRDIM;
  else
    *x= A->elm[row-1][col-1];
}


void Mat_Copy(const Mat_Ptr source, Mat_Ptr dest)
{ memmove((void *)dest, (void *)source, (size_t)sizeof(Mat_Typ)); }


void Mat_Zero(Mat_Ptr A)
{ Mat_Ind i,j;

  for (i= 0; i < A->nrow; ++i)
    for (j= 0; j < A->ncol; ++j)
      A->elm[i][j]= 0.0;
}


void Mat_Unify(Mat_Ptr A)
{ Mat_Ind i;

  if (A->nrow != A->ncol)
    Mat_ErrCod= Mat_ERRDIM;
  else {
    Mat_Zero(A);
    for (i= 0; i < A->nrow; ++i)
      A->elm[i][i]= 1.0;
} }


void Mat_Add(const Mat_Ptr A, const Mat_Ptr B, Mat_Ptr result)
{ Mat_Ind i,j;

  if ((A->nrow != B->nrow) || (A->ncol != B->ncol))
    Mat_ErrCod= Mat_ERRDIM;
  else {
    Mat_Dim(result, A->nrow, A->ncol);
    for (i=0; i < A->nrow; ++i) {
      for (j=0; j < A->ncol; ++j) {
        result->elm[i][j]= A->elm[i][j] + B->elm[i][j];
} } } }


void Mat_Subtract(const Mat_Ptr A, const Mat_Ptr B, Mat_Ptr result)
{ Mat_Ind i,j;

  if ((A->nrow != B->nrow) || (A->ncol != B->ncol))
    Mat_ErrCod= Mat_ERRDIM;
  else {
    Mat_Dim(result, A->nrow, A->ncol);
    for (i=0; i < A->nrow; ++i) {
      for (j=0; j < A->ncol; ++j) {
        result->elm[i][j]= A->elm[i][j] - B->elm[i][j];
} } } }


void Mat_ScalarProd(Mat_ElmTyp factor, const Mat_Ptr A, Mat_Ptr result)
{ Mat_Ind i,j;

  Mat_Dim(result, A->nrow, A->ncol);
  for (i=0; i < A->nrow; ++i) {
    for (j=0; j < A->ncol; ++j) {
      result->elm[i][j]= factor * A->elm[i][j];
} } }


void Mat_MatrixProd(const Mat_Ptr A, const Mat_Ptr B, Mat_Ptr result)
{ Mat_Ind i,j,k; Mat_Ptr tmp;

  if (A->ncol != B->nrow)
    Mat_ErrCod= Mat_ERRDIM;
  else {
    tmp= _Mat_Alloc(); Mat_Dim(tmp, A->nrow, B->ncol);
    Mat_Zero(tmp);
    for (i=0; i < A->nrow; ++i) {
      for (j=0; j < B->ncol; ++j) {
        for (k=0; k < A->ncol; ++k) {
          tmp->elm[i][j]= tmp->elm[i][j] + A->elm[i][k] * B->elm[k][j];
    } } }
    Mat_Copy(tmp,result);
    free(tmp);
} }


void Mat_Transpose(const Mat_Ptr A, Mat_Ptr result)
{ Mat_Ind i,j; Mat_Ptr tmp;

  tmp= _Mat_Alloc(); Mat_Dim(tmp, A->ncol, A->nrow);
  for (i= 0; i < A->nrow; ++i) {
    for (j= 0; j < A->ncol; ++j) {
      tmp->elm[j][i]= A->elm[i][j];
  } }
  Mat_Copy(tmp,result);
  free(tmp);
}


void Mat_Power(const Mat_Ptr A, int exponent, Mat_Ptr result)
{ int i; Mat_Ptr tmp;

  if      (A->nrow != A->ncol) Mat_ErrCod= Mat_ERRDIM;
  else if (exponent < 0)       Mat_ErrCod= Mat_ERREXP;
  else {
    tmp= _Mat_Alloc();
    Mat_Dim(tmp, A->nrow, A->ncol);
    Mat_Dim(result, A->nrow, A->ncol);
    Mat_Unify(result);
    for (i=0; i < exponent; ++i) {
      Mat_MatrixProd(result, A, tmp);
      Mat_Copy(tmp,result);
    }
    free(tmp);
} }


void Mat_Ssq(const Mat_Ptr A, Mat_ElmTyp *ssq)
{ Mat_Ptr AT, prod;

  AT= _Mat_Alloc(); prod= _Mat_Alloc();
  Mat_Transpose(A, AT);
  if      (A->ncol == 1) Mat_MatrixProd(AT, A , prod);
  else if (A->nrow == 1) Mat_MatrixProd(A , AT, prod);
  else Mat_ErrCod= Mat_ERRDIM;
  Mat_Retrieve(prod, 1, 1, ssq);
  free(AT); free(prod);
}


void Mat_GaussJordan(Mat_Ptr A, Mat_Ptr A1, Mat_ElmTyp *det)
{ Mat_Ind    i,r,k,m;
  Mat_ElmTyp e;

  if (A->nrow != A->ncol) Mat_ErrCod= Mat_ERRDIM;
  else {
    Mat_Dim(A1, A->nrow, A->ncol);
    Mat_Unify(A1);
    i=0;
    *det= 1.0;
    while ((i < A->ncol) && (fabs(*det) >= Mat_EPS)) {
      m= i;
      e= fabs(A->elm[i][i]);
      for (r= i+1; r < A->nrow; ++r)
        if (fabs(A->elm[r][i]) > e) {
          e= fabs(A->elm[r][i]);
          m= r;
        }
      if (m != i) {
        *det= -*det;
        for (k= i; k < A->ncol; ++k) {
          e           = A->elm[i][k];
          A->elm[i][k]= A->elm[m][k];
          A->elm[m][k]= e;
        }
        for (k= 0; k < A1->ncol; ++k) {
          e            = A1->elm[i][k];
          A1->elm[i][k]= A1->elm[m][k];
          A1->elm[m][k]= e;
      } }
      e= A->elm[i][i];
      if (fabs(e) < Mat_EPS) *det= 0.0;
      else {
        *det *= e;
        for (k= i; k < A->ncol; ++k) A ->elm[i][k] /= e;
        for (k= 0; k < A->ncol; ++k) A1->elm[i][k] /= e;
        for (r= 0; r < A->nrow; ++r) {
          if (r != i) {
            e= A->elm[r][i];
            for (k= i; k < A->ncol; ++k) A ->elm[r][k] -= e * A ->elm[i][k];
            for (k= 0; k < A->ncol; ++k) A1->elm[r][k] -= e * A1->elm[i][k];
      } } }
      ++i;
} } }


void Mat_Invert(Mat_Ptr A, Mat_Ptr result, int *singular)
{ Mat_ElmTyp det; Mat_Ptr A1;

  A1= _Mat_Alloc(); Mat_Copy(A,A1);
  Mat_GaussJordan(A1, result, &det);
  *singular= fabs(det) < Mat_EPS;
  free(A1);
}


void Mat_Det(Mat_Ptr A, Mat_ElmTyp *det)
{ Mat_Ptr A1, A2;

  A1= _Mat_Alloc(); A2= _Mat_Alloc(); Mat_Copy(A,A1);
  Mat_GaussJordan(A1, A2, det);
  free(A2); free(A1);
}

void Mat_Solve(Mat_Ptr A, const Mat_Ptr b, Mat_Ptr x, int *singular)
{ Mat_Ptr Ainv;

  Ainv= _Mat_Alloc();
  Mat_Invert(A, Ainv, singular);
  if ((Mat_ErrCod == Mat_ERROK) && (*singular == 0))
    Mat_MatrixProd(Ainv, b, x);
  free(Ainv);
}
