/* EVLLIB.C : Float expression evaluation

  Title   : EVLLIB
  Version : 4.0
  Date    : Nov 23,1996
  Language: Turbo C 2.0, Turbo C++ 3.1 for Windows
  Author  : J.R. Ferguson
  Usage   : Function library
*/

#include <string.h>
#include <math.h>
#include <stdio.h>
#include <ctype.h>
#include "evllib.h"


/* --- local definitions --- */


/* constants */

#define   evl_MAXF  10    /* number of functions */
#define   evl_FLEN   6    /* max. length function identifier */
#define   evl_MAXS  20    /* max. depth expression() stack */

#define   evl_SINIDN     0
#define   evl_COSIDN     1
#define   evl_TANIDN     2
#define   evl_COTIDN     3
#define   evl_ATNIDN     4
#define   evl_ABSIDN     5
#define   evl_EXPIDN     6
#define   evl_LOGIDN     7
#define   evl_SQTIDN     8
#define   evl_SQRIDN     9

#define   evl_ILLSYM    10
#define   evl_PLSSYM    11
#define   evl_MINSYM    12
#define   evl_MULSYM    13
#define   evl_DIVSYM    14
#define   evl_LPAREN    15
#define   evl_RPAREN    16
#define   evl_NUMBER    17
#define   evl_VARIDN    18
#define   evl_ENDSYM    19


/* variables */

static const char *evl_ftab [evl_SQRIDN + 1] =
 {"sin", "cos", "tan", "cot", "arctan", "abs", "exp", "ln", "sqrt", "sqr"};

static int          evl_sptr;
static float        evl_stack [evl_MAXS];

static size_t       i;
static size_t       l;
static char         c;
static int          sym;
static float        tmp;
static float        value;
static int          evl_error;
static const char  *evl_expr;
static float        evl_x;

/* --- local functions --- */

#define tenpower(n)   (float)(pow(10.0,n))

static void chgsgn(void);
static void push(void);
static void pop(void);
static void factor(void);
static void term(void);
static void argument(void);



#ifdef TEST
static void dbgln(void)  { if (Dbg_DebugMode) printf("\n"); }
static void dbgval(void) { if (Dbg_DebugMode) printf(" val=%f ",value); }
static void dbgchr(void) { if (Dbg_DebugMode) printf(" c='%c' i=%i ",c,i); }
static void dbgstk(void) { if (Dbg_DebugMode) printf(" sp=%i ",evl_sptr); }


static void dbgsym(void)
{
  if (Dbg_DebugMode) {
    printf("sym=");
    switch (sym) {
      case evl_ILLSYM: printf("ILLSYM"); break;
      case evl_PLSSYM: printf("PLSSYM"); break;
      case evl_MINSYM: printf("MINSYM"); break;
      case evl_MULSYM: printf("MULSYM"); break;
      case evl_DIVSYM: printf("DIVSYM"); break;
      case evl_LPAREN: printf("LPAREN"); break;
      case evl_RPAREN: printf("RPAREN"); break;
      case evl_SINIDN: printf("SINIDN"); break;
      case evl_COSIDN: printf("COSIDN"); break;
      case evl_TANIDN: printf("TANIDN"); break;
      case evl_COTIDN: printf("COTIDN"); break;
      case evl_ATNIDN: printf("ATNIDN"); break;
      case evl_ABSIDN: printf("ABSIDN"); break;
      case evl_EXPIDN: printf("EXPIDN"); break;
      case evl_LOGIDN: printf("LOGIDN"); break;
      case evl_SQTIDN: printf("SQTIDN"); break;
      case evl_SQRIDN: printf("SQRIDN"); break;
      case evl_NUMBER: printf("NUMBER"); break;
      case evl_VARIDN: printf("VARIDN"); break;
      case evl_ENDSYM: printf("ENDSYM"); break;
    }
  }
}


static void dbgall(void)
{
  dbgsym();
  dbgchr();
  dbgstk();
  dbgval();
}
#endif /* TEST */


static void errmsg(const char *msg)
{ int n;
  if (evl_error == 0) {
    printf("\n%s\n",evl_expr);
    for (n= i; n > 0; n--) putchar(' '); printf("^\n");
    printf("EVAL: %s\n",msg);
#ifdef TEST
    dbgall(); dbgln();
#endif
    evl_error= 1;
  }
}


static void need_lparen(void)
{
#ifdef TEST
  Dbg_Trace("need_lparen");
#endif
  if (sym != evl_LPAREN) errmsg("missing '('");
}


static void need_rparen(void)
{
#ifdef TEST
  Dbg_Trace("need_rparen");
#endif
  if (sym != evl_RPAREN) errmsg("missing ')'");
}


static void getchr(void)
{
#ifdef TEST
  Dbg_Trace("getchr");
#endif
  c= tolower( (i == l) ? '\0' : evl_expr[i++] );
#ifdef TEST
  dbgchr();
#endif
}


static int compare(int t)
/* return 0 if matched, 1 if not */
{ char    c0;
  size_t  i0;
  int     f,fl;

#ifdef TEST
  Dbg_Trace("compare");
#endif
  c0= c; i0= i; f= 0; fl= strlen(evl_ftab[t]) - 1;
  while ((c == evl_ftab[t][f]) && (f < fl)) { f++; getchr(); }
  if (c != evl_ftab[t][f]) {
    i= i0; c= c0; return 1;
  }
  else return 0;
}


static void need_fidn(void)
{ int nomatch;

#ifdef TEST
  Dbg_Trace("need_fidn");
#endif
  sym= evl_SINIDN;
  nomatch= compare(sym);
  while (nomatch && (sym < evl_SQRIDN)) nomatch= compare(++sym);
  if (nomatch) sym= evl_ILLSYM;
}


/* int digval(char c) */
#define  digval(c)  ((c) - '0')


static void exponent(void)
{
  tmp= value; pop(); value *= tenpower((int)floor(tmp));
#ifdef TEST
  Dbg_Trace("exponent"); dbgval();
#endif
}


static void unsint(void)
{
  value= 0.0;
  while (isdigit(c)) { value= 10.0*value + (float)digval(c); getchr(); }
#ifdef TEST
  Dbg_Trace("unsint"); dbgval();
#endif
}


static void fracprt(void)
{ float w;

  w= 1.0;
  while (isdigit(c)) {
    w/= 10; value+= w * (float)digval(c); getchr();
  }
#ifdef TEST
  Dbg_Trace("fracprt"); dbgval();
#endif
}


static void fpnum(void)
{
  unsint(); if (c == '.') { getchr(); fracprt(); }
#ifdef TEST
  Dbg_Trace("fpnum"); dbgval();
#endif
}


static void expprt(void)
{
  if (c == '-') {
    getchr(); unsint(); chgsgn();
  }
  else {
    if (c == '+') getchr();
    unsint();
  }
  exponent();
#ifdef TEST
  Dbg_Trace("expprt"); dbgval();
#endif
}


static void unsnum(void)
{
  fpnum();
  if (c == 'e') { push(); getchr(); expprt(); }
#ifdef TEST
  Dbg_Trace("unsnum"); dbgval();
#endif
}


static void getsym(void)
{
  if (evl_error == 0) {
#ifdef TEST
    Dbg_Trace("getsym");
#endif
    while (c == ' ') getchr();
    if (isdigit(c) || (c == '.')) { unsnum(); sym= evl_NUMBER; }
    else if (c == '\0') sym= evl_ENDSYM;
    else {
      switch (c) {
        case '+' : sym= evl_PLSSYM; break;
        case '-' : sym= evl_MINSYM; break;
        case '*' : sym= evl_MULSYM; break;
        case '/' : sym= evl_DIVSYM; break;
        case '(' : sym= evl_LPAREN; break;
        case ')' : sym= evl_RPAREN; break;
        case 'x' : sym= evl_VARIDN; break;
        default  : need_fidn();
      }
#ifdef TEST
      dbgall();
#endif
      getchr();
    }
  }
}


static void push(void)
{
#ifdef TEST
  Dbg_Trace("push"); dbgval();
#endif
  if (evl_sptr < evl_MAXS) evl_stack[evl_sptr++]= value;
  else errmsg("stack full");
}


static void pop(void)
{
#ifdef TEST
  Dbg_Trace("pop");
#endif
  if (evl_sptr > 0) {
    value= evl_stack[--evl_sptr];
#ifdef TEST
    dbgval();
#endif
  }
  else errmsg("stack empty");
}


static void chgsgn(void)
{
  value= -value;
#ifdef TEST
  Dbg_Trace("chgsgn"); dbgval();
#endif
}


static void add(void)
{
  tmp= value; pop(); value+= tmp;
#ifdef TEST
  Dbg_Trace("add"); dbgval();
#endif
}


static void subtract(void)
{
  chgsgn(); add();
#ifdef TEST
  Dbg_Trace("subtract"); dbgval();
#endif
}


static void mult(void)
{
  tmp= value; pop(); value*= tmp;
#ifdef TEST
  Dbg_Trace("mult"); dbgval();
#endif
}


static void divide(void)
{
  tmp= value; pop(); value/= tmp;
#ifdef TEST
  Dbg_Trace("divide"); dbgval();
#endif
}


static void expression(void)
{
#ifdef TEST
  Dbg_Trace("expression");
#endif
  term();
  while ((sym == evl_PLSSYM) || (sym == evl_MINSYM)) {
    if (sym == evl_PLSSYM) { push(); getsym(); term(); add(); }
    else                   { push(); getsym(); term(); subtract(); }
  }
}


static void term(void)
{
#ifdef TEST
  Dbg_Trace("term");
#endif
  factor();
  while ((sym == evl_MULSYM) || (sym == evl_DIVSYM)) {
    if (sym == evl_MULSYM) { push(); getsym(); factor(); mult(); }
    else                   { push(); getsym(); factor(); divide(); }
  }
}


static void unsfac(void)
{
#ifdef TEST
  Dbg_Trace("unsfac");
#endif
  switch (sym) {
    case evl_NUMBER: /*nothing*/                            break;
    case evl_VARIDN: value= evl_x;                          break;
    case evl_SINIDN: argument(); value= sin(value);         break;
    case evl_COSIDN: argument(); value= cos(value);         break;
    case evl_TANIDN: argument(); value= tan(value);         break;
    case evl_COTIDN: argument(); value= 1.0/tan(value);     break;
    case evl_ATNIDN: argument(); value= atan(value);        break;
    case evl_ABSIDN: argument(); value= fabs(value);        break;
    case evl_EXPIDN: argument(); value= exp(value);         break;
    case evl_LOGIDN: argument(); value= log(value);         break;
    case evl_SQTIDN: argument(); value= sqrt(value);        break;
    case evl_SQRIDN: argument(); value= value * value;      break;
    case evl_LPAREN: getsym(); expression(); need_rparen(); break;
    case evl_ENDSYM: errmsg("syntax error");                break;
    default:         errmsg("illegal character");
  }
  getsym();
}


static void factor(void)
{
#ifdef TEST
  Dbg_Trace("factor");
#endif
  switch (sym) {
    case evl_PLSSYM: getsym(); unsfac();           break;
    case evl_MINSYM: getsym(); unsfac(); chgsgn(); break;
    default        : unsfac();
  }
}


static void argument(void)
{
#ifdef TEST
  Dbg_Trace("argument");
#endif
  getsym(); need_lparen(); getsym(); expression(); need_rparen();
}



/* ======================= GLOBAL FUNCTIONS ============================ */


float eval(const char *expr, float x, int *error)
{
  evl_sptr= 0; i= 0; l= strlen(expr); c= ' ';
  evl_error= 0; evl_expr= expr; evl_x= x;
  getsym(); expression();
#ifdef TEST
  dbgln(); dbgall(); dbgln();
#endif
  if (sym != evl_ENDSYM) errmsg("syntax error");
  *error= evl_error;
  return value;
}
